/**********************************************************************
 * Copyright (c) 2003, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: XMLLoader.java,v 1.2 2008/01/24 02:28:17 apnan Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.util;

import java.io.BufferedOutputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.loaders.hierarchy.Constants;
import org.eclipse.hyades.loaders.hierarchy.IgnoredXMLFragmentLoader;
import org.eclipse.hyades.models.hierarchy.CorrelationSourceInfo;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCCollectionMode;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.UnresolvedCorrelation;
import org.eclipse.hyades.models.hierarchy.util.HierarchyResourceSetImpl;
import org.eclipse.hyades.models.hierarchy.util.MonitoredInputStream;
import org.eclipse.hyades.models.hierarchy.util.PerfUtil;
import org.eclipse.hyades.models.hierarchy.util.SaveUtil;
import org.eclipse.hyades.models.hierarchy.util.StringUtil;
import org.eclipse.hyades.models.hierarchy.util.internal.CrimsonFragmentHandler;
import org.eclipse.hyades.models.hierarchy.util.internal.IExtendedLoader;
import org.eclipse.hyades.models.hierarchy.util.internal.InvalidEventException;
import org.eclipse.hyades.models.hierarchy.util.internal.SimpleFragmentHandler;
import org.eclipse.hyades.models.util.ModelDebugger;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 * @author slavescu
 */
public class XMLLoader implements IExtendedLoader {
	//~ Instance fields
	// ----------------------------------------------------------------------------
	protected BufferedOutputStream file;

	protected HierarchyContext context;

	protected InvalidXMLException error;

	protected Map loaders = new HashMap();

	protected String currentAttributeName;

	protected String currentElementName;

	protected XMLFragmentHandler scanner;

	protected XMLFragmentLoader currentLoader;

	protected XMLFragmentLoader defaultLoader = new IgnoredXMLFragmentLoader();

	protected boolean startDocument;

	protected int depth = 0;

	protected int fragmentsCount;

	protected PerfUtil p;

	protected boolean debug;

	protected static final char[] LE_IN = new char[] { 'l', 'E', '>' };

	protected static final char[] LE_OUT = new char[] { 'l', 'E', '<' };

	protected StringBuffer sb = new StringBuffer();

	protected String rootElement;

	//~ Constructors
	// -------------------------------------------------------------------------------
	public XMLLoader(TRCAgentProxy agentProxy) {
		makeScanner();
		TRCAgent agent = agentProxy.getAgent();
		if ((agent != null) && (agent.eResource() != null)) {
			context = LoadersUtils.locateHierarchyContext(agent);
		}
		if (context == null) {
			context = new HierarchyContext();
			if (agentProxy != null) {
				context.setMonitor(agentProxy.getProcessProxy().getNode().getMonitor());
				context.setNode(agentProxy.getProcessProxy().getNode());
				context.setProcessProxy(agentProxy.getProcessProxy());
				context.setAgentProxy(agentProxy);
			}
			LookupServiceExtensions.getInstance().register(null, context);
		} else {
			if (context.getAgentProxy() == null) {
				context.setAgentProxy(agentProxy);
			}
			if (context.getProcessProxy() == null) {
				context.setProcessProxy(agentProxy.getProcessProxy());
			}
			if (context.getMonitor() == null) {
				context.setMonitor(agentProxy.getProcessProxy().getNode().getMonitor());
			}
			if (context.getNode() == null) {
				context.setNode(agentProxy.getProcessProxy().getNode());
			}
		}
		defaultLoader = getLoader("DefaultXMLFragmentLoader");
	}

	public XMLLoader(TRCAgent agent) {
		super();
		makeScanner();
		if ((agent != null)) {
			context = LoadersUtils.locateHierarchyContext(agent);
		}
		if (context == null) {
			context = new HierarchyContext();
			context.setMonitor(agent.getAgentProxy().getProcessProxy().getNode().getMonitor());
			context.setNode(agent.getAgentProxy().getProcessProxy().getNode());
			context.setProcessProxy(agent.getAgentProxy().getProcessProxy());
			context.setAgentProxy(agent.getAgentProxy());
			context.setAgent(agent);
			if (agent.eResource() == null) {
				try {
					addToResource(agent, context);
				} catch (Exception e) {
					// TODO MS - add proper handling
				}
			}
			LookupServiceExtensions.getInstance().register(null, context);
		} else {
			if (context.getAgentProxy() == null) {
				context.setAgentProxy(agent.getAgentProxy());
				if (context.getProcessProxy() == null) {
					context.setProcessProxy(agent.getAgentProxy().getProcessProxy());
				}
				if (context.getMonitor() == null) {
					context.setMonitor(agent.getAgentProxy().getProcessProxy().getNode().getMonitor());
				}
				if (context.getNode() == null) {
					context.setNode(agent.getAgentProxy().getProcessProxy().getNode());
				}
			}
		}
		defaultLoader = getLoader("DefaultXMLFragmentLoader");
	}

	public XMLLoader(TRCMonitor monitor) {
		super();
		//        if ((monitor != null) && (monitor.eResource() != null)) {
		//            context = (HierarchyContext)
		// LookupServiceExtensions.getInstance().locate(null,
		// HierarchyContext.class, monitor.eResource().getURI().toString());
		//        }
		//
		if (context == null) {
			context = new HierarchyContext();
			context.setMonitor(monitor);
			//            LookupServiceExtensions.getInstance().register(null, context);
		}
		makeScanner();
		defaultLoader = getLoader("DefaultXMLFragmentLoader");
	}

	//~ Methods
	// ------------------------------------------------------------------------------------
	public void setCollectionMode(int collectionMode) {
		context.setCollectionMode(TRCCollectionMode.get(collectionMode));
	}

	public HierarchyContext getContext() {
		return context;
	}

	public int getProcessedFragments() {
		return fragmentsCount;
	}

	public void attributeName(String name) {
		if (isIgnoredElement()) {
			return;
		}
		currentAttributeName = name;
	}

	public void attributeValueCharacters(String attributeValue) {
		if (isIgnoredElement()) {
			return;
		}
		try {
			currentLoader.addAttribute(currentAttributeName, attributeValue);
		} catch (Exception e) {
			log(e);
		}
	}

	public void characters(char[] ch, int start, int length) {
		if (isIgnoredElement()) {
			String s = new String(ch,start,length).trim();
			if(s.length()==0)
				return;
				
			if(currentElementName!=null)
			{
				if(getContext().getAgent()==null)
				{
					LoadersUtils.createAgent(getContext());
				}
				//TODO check if this is still required
//				CBECommonBaseEvent commonBaseEvent = CBEFactory.eINSTANCE.createCBECommonBaseEvent();
//				commonBaseEvent.setMsg(s);
//				commonBaseEvent.setExtensionName(currentElementName);
//				commonBaseEvent.setCreationTime(new Date().getTime() * 1000);
//				getContext().getAgent().getDefaultEvents().add(commonBaseEvent);
			}
			return;
		}
		try {
			if (currentLoader != null) {
				currentLoader.addCharacters(ch, start, length);
			}
		} catch (Exception e) {
			log(e);
		}
	}

	public synchronized void cleanUp() {
		if (scanner == null) {
			return;
		}
		if (ModelDebugger.INSTANCE.debug) {
			System.out.println("XMLLoader.cleanUp()");
		}
		printLoadInfo("cleanup()");
		try {
			scanner.terminateParser();
		} catch (Exception e) {
			// TODO MS - add proper handling
			log(e);
		}
		scanner = null;
		if (file != null) {
			try {
				file.close();
			} catch (IOException e) {
				log(e);
			}
		}
		file = null;
		error = null;
		depth = 0;
		startDocument = true;
		fragmentsCount = 0;
		List processed = new ArrayList();
		for (Iterator iter = loaders.entrySet().iterator(); iter.hasNext();) {
			Map.Entry entry = (Map.Entry) iter.next();
			if (!processed.contains(entry.getValue())) {
				((XMLFragmentLoader) entry.getValue()).cleanUp();
				processed.add(entry.getValue());
			}
		}
		loaders.clear();
		if (getContext().getGlobalForwardReferences().size() > 0) {
			processGlobalForwardReferences();
		}
		
	}

	/**
	 *  
	 */
	protected void printLoadInfo(String prefix) {
		String contextInfo = "contextInfo=";
		if (context != null)
			contextInfo += context.getContextURI();
		p.stopAndPrintStatus(prefix + ", fragmentsCount=" + fragmentsCount + ", " + contextInfo);
		p.setMessage("XMLLoader loadEvents +++");
	}

	public void endDocument(Object object, int i) {
		//        cleanUp();
		printLoadInfo("endDocument()");
	}

	public void endElement(String elementName, int currentOffset) {
		currentElementName = elementName;
		try {
			depth--;
			if (depth == 0) {
				currentLoader.addYourselfInContext();
				if ((getContext().getAgent() != null) && (getContext().getAgent().eResource() != null) && !getContext().getAgent().eResource().isModified()) {
					getContext().getAgent().eResource().setModified(true);
				}
				currentElementName = rootElement;
			} else if (depth > 0)
				currentLoader.endChild(currentElementName);
				
		} catch (Exception e) {
			log(e, "fragmentsCount=" + fragmentsCount);
			error = new InvalidXMLException(e);
			throw error;
		}
	}

	public void error(InvalidXMLException exception) {
		error = exception;
		error.fillInStackTrace();
		//        if(exception.getEnclosedException()!=null)
		//        	log(exception.getEnclosedException());
		throw error;
	}

	public void loadEvent(byte[] buffer, int offset, int length, boolean loadToModel, boolean toProfileFile) throws InvalidXMLException {
		checkForStopProcessing();
		if (toProfileFile) {
			if (ModelDebugger.INSTANCE.debugEventsToFile)
				writeToBinaryOutputFile(buffer, 0, length);
		} else {
			loadEvent(buffer, offset, length, loadToModel);
		}
	}

	public void loadEvent(byte[] buffer, int offset, int length, boolean loadToModel) throws InvalidXMLException {
		checkForStopProcessing();
//		if ((getContext().getAgent() != null) && (getContext().getAgent().eResource() != null) && !getContext().getAgent().eResource().isModified()) {
//			getContext().getAgent().eResource().setModified(true);
//		}
		setLoadToModel(loadToModel);
		try {
			//			LoadersUtils.log("loadEvent1(length)="+length);
			//LoadersUtils.log("loadEvent(byte[], int, boolean)======\n" +
			// LoadersUtils.makeString(buffer,offset,length));
			//LoadersUtils.log("loadEvent(byte[], int, boolean)******");
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(LE_IN);
				sb.setLength(0);
				sb.append(fragmentsCount);
				System.out.println(LoadersUtils.getChars(sb));
			}
			//			System.out.println(LoadersUtils.makeString(buffer,offset,length));
			if (ModelDebugger.INSTANCE.debugEventsToFile) {
				//				if (!startDocument && context!=null &&
				// context.getAgentProxy() !=null) {
				//					context.getAgentProxy().setProfileFile("c:/log_" + (new
				// java.util.Date()).getTime() + ".xml");
				//				}
				//				System.out.println(LoadersUtils.makeString(buffer,offset,length));
				writeToBinaryOutputFile(buffer, offset, length);
			} else {
				//				System.out.write(buffer,offset,length);
				//				System.out.print("\n");
				scanner.scanContent(buffer, offset, length);
				//				writeToBinaryOutputFile(buffer, offset, length);
			}
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.println(LE_OUT);
			}
		} catch (Exception e) {
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(LE_OUT);
				System.out.println(" = " + e.getLocalizedMessage());
			}
			if (error == null) {
				log(e);
			}
		}
		if (error != null) {
			throw error;
		}
	}

	protected void checkForStopProcessing() {
		//		LoadersUtils.log("loadEvent(length)="+length+" error="+error+"
		// scanner="+scanner);
		if (error != null) {
			ModelDebugger.log(error);
			throw error;
		}
	}

	public void loadEvent(byte[] buffer, int offset, int length) throws InvalidXMLException {
		this.loadEvent(buffer, offset, length, true);
	}

	public void restartParser() {
		if (scanner != null) {
			try {
				scanner.terminateParser();
			} catch (Exception e) {
				// TODO MS - add proper handling
				log(e);
			}
		}
		error = null;
		startDocument = true;
		depth = 0;
		makeScanner();
	}

	public void startDocument() {
		startDocument = true;
	}

	public void startElement(String elementName, boolean hasAttributes, boolean isEmpty) {
		currentElementName = elementName;
		if (isIgnoredElement()) {
			return;
		}
		if (depth == 0) {
			currentLoader = getLoader(getName(currentElementName));
			currentLoader.initialize(getContext(), currentElementName);
			fragmentsCount++;
		} else {
			try {
				currentLoader.startChild(currentElementName);
			} catch (Exception e) {
				log(e);
				error = new InvalidXMLException(e);
				throw error;
			}
		}
		depth++;
	}

	protected boolean isIgnoredElement() {
		if (startDocument) {
				
			if (currentElementName == rootElement) {
				return true;
			} else if(rootElement!=null)
				return false;
				
			rootElement = currentElementName;
			if (currentElementName.equals("TRACE") || currentElementName.equals("CommonBaseEvents") || currentElementName.equals("Statistic")) {
				return true;
			} else {
				startDocument = false;
			}
		} else if(depth==0 && currentElementName==rootElement)
			return true;
		return false;
	}

	protected XMLFragmentLoader getLoader(String currentElementName) {
		XMLFragmentLoader loader = (XMLFragmentLoader) loaders.get(currentElementName);
		if (loader == null) {
			loader = (XMLFragmentLoader) LoaderExtensions.getInstance().get(currentElementName);
			if (loader != null) {
				try {
					loader = (XMLFragmentLoader) loader.getClass().newInstance();
					loaders.put(currentElementName, loader);
					return loader;
				} catch (InstantiationException e) {
					// TODO MS - add proper handling
				} catch (IllegalAccessException e) {
					//					 TODO MS - add proper handling
				}
			}
			loader = defaultLoader;
		}
		return loader;
	}

	protected String getName(String currentElementName) {
		int column = currentElementName.lastIndexOf(":");
		if (column != -1) {
			return currentElementName.substring(column);
		}
		return currentElementName;
	}

	protected void log(Exception e) {
		log(e, null);
	}

	protected void log(Exception e, String msg) {
		if (e instanceof SAXParseException) {
			SAXParseException spx = (SAXParseException) e;
			System.err.println(spx.getLocalizedMessage());
			System.err.println("at line: " + spx.getLineNumber());
			System.err.println("at column: " + spx.getColumnNumber());
			spx.fillInStackTrace();
			ModelDebugger.log(spx, msg);
		} else {
			ModelDebugger.log(e, msg);
		}
	}

	protected void makeScanner() {
		p = PerfUtil.createInstance("XMLLoader loadEvents", true);
		try {
			String s = System.getProperties().getProperty("XMLLoader.debug"); // use
			debug = Boolean.valueOf(s).booleanValue();
			if (!debug)
				debug = ModelDebugger.INSTANCE.debug;
		} catch (Exception e) {
			//ignore and leave the default false
		}
		scanner = (XMLFragmentHandler) XMLFragmentHandlerExtensions.getInstance().get(XMLFragmentHandlerRegistryReader.TAG_HANDLER);
		if (scanner != null) {
			try {
				scanner = (XMLFragmentHandler) scanner.getClass().newInstance();
				scanner.setXMLLoader(this);
			} catch (Exception e) {
				log(e);
				scanner = new SAXFragmentHandler();
				scanner.setXMLLoader(this);
			}
		} else {
			if (ModelDebugger.INSTANCE.debugUseOptimizedScanner) {
				scanner = new SimpleFragmentHandler();
			} else {
				SAXParser p = null;
				try {
					p = makeParser();
				} catch (Exception e) {
					// TODO: handle exception
				}
				if (p != null && p.getClass().getName().equals("org.apache.crimson.jaxp.SAXParserImpl")) {
					//fixes the problems with Crimson parser buffering
					scanner = new CrimsonFragmentHandler();
				} else
					scanner = new SAXFragmentHandler();
			}
			scanner.setXMLLoader(this);
		}
	}

	public static SAXParser makeParser() throws ParserConfigurationException, SAXException {
		//LoadersUtils.log("makeParser()");
		SAXParserFactory factory = SAXParserFactory.newInstance();
		//      depth = -1;
		//      rootTagRequired = false;
		//      SAXParserFactoryImpl factory = new SAXParserFactoryImpl();
		//      factory.setNamespaceAware(true);
		factory.setFeature("http://xml.org/sax/features/string-interning", true); //$NON-NLS-1$ 
		factory.setValidating(false);
		return factory.newSAXParser();
	}

	protected void processGlobalForwardReferences() {
		for (Iterator iter = context.getAgent().getUnresolvedCorrelations().iterator(); iter.hasNext();) {
			UnresolvedCorrelation unresolvedCorrelation = (UnresolvedCorrelation) iter.next();
			for (Iterator iterator = unresolvedCorrelation.getSourceInfos().iterator(); iterator.hasNext();) {
				CorrelationSourceInfo correlationSourceInfo = (CorrelationSourceInfo) iterator.next();
				EObject owner = correlationSourceInfo.getOwner();
				EReference reference = correlationSourceInfo.getReference();
				EObject o = (EObject) LookupServiceExtensions.getInstance().locate(null, reference.getEReferenceType().getInstanceClass(), LoadersUtils.getLookUpKey(unresolvedCorrelation.getContextId()));
				if (o != null) {
					if (reference.isMany()) {
						((EList) owner.eGet(reference)).add(o);
					} else {
						owner.eSet(reference, o);
					}
					iterator.remove();
				}
			}
			if (unresolvedCorrelation.getSourceInfos().size() == 0) {
				iter.remove();
			}
		}
	}

	protected void setLoadToModel(boolean loadToModel) {
		if (context.isLoadToModel() != loadToModel) {
			context.setLoadToModel(loadToModel);
		}
	}

	protected void addToResource(TRCAgent agent, HierarchyContext context) {
		String monitorFolder = LoadersUtils.getMonitorFolder(context.getMonitor());
		Resource agDoc = null;
		if (agent.getName() == null) {
			agent.setName(Constants.UNKNOWN);
		}
		String aName = StringUtil.change(context.getMonitor().getName().trim(), " ", "") + "_" + context.getNode().getName().trim() + "_" + context.getProcessProxy().getPid() + "_" + context.getProcessProxy().getRuntimeId() + "_" + StringUtil.change(agent.getName().trim(), " ", "");
		String pFileName = monitorFolder + aName;
		agDoc = Resource.Factory.Registry.INSTANCE.getFactory(SaveUtil.createURI(pFileName + ".trcaxmi")).createResource(SaveUtil.createURI(pFileName + ".trcaxmi"));
		agDoc.setModified(true);
		HierarchyResourceSetImpl.getInstance().getResources().add(agDoc);
		//        SaveUtil.addDocument(agDoc);
		if (agDoc != null) {
			agDoc.getContents().add(agent);
		}
	}

	protected void writeToTextOutputFile(String xml) {
		try {
			if (createOutputFile()) {
				file.write(xml.getBytes(), 0, xml.getBytes().length);
				file.write('\n');
				file.flush();
			}
		} catch (IOException e) {
			error = new InvalidXMLException(e);
			throw error;
		}
	}

	protected void writeToBinaryOutputFile(byte[] buffer, int offset, int length) {
		try {
			if (createOutputFile()) {
				file.write(buffer, offset, length);
				file.write('\n');
				file.flush();
			}
		} catch (IOException e) {
			error = new InvalidXMLException(e);
			ModelDebugger.log(e);
			throw error;
		}
	}

	/**
	 * @throws FileNotFoundException
	 */
	protected boolean createOutputFile() {
		try {
			if (file == null) {
				String fileName = "/XMLLoader_toFile_" + (new java.util.Date()).getTime() + ".out";
				if (context != null && context.getAgentProxy() != null) {
					context.getAgentProxy().setProfileFile(fileName);
				}
				file = new BufferedOutputStream(new FileOutputStream(fileName));
			}
		} catch (Exception e) {
			return false;
		}
		return true;
	}

	public void loadEvent(byte[] buffer, int length, boolean loadToModel, boolean toProfileFile) throws InvalidXMLException {
		loadEvent(buffer, 0, length, loadToModel, toProfileFile);
	}

	public void loadEvent(byte[] buffer, int length, boolean loadToModel) throws InvalidXMLException {
		loadEvent(buffer, 0, length, loadToModel);
	}

	public void loadEvent(byte[] buffer, int length) throws InvalidXMLException {
		loadEvent(buffer, 0, length);
	}

	public void attributeName(int nameId) {
		// TODO Auto-generated method stub

	}

	public void attributeValueCharacters(char[] attributeValue) {
		// TODO Auto-generated method stub

	}

	public void endElement(int elementNameId, int currentOffset) {
		// TODO Auto-generated method stub

	}

	public void error(InvalidEventException exception) {
		// TODO Auto-generated method stub

	}

	public void startElement(int elementNameId, boolean hasAttributes, boolean isEmpty) {
		//		currentElementId = elementNameId;
		//		if (depth == 0) {
		//			currentLoader = getLoader(getName(currentElementName));
		//			currentLoader.initialize(getContext(), currentElementName);
		//			fragmentsCount++;
		//		} else {
		//			try {
		//				currentLoader.startChild(currentElementName);
		//			} catch (Exception e) {
		//				log(e);
		//				error = new InvalidXMLException(e);
		//				throw error;
		//			}
		//		}
		//		depth++;
	}

	public void loadEvents(InputStream inputStream, long offset, long length) throws InvalidEventException {
		if (scanner != null) {
			if (!(inputStream instanceof MonitoredInputStream) && !(offset == 0 && length == -1)) {
				final long startPos = (long) offset;
				final long endPos = (long) offset + length;

				if (startPos > 0)
					getContext().setLoadToModel(false);
				inputStream = new MonitoredInputStream(inputStream) {
					int state = 0;

					long totalBytesRead;

					protected int afterRead(int readBytes) {
						if (state == 2)
							return -1;
						totalBytesRead += readBytes;
						return readBytes;
					}

					protected int beforeRead(int maxBytesToRead) {
						if (state == 2)
							return 0;
						switch (state) {
						case 0:
							if (totalBytesRead == startPos) {
								getContext().setLoadToModel(true);
								state++;
							} else if (totalBytesRead + maxBytesToRead > startPos) {
								maxBytesToRead = (int) (startPos - totalBytesRead);
							}
							break;
						case 1:
							if (totalBytesRead == endPos) {
								getContext().setLoadToModel(false);
								maxBytesToRead = 0;
								state++;
							} else if (totalBytesRead + maxBytesToRead > endPos) {
								maxBytesToRead = (int) (endPos - totalBytesRead);
							}
							break;
						default:
							break;
						}
						return maxBytesToRead;
					}

					protected boolean isCanceled() {
						if (state == 2)
							return true;
						return false;
					}

				};
			}

			scanner.scanContent(inputStream, offset, length);
		}
	}

	public String toString() {
		return super.toString() + ", fragmentsProcessed=" + fragmentsCount;
	}
}