/**********************************************************************
 * Copyright (c) 2003, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EMFUtil.java,v 1.3 2009/04/27 15:21:37 paules Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.models.hierarchy.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.EMap;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.InternalEObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreEList;
import org.eclipse.emf.ecore.util.EcoreEMap;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.util.InternalEList;
import org.eclipse.emf.ecore.util.EcoreUtil.EqualityHelper;

public class EMFUtil {
	// ~ Methods
	// ------------------------------------------------------------------------------------

	/**
	 * Modified version of EMF EqualityHelper which ignores the container (the
	 * compared objects could be in separate resource or not contained at all)
	 */
	public static class EqualityHelperIgnoreContainer extends EqualityHelper {
		protected static final long serialVersionUID = -2447028134489769357L;

		protected boolean haveEqualFeature(EObject eObject1, EObject eObject2, EStructuralFeature feature) {
			if (feature instanceof EReference && ((EReference) feature).isContainer())
				// ignore the container feature, allows to check objects that
				// are not contained yet
				return true;
			return super.haveEqualFeature(eObject1, eObject2, feature);
		}

	}

	public static class ReusableValuesAdapter extends AdapterImpl {
		Map valuesByFeatureMap = new HashMap();

		public Map getValuesByFeatureMap() {
			return valuesByFeatureMap;
		}

	}

	public static Set delete(EObject object) {
		if (object == null || object.eResource() == null) {
			return Collections.EMPTY_SET;
		}

		return delete(object, object.eResource().getContents());
	}

	public static Set delete(final EObject object, Collection rootObjects) {
		if (object == null || object.eResource() == null) {
			return Collections.EMPTY_SET;
		}
		final boolean resolve = false;
		final ContainmentTraverser containmentTraverser = new ContainmentTraverser(rootObjects, resolve);
		final Set changedResources = delete(object, resolve, containmentTraverser);

		return changedResources;
	}

	public static Set delete(final EObject object, final boolean resolve, final ContainmentTraverser containmentTraverser) {

		final Set changedResources = new HashSet();
		final List removeList = new FastList();
		final List deleteList = new FastList();

		EObjectVisitor objectVisitor = new EObjectVisitor() {
			public boolean beforeChildren(EObject element) {
				return true;
			}

			public boolean afterChildren(EObject element) {
				if (isContainedIn(element, object)) {
					deleteList.add(element);
				} else {
					processObjectReferences(object, changedResources, resolve, true, removeList, containmentTraverser, element);
				}
				return true;
			}

		};

		containmentTraverser.registerVisitors(objectVisitor);

		containmentTraverser.traverse();
		if (object.eResource() != null)
			changedResources.add(object.eResource());

		deleteList.add(object);
		for (int i = deleteList.size() - 1; i >= 0; i--) {
			unsetAllFeatures((EObject) deleteList.get(i));
		}
		deleteList.clear();
		object.eAdapters().clear();

		return changedResources;
	}

	public static Set delete(EObject object, ResourceSet resourceSet) {
		if (object == null || object.eResource() == null) {
			return Collections.EMPTY_SET;
		}
		if (resourceSet == null)
			return delete(object, object.eResource().getContents());

		return delete(object, getResourceSetRoots(resourceSet, null));
	}

	public static Set delete(Resource resource) {
		if (resource == null) {
			return Collections.EMPTY_SET;
		}

		return delete(resource, resource.getResourceSet());
	}

	public static Set delete(final Resource resource, ResourceSet resourceSet) {
		if ((resource == null) || (resourceSet == null)) {
			return Collections.EMPTY_SET;
		}
		final Set changedResources = new HashSet();
		final List resourceSetRoots = getResourceSetRoots(resourceSet, resource);
		final boolean resolve = false;
		final List removeList = new FastList();
		// final List removeMap = new FastList();
		final ContainmentTraverser containmentTraverser = new ContainmentTraverser(resourceSetRoots, resolve);

		EObjectVisitor objectVisitor = new EObjectVisitor() {
			public boolean beforeChildren(EObject element) {
				return true;
			}

			public boolean afterChildren(EObject element) {
				processObjectReferences(resource, changedResources, resolve, true, removeList, containmentTraverser, element);
				return true;
			}

		};

		containmentTraverser.registerVisitors(objectVisitor);

		containmentTraverser.traverse();

		resource.getContents().clear();

		resourceSet.getResources().remove(resource);

		return changedResources;

	}

	public static boolean processObjectReferences(final Notifier container, final Set changedResources, final boolean resolve, final boolean delete,
			final Collection collectedEntries, final ContainmentTraverser containmentTraverser, EObject element) {
		boolean modified = false;
		List references = element.eClass().getEAllReferences();
		for (int i = references.size() - 1; i >= 0; i--) {
			EReference ref = (EReference) references.get(i);
			if (!ref.isContainment() && /* && ref.isTransient() && */
			element.eIsSet(ref)) {
				Object o = element.eGet(ref, resolve);
				if (o instanceof EObject) {
					if (isContainedIn((EObject) o, container)) {
						if (delete) {
							if (ref.isChangeable())
								element.eUnset(ref);
						} else
							collectedEntries.add(o);
						modified = true;
					}
				} else if (o instanceof EMap) {
					// MS fix for EMap in generic correlation case
					// TODO: MS add support for lists/maps as keys (mutable
					// keys)
					EMap l = (EMap) o;
					if (delete) {
						collectedEntries.clear();
					}
					for (Iterator iter2 = containmentTraverser.getIterator(l); iter2.hasNext();) {
						Map.Entry entry = (Map.Entry) iter2.next();
						if (!(entry.getKey() instanceof EObject)) {
							// ignore any EMap that doesn't have EObject as key
							continue;
						}

						EObject key = (EObject) entry.getKey();
						if (key == null)
							continue;
						if (isContainedIn(key, container)) {
							collectedEntries.add(entry);
							modified = true;
						} else {
							EList value = (EList) entry.getValue();
							if (value != null) {
								List removeEntries1 = new FastList();
								// removeEntries1.clear();
								for (Iterator iter3 = containmentTraverser.getIterator(value); iter3.hasNext();) {
									EObject entry1 = (EObject) iter3.next();
									if (isContainedIn(entry1, container)) {
										removeEntries1.add(entry1);
										modified = true;
									}
								}
								if (delete) {
									for (Iterator iter = removeEntries1.iterator(); iter.hasNext();) {
										EObject entry1 = (EObject) iter.next();
										if (entry1.eIsProxy() && !resolve)
											((InternalEList) value).basicRemove(entry, null);
										else
											l.remove(entry);
									}
									removeEntries1.clear();
								} else {
									collectedEntries.addAll(removeEntries1);
								}
							}
						}
					}
					if (delete) {
						for (Iterator iter = collectedEntries.iterator(); iter.hasNext();) {
							Map.Entry entry = (Map.Entry) iter.next();
							if (entry.getKey() == null)
								continue;
							if (((EObject) entry.getKey()).eIsProxy() && !resolve) {
								((EcoreEMap) l).basicRemove(entry.getKey(), null);
							} else
								l.removeKey(entry.getKey());
						}
						collectedEntries.clear();
					}

				} else if (o instanceof EList) {
					EList l = (EList) o;
					if (delete) {
						collectedEntries.clear();
					}
					for (Iterator iter2 = containmentTraverser.getIterator(l); iter2.hasNext();) {
						EObject entry = (EObject) iter2.next();
						if (isContainedIn((EObject) entry, container)) {
							collectedEntries.add(entry);
							modified = true;
						}
					}
					if (delete) {
						for (Iterator iter = collectedEntries.iterator(); iter.hasNext();) {
							EObject entry = (EObject) iter.next();
							if (entry.eIsProxy() && !resolve)
								((InternalEList) l).basicRemove(entry, null);
							else
								l.remove(entry);
						}
						collectedEntries.clear();
					}
				}
			}
		}
		if (modified)
			changedResources.add(element.eResource());
		return modified;
	}

	public static Set unload(final Resource resource, ResourceSet resourceSet) {
		if ((resource == null) || (resourceSet == null)) {
			return Collections.EMPTY_SET;
		}
		final Set changedResources = new HashSet();
		final Set proxyObjects = new HashSet();
		final List resourceSetRoots = getResourceSetRoots(resourceSet, resource);
		final boolean resolve = false;
		final ContainmentTraverser containmentTraverser = new ContainmentTraverser(resourceSetRoots, resolve);

		EObjectVisitor objectVisitor = new EObjectVisitor() {
			public boolean beforeChildren(EObject element) {
				return true;
			}

			public boolean afterChildren(EObject element) {
				processObjectReferences(resource, changedResources, resolve, false, proxyObjects, containmentTraverser, element);
				return true;
			}

		};

		containmentTraverser.registerVisitors(objectVisitor);

		containmentTraverser.traverse();
		proxyObjects.addAll(resource.getContents());

		for (Iterator iter = proxyObjects.iterator(); iter.hasNext();) {
			EObject element = (EObject) iter.next();
			convertToProxy((InternalEObject) element);
		}

		resource.getContents().clear();

		resourceSet.getResources().remove(resource);

		return changedResources;

	}

	public static void unsetFeaturesInSameResource(EObject object, boolean resolve) {
		Resource resource = object.eResource();
		if (resource == null)
			return;
		List features = object.eClass().getEAllAttributes();
		for (int i = features.size() - 1; i >= 0; i--) {
			EStructuralFeature feature = (EStructuralFeature) features.get(i);
			if (!feature.isChangeable())
				continue;
			object.eUnset(feature);
		}
		features = object.eClass().getEAllReferences();
		for (int i = features.size() - 1; i >= 0; i--) {
			EReference feature = (EReference) features.get(i);
			if (!feature.isChangeable())
				continue;
			if (feature != object.eContainmentFeature()) {

				Object o = object.eGet(feature, resolve);
				if (feature.getEOpposite() != null) {
					if (o instanceof EObject) {
						if (isContainedIn((EObject) o, resource)) {
							object.eUnset(feature);
						}
					} else if (o instanceof EcoreEList) {
						EcoreEList l = (EcoreEList) o;
						if (isContainedIn((EObject) l.getNotifier(), resource)) {
							object.eUnset(feature);
						}
					}

				} else {
					if (o instanceof EObject) {
						if (isContainedIn((EObject) o, resource)) {
							object.eUnset(feature);
						}
					} else if (o instanceof EcoreEList) {
						EcoreEList l = (EcoreEList) o;
						if (isContainedIn((EObject) l.getNotifier(), resource)) {
							object.eUnset(feature);
						}
					}
				}
			}
		}

		if (object.eContainer() != null)
			EcoreUtil.remove(object.eContainer(), object.eContainmentFeature(), object);
	}

	/**
	 * @param object
	 */
	public static void convertToProxy(InternalEObject object) {
		object.eSetProxyURI(EcoreUtil.getURI(object));
		unsetFeaturesInSameResource(object, false);
		object.eAdapters().clear();
	}

	public static void unsetAllFeatures(EObject object) {
		List features = object.eClass().getEAllStructuralFeatures();
		for (int i = features.size() - 1; i >= 0; i--) {
			EStructuralFeature feature = (EStructuralFeature) features.get(i);
			if (feature.isChangeable())
				object.eUnset(feature);
		}
		if (object.eContainer() != null)
			EcoreUtil.remove(object.eContainer(), object.eContainmentFeature(), object);
	}

	public final static boolean isContainedIn(EObject object, Notifier container) {
		if (container instanceof Resource)
			return isContainedIn(object, (Resource) container);
		else
			return isContainedIn(object, (EObject) container);

	}

	public final static boolean isContainedIn(EObject object, Resource resource) {
		if (!object.eIsProxy())
			return resource == object.eResource();
		else
			// this should work for fragments but it won't work with ID's.
			return ((InternalEObject) object).eProxyURI().trimFragment().equals(resource.getURI());

	}

	public final static boolean isContainedIn(EObject object, EObject container) {
		if (!object.eIsProxy())
			return object == container || EcoreUtil.isAncestor(container, object);
		else
			// this should work for fragments but it won't work with ID's.
			return object == container || EcoreUtil.getURI(object).toString().indexOf(EcoreUtil.getURI(container).toString()) > 0;

	}

	public static List getResourceSetRoots(ResourceSet resourceSet, Resource ignoredResource) {
		List list = new BasicEList();
		// Changing from an iterator here to array use to avoid
		// concurrentmodificationexceptions.
		Resource[] resources = (Resource[]) resourceSet.getResources().toArray(new Resource[]{});
		for (int i=0; i<resources.length; i++) {
			
			Resource resource = resources[i];
			// TODO MS - replace this temp fix for large log support
			//if workspace is out of sync (needs refreshing), resource can be null
			//for that reason, adding this null check
			if (resource!=null && resource != ignoredResource && !resource.getClass().getName().endsWith("DBResourceImpl"))
				list.addAll(resource.getContents());
		}
		return list;
	}

	public static Set unload(Resource resource) {
		if (resource == null) {
			return Collections.EMPTY_SET;
		}
		return unload(resource, resource.getResourceSet());
	}

	public static EObject getAddReusableValue(EObject container, EStructuralFeature feature, EObject newValue) {
		ReusableValuesAdapter reusableValuesAdapter = null;
		List adapters = null;
		if (container.eResource() == null) {
			// used at event load time
			adapters = container.eAdapters();
		} else {
			// used at resource load time
			adapters = container.eResource().eAdapters();
		}
		for (Iterator iter = adapters.iterator(); iter.hasNext();) {
			Adapter element = (Adapter) iter.next();
			if (element instanceof ReusableValuesAdapter) {
				reusableValuesAdapter = (ReusableValuesAdapter) element;
				break;
			}
		}
		if (reusableValuesAdapter == null) {
			if (container.eResource() != null) {
				reusableValuesAdapter = new ReusableValuesAdapter();
				container.eResource().eAdapters().add(reusableValuesAdapter);
			} else
				return newValue;
		}

		List reusableValues = (List) reusableValuesAdapter.getValuesByFeatureMap().get(feature);
		if (reusableValues == null) {
			reusableValues = new ArrayList(1);
			reusableValuesAdapter.getValuesByFeatureMap().put(feature, reusableValues);
		}
		EqualityHelperIgnoreContainer equalityHelperIgnoreContainer = new EqualityHelperIgnoreContainer();
		for (int i = reusableValues.size(); --i >= 0;) {
			if (equalityHelperIgnoreContainer.equals((EObject) reusableValues.get(i), newValue)) {
				return (EObject) reusableValues.get(i);
			}
			equalityHelperIgnoreContainer.clear();
		}
		reusableValues.add(newValue);
		return newValue;
	}

	/**
	 * Uses a special equal method to find the source in the target using
	 * feature by feature comparation.
	 * 
	 * @param source
	 * @param target
	 * @return
	 */
	public static boolean containsObjectUsingSpecialEqual(final EObject source, EObject target) {
		final boolean[] res = new boolean[1];
		res[0] = false;
		final List resourceSetRoots = new ArrayList();
		resourceSetRoots.add(target);
		final boolean resolve = false;
		final ContainmentTraverser containmentTraverser = new ContainmentTraverser(resourceSetRoots, resolve);
		final EqualityHelperIgnoreContainer equalityHelperIgnoreContainer = new EqualityHelperIgnoreContainer();
		EObjectVisitor objectVisitor = new EObjectVisitor() {
			public boolean beforeChildren(EObject element) {
				if (equalityHelperIgnoreContainer.equals(source, element)) {
					res[0] = true;
					equalityHelperIgnoreContainer.clear();
					return false;
				}
				equalityHelperIgnoreContainer.clear();
				return true;
			}

			public boolean afterChildren(EObject element) {
				return true;
			}

		};

		containmentTraverser.registerVisitors(objectVisitor);

		containmentTraverser.traverse();

		return res[0];
	}
}
