/**********************************************************************
 * Copyright (c) 2003, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HierarchyURIConverterImpl.java,v 1.3 2008/01/24 02:28:17 apnan Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.models.hierarchy.util;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.plugin.EcorePlugin;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.ResourceImpl;
import org.eclipse.emf.ecore.resource.impl.URIConverterImpl;
/**
 * A highly functional and extensible URI converter implementation.
 * <p>
 * This implementation provides seamless transparent Eclipse integration by
 * supporting the <code>platform:/resource</code> mechanism both inside of
 * Eclipse and outside of Eclipse. Furthermore, although the implementation
 * imports both {@link org.eclipse.core.runtime} and
 * {@link org.eclipse.core.resources}, and hence requires the Eclipse libraries
 * at development time, the implementation does <b>not</b> require them at
 * runtime. Clients of this implementation must be cautious if they wish to
 * maintain this platform neutral behaviour.
 * </p>
 */
public class HierarchyURIConverterImpl extends URIConverterImpl {

	// ~ Inner Classes
	// ------------------------------------------------------------------------------
	public static class HierarchyPipedInputStream extends InputStream {
		protected byte[] inBuf;
		protected int inCount;
		protected int inPos;
		protected long inTotalLength;
		/**
		 * Constructor MyPipedInputStream.
		 * 
		 * @param buf
		 */
		public HierarchyPipedInputStream(byte[] buf, int length) {
			inBuf = buf;
			inCount = length;
			inTotalLength = length;
			inPos = 0;
		}
		/**
		 * Sets the inBuf.
		 * 
		 * @param inBuf
		 *            The inBuf to set
		 */
		public void setInBuf(byte[] buf, int length) {
			inBuf = buf;
			inCount = length;
			inTotalLength = length;
			inPos = 0;
		}
		/**
		 * @see java.io.InputStream#available()
		 */
		public int available() throws IOException {
			return inCount;
		}
		/**
		 * @see java.io.InputStream#markSupported()
		 */
		public boolean markSupported() {
			return false;
		}
		/**
		 * @see java.io.InputStream#read()
		 */
		public int read() throws IOException {
			if (available() == 0) {
				return -1;
			} else {
				inCount--;
				return inBuf[inPos++];
			}
		}
		/**
		 * @see java.io.InputStream#read(byte[], int, int)
		 */
		public int read(byte[] outBuf, int offset, int length) throws IOException {
			if (available() == 0) {
				return -1;
			}
			// int readBytes = 0;
			// while ((length > 0) && (inPos < inLastPos)) {
			// outBuf[offset++] = inBuf[inPos++];
			// length--;
			// readBytes++;
			// }
			int readBytes = min(inCount, length);
			System.arraycopy(inBuf, inPos, outBuf, offset, readBytes);
			inPos = inPos + readBytes;
			inCount = inCount - readBytes;
			return readBytes;
		}
		/**
		 * @see java.io.InputStream#read(byte[])
		 */
		public int read(byte[] arg0) throws IOException {
			return read(arg0, 0, arg0.length);
		}
		/**
		 * @see java.io.InputStream#skip(long)
		 */
		public long skip(long arg0) throws IOException {
			arg0 = arg0 + inPos;
			if (arg0 < inTotalLength) {
				inPos = (int) arg0;
				return arg0;
			} else {
				return -1;
			}
		}
		protected int min(int a, int b) {
			if (a < b) {
				return a;
			}
			return b;
		}
	}
	public static class HierarchyPlatformResourceOutputStream extends OutputStream {
		protected HierarchyPipedInputStream pipedInputStream;
		protected IFile file;
		protected IProgressMonitor progressMonitor;
		protected boolean firstCall = true;
		protected boolean force;
		protected boolean keepHistory;
		protected int SAVE_TRESHOLD = org.eclipse.hyades.models.hierarchy.util.HierarchyXMISaveImpl.BUFFER_SIZE;
		protected byte[] outBuf = new byte[SAVE_TRESHOLD];
		protected int outPos;
		public HierarchyPlatformResourceOutputStream(IFile file, boolean force, boolean keepHistory, IProgressMonitor monitor) throws IOException {
			this.file = file;
			this.force = force;
			this.keepHistory = keepHistory;
			this.progressMonitor = monitor;
			// if (file.exists())
			// {
			// try {
			// file.delete(force,keepHistory,progressMonitor);
			// } catch (CoreException e) {
			// throw new ResourceImpl.IOWrappedException(e);
			// }
			// }
			// else
			createContainer(file.getParent());
			pipedInputStream = new HierarchyPipedInputStream(outBuf, 0);
		}
		/**
		 * @see java.io.OutputStream#close()
		 */
		public void close() throws IOException {
			flush();
		}
		/**
		 * @see java.io.OutputStream#flush()
		 */
		public void flush() throws IOException {
			if (outPos > 0) {
				writeToWorkspaceResource();
				outPos = 0;
			}
		}
		/**
		 * @see java.io.OutputStream#write(byte[], int, int)
		 */
		public void write(byte[] data, int offset, int length) throws IOException {
			// while (length > 0) {
			// while ((outPos < SAVE_TRESHHOLD) && (length > 0)) {
			// outBuf[outPos++] = data[offset++];
			// length--;
			// }
			//
			// if (outPos == SAVE_TRESHHOLD) {
			// flush();
			// }
			// }
			while (length > 0) {
				int writeBytes = min(SAVE_TRESHOLD - outPos, length);
				System.arraycopy(data, offset, outBuf, outPos, writeBytes);
				outPos = outPos + writeBytes;
				offset = offset + writeBytes;
				length = length - writeBytes;
				if (outPos == SAVE_TRESHOLD) {
					flush();
				}
			}
		}
		/**
		 * @see java.io.OutputStream#write(byte[])
		 */
		public void write(byte[] arg0) throws IOException {
			write(arg0, 0, arg0.length);
		}
		/**
		 * @see java.io.OutputStream#write(int)
		 */
		public void write(int arg0) throws IOException {
			outBuf[outPos++] = (byte) arg0;
			if (outPos == SAVE_TRESHOLD) {
				flush();
			}
		}
		protected void createContainer(IContainer container) throws IOException {
			if (!container.exists()) {
				if (container.getType() == IResource.FOLDER) {
					createContainer(container.getParent());
					try {
						((IFolder) container).create(force, keepHistory, progressMonitor);
					} catch (CoreException exception) {
						throw new ResourceImpl.IOWrappedException(exception);
					}
				}
			}
		}
		protected int min(int a, int b) {
			if (a < b) {
				return a;
			}
			return b;
		}
		/**
		 * Method writeToResource.
		 */
		protected void writeToWorkspaceResource() throws IOException {
			pipedInputStream.setInBuf(outBuf, outPos);
			try {
				if (!file.exists()) {
					file.create(pipedInputStream, force, progressMonitor);
					firstCall = false;
				} else {
					if (!file.isSynchronized(IResource.DEPTH_ONE)) {
						file.refreshLocal(IResource.DEPTH_ONE, progressMonitor);
					}
					if (firstCall) {
						file.setContents(pipedInputStream, force, keepHistory, progressMonitor);
						firstCall = false;
					} else {
						file.appendContents(pipedInputStream, force, keepHistory, progressMonitor);
					}
				}
			} catch (CoreException exception) {
				throw new Resource.IOWrappedException(exception);
			}
		}
	}
	/**
	 * Isolated Eclipse workbench utilities.
	 */
	public static class WorkbenchHelper {
		/**
		 * Creates an input stream for the given {@link IFile} path.
		 * <p>
		 * This implementation uses {@link IFile#getContents IFile.getContents}.
		 * </p>
		 * 
		 * @return an open input stream.
		 * @see IWorkspaceRoot#getFile(IPath)
		 * @see IFile#getContents
		 * @exception IOException
		 *                if there is a problem obtaining an open input stream.
		 */
		public static InputStream createPlatformResourceInputStream(String platformResourcePath) throws IOException {
			IFile file = workspaceRoot.getFile(new Path(platformResourcePath));
			try {
				if (!file.isSynchronized(IResource.DEPTH_ONE)) {
					file.refreshLocal(IResource.DEPTH_ONE, null);
				}
				return file.getContents();
			} catch (CoreException exception) {
				throw new Resource.IOWrappedException(exception);
			}
		}
		/**
		 * Creates an output stream for the given {@link IFile} path.
		 * <p>
		 * This implementation uses a
		 * {@link PerftraceURIConverterImpl.PlatformResourceOutputStream}.
		 * </p>
		 * 
		 * @return an open output stream.
		 * @exception IOException
		 *                if there is a problem obtaining an open output stream.
		 * @see IWorkspaceRoot#getFile(IPath)
		 * @see PerftraceURIConverterImpl.PlatformResourceOutputStream
		 * @see IFile#setContents(InputStream, boolean, boolean,
		 *      IProgressMonitor)
		 */
		public static OutputStream createPlatformResourceOutputStream(String platformResourcePath) throws IOException {
			IFile file = workspaceRoot.getFile(new Path(platformResourcePath));
			return new HierarchyPlatformResourceOutputStream(file, false, false, null);
		}
	}
	
	  /**
	   * Creates an output stream for the platform resource path and returns it.
	   * <p>
	   * This implementation does one of two things, depending on the runtime environment.
	   * If there is an Eclipse workspace, it delegates to 
	   * {@link WorkbenchHelper#createPlatformResourceOutputStream WorkbenchHelper.createPlatformResourceOutputStream},
	   * which gives the expected Eclipse behaviour.
	   * Otherwise, the {@link EcorePlugin#resolvePlatformResourcePath resolved} URI 
	   * is delegated to {@link #createOutputStream createOutputStream}
	   * for recursive processing.
	   * @return an open output stream.
	   * @exception IOException if there is a problem obtaining an open output stream or a valid interpretation of the path.
	   * @see EcorePlugin#resolvePlatformResourcePath(String)
	   */
	  protected OutputStream createPlatformResourceOutputStream(String platformResourcePath) throws IOException
	  {
	    // ECLIPSE-DEPEND-BEGIN
	    if (workspaceRoot != null)
	    {
	      return WorkbenchHelper.createPlatformResourceOutputStream(platformResourcePath);
	    }
	    else
	    // ECLIPSE-DEPEND-END
	    {
	      URI resolvedLocation = EcorePlugin.resolvePlatformResourcePath(platformResourcePath);
	      if (resolvedLocation != null)
	      {
	        return createOutputStream(resolvedLocation);
	      }

	      throw new IOException("The path '" + platformResourcePath + "' is unmapped");
	    }
	  }
	  /**
	   * Creates an input stream for the platform resource path and returns it.
	   * <p>
	   * This implementation does one of two things, depending on the runtime environment.
	   * If there is an Eclipse workspace, it delegates to 
	   * {@link WorkbenchHelper#createPlatformResourceInputStream WorkbenchHelper.createPlatformResourceInputStream},
	   * which gives the expected Eclipse behaviour.
	   * Otherwise, the {@link EcorePlugin#resolvePlatformResourcePath resolved} URI 
	   * is delegated to {@link #createInputStream createInputStream}
	   * for recursive processing.
	   * @return an open input stream.
	   * @exception IOException if there is a problem obtaining an open input stream or a valid interpretation of the path.
	   * @see EcorePlugin#resolvePlatformResourcePath(String)
	   */
	  protected InputStream createPlatformResourceInputStream(String platformResourcePath) throws IOException
	  {
	    // ECLIPSE-DEPEND-BEGIN
	    if (workspaceRoot != null)
	    {
	      return WorkbenchHelper.createPlatformResourceInputStream(platformResourcePath);
	    }
	    else
	    // ECLIPSE-DEPEND-END
	    {
	      URI resolvedLocation = EcorePlugin.resolvePlatformResourcePath(platformResourcePath);
	      if (resolvedLocation != null)
	      {
	        return createInputStream(resolvedLocation);
	      }

	      throw new IOException("The path '" + platformResourcePath + "' is unmapped");
	    }
	  }
	public String toString(URI testUri) {
		
		return super.toString();
	}
}
