/**********************************************************************
 * Copyright (c) 2003, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CrimsonFragmentHandler.java,v 1.2 2008/01/24 02:28:17 apnan Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.models.hierarchy.util.internal;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import org.eclipse.hyades.loaders.util.IXMLLoader;
import org.eclipse.hyades.loaders.util.InvalidXMLException;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.loaders.util.XMLFragmentHandler;
import org.eclipse.hyades.loaders.util.XMLLoader;
import org.eclipse.hyades.models.util.ModelDebugger;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;
/**
 * @author slavescu
 * 
 * This class wraps an XMLLoader with a SAX DefaultHandler. 
 * Tries to overcome the Crimson buffering problem (until Crimson's internal buffer is not full we don't get any SAX events). 
 */
public class CrimsonFragmentHandler extends DefaultHandler implements XMLFragmentHandler {
	protected static final String IGNORED_ELEMENT_NAME = "M69";
	protected static final char[] END_ELEMENT = "endElement=".toCharArray();
	protected static final char[] START_ELEMENT = "startElement=".toCharArray();
	protected static final char[] START_DOCUMENT = "startDocument=".toCharArray();
	protected static final char[] END_DOCUMENT = "endDocument=".toCharArray();
	protected static final char[] CHARACTERS = "characters=".toCharArray();
//	private static final char[] CHARACTERS_WITH_IGNORED = "charactersWithIgnored=".toCharArray();
	protected InputSource inputSource = null;
	protected InputStream xmlStream = null;
	protected ParserPipedInputStream pipedXmlStream = null;
	protected SAXParser parser;
	protected Thread parseThread;
	protected IXMLLoader handler;
	protected int depth;
	protected static final int WAIT_TIME = 1000;
	protected static final char[] WB_IN = "wB>".toCharArray();
	protected static final char[] WB_OUT = "wB<".toCharArray();
	protected char[] IGNORE_CHAR_LENGTH = "rB>ignoreCharsLength=".toCharArray();
	protected static final byte[] IGNORE_ELEMENT_PREFIX = ("<" + IGNORED_ELEMENT_NAME + " S88=\"").getBytes();
	protected static final byte[] IGNORE_ELEMENT_POSTFIX = "\"/>".getBytes();
	protected static final int MIN_INGNORED_ELEMENT_LENGTH = IGNORE_ELEMENT_PREFIX.length + IGNORE_ELEMENT_POSTFIX.length;
	protected int ignoredCount = 0;
	protected byte[] ignoredAttributeValue;
	protected boolean ignoreOn;
	protected boolean initRequired;
	final private static int CRIMSON_BUFSIZ = 8 * 1024;
	protected final String lockForIgnored = new String("LockForIgnoredEvents");
	public void setXMLLoader(IXMLLoader loader) {
		handler = loader;
		initRequired=true;
	}
	public void init() {
		try {
			initRequired=false;
			depth = -1;
			parser = makeParser();
			pipedXmlStream = new ParserPipedInputStream(this);
			xmlStream = pipedXmlStream;
			parseThread = new Thread(Thread.currentThread().getThreadGroup(), "xmlParserThread") {
				public void run() {
					if (xmlStream == null) {
						return;
					}
					try {
						parse(xmlStream, new SpecialHandler());
					} catch (Exception e) {
						// ignore error
					}
					xmlStream = null;
				}
			};
			parseThread.start();
		} catch (Exception e) {
			throw new InvalidXMLException(e);
		}
	}
	public void scanContent(byte[] fragment, int offset, int length) throws InvalidXMLException {
		if (pipedXmlStream == null) 
		{
			if(initRequired)
				init();
			else
				return;
		}
		pipedXmlStream.writeBuf(fragment, offset, length);
	}
	public void terminateParser() {
		depth = -1;
		if (xmlStream != null) {
			try {
				if (ModelDebugger.INSTANCE.debug) {
					ModelDebugger.log("SAXFragmentHandler.terminateParser() - close ParserPipedInputStream");
				}
				if(pipedXmlStream!=null)
				{
					ParserPipedInputStream temp = pipedXmlStream;
					pipedXmlStream = null;
					temp.makeClosed();
				}
				xmlStream = null;
			} catch (Exception e) {
				if (ModelDebugger.INSTANCE.debug) {
					ModelDebugger.log(e, "SAXFragmentHandler.terminateParser()");
				}
				// ignore exception
			}
		}
		synchronized (this) {
			notifyAll();
			if(parseThread!=null)
				parseThread.interrupt();
			if (ModelDebugger.INSTANCE.debug) {
				ModelDebugger.log("SAXFragmentHandler.terminateParser() - done");
			}
		}
	}
	/**
	 * Make either a validating or non-validating parser; throw an if one could
	 * not be made.
	 */
	protected SAXParser makeParser() throws ParserConfigurationException, SAXException {
		parser = XMLLoader.makeParser();
//		try {
//			parser.setProperty("http://apache.org/xml/properties/entity-expansion-limit",new Integer(100000));
//		} catch (SAXNotRecognizedException e) {
//			// TODO Auto-generated catch block
//			e.printStackTrace();
//		} catch (SAXNotSupportedException e) {
//			// TODO Auto-generated catch block
//			e.printStackTrace();
//		}
		return parser;
	}
	protected synchronized void parse(InputStream inputStream, DefaultHandler theHandler) throws InvalidXMLException {
		try {
//			inputSource = new InputSource(inputStream);
			parser.parse(inputStream,theHandler);
		} catch (Exception e) {
			if(ModelDebugger.INSTANCE.debug)
			{
				log(e,null);
			}
			InvalidXMLException i;
			if (e instanceof InvalidXMLException)
				i = (InvalidXMLException) e;
			else
				i = new InvalidXMLException(e);
			handler.error(i);
		}
	}
	
	public class SpecialHandler extends DefaultHandler
	{

		public void setDocumentLocator(Locator locator) {
		}

		public void characters(char[] ch, int start, int length) throws SAXException {
			handler.characters(ch, start, length);
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(CHARACTERS);
				System.out.println(ignoredCount + "," + length + "=" + new String(ch, start, length));
			}
		}

		public void endDocument() throws SAXException {
			if (depth == 0 && xmlStream != null) {
				if(pipedXmlStream !=null)
				{
					pipedXmlStream.makeClosed();
					pipedXmlStream = null;
				}
				xmlStream = null;
			}
			depth = -1;
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(END_DOCUMENT);
				System.out.println();
			}
			handler.endDocument(null, 0);
		}

		public void endElement(String uri, String localName, String qName) throws SAXException {
			synchronized (lockForIgnored) {
				if (ignoredCount > 0) {
					if (IGNORED_ELEMENT_NAME.equals(qName)) {
						ignoredCount--;
						if (ignoredCount == 0)
							ignoreOn = false;
						return;
					}
				}
				lockForIgnored.notify();
			}
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(END_ELEMENT);
				System.out.println(qName);
			}
			handler.endElement(qName, 0);
			depth--;
			if (depth == 0) {
				if (xmlStream != null) {
					if(pipedXmlStream !=null)
					{
						pipedXmlStream.makeClosed();
						pipedXmlStream = null;
					}
					xmlStream = null;
				}
			}
		}

		public void endPrefixMapping(String prefix) throws SAXException {
		}

		public void error(SAXParseException e) throws SAXException {
			throw e;
		}

		public void fatalError(SAXParseException e) throws SAXException {
			throw e;
		}

		public void ignorableWhitespace(char[] ch, int start, int length) throws SAXException {
		}

		public void notationDecl(String name, String publicId, String systemId) throws SAXException {
		}

		public void processingInstruction(String target, String data) throws SAXException {
		}

		public InputSource resolveEntity(String publicId, String systemId) throws SAXException {
			return null;
		}

		public void skippedEntity(String name) throws SAXException {
		}

		public void startDocument() throws SAXException {
			depth = 0;
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(START_DOCUMENT);
				System.out.println();
			}
			handler.startDocument();
		}

		public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
			synchronized (lockForIgnored) {
				if (ignoredCount > 0) {
					if (IGNORED_ELEMENT_NAME.equals(qName)) {
						return;
					}
				}
				lockForIgnored.notify();
			}
			depth++;
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(START_ELEMENT);
				System.out.println(qName);
			}
			handler.startElement(qName, false, attributes.getLength() == 0);
			for (int i = 0; i < attributes.getLength(); i++) {
				handler.attributeName(attributes.getQName(i));
				handler.attributeValueCharacters(attributes.getValue(i));
			}
		}

		public void startPrefixMapping(String prefix, String uri) throws SAXException {
		}

		public void unparsedEntityDecl(String name, String publicId, String systemId, String notationName) throws SAXException {
		}

		public void warning(SAXParseException e) throws SAXException {
		}
		
	}
	
	/**
	 * 
	 * @author slavescu
	 *
	 * Input stream that can be feed from a different thread, see writeBuf method.
	 *
	 */
	public class ParserPipedInputStream extends InputStream {
		protected byte[] inBuf;
		protected boolean closed = false;
		protected int inCount;
		protected long inTotalLength;
		protected int inPartLength;
		protected int inPos;
//		private CrimsonFragmentHandler handler;
		protected StringBuffer sb = new StringBuffer();
		/**
		 * Constructor MyPipedInputStream.
		 * 
		 * @param buf
		 */
		public ParserPipedInputStream(CrimsonFragmentHandler handler) {
//			this.handler = handler;
			inTotalLength = 0;
			inPartLength = 0;
			inCount = 0;
			inPos = 0;
			ignoredAttributeValue = new byte[CRIMSON_BUFSIZ];
			Arrays.fill(ignoredAttributeValue, (byte) 'S');
		}
		protected synchronized void makeOpened() {
			inPartLength = 0;
			inBuf = null;
			inCount = 0;
			inPos = 0;
			closed = false;
			notifyAll();
		}
		/**
		 * @see java.io.InputStream#available()
		 */
		public synchronized int available() throws IOException {
			waitForNewData();
			return inCount;
		}
		public void close() throws IOException {
			super.close();
			closed = true;
		}
		public synchronized boolean hasEmptyBuffer() {
			return inCount == 0;
		}
		public synchronized void mark(int arg0) {
			super.mark(arg0);
		}
		/**
		 * @see java.io.InputStream#markSupported()
		 */
		public boolean markSupported() {
			return false;
		}
		/**
		 * @see java.io.InputStream#read()
		 */
		public int read() throws IOException {
			synchronized (lockForIgnored) {
				if (available() == 0) {
					return -1;
				} else {
					if (ignoreOn) {
						if (ModelDebugger.INSTANCE.debugEventFlow) {
							System.out.print(IGNORE_CHAR_LENGTH);
							System.out.println("1,1");
						}
						inCount = 0;
						inPartLength = 0;
						if (ModelDebugger.INSTANCE.debugEventFlow) {
							ModelDebugger.INSTANCE.writeBinaryLog("PARSER_READ", ignoredAttributeValue, 0, 1);
						}
						lockForIgnored.notify();
						return ignoredAttributeValue[0];
					} else {
						inCount--;
						if (ModelDebugger.INSTANCE.debugEventFlow) {
							ModelDebugger.INSTANCE.writeBinaryLog("PARSER_READ", inBuf, inPos, 1);
						}
						return inBuf[inPos++];
					}
				}
			}
		}
		/**
		 * @see java.io.InputStream#read(byte[], int, int)
		 */
		public int read(byte[] outBuf, int offset, int length) throws IOException {
			synchronized (lockForIgnored) {
				if (available() == 0) {
					return -1;
				}
				if (ignoreOn) {
					int ignoreElementLength = length;// - inPartLength;
					//										if (ignoreElementLength < 0)
					//											ignoreElementLength = length - inPartLength % CRIMSON_BUFSIZ + 1;
					//										if (ignoreElementLength < MIN_INGNORED_ELEMENT_LENGTH) {
					//											ignoreElementLength = MIN_INGNORED_ELEMENT_LENGTH;
					//										}
					if (ModelDebugger.INSTANCE.debugEventFlow) {
						System.out.print(IGNORE_CHAR_LENGTH);
						System.out.print(ignoreElementLength);
						System.out.print(',');
						System.out.println(length);
					}
					System.arraycopy(IGNORE_ELEMENT_PREFIX, 0, outBuf, offset, IGNORE_ELEMENT_PREFIX.length);
					if (ignoreElementLength > MIN_INGNORED_ELEMENT_LENGTH)
						System.arraycopy(ignoredAttributeValue, 0, outBuf, offset + IGNORE_ELEMENT_PREFIX.length, ignoreElementLength - MIN_INGNORED_ELEMENT_LENGTH);
					System.arraycopy(IGNORE_ELEMENT_POSTFIX, 0, outBuf, offset + ignoreElementLength - IGNORE_ELEMENT_POSTFIX.length, IGNORE_ELEMENT_POSTFIX.length);
					inCount = 0;
					inPartLength = 0;
//					if (ModelDebugger.INSTANCE.debugEventFlow) {
//						ModelDebugger.INSTANCE.writeBinaryLog("PARSER_READ", outBuf, offset, ignoreElementLength);
//					}
					ignoredCount++;
					ignoreOn = false;
					lockForIgnored.notify();
					return ignoreElementLength;
				} else {
					int readBytes = Math.min(inCount, length);
					System.arraycopy(inBuf, inPos, outBuf, offset, readBytes);
					inPos = inPos + readBytes;
					inCount = inCount - readBytes;
//					if (ModelDebugger.INSTANCE.debugEventFlow) {
//						ModelDebugger.INSTANCE.writeBinaryLog("PARSER_READ", outBuf, offset, readBytes);
//					}
					return readBytes;
				}
			}
		}
		/**
		 * @see java.io.InputStream#read(byte[])
		 */
		public int read(byte[] arg0) throws IOException {
			return read(arg0, 0, arg0.length);
		}
		public synchronized void reset() throws IOException {
			super.reset();
		}
		/**
		 * @see java.io.InputStream#skip(long)
		 */
		public synchronized long skip(long arg0) throws IOException {
			arg0 = arg0 + inPos;
			if (arg0 < inTotalLength) {
				inPos = (int) arg0;
				return arg0;
			} else {
				return -1;
			}
		}
		/**
		 * Sets the inBuf.
		 * 
		 */
		public synchronized void writeBuf(byte[] buf, int offset, int length) {
			if ((buf == null) || (length == 0)) {
				return;
			}
			synchronized (this) {
				while (ignoreOn) {
					try {
						wait();
					} catch (InterruptedException e) {
						Thread.currentThread().interrupt();
					}
				}
				inBuf = buf;
				inPos = offset;
				inTotalLength = inTotalLength + length;
				inPartLength = inPartLength + length;
				inCount = length;
			}
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(WB_IN);
				sb.setLength(0);
				sb.append(length);
				System.out.println(LoadersUtils.getChars(sb));
				System.out.write(buf, offset, length);
				System.out.write(buf, offset, length);
				System.out.println();
			}
			notifyAll();
			waitForEmptyBuffer();
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.println(WB_OUT);
			}
		}
		protected synchronized void makeClosed() {
			inPartLength = 0;
			inBuf = null;
			inCount = 0;
			inPos = 0;
			closed = true;
			notifyAll();
		}
		protected void waitForEmptyBuffer() {
			while (!hasEmptyBuffer() && !closed) {
				try {
					wait(WAIT_TIME);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					if (hasEmptyBuffer() || closed) {
						return;
					}
				}
			}
		}
		protected synchronized void waitForNewData() {
			boolean notify = true;
			while (hasEmptyBuffer()) {
				if (notify) {
					notify();
					notify = false;
				}
				if (closed) {
					return;
				}
				try {
					wait(WAIT_TIME);
					synchronized (this) {
						if (hasEmptyBuffer() && !ignoreOn && inPartLength > 0) {
							inCount = Integer.MAX_VALUE;
							ignoreOn = true;
							return;
						}
					}
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					if (closed) {
						return;
					}
				}
			}
		}
	}

	public class RegularHandler extends DefaultHandler
	{
	
		public void setDocumentLocator(Locator locator) {
		}
	
		public void characters(char[] ch, int start, int length) throws SAXException {
			handler.characters(ch, start, length);
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(CHARACTERS);
				System.out.println(length + "=" + new String(ch, start, length));
			}
		}
	
		public void endDocument() throws SAXException {
			if (depth == 0 && xmlStream != null) {
				if(pipedXmlStream !=null)
				{
					pipedXmlStream.makeClosed();
					pipedXmlStream = null;
				}
				xmlStream = null;
			}
			depth = -1;
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(END_DOCUMENT);
				System.out.println();
			}
			handler.endDocument(null, 0);
		}
	
		public void endElement(String uri, String localName, String qName) throws SAXException {
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(END_ELEMENT);
				System.out.println(qName);
			}
			handler.endElement(qName, 0);
			depth--;
			if (depth == 0) {
				if (xmlStream != null) {
					if(pipedXmlStream !=null)
					{
						pipedXmlStream.makeClosed();
						pipedXmlStream = null;
					}
					xmlStream = null;
				}
			}
		}
	
		public void endPrefixMapping(String prefix) throws SAXException {
		}
	
		public void error(SAXParseException e) throws SAXException {
			throw e;
		}
	
		public void fatalError(SAXParseException e) throws SAXException {
			throw e;
		}
	
		public void ignorableWhitespace(char[] ch, int start, int length) throws SAXException {
		}
	
		public void notationDecl(String name, String publicId, String systemId) throws SAXException {
		}
	
		public void processingInstruction(String target, String data) throws SAXException {
		}
	
		public InputSource resolveEntity(String publicId, String systemId) throws SAXException {
			return null;
		}
	
		public void skippedEntity(String name) throws SAXException {
		}
	
		public void startDocument() throws SAXException {
			depth = 0;
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(START_DOCUMENT);
				System.out.println();
			}
			handler.startDocument();
		}
	
		public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
			depth++;
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(START_ELEMENT);
				System.out.println(qName);
			}
			handler.startElement(qName, false, attributes.getLength() == 0);
			for (int i = 0; i < attributes.getLength(); i++) {
				handler.attributeName(attributes.getQName(i));
				handler.attributeValueCharacters(attributes.getValue(i));
			}
		}
	
		public void startPrefixMapping(String prefix, String uri) throws SAXException {
		}
	
		public void unparsedEntityDecl(String name, String publicId, String systemId, String notationName) throws SAXException {
		}
	
		public void warning(SAXParseException e) throws SAXException {
		}
		
	}

	protected void log(Exception e, String msg) {
		if (e instanceof SAXParseException) {
			SAXParseException spx = (SAXParseException) e;
			System.err.println(spx.getLocalizedMessage());
			System.err.println("at line: " + spx.getLineNumber());
			System.err.println("at column: " + spx.getColumnNumber());
			spx.fillInStackTrace();
			ModelDebugger.log(spx, msg);
		} else {
			ModelDebugger.log(e, msg);
		}
	}
	
	public void scanContent(InputStream inputStream, long offset, long length) {
		try {
			initRequired=false;
			depth = -1;
			parser = makeParser();
			xmlStream = inputStream;//new BufferedInputStream(inputStream,256*1024);
			parse(xmlStream,new RegularHandler());
		} catch (Exception e) {
			InvalidXMLException i;
			if (e instanceof InvalidXMLException)
				i = (InvalidXMLException) e;
			else
				i = new InvalidXMLException(e);
			handler.error(i);
		}
	}
}