/********************************************************************** 
 * Copyright (c) 2005, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: EMFWorkspaceUtil.java,v 1.3 2008/01/24 02:28:17 apnan Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.models.hierarchy.util.internal;

import java.io.File;
import java.io.IOException;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.URI;

/**
 * This internal class contains utility methods for EMF that are
 * workspace specific.  Do not access this class from code that may be
 * run without access to the org.eclipse.core.resources plugin, as the
 * import statements above will cause a class load failure.
 * 
 * @author jtoomey
 */
public class EMFWorkspaceUtil {
	/**
	 * Protocol for platform uri's. i.e. "platform:/..."
	 */
	public static final String PLATFORM_PROTOCOL = "platform"; //$NON-NLS-1$

	/**
	 * Resource indication in platform protocol. Indicates url is for a resource in the workspace. i.e. "platform:/resource/projectname/..."
	 * 
	 */
	public static final String PLATFORM_RESOURCE = "resource"; //$NON-NLS-1$
	
	/**
	 * Does the passed URI have the form platform:/resource/... ?
	 * 
	 * @param uri
	 * @return <code>true</code> if it is a platform resource protocol.
	 * 
	 * @since (from JEM WorkbenchResourceHelperBase.java)
	 */
	public static boolean isPlatformResourceURI(URI uri) {
		return PLATFORM_PROTOCOL.equals(uri.scheme()) && PLATFORM_RESOURCE.equals(uri.segment(0));
	}

	/**
	 * Does the file specified by the uri exist.  (Supports both file and
	 * platform URIs.)
	 * @param uri
	 * @return
	 */
	public static boolean exists(URI uri) {
		File file = getFileFromURI(uri);
		return file.exists();
	}
	
	/**
	 * This method returns a string representation of a platform URI
	 * @param uri
	 * @return a string of the form "/project/path", given an EMF platform URI
	 */
	public static String getPlatformResourcePathFromPlatformURI(URI uri) {
		String scheme = uri.scheme();
		StringBuffer platformResourcePath = new StringBuffer();
		if ("platform".equals(scheme) && uri.segmentCount() > 1 && "resource".equals(uri.segment(0)))
		{
			for (int i = 1, size = uri.segmentCount(); i < size; ++i)
			{
				platformResourcePath.append('/');
				platformResourcePath.append(URI.decode(uri.segment(i)));
			}
		}
		return platformResourcePath.toString();
	}

	/**
	 * This method returns a java File object given an EMF URI (either file
	 * URI or Platform URI.) 
	 * @param uri
	 * @return
	 */
	public static File getFileFromURI(URI uri) {
		File file = null;
		if (isPlatformResourceURI(uri)) {
			String platformResourcePath = getPlatformResourcePathFromPlatformURI(uri);
			IFile ifile = ResourcesPlugin.getWorkspace().getRoot().getFile(new
				    Path(platformResourcePath));
			if (ifile != null) {
				IPath path = ifile.getLocation();
				if (path != null) {
					file = ifile.getLocation().toFile();
				}						 			
			}			
		}
		else
			file = getFileFromFileURI(uri);
		return file;
	}
	
	protected static File getFileFromFileURI(URI uri) {
		File file = new File(uri.toFileString().substring("file:".length()));
		return file;
	}

	/**
	 * @param uri
	 * @throws CoreException 
	 */
	public static void refreshLocal(URI uri) throws IOException {
		IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
		IFile file = root.getFile(new Path(EMFWorkspaceUtil.getPlatformResourcePathFromPlatformURI(uri)));
		if (file != null) {
			try {
				file.refreshLocal(IResource.DEPTH_ZERO,
						null);
			} catch (CoreException e) {
				IOException ex = new IOException();
				ex.initCause(e);
				throw ex;
			}
		}								
	}
	
}
