/**********************************************************************
 * Copyright (c) 2003, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 *
 * $Id: LongToObjectMapImpl.java,v 1.3 2008/01/24 02:28:17 apnan Exp $
 **********************************************************************/
package org.eclipse.hyades.models.hierarchy.util.internal;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.eclipse.hyades.models.hierarchy.util.LongToObjectMap;
import org.eclipse.hyades.models.hierarchy.util.PerfUtil;
import org.eclipse.hyades.models.util.ModelDebugger;

/**
 * A dual map with (non-zero) long as key, long -> Object (handles also Map interface, converts the key to hashCode).
 * 
 * @author Alex Nan (apnan@ca.ibm.com)
 * @author Marius Slavescu (slavescu@ca.ibm.com)
 * @since 4.1
 * 
 */
public class LongToObjectMapImpl implements LongToObjectMap {

	public static final float GROWTH_FACTOR = SizesAsPrimes.GROWTH_FACTOR; //2.0f; //

	public static final long MASK = 0x7FFFFFFFFFFFFFFFL;

	public static final long MISSING_KEY = 0;

	public static final Object MISSING_VALUE = null;
	protected PerfUtil p = PerfUtil.createInstance();
	
	public static void main(String[] args) {
		int nrKeys = 25;
		LongToObjectMap longMap = new LongToObjectMapImpl();
		for (int i = 1; i <= nrKeys; i++) {
			//		for (int i = nrKeys; i >0; i--) {
			try {
				longMap.put(i, new Integer(i));
			} catch (Exception e) {
				System.out.println("i=" + i);
				e.printStackTrace();
				System.exit(-1);
			}
		}
		System.out.println(longMap);
		longMap.remove(10);
		longMap.remove(5);
		System.out.println(longMap);
		System.out.println("List keys:");
		for (int i = 1; i <= nrKeys; i++) {
			try {
				System.out.println(i + " -> " + longMap.get(i));
				System.out.println(i + " -> " + longMap.get(new Integer(i)));
			} catch (Exception e) {
				System.out.println("i=" + i);
				e.printStackTrace();
				System.exit(-1);
			}
		}
		longMap.put(new Integer(10), new Integer(10));
		longMap.put(new Integer(5), new Integer(5));
		System.out.println(longMap);

		Set set = longMap.entrySet();
		System.out.println("List entrySet.size=" + set.size() + ", entries:");
		for (Iterator iter = set.iterator(); iter.hasNext();) {
			Map.Entry element = (Map.Entry) iter.next();
			System.out.println(element.getKey() + " -> " + element.getValue());
		}
		set = longMap.keySet();
		System.out.println("List keySet.size=" + set.size() + ", entries:");
		for (Iterator iter = set.iterator(); iter.hasNext();) {
			Object element = iter.next();
			System.out.println(element + " -> " + longMap.get(element));
			System.out.println(element.hashCode() + " -> " + longMap.get(element.hashCode()));
		}

		longMap.clear();
		nrKeys = 10000000;
		PerfUtil p = PerfUtil.createInstance();
		p.setDebug(true);
		p.setMessageAndStart("LongToObjectMap generate nrKeys=" + nrKeys);
		Integer[] keys = new Integer[nrKeys + 1];
		//		Integer sigleValue = new Integer(nrKeys);
		for (int i = nrKeys + 1; i-- > 1;) {
			keys[i] = new Integer(i);
		}
		p.stopAndPrintStatus();
		for (int j = 0; j < 10; j++) {
			System.gc();
			try {
				Thread.sleep(3000);
			} catch (InterruptedException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			}
			System.out.println("LongToObjectMapImpl.main() - j=" + j);
			p.setMessageAndStart("LongToObjectMap add long descending nrKeys=" + nrKeys);
			for (int i = nrKeys + 1; i-- > 1;) {
				longMap.put(i, keys[i]);
				if ((nrKeys - i + 1) != longMap.size())
					System.out.println("Error at key=" + i);
			}
			p.stopAndPrintStatus("keyTable.length=" + longMap.getKeys().length);
			longMap.clear();
			p.setMessageAndStart("LongToObjectMap add long ascending nrKeys=" + nrKeys);
			nrKeys++;
			for (int i = 1; i < nrKeys; i++) {
				longMap.put(i, keys[i]);
			}
			p.stopAndPrintStatus("keyTable.length=" + longMap.getKeys().length);
			nrKeys--;
			p.setMessageAndStart("LongToObjectMap get long descending nrKeys=" + nrKeys);
			for (int i = nrKeys + 1; i-- > 1;) {

				Object v = longMap.get(i);
				try {
					if (v == null || ((Integer) v).intValue() != i)
						System.out.println("Invalid entry at: " + i);
				} catch (Exception e) {
					System.err.println("Invalid entry at: " + i);
					e.printStackTrace();
				}
			}
			p.stopAndPrintStatus("keyTable.length=" + longMap.getKeys().length);
			p.setMessageAndStart("LongToObjectMap remove and compact long descending nrKeys=" + nrKeys);
			for (int i = nrKeys + 1; i-- > 1;) {
				longMap.remove(i);
			}
			longMap.compact();
			p.stopAndPrintStatus("keyTable.length=" + longMap.getKeys().length);

			Map hMap = new HashMap(89);
			p.setMessageAndStart("HashMap add descending nrKeys=" + nrKeys);
			for (int i = nrKeys + 1; i-- > 1;) {
				hMap.put(keys[i], keys[i]);
			}
			p.stopAndPrintStatus();
			hMap.clear();
			p.setMessageAndStart("HashMap add ascending nrKeys=" + nrKeys);
			nrKeys++;
			for (int i = 1; i < nrKeys; i++) {
				hMap.put(keys[i], keys[i]);
			}
			p.stopAndPrintStatus();
			nrKeys--;
			p.setMessageAndStart("HashMap get descending nrKeys=" + nrKeys);
			for (int i = nrKeys + 1; i-- > 1;) {
				hMap.get(keys[i]);
			}
			p.stopAndPrintStatus();
			p.setMessageAndStart("HashMap remove descending nrKeys=" + nrKeys);
			for (int i = nrKeys + 1; i-- > 1;) {
				hMap.remove(keys[i]);
			}
			p.stopAndPrintStatus();

			p.setMessageAndStart("LongToObjectMap add object descending nrKeys=" + nrKeys);
			for (int i = nrKeys + 1; i-- > 1;) {
				longMap.put(keys[i], keys[i]);
			}
			p.stopAndPrintStatus("keyTable.length=" + longMap.getKeys().length);
			longMap.clear();
			p.setMessageAndStart("LongToObjectMap add object ascending nrKeys=" + nrKeys);
			nrKeys++;
			for (int i = 1; i < nrKeys; i++) {
				longMap.put(keys[i], keys[i]);
			}
			p.stopAndPrintStatus();
			nrKeys--;
			p.setMessageAndStart("LongToObjectMap get object descending nrKeys=" + nrKeys);
			for (int i = nrKeys + 1; i-- > 1;) {
				longMap.get(keys[i]);
			}
			p.stopAndPrintStatus("keyTable.length=" + longMap.getKeys().length);
			p.setMessageAndStart("LongToObjectMap remove and compact object descending nrKeys=" + nrKeys);
			for (int i = nrKeys + 1; i-- > 1;) {
				longMap.remove(keys[i]);
			}
			longMap.compact();
			p.stopAndPrintStatus("keyTable.length=" + longMap.getKeys().length);
		}
	}

	protected int elementSize; // number of elements in the table

	protected int putIterations;

	protected long[] keyTable;

	protected int threshold;

	protected Object[] valueTable;

	protected int getIterations;

	public LongToObjectMapImpl() {
		init(13); //default 13
	}

	public LongToObjectMapImpl(int size) {
		init(size); // recomended value is 89
	}

	public void clear() {
		init(13);
	}

	/**
	 * Removes missing keys and values, and resize the arrays
	 */
	public int compact() {
		long[] kt = keyTable;
		Object[] vt = valueTable;
		int newElementSize = 0;
		for (int i = kt.length; i-- > 0;) {
			if (kt[i] != MISSING_KEY && vt[i] != MISSING_VALUE)
				newElementSize++;
		}
		elementSize = newElementSize;
		rehash();
		return newElementSize;
	}

	protected int computeNewSize(int oldSize) {
		int extraRoom = (int) (oldSize * GROWTH_FACTOR);
		if (threshold == extraRoom)
			extraRoom++;
//		if (extraRoom % 2 == 0)
//			extraRoom += 3;
		extraRoom = SizesAsPrimes.getSize(GROWTH_FACTOR,extraRoom);

		return extraRoom;
	}

	public boolean containsKey(long key) {
		if(key == MISSING_KEY) // adjustment for missing key case, it should't happen
			key--;
		int index = getIndex(key);
		return index == -1 ? false : true;
	}

	public boolean containsKey(Object key) {
		if (key == null)
			return false;
		return containsKey(key.hashCode());
	}

	public boolean containsValue(Object value) {
		if (value == MISSING_VALUE)
			return false;
		long[] kt = keyTable;
		Object[] vt = valueTable;
		for (int i = vt.length; i-- > 0;) {
			if (kt[i] != MISSING_KEY && vt[i] == value)
				return true;
		}
		return false;
	}

	public Set entrySet() {
		Set entries = new HashSet();
		for (int i = 0; i < keyTable.length; i++) {
			if (keyTable[i] != MISSING_KEY) {
				final long currentKey = keyTable[i];
				Map.Entry entry = new Map.Entry() {
					Object key;

					Object value;

					public Object getKey() {
						return key;
					}

					public Object getValue() {
						return value;
					}

					public Object setValue(Object value) {
						if (key != null)
							throw new UnsupportedOperationException();
						this.value = value;
						this.key = new Long(currentKey);
						return value;
					}
				};
				entry.setValue(valueTable[i]);
				entries.add(entry);
			}
		}
		return entries;
	}

	public Object get(long key) {
		if(key == MISSING_KEY) // adjustment for missing key case, it should't happen
			key--;
		int index = getIndex(key);
		if (index != -1) {
			return valueTable[index];
		}
		return MISSING_VALUE;
	}

	public Object get(Object key) {
		if (key == null)
			return MISSING_VALUE;
		return get(key.hashCode());
	}

	/**
	 * Finds the next available spot for insertion
	 * @return If the returned value is negative then it is an update
	 */
	protected int getFreeSpot(long key, long[] kt) {
		int length = kt.length;
		int index = (int) ((key & MASK) % length);
		long currentKey;
		while ((currentKey = kt[index]) != MISSING_KEY) {
			if (currentKey == key) {
				return -index;
			}
			index = (index + 1) % length;
			putIterations++;
		}
		return index;
	}

	protected int getIndex(long key) {
		int length = keyTable.length;
		int index = (int) ((key & MASK) % length);
		long currentKey;
		long[] kt = keyTable;
		while ((currentKey = kt[index]) != MISSING_KEY) {
			if (currentKey == key)
				return index;
			index = (index + 1) % length;
			getIterations++;
		}
		return -1;
	}

	public long[] getKeys() {
		return keyTable;
	}

	public Object[] getValues() {
		return valueTable;
	}

	protected void init(int size) {
		elementSize = 0;
		threshold = size; // size represents the expected number of elements
		int extraRoom = computeNewSize(size);

		keyTable = new long[extraRoom];
		valueTable = new Object[extraRoom];
	}

	public boolean isEmpty() {
		return elementSize == 0;
	}

	public Set keySet() {
		Set keys = new HashSet();
		for (int i = 0; i < keyTable.length; i++) {
			if (keyTable[i] != MISSING_KEY)
				keys.add(new Long(keyTable[i]));
		}
		return keys;
	}

	public Object put(long key, Object value) {
		if(key == MISSING_KEY) // adjustment for missing key case, it should't happen
			key--;
		int index = getFreeSpot(key, keyTable);
		if (index < 0) {
			index = -index;
			Object oldValue = valueTable[index];
			valueTable[index] = value;
			return oldValue;
		}
		keyTable[index] = key;
		Object oldValue = valueTable[index];
		valueTable[index] = value;

		// assumes the threshold is never equal to the size of the table
		if (++elementSize > threshold)
			rehash();
		return oldValue;
	}

	public Object put(Object key, Object value) {
		if (key == null)
			return value;
		return put(key.hashCode(), value);
	}

	public void putAll(Map t) {
		for (Iterator iter = entrySet().iterator(); iter.hasNext();) {
			Map.Entry element = (Map.Entry) iter.next();
			put(element.getKey(), element.getValue());
		}
	}

	protected void rehash() {
		if(ModelDebugger.INSTANCE.debugCustomMaps)
			p.setMessageAndStart("IntToObjectMapImpl.rehash() firstValueType="+getFirstValueTypeName()+", elementSize=" + elementSize + ", oldGetIterations=" + getIterations + ", oldPutIterations=" + putIterations + ", oldKeyTable.lenght=" + keyTable.length);

		threshold = (int) (elementSize * GROWTH_FACTOR);
		int extraRoom = computeNewSize(threshold);
		getIterations = 0;
		putIterations = 0;
		rehash(extraRoom);
		if(ModelDebugger.INSTANCE.debugCustomMaps)		
			p.stopAndPrintStatus("newGetIterations=" + getIterations + ", newPutIterations=" + putIterations + ", newKeyTable.lenght=" + keyTable.length);
		putIterations = 0;
		getIterations = 0;
	}

	protected String getFirstValueTypeName() {
		if(ModelDebugger.INSTANCE.debug)
		{		
			for (int i = 0; i < valueTable.length; i++) {
				if(valueTable[i]!=MISSING_VALUE)
					return valueTable[i].getClass().getName();
			}
			return "null";
		}
		else
			return "";
	}

	/**
	 * Rehash will throw away the entries with MISSING_KEY and MISSING_VALUE
	 * @param newSize
	 */
	protected void rehash(int newSize) {
		long[] oKT = keyTable;
		Object[] oVT = valueTable;
		int oL = oKT.length;

		long[] nKT = new long[newSize];
		Object[] nVT = new Object[newSize];
		elementSize = 0;
		for (int i = oL; i-- > 0;) {
			if (oKT[i] != MISSING_KEY && oVT[i] != MISSING_VALUE) {
				long o = oKT[i];
				int index = getFreeSpot(o, nKT);
				if (index < 0) {
					index = -index;
				}
				nKT[index] = o;
				nVT[index] = oVT[i];
				elementSize++;
			}
		}

		keyTable = nKT;
		valueTable = nVT;
	}

//	protected void rehashKey(int key) {
//		// TODO Auto-generated method stub
//
//	}

	/**
	 * remove will just set the value on MISSING_VALUE
	 */
	public Object remove(long key) {
		if(key == MISSING_KEY) // adjustment for missing key case, it should't happen
			key--;
		int index = getIndex(key);
		if (index != -1) {
			Object oldValue = valueTable[index];
			//				keyTable[index]=MISSING_KEY;
			valueTable[index] = MISSING_VALUE;
			//				elementSize--;
			//				rehashKey(key);
			return oldValue;
		}
		return MISSING_VALUE;
	}

	public Object remove(Object key) {
		if (key == null)
			return MISSING_VALUE;
		return remove(key.hashCode());
	}

	public int size() {
		return elementSize;
	}

	public String toString() {
		String s = "LongToObjectMapImpl elementSize=" + elementSize + ", keyTable.length=" + keyTable.length + ", entries:\n"; //$NON-NLS-1$
		Object object;
		for (int i = 0, length = keyTable.length; i < length; i++)
			if (keyTable[i] != MISSING_KEY && (object = valueTable[i]) != MISSING_VALUE)
				s += "" + keyTable[i] + " -> " + object + "\n"; //$NON-NLS-2$ //$NON-NLS-1$
		return s;
	}

	public Collection values() {
		return Arrays.asList(valueTable);
	}
}
