/**********************************************************************
 * Copyright (c) 2003, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 *
 * $Id: ObjectToIntMapImpl.java,v 1.1 2007/11/27 04:28:39 apnan Exp $
 **********************************************************************/
package org.eclipse.hyades.models.hierarchy.util.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.eclipse.hyades.models.hierarchy.util.ObjectToIntMap;
import org.eclipse.hyades.models.hierarchy.util.PerfUtil;
import org.eclipse.hyades.models.util.ModelDebugger;

/**
 * A dual map with values as int >= 0 and Object as keys, Object -> int (handles also Map interface, converts the value to Integer).
 * 
 * @author Marius Slavescu (slavescu@ca.ibm.com)
 * @author Alex Nan (apnan@ca.ibm.com)
 * @since 4.1
 * 
 */
public class ObjectToIntMapImpl implements ObjectToIntMap {
	protected static final float GROWTH_FACTOR = SizesAsPrimes.GROWTH_FACTOR;//2.0f; //

	public static final int MASK = 0x7fffffff;

	protected PerfUtil p = PerfUtil.createInstance();
	
	public static void main(String[] args) {
		int nrKeys = 25;
		ObjectToIntMap intMap = new ObjectToIntMapImpl();
		for (int i = 1; i <= nrKeys; i++) {
			//		for (int i = nrKeys; i >0; i--) {
			try {
				intMap.put(new Integer(i),i);
			} catch (Exception e) {
				System.out.println("i=" + i);
				e.printStackTrace();
				System.exit(-1);
			}
		}
		System.out.println(intMap);
		intMap.remove(new Integer(10));
		intMap.remove(new Integer(5));
		System.out.println(intMap);
		System.out.println("List keys:");
		for (int i = 1; i <= nrKeys; i++) {
			try {
				System.out.println(i + " -> " + intMap.get(new Integer(i)));
			} catch (Exception e) {
				System.out.println("i=" + i);
				e.printStackTrace();
				System.exit(-1);
			}
		}
		intMap.put(new Integer(10), 10);
		intMap.put(new Integer(5), 5);
		System.out.println(intMap);

		Set set = intMap.entrySet();
		System.out.println("List entrySet.size=" + set.size() + ", entries:");
		for (Iterator iter = set.iterator(); iter.hasNext();) {
			Map.Entry element = (Map.Entry) iter.next();
			System.out.println(element.getKey() + " -> " + element.getValue());
		}
		set = intMap.keySet();
		System.out.println("List keySet.size=" + set.size() + ", entries:");
		for (Iterator iter = set.iterator(); iter.hasNext();) {
			Object element = iter.next();
			System.out.println(element + " -> " + intMap.get(element));
//			System.out.println(element.hashCode() + " -> " + intMap.get(element));
		}

		intMap.clear();
		nrKeys = 10000000;
		PerfUtil p = PerfUtil.createInstance();
		p.setDebug(true);
		p.setMessageAndStart("ObjectToIntMap generate nrKeys=" + nrKeys);
		Integer[] keys = new Integer[nrKeys + 1];
		//		Integer sigleValue = new Integer(nrKeys);
		for (int i = nrKeys + 1; i-- > 1;) {
			keys[i] = new Integer(i);
		}
		p.stopAndPrintStatus();
		for (int j = 0; j < 10; j++) {
			System.gc();
			try {
				Thread.sleep(3000);
			} catch (InterruptedException e1) {
				e1.printStackTrace();
			}
			System.out.println("ObjectToIntMapImpl.main() - j=" + j);
			p.setMessageAndStart("ObjectToIntMap add int descending nrKeys=" + nrKeys);
			for (int i = nrKeys + 1; i-- > 1;) {
				intMap.put(keys[i],i);
				if ((nrKeys - i + 1) != intMap.size())
					System.out.println("Error at key=" + i);
			}
			p.stopAndPrintStatus("keyTable.length=" + intMap.getKeys().length);
			intMap.clear();
			p.setMessageAndStart("ObjectToIntMap add int ascending nrKeys=" + nrKeys);
			nrKeys++;
			for (int i = 1; i < nrKeys; i++) {
				intMap.put(keys[i],i);
			}
			p.stopAndPrintStatus("keyTable.length=" + intMap.getKeys().length);
			nrKeys--;
			p.setMessageAndStart("ObjectToIntMap get int descending nrKeys=" + nrKeys);
			for (int i = nrKeys + 1; i-- > 1;) {

				int v = intMap.getInt(keys[i]);
				try {
					if (v == -1)
						System.out.println("Invalid entry at: " + i);
				} catch (Exception e) {
					System.err.println("Invalid entry at: " + i);
					e.printStackTrace();
				}
			}
			p.stopAndPrintStatus("keyTable.length=" + intMap.getKeys().length);
			p.setMessageAndStart("ObjectToIntMap remove and compact int descending nrKeys=" + nrKeys);
			for (int i = nrKeys + 1; i-- > 1;) {
				intMap.removeInt(keys[i]);
			}
			intMap.compact();
			p.stopAndPrintStatus("keyTable.length=" + intMap.getKeys().length);

			Map hMap = new HashMap(89);
			p.setMessageAndStart("HashMap add descending nrKeys=" + nrKeys);
			for (int i = nrKeys + 1; i-- > 1;) {
				hMap.put(keys[i], keys[i]);
			}
			p.stopAndPrintStatus();
			hMap.clear();
			p.setMessageAndStart("HashMap add ascending nrKeys=" + nrKeys);
			nrKeys++;
			for (int i = 1; i < nrKeys; i++) {
				hMap.put(keys[i], keys[i]);
			}
			p.stopAndPrintStatus();
			nrKeys--;
			p.setMessageAndStart("HashMap get descending nrKeys=" + nrKeys);
			for (int i = nrKeys + 1; i-- > 1;) {
				hMap.get(keys[i]);
			}
			p.stopAndPrintStatus();
			p.setMessageAndStart("HashMap remove descending nrKeys=" + nrKeys);
			for (int i = nrKeys + 1; i-- > 1;) {
				hMap.remove(keys[i]);
			}
			p.stopAndPrintStatus();

			p.setMessageAndStart("ObjectToIntMap add object descending nrKeys=" + nrKeys);
			for (int i = nrKeys + 1; i-- > 1;) {
				intMap.put(keys[i], keys[i]);
			}
			p.stopAndPrintStatus("keyTable.length=" + intMap.getKeys().length);
			intMap.clear();
			p.setMessageAndStart("ObjectToIntMap add object ascending nrKeys=" + nrKeys);
			nrKeys++;
			for (int i = 1; i < nrKeys; i++) {
				intMap.put(keys[i], keys[i]);
			}
			p.stopAndPrintStatus();
			nrKeys--;
			p.setMessageAndStart("ObjectToIntMap get object descending nrKeys=" + nrKeys);
			for (int i = nrKeys + 1; i-- > 1;) {
				intMap.get(keys[i]);
			}
			p.stopAndPrintStatus("keyTable.length=" + intMap.getKeys().length);
			p.setMessageAndStart("ObjectToIntMap remove and compact object descending nrKeys=" + nrKeys);
			for (int i = nrKeys + 1; i-- > 1;) {
				intMap.remove(keys[i]);
			}
			intMap.compact();
			p.stopAndPrintStatus("keyTable.length=" + intMap.getKeys().length);
		}
	}

	protected int elementSize; // number of elements in the table

	protected int getIterations;

	protected int putIterations;

	protected Object[] keyTable;

	protected int threshold;

	protected int[] valueTable;

	public ObjectToIntMapImpl() {
		init(13); //default 13
	}

	public ObjectToIntMapImpl(int size) {
		init(size); // recomended value is 89
	}

	public void clear() {
		init(13);
	}

	/**
	 * Removes missing keys and values, and resize the arrays
	 */
	public int compact() {
		Object[] kt = keyTable;
		int[] vt = valueTable;
		int newElementSize = 0;
		for (int i = kt.length; i-- > 0;) {
			if (kt[i] != MISSING_KEY && vt[i] != MISSING_VALUE)
				newElementSize++;
		}
		elementSize = newElementSize;
		rehash();
		return newElementSize;
	}

	protected int computeNewSize(int size) {
		int extraRoom = (int) (size * GROWTH_FACTOR);
		if (threshold == extraRoom)
			extraRoom++;
//		if (extraRoom % 2 == 0)
//			extraRoom += 3;
		extraRoom = SizesAsPrimes.getSize(GROWTH_FACTOR,extraRoom);
		return extraRoom;
	}

	public boolean containsKey(Object key) {
		if (key == MISSING_KEY)
			return false;
		int index = getIndex(key);
		return index == -1 ? false : true;
	}

	public boolean containsValue(Object value) {
		if (value == null)
			return false;
		Object[] kt = keyTable;
		int[] vt = valueTable;
		for (int i = vt.length; i-- > 0;) {
			if (kt[i] != MISSING_KEY && new Integer(vt[i]).equals(value))
				return true;
		}
		return false;
	}

	public Set entrySet() {
		Set entries = new HashSet();
		for (int i = 0; i < keyTable.length; i++) {
			if (keyTable[i] != MISSING_KEY) {
				final Object currentKey = keyTable[i];
				Map.Entry entry = new Map.Entry() {
					Object key;

					Object value;

					public Object getKey() {
						return key;
					}

					public Object getValue() {
						return value;
					}

					public Object setValue(Object value) {
						if (key != null)
							throw new UnsupportedOperationException();
						this.value = value;
						this.key = currentKey;
						return value;
					}
				};
				entry.setValue(new Integer(valueTable[i]));
				entries.add(entry);
			}
		}
		return entries;
	}

	public int getInt(Object key) {
		int index = getIndex(key);
		if (index != -1) {
			return valueTable[index];
		}
		return MISSING_VALUE;
	}

	public Object get(Object key) {
		if (key == null)
			return null;
		return new Integer(getInt(key));
	}

	/**
	 * Finds the next available spot for insertion
	 * @return If the returned value is negative then it is an update
	 */
	protected int getFreeSpot(Object key, Object[] kt) {
		int length = kt.length;
		int index = (key.hashCode() & MASK) % length;
		Object currentKey;
		while ((currentKey = kt[index]) != MISSING_KEY) {
			if (currentKey == key) {
				return -index;
			}
			index = (index + 1) % length;
			putIterations++;
		}
		return index;
	}

	protected int getIndex(Object key) {
		Object[] kt = keyTable;
		int length = kt.length;
		int index = (key.hashCode() & MASK) % length;
		Object currentKey;
		while ((currentKey = kt[index]) != MISSING_KEY) {
			if (currentKey == key || (currentKey !=null && currentKey.equals(key)))
				return index;
			index = (index + 1) % length;
			getIterations++;
		}
		return -1;
	}

	public Object[] getKeys() {
		return keyTable;
	}

	public int[] getValues() {
		return valueTable;
	}

	protected void init(int size) {
		elementSize = 0;
		threshold = size; // size represents the expected number of elements
		int extraRoom = computeNewSize(size);

		valueTable = new int[extraRoom];
		keyTable = new Object[extraRoom];
	}

	public boolean isEmpty() {
		return elementSize == 0;
	}

	public Set keySet() {
		Set keys = new HashSet();
		for (int i = 0; i < keyTable.length; i++) {
			if (keyTable[i] != MISSING_KEY)
				keys.add(keyTable[i]);
		}
		return keys;
	}

	public int put(Object key, int value) {

		int index = getFreeSpot(key, keyTable);
		if (index < 0) {
			index = -index;
			int oldValue = valueTable[index];
			valueTable[index] = value;
			return oldValue;
		}
		keyTable[index] = key;
		int oldValue = valueTable[index];
		valueTable[index] = value;

		// assumes the threshold is never equal to the size of the table
		if (++elementSize > threshold)
			rehash();
		return oldValue;
	}

	public Object put(Object key, Object value) {
		if (key == null)
			return value;
		return new Integer(put(key, value.hashCode()));
	}

	public void putAll(Map t) {
		for (Iterator iter = entrySet().iterator(); iter.hasNext();) {
			Map.Entry element = (Map.Entry) iter.next();
			put(element.getKey(), element.getValue());
		}
	}

	protected void rehash() {
		if(ModelDebugger.INSTANCE.debugCustomMaps)
			p.setMessageAndStart("ObjectToIntMapImpl.rehash() oldGetIterations=" + getIterations + ", oldPutIterations=" + putIterations + ", oldElementSize=" + elementSize + ", oldKeyTable.lenght=" + keyTable.length);

		threshold = (int) (elementSize * GROWTH_FACTOR);
		int extraRoom = computeNewSize(threshold);
		getIterations = 0;
		putIterations = 0;
		rehash(extraRoom);
		if(ModelDebugger.INSTANCE.debugCustomMaps)
			p.stopAndPrintStatus("newGetIterations=" + getIterations + ", newPutIterations=" + putIterations + ", newElementSize=" + elementSize + ", newKeyTable.lenght=" + keyTable.length);
		putIterations = 0;
		getIterations = 0;
	}

	/**
	 * Rehash will throw away the entries with MISSING_KEY and MISSING_VALUE
	 * @param newSize
	 */
	protected void rehash(int newSize) {
		Object[]oKT = keyTable;
		int[] oVT = valueTable;
		int oL = oKT.length;

		Object[] nKT = new Object[newSize];
		 int[] nVT = new int[newSize];
		int nElementSize = 0;
		for (int i = oL; i-- > 0;) {
			if (oKT[i] != MISSING_KEY && oVT[i] != MISSING_VALUE) {
				Object o = oKT[i];
				int index = getFreeSpot(o, nKT);
				if (index < 0) {
					index = -index;
				}
				nKT[index] = o;
				nVT[index] = oVT[i];
				nElementSize++;
			}
		}
		elementSize = nElementSize;
		keyTable = nKT;
		valueTable = nVT;
	}

//	protected void rehashKey(int key) {
//		// TODO Auto-generated method stub
//
//	}

	/**
	 * remove will just set the value on MISSING_VALUE
	 */
	public int removeInt(Object key) {
		int index = getIndex(key);
		if (index != -1) {
			int oldValue = valueTable[index];
			//				keyTable[index]=MISSING_KEY;
			valueTable[index] = MISSING_VALUE;
			//				elementSize--;
			//				rehashKey(key);
			return oldValue;
		}
		return MISSING_VALUE;
	}

	public Object remove(Object key) {
		if (key == null)
			return null;
		return new Integer(removeInt(key));
	}

	public int size() {
		return elementSize;
	}

	public String toString() {
		String s = "ObjectToIntMapImpl elementSize=" + elementSize + ", keyTable.length=" + keyTable.length + ", entries:\n"; //$NON-NLS-1$
		int object;
		for (int i = 0, length = keyTable.length; i < length; i++)
			if (keyTable[i] != MISSING_KEY && (object = valueTable[i]) != MISSING_VALUE)
				s += "" + keyTable[i] + " -> " + object + "\n"; //$NON-NLS-2$ //$NON-NLS-1$
		return s;
	}

	public Collection values() {
		ArrayList res = new ArrayList();
		for (int i = 0; i < valueTable.length; i++) {
			res.add(new Integer(valueTable[i]));
		}
		return res;
	}
}
