/**********************************************************************
 * Copyright (c) 2003, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 *
 * $Id: QueryUtils.java,v 1.3 2010/07/27 02:06:59 jwest Exp $
 **********************************************************************/
package org.eclipse.hyades.models.hierarchy.util.internal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.hierarchy.extensions.BinaryExpression;
import org.eclipse.hyades.models.hierarchy.extensions.ExtensionsFactory;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.Operand;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleOperand;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.hierarchy.extensions.TimeBasedCorrelationQuery;
import org.eclipse.hyades.models.hierarchy.extensions.WhereExpression;
import org.eclipse.hyades.models.hierarchy.util.ContainmentTraverser;
/**
 * @author Marius Slavescu (slavescu@ca.ibm.com)
 * @since 3.3
 *  
 */
public class QueryUtils {
	public static final String DOLLAR_SIGN = "$";
	private static Map<EClass, Set<EClass>> _subTypesIndex = new HashMap<EClass, Set<EClass>>();
	
	
	public static SimpleSearchQuery removeClasses(SimpleSearchQuery simpleSearchQuery, final Set<EClass> classes) {
		SimpleSearchQuery query = EcoreUtil.copy(simpleSearchQuery);
		final Map<EObject, Object> alreadyProcessed = new HashMap<EObject, Object>();
		final Set<EObject> removeSet = new HashSet<EObject>();
		List<WhereExpression> roots = new ArrayList<WhereExpression>();
		
		roots.add(query.getWhereExpression());
		ContainmentTraverser containmentTraverser = new ContainmentTraverser(roots) {
			protected boolean beforeChildren(EObject element) {
				String name = getElementName(element);
				if (name != null && name.startsWith(DOLLAR_SIGN))
					return super.beforeChildren(element);
				if (alreadyProcessed.containsKey(element))
					return super.beforeChildren(element);
				// initialize evaluation result
				alreadyProcessed.put(element, null);
				if (element instanceof SimpleOperand) {
					SimpleOperand o = (SimpleOperand) element;
					EClass c = o.getType();
					if (c == null && o.getFeature() != null) {
						c = o.getFeature().getEContainingClass();
					}
					if (c != null && classes.contains(c)) {
						BinaryExpression be = getParentBinaryExpression(o);
						if (be != null)
							removeSet.add(be);
					}
				}
				return super.beforeChildren(element);
			}
		};
		containmentTraverser.traverse();
		removeElementsAndCompactQuery(query, removeSet);
		return query;
	}
	public static SimpleSearchQuery removeFeatures(SimpleSearchQuery simpleSearchQuery, final Set features, final boolean include) {
		SimpleSearchQuery query = EcoreUtil.copy(simpleSearchQuery);
		final Map<EObject, Object> alreadyProcessed = new HashMap<EObject, Object>();
		final Set<EObject> removeSet = new HashSet<EObject>();
		List<WhereExpression> roots = new ArrayList<WhereExpression>();
		roots.add(query.getWhereExpression());
		ContainmentTraverser containmentTraverser = new ContainmentTraverser(roots) {
			protected boolean beforeChildren(EObject element) {
				String name = getElementName(element);
				if (name != null && name.startsWith(DOLLAR_SIGN))
					return super.beforeChildren(element);
				if (alreadyProcessed.containsKey(element))
					return super.beforeChildren(element);
				// initialize evaluation result
				alreadyProcessed.put(element, null);
				if (element instanceof SimpleOperand) {
					SimpleOperand o = (SimpleOperand) element;
					EStructuralFeature f = o.getFeature();
					if (f != null)
					{
						if(features.contains(f)) {
							if(include)
							{
								BinaryExpression be = getParentBinaryExpression(o);
								if (be != null)
									removeSet.add(be);
							}
						}
						else if(!include)
						{
							BinaryExpression be = getParentBinaryExpression(o);
							if (be != null)
								removeSet.add(be);
						}
					}
				}
				return super.beforeChildren(element);
			}
		};
		containmentTraverser.traverse();
		removeElementsAndCompactQuery(query, removeSet);
		return query;
	}
	
	public static BinaryExpression getParentBinaryExpression(EObject o) {
		if (o != null) {
			if (o instanceof BinaryExpression)
				return (BinaryExpression) o;
			else
				return getParentBinaryExpression(o.eContainer());
		}
		return null;
	}
	
	public static void removeElementsAndCompactQuery(SimpleSearchQuery query, Set<EObject> elements) {
		for (Iterator<EObject> iter = elements.iterator(); iter.hasNext();) {
			EObject element = iter.next();
			removeElementAndCompactContainers(element);
		}
	}
	
	public static void removeElementAndCompactContainers(EObject element) {
		if (element == null)
			return;
		EObject container = element.eContainer();
		if (container == null)
			return;
		if (container instanceof LogicalExpression) {
			LogicalExpression le = (LogicalExpression) container;
			le.getArguments().remove(element);
			if (le.getArguments().isEmpty()) {
				removeElementAndCompactContainers(le);
			} else {
				compactWhereExpression(le);
			}
		}
		//		else if (container instanceof SimpleSearchQuery) {
		//			((SimpleSearchQuery) container).setWhereExpression(null);
		//		}
	}
	
	public static void compactWhereExpression(WhereExpression we) {
		if (we instanceof LogicalExpression && ((LogicalExpression) we).getArguments().size() > 0) {
			compactLogicalExpression((LogicalExpression) we);
			return;
		} else {
			compactBinaryExpression((BinaryExpression) we);
		}
	}
	
	public static void compactLogicalExpression(LogicalExpression le) {
		if (le.getArguments().size() == 1 && le.getOperator() == LogicalOperators.NOT_LITERAL) {
			return;
		}
		if (le.getArguments().size() > 1) {
			return;
		}
		EObject container = le.eContainer();
		if (container == null)
			return;
		if (container instanceof LogicalExpression) {
			LogicalExpression c = (LogicalExpression) container;
			if (le.getArguments().size() == 1)
				c.getArguments().set(c.getArguments().indexOf(le), le.getArguments().get(0));
			else
				c.getArguments().remove(le);
			compactLogicalExpression(c);
		}
	}
	
	public static void compactBinaryExpression(BinaryExpression be) {
		if (be.getRightOperands().size() > 0) {
			return;
		}
		EObject container = be.eContainer();
		if (container == null)
			return;
		if (container instanceof LogicalExpression) {
			LogicalExpression c = (LogicalExpression) container;
			c.getArguments().remove(be);
			compactLogicalExpression(c);
		}
	}
	
	public static String getElementName(EObject element) {
		if (element instanceof WhereExpression)
			return ((WhereExpression) element).getName();
		else if (element instanceof Operand)
			return ((Operand) element).getName();
		return element.toString();
	}
	
	public static Set<EClass> getAllSubTypes(EClass c) {
		Set<EClass> s = _subTypesIndex.get(c);
		if (s == null) {
			s = new HashSet<EClass>();
			_subTypesIndex.put(c, s);
			for (Iterator<Map.Entry<String, Object>> iter = EPackage.Registry.INSTANCE.entrySet().iterator(); iter.hasNext();) {
				Map.Entry<String, Object> packageEntry = iter.next();
				if (packageEntry.getValue() == null || !(packageEntry.getValue() instanceof EPackage))
					continue;
				for (Iterator<EClassifier> iterator = ((EPackage) packageEntry.getValue()).getEClassifiers().iterator(); iterator.hasNext();) {
					EClassifier classifier = iterator.next();
					if (classifier instanceof EClass) {
						EClass class1 = (EClass) classifier;
						if (class1.getEAllSuperTypes().contains(c)) {
							s.add(class1);
						}
					}
				}
			}
		}
		return s;
	}
	public static void addSubTypes(EClass c, Map<EClass, Collection<EObject>> m, Collection<EObject> l) {

		// Developers Note: SimpleSearchQueryEngine only seems to ever call this method with a null l - jgw
				
		// For each subtype of c (which are EClasses)
		for (Iterator<EClass> iter = QueryUtils.getAllSubTypes(c).iterator(); iter.hasNext();) {
			EClass element = iter.next();
			
			// Query the given map for the subtype, and then add l to the map as the value
			Collection<EObject> l1 = m.get(element);
			if (l1 != null)
				l1.addAll(l);
			else
				m.put(element, l);
		}
	}
	
	public static void addSuperTypes(EClass c, Map<EClass, Collection<EObject>> m, Collection<EObject> l) {
		
		// Developers Note: SimpleSearchQueryEngine doesn't seem to invoke this method in the cases I've seen so far - jgw		
		
		// For each supertype of c (which are EClasses)
		for (Iterator<EClass> iter = c.getEAllSuperTypes().iterator(); iter.hasNext();) {
			EClass element = iter.next();
			
			// Query the given m for the EClass, and add the elements of l to it as the value
			Collection<EObject> l1 = m.get(element);
			if (l1 != null)
				l1.addAll(l);
			else
				m.put(element, l);
		}
	}
	public static void addSuperTypes(EClass c, Collection<Object> s) {
		for (Iterator<EClass> iter = c.getEAllSuperTypes().iterator(); iter.hasNext();) {
			EClass element = iter.next();
			s.add(element);
		}
	}
	public static void addSubTypes(EClass c, Collection<Object> s) {
		for (Iterator<EClass> iter = QueryUtils.getAllSubTypes(c).iterator(); iter.hasNext();) {
			EClass element = iter.next();
			s.add(element);
		}
	}
	public static Map<EClass, Collection<EObject>> getClassPredicatesIndex(SimpleSearchQuery query) {
		final Map<EObject, Object> whereExpressionValuesIndex = new HashMap<EObject, Object>();
		final Map<EClass, Collection<EObject>> classPredicatesIndex = new HashMap<EClass, Collection<EObject>>();
		final Map<String, EObject> namesIndex = new HashMap<String, EObject>();
		List<WhereExpression> roots = new ArrayList<WhereExpression>();
		roots.add(query.getWhereExpression());
		
		ContainmentTraverser containmentTraverser = new ContainmentTraverser(roots) {
			protected boolean beforeChildren(EObject element) {
				String name = QueryUtils.getElementName(element);
				if (name != null && name.startsWith(QueryUtils.DOLLAR_SIGN))
					return super.beforeChildren(element);
				if (whereExpressionValuesIndex.containsKey(element))
					return super.beforeChildren(element);
				// initialize evaluation result
				whereExpressionValuesIndex.put(element, null);
				if (element instanceof SimpleOperand) {
					SimpleOperand o = (SimpleOperand) element;
					EClass c = o.getType();
					if (c != null) {
						Collection<EObject> l = classPredicatesIndex.get(c);
						if (l == null) {
							l = new HashSet<EObject>();
							classPredicatesIndex.put(c, l);
							addSubTypes(c, classPredicatesIndex, l);
						}
						//						if (!l.contains(element))
						l.add(element);
					} else if (o.getFeature() != null) {
						c = o.getFeature().getEContainingClass();
						Collection<EObject> l = classPredicatesIndex.get(c);
						if (l == null) {
							l = new HashSet<EObject>();
							classPredicatesIndex.put(c, l);
							addSubTypes(c, classPredicatesIndex, l);
						}
						//						if (!l.contains(element))
						l.add(element);
					}
				}
				if (name != null)
					namesIndex.put(name, element);
				return super.beforeChildren(element);
			}
		};
		containmentTraverser.traverse();
		return classPredicatesIndex;
	}
	
	private static Map<EClass, List<EObject>> getClassPredicatesNoSubClassesIndex(SimpleSearchQuery query) {
		
		final Map<EObject, Object> whereExpressionValuesIndex = new HashMap<EObject, Object>();
		final Map<EClass, List<EObject>> classPredicatesIndex = new HashMap<EClass, List<EObject>>();
		final Map<String, EObject> namesIndex = new HashMap<String, EObject>();
		
		List<WhereExpression> roots = new ArrayList<WhereExpression>();
		if (query.getWhereExpression() == null)
			return classPredicatesIndex;
		roots.add(query.getWhereExpression());
		ContainmentTraverser containmentTraverser = new ContainmentTraverser(roots) {
			protected boolean beforeChildren(EObject element) {
				String name = QueryUtils.getElementName(element);
				if (name != null && name.startsWith(QueryUtils.DOLLAR_SIGN))
					return super.beforeChildren(element);
				if (whereExpressionValuesIndex.containsKey(element))
					return super.beforeChildren(element);
				// initialize evaluation result
				whereExpressionValuesIndex.put(element, null);
				if (element instanceof SimpleOperand) {
					SimpleOperand o = (SimpleOperand) element;
					EClass c = o.getType();
					if (c != null) {
						List<EObject> l = classPredicatesIndex.get(c);
						if (l == null) {
							l = new ArrayList<EObject>();
							classPredicatesIndex.put(c, l);
						}
						if (!l.contains(element))
							l.add(element);
					} else if (o.getFeature() != null) {
						c = o.getFeature().getEContainingClass();
						List<EObject> l = classPredicatesIndex.get(c);
						if (l == null) {
							l = new ArrayList<EObject>();
							classPredicatesIndex.put(c, l);
						}
						if (!l.contains(element))
							l.add(element);
					}
				}
				if (name != null)
					namesIndex.put(name, element);
				return super.beforeChildren(element);
			}
		};
		containmentTraverser.traverse();
		return classPredicatesIndex;
	}
	public static void removeClass(SimpleSearchQuery searchQuery, EClass clazz) {
		Set<EClass> s = new HashSet<EClass>();
		s.add(clazz);
		removeClasses(searchQuery, s);
	}
	/**
	 * @return
	 */
	public static SimpleSearchQuery getEmptyQuery() {
		SimpleSearchQuery query = ExtensionsFactory.eINSTANCE.createSimpleSearchQuery();
		query.setName("EmptyFilter");
		LogicalExpression logicalExpression = ExtensionsFactory.eINSTANCE.createLogicalExpression();
		logicalExpression.setOperator(LogicalOperators.AND_LITERAL);
		query.setWhereExpression(logicalExpression);
		return query;
	}
	public static TimeBasedCorrelationQuery getEmptyTimeBasedCorrelationQuery() {
		TimeBasedCorrelationQuery query = ExtensionsFactory.eINSTANCE.createTimeBasedCorrelationQuery();
		query.setName("EmptyFilter");
		LogicalExpression logicalExpression = ExtensionsFactory.eINSTANCE.createLogicalExpression();
		logicalExpression.setOperator(LogicalOperators.AND_LITERAL);
		query.setWhereExpression(logicalExpression);
		return query;
	}
	
	public  static int like(String lhs, String rhs)
	{
		int lhs_pos = 0;
		int rhs_pos = 0;
		int lhs_len = lhs.length();
		int rhs_len = rhs.length();
		int wildcard = -1;
		int curent_lhs_pos = -1;
		while (true) {
			if (rhs_pos < rhs_len && rhs.charAt(rhs_pos) == '*') {
				wildcard = ++rhs_pos;
				curent_lhs_pos = lhs_pos;
			} else if (lhs_pos == lhs_len) {
				if (rhs_pos < rhs_len)
					return -1;
				else
					return rhs_pos == rhs_len ? 0 : 1;
			} else if (rhs_pos < rhs_len && (lhs.charAt(lhs_pos) == rhs.charAt(rhs_pos) || rhs.charAt(rhs_pos) == '?')) {
				lhs_pos = lhs_pos + 1;
				rhs_pos = rhs_pos + 1;
			} else if (wildcard >= 0) {
				lhs_pos = ++curent_lhs_pos;
				rhs_pos = wildcard;
			} else {
				if(rhs_pos==rhs_len)
					return 1;
				else
					return lhs.charAt(lhs_pos) > rhs.charAt(rhs_pos) ? 1 : -1;
			}
		}
	}
	public static void main(String[] args) {
		System.out.println("QueryUtils.like() = "+like("ABD","ABC"));
		System.out.println("QueryUtils.like() = "+like("ABC","ABD"));
		System.out.println("QueryUtils.like() = "+like("ABC","ABC"));
		System.out.println("QueryUtils.like() = "+like("ABC","ABC*"));
		System.out.println("QueryUtils.like() = "+like("ABC*","ABC"));
		System.out.println("QueryUtils.like() = "+like("ABC","ABCD"));
		System.out.println("QueryUtils.like() = "+like("ABCDGH","ABC?G"));
		System.out.println("QueryUtils.like() = "+like("ABCDGHIFFFLMNTLMN","ABC*GHI*LMN"));
	}
}