/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 *
 * $Id: TPTPApplicationManager.java,v 1.6 2008/06/06 21:49:53 jkubasta Exp $
 **********************************************************************/
package org.eclipse.tptp.platform.extensions.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.hyades.loaders.util.HyadesConfigurationElement;
import org.eclipse.hyades.loaders.util.RegistryReader;
import org.eclipse.hyades.models.hierarchy.plugin.ModelsHierarchyPlugin;
import org.eclipse.hyades.models.util.ModelDebugger;
import org.eclipse.tptp.platform.extensions.IApplicationManager;
import org.eclipse.tptp.platform.extensions.IApplicationManagerListener;
import org.eclipse.tptp.platform.extensions.IHandlerElement;

/**
 * 
 * @author slavescu
 * 
 */
public class TPTPApplicationManager implements IApplicationManager {

	protected static final String DEFAULT_HANDLER = "defaultHandler";
	protected Map defaultHandlersByApplicationFunction = new HashMap();
	protected Map handlersByUniqueId = new HashMap();
	protected Map handlersByApplicationFunction = new HashMap();
	protected Set disabledOrReplacedHandlers = new HashSet();
	protected GenericHandlerRegistryReader registryReader;

	public TPTPApplicationManager() {
		init();
	}

	protected void init() {
		readRegistry();
		setDefaults();
	}

	protected void setDefaults() {
		for (Iterator iterator = handlersByApplicationFunction.entrySet().iterator(); iterator.hasNext();) {
			Map.Entry entry = (Map.Entry) iterator.next();
			if (entry.getKey() != null && entry.getValue() != null) {
				String defaultHandlerUniqueId = registryReader.getPluginCustomizationString(ModelsHierarchyPlugin.getPlugin().getBundle().getSymbolicName(),((entry.getKey() + "/" + DEFAULT_HANDLER)).replace('/', '_'));
				List handlers = (List) entry.getValue();
				IHandlerElement maxPriorityHandler = null;
				IHandlerElement defaultHandler = null;
				for (Iterator iter2 = handlers.iterator(); iter2.hasNext();) {
					IHandlerElement handler = (IHandlerElement) iter2.next();
					if (IHandlerElement.DISABLED.equals(handler.getModifier()))
						continue;
					String handlerId = handler.getUniqueID();
					if (disabledOrReplacedHandlers.contains(handlerId)) {
						if (IHandlerElement.REPLACE.equals(handler.getModifier())) {
							handlerId = handler.getTargetId();
							if (handlerId.equals(defaultHandlerUniqueId)) {
								defaultHandler = handler;
								break;
							}
						} else
							continue;
					}
					if (handler.getUniqueID().equals(defaultHandlerUniqueId)) {
						defaultHandler = handler;
						break;
					}
					if (maxPriorityHandler == null || maxPriorityHandler.getPriority() < handler.getPriority())
						maxPriorityHandler = handler;
				}
				IHandlerElement existingHandler = (IHandlerElement) defaultHandlersByApplicationFunction.get(entry.getKey());
				if (defaultHandler != null) {
					if (existingHandler == null || existingHandler.getPriority() < defaultHandler.getPriority())
						defaultHandlersByApplicationFunction.put(entry.getKey(), defaultHandler);
				} else if (maxPriorityHandler != null) {
					if (existingHandler == null || existingHandler.getPriority() < maxPriorityHandler.getPriority())
						defaultHandlersByApplicationFunction.put(entry.getKey(), maxPriorityHandler);
				}
			}
		}

	}

	protected void readRegistry() {
		registryReader = new GenericHandlerRegistryReader();
		registryReader.readRegistry();
	}

	public void addListener(IApplicationManagerListener listener, int targetEventType) {
		// TODO Auto-generated method stub

	}

	public Object createDefaultHandlerInstance(String applicationAndFunction) {
		return createHandlerInstance(getDefaultHandlerElement(applicationAndFunction));
	}

	public Object createHandlerInstance(IHandlerElement handlerElement) {
		
		try {
			return registryReader.createPluginClassDescriptor(((HandlerElementImpl) handlerElement).getConfigurationElement(), IHandlerElement.CLASS).createInstance();	
		} catch (Throwable e) {			
			//e.printStackTrace();
		}
		return null;
	}

	public List findHandlerElements(IHandlerElement templateQuery, boolean defaultsOnly) {
		// TODO add support for other cases
		if(templateQuery!=null && templateQuery.getApplication()!=null && templateQuery.getFunction()!=null)
		{
			List result =  new ArrayList();
			if(templateQuery.getId()!=null)
			{
				result.add(handlersByUniqueId.get(templateQuery.getUniqueID()));
			}
			else
			{
				for (Iterator iterator = ((List)handlersByApplicationFunction.get(templateQuery.getApplication()+"/"+templateQuery.getFunction())).iterator(); iterator.hasNext();) {
					IHandlerElement handler = (IHandlerElement) iterator.next();
					if(handler!=null)
						result.add(handler);
				}
			}
			return result;
		}
		else
			return Arrays.asList(handlersByUniqueId.values().toArray());
	}

	public IHandlerElement getDefaultHandlerElement(String applicationAndFunction) {
		return (IHandlerElement) defaultHandlersByApplicationFunction.get(applicationAndFunction);
	}

	public void removeListener(IApplicationManagerListener listener, int targetEventType) {
		// TODO Auto-generated method stub

	}

	public static IApplicationManager createInstance() {
		return new TPTPApplicationManager();
	}

	/**
	 * A plugin extension reader that populates the loaders registry. Clients
	 * are not expected to use this class directly.
	 */
	class GenericHandlerRegistryReader extends RegistryReader {
		static final String TAG_LOADER = "handler";

		// ~ Constructors
		// -------------------------------------------------------------------------------

		public GenericHandlerRegistryReader() {
			super(ModelsHierarchyPlugin.INSTANCE.getSymbolicName(), RegistryReader.GENERIC_HANDLER_PPID);
		}

		// ~ Methods
		// ------------------------------------------------------------------------------------

		protected boolean readElement(HyadesConfigurationElement element) {
			if (element.getName().equals(TAG_LOADER)) {
				if (element.getAttribute(IHandlerElement.CLASS) == null) {
					logMissingAttribute(element, IHandlerElement.CLASS);
				} else {
					try {
						HandlerElementImpl handler = new HandlerElementImpl();
						handler.setConfigurationElement(element);

						if (element.getAttribute(IHandlerElement.ID) != null) {
							handler.setId(element.getAttribute(IHandlerElement.ID));
						}
						if (element.getAttribute(IHandlerElement.APPLICATION) != null) {
							handler.setApplication(element.getAttribute(IHandlerElement.APPLICATION));
						}
						if (element.getAttribute(IHandlerElement.FUNCTION) != null) {
							handler.setFunction(element.getAttribute(IHandlerElement.FUNCTION));
						}

						if (element.getAttribute(IHandlerElement.CLASS) != null) {
							handler.setClassName(element.getAttribute(IHandlerElement.CLASS));
						}
						if (element.getAttribute(IHandlerElement.MODIFIER) != null) {
							handler.setModifier(element.getAttribute(IHandlerElement.MODIFIER));
						}
						if (element.getAttribute(IHandlerElement.NAME) != null) {
							handler.setName(element.getAttribute(IHandlerElement.NAME));
						}
						if(element.getElement() instanceof IConfigurationElement)
							handler.setOwner(((IConfigurationElement)element.getElement()).getContributor().getName());
						
						if (element.getAttribute(IHandlerElement.PARENT_ID) != null) {
							handler.setParentId(element.getAttribute(IHandlerElement.PARENT_ID));
						}
						if (element.getAttribute(IHandlerElement.PRIORITY) != null) {
							try {
								handler.setPriority(Float.parseFloat(element.getAttribute(IHandlerElement.PRIORITY)));
							} catch (Exception e) {
								ModelDebugger.log(e, "Error when reading extension point " + element.getDeclaringExtensionName() + ", attribute " + IHandlerElement.PRIORITY);
							}
						}
						if (element.getAttribute(IHandlerElement.TARGET_ID) != null) {
							handler.setTargetId(element.getAttribute(IHandlerElement.TARGET_ID));
							if (IHandlerElement.DISABLED.equals(handler.getModifier()) || IHandlerElement.REPLACE.equals(handler.getModifier())) {
								disabledOrReplacedHandlers.add(handler.getTargetId());
							}
						} else if(handler.getModifier()!=null){
							ModelDebugger.log("Error when reading extension point " + element.getDeclaringExtensionName() + ", attribute targetId is empty!");
						}

						if (element.getAttribute(IHandlerElement.TARGET_EXTENSION) != null) {
							handler.setTargetExtension(element.getAttribute(IHandlerElement.TARGET_EXTENSION));
						}
						if (element.getAttribute(IHandlerElement.TYPE) != null) {
							handler.setType(element.getAttribute(IHandlerElement.TYPE));
						}
						if (element.getAttribute(IHandlerElement.VERSION) != null) {
							handler.setVersion(element.getAttribute(IHandlerElement.VERSION));
						}

						for (int i = 0; i < element.getChildren().length; i++) {
							HyadesConfigurationElement child = element.getChildren()[i];
							if (child.getName().equals(IHandlerElement.PARAMETER)) {
								if (child.getAttribute(IHandlerElement.NAME) != null) {
									handler.getParametersNames().add(child.getAttribute(IHandlerElement.NAME));
									handler.getParametersValues().add(child.getAttribute(IHandlerElement.VALUE));
								}
							}
						}

						String disabled = ModelsHierarchyPlugin.getPlugin().getPluginPreferences().getString(handler.getUniqueID() + "." + IHandlerElement.DISABLED);
						if (Boolean.valueOf(disabled).booleanValue())
						{
							disabledOrReplacedHandlers.add(handler.getUniqueID());
						}
						else
						{
							List handlers = (List) handlersByApplicationFunction.get(handler.getApplication() + "/" + handler.getFunction());
							if (handlers == null) {
								handlers = new ArrayList();
								handlersByApplicationFunction.put(handler.getApplication() + "/" + handler.getFunction(), handlers);
							}
							IHandlerElement existingHandler = (IHandlerElement) handlersByUniqueId.get(handler.getUniqueID());
							if (existingHandler == null) {
								handlersByUniqueId.put(handler.getUniqueID(), handler);
								handlers.add(handler);
							} else {
								if (existingHandler.getVersion().compareTo(handler.getVersion()) > 0) {
									handlersByUniqueId.put(handler.getUniqueID(), handler);
									handlers.remove(existingHandler);
									disabledOrReplacedHandlers.add(existingHandler);
									handlers.add(handler);
								}
							}
						}
					} catch (Exception e) {
						logError(element, e.getLocalizedMessage());
					}

					return true;
				}
			}
			return false;
		}
	}
}
