/*******************************************************************************
 * Copyright (c) 2008 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Stanislav Polevic, Intel - Initial API and Implementation 
 * 	  
 * $Id: BFHeader.java,v 1.3 2008/10/09 18:15:48 jwest Exp $ 
 *******************************************************************************/
package org.eclipse.hyades.loaders.internal.binary;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

public class BFHeader {
	
	public static final int LENGTH = 12;
	
	private Magic magic = Magic.TBF_INDENT;
	
	private Version version = Version.VERSION_1_0;

	private Platform platform = Platform.IA32;
	
	private Endianness endianness = Endianness.BIG_ENDIAN;
	
	public BFHeader() {
	}
	
	public boolean parse(InputStream inputStream, Offset offset) throws IOException {
		byte[] header = new byte[LENGTH];
		if (inputStream.read(header) < 0) {
			throw new IllegalArgumentException("Malformed header");
		}
		return parse(header, 0, offset);
	}
	
	public boolean parse(byte[] header, int start, Offset offset) {
		Offset marker = new Offset(start, LENGTH);

		// indent is ignored
		marker.increaseOffset(1);
		magic = Magic.read(header, marker);
		if (magic == null) {
			return false;
		}
		version = Version.read(header, marker);
		if (version == null) {
			throw new IllegalArgumentException("Version is not supported");
		}
		platform = Platform.read(header, marker);
		if (platform == null) {
			throw new IllegalArgumentException("Platform is not supported");
		}
		endianness = Endianness.read(header, marker);
		if (endianness == null) {
			throw new IllegalArgumentException("Endianness is not supported");
		}
		// offset to data is ignored at the moment
		
		offset.increaseOffset(LENGTH);
		return true;
	}

	public Magic getMagic() {
		return magic;
	}

	public Platform getPlatform() {
		return platform;
	}

	public Endianness getEndianness() {
		return endianness;
	}

	public Version getVersion() {
		return version;
	}

	public BFReader visit(BFReader reader) {
		if (Platform.IA32.equals(platform) && Endianness.LITTLE_ENDIAN.equals(endianness)) {
			reader.setNumberReader(BFReader.LE_64_READER);
		} else if (Platform.IA64.equals(platform) && Endianness.LITTLE_ENDIAN.equals(endianness)) {
			reader.setNumberReader(BFReader.LE_64_READER);
		}
		//Add support for big endianness
		else if (Endianness.BIG_ENDIAN.equals(endianness)) {
			reader.setNumberReader(BFReader.BE_64_READER);
		}

		return reader;
	}
	
	public static class Magic {

		private static Map entries = new HashMap();
		
		public static int LENGTH = 3;
		
		public static final Magic TBF_INDENT = new Magic("TBF");
			
		private final String magic;
		
		public Magic(String magic) {
			this.magic = magic;
			entries.put(magic, this);
		}

		public String getMagic() {
			return magic;
		}
		
		public static Magic read(byte[] header, Offset offset) {
			String magic = new String(header, offset.increaseOffset(LENGTH), LENGTH);
			return (Magic) entries.get(magic);
		}
	}
	
	public static class Platform {

		private static Map entries = new HashMap();
		
		public static final Platform IA32 = new Platform(0);

		public static final Platform IA64 = new Platform(1);
		

		private final int code;
		
		public Platform(int code) {
			this.code = code;
			entries.put(new Integer(code), this);
		}

		public int getCode() {
			return code;
		}

		public static Platform read(byte[] header, Offset offset) {
			int code = header[offset.increaseOffset(1)];
			return (Platform) entries.get(new Integer(code));
		}
	}
	
	public static class Endianness {
		private static Map entries = new HashMap();

		//Previously, the value of BIG_ENDIAN was 0 and the value of LITTLE_ENDIAN was 1
		//However, JVMTI agents did not adhere to this and set a value of 0 for all platforms
		//to retain compatibility with older JVMTI agents, BIG_ENDIAN is now defined as 1
		//and LITTLE_ENDIAN as 0
		
		public static final Endianness BIG_ENDIAN = new Endianness(1);
		
		public static final Endianness LITTLE_ENDIAN = new Endianness(0);

		private final int code;
		
		public Endianness(int code) {
			this.code = code;
			entries.put(new Integer(code), this);
		}

		public int getCode() {
			return code;
		}
		
		public static Endianness read(byte[] header, Offset offset) {
			int code = header[offset.increaseOffset(1)];
			return (Endianness) entries.get(new Integer(code));
		}
	}
	
	public static class Version {
		private static Map entries = new HashMap();

		public static final Version VERSION_1_0 = new Version(1.0);
		
		private final double version;
		
		public Version(double version) {
			this.version = version;
			entries.put(new Double(version), this);
		}

		public double getVersion() {
			return version;
		}

		public static Version read(byte[] header, Offset offset) {
			int major = header[offset.increaseOffset(1)];
			int minor = header[offset.increaseOffset(1)];
			double ver = major + (minor * 0.1);
			return (Version) entries.get(new Double(ver));
		}
	}
	
	public static class Encoding {
		private static Map entries = new HashMap();

		public static final Encoding UTF8 = new Encoding(0);
		
		public static final Encoding ASCII = new Encoding(1);

		private final int code;
		
		public Encoding(int code) {
			this.code = code;
			entries.put(new Integer(code), this);
		}

		public int getCode() {
			return code;
		}
		
		public static Encoding read(byte[] header, Offset offset) {
			int code = header[offset.increaseOffset(1)];
			return (Encoding) entries.get(new Integer(code));
		}
	}
}
