/**********************************************************************
 * Copyright (c) 2003, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LoaderExtensions.java,v 1.3 2008/01/24 02:28:17 apnan Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.util;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;


/**
 * @author slavescu
 */
public class LoaderExtensions extends HashMap {
    //~ Static fields/initializers -----------------------------------------------------------------

    /**
	 * 
	 */
	protected static final long serialVersionUID = 3257283621701498422L;
    protected static LoaderExtensions instance;

    //~ Constructors -------------------------------------------------------------------------------

    protected LoaderExtensions() {
        super();
    }

    //~ Methods ------------------------------------------------------------------------------------

    public static LoaderExtensions getInstance() {
        if (instance == null) {
            instance = new LoaderExtensions();
            new LoaderRegistryReader().readRegistry();
            new LoadersFactoryRegistryReader().readRegistry();
        }

        return instance;
    }
}


/**
 * A plugin extension reader that populates the
 * loaders registry.
 * Clients are not expected to use this class directly.
 */
class LoaderRegistryReader extends RegistryReader {
    //~ Static fields/initializers -----------------------------------------------------------------

    static final String TAG_LOADER = "loader";
    static final String ATT_ELEMENT_NAME = "elementName";
    public LoaderRegistryReader() {
        super(RegistryReader.LOADER_PPID);
    }

    //~ Methods ------------------------------------------------------------------------------------

    protected boolean readElement(HyadesConfigurationElement element) {
        if (element.getName().equals(TAG_LOADER)) {
            String elementName = element.getAttribute(ATT_ELEMENT_NAME);

            if (elementName == null) {
                logMissingAttribute(element, ATT_ELEMENT_NAME);
            } else if (element.getAttribute(ATT_CLASS) == null) {
                logMissingAttribute(element, ATT_CLASS);
            } else {
            	float priority = 0;
                if (element.getAttribute(ATT_PRIORITY) != null) {
                	priority = Float.parseFloat(element.getAttribute(ATT_PRIORITY));
                }
               	if(LoaderExtensions.getInstance().containsKey(elementName) && ((Float)getPriorities().get(elementName)).floatValue() >= priority){
            		logError(element, "A loader for this element with a higher or equal priority already exists, loader \""+element.getAttribute(ATT_CLASS)+"\" will be ignored for elementName \""+elementName+"\".");
               	} else {
                    try {
                        IHyadesPluginClassDescriptor classDescriptor = createPluginClassDescriptor(element, ATT_CLASS);

                        LoaderExtensions.getInstance().put(elementName, (XMLFragmentLoader) classDescriptor.createInstance());
                        getPriorities().put(elementName,new Float(priority));
                    } catch (Exception e) {
                        logError(element, e.getLocalizedMessage());
                    }
                }
                return true;
            }
        }

        return false;
    }
}


class LoadersFactoryRegistryReader extends RegistryReader {
    //~ Static fields/initializers -----------------------------------------------------------------

    static final String TAG_LOADERS_FACTORY = "loadersFactory";
    public LoadersFactoryRegistryReader() {
        super(RegistryReader.LOADERS_FACTORY_PPID);
    }

    //~ Methods ------------------------------------------------------------------------------------

    protected boolean readElement(HyadesConfigurationElement element) {
        if (element.getName().equals(TAG_LOADERS_FACTORY)) {
            if (element.getAttribute(ATT_CLASS) == null) {
                logMissingAttribute(element, ATT_CLASS);
            } else {
                try {
                	float priority = 0;
                    if (element.getAttribute(ATT_PRIORITY) != null) {
                    	priority = Float.parseFloat(element.getAttribute(ATT_PRIORITY));
                    }
                    IHyadesPluginClassDescriptor factoryClassDescriptor = createPluginClassDescriptor(element, ATT_CLASS);
                    XMLFragmentLoadersFactory factory = (XMLFragmentLoadersFactory) factoryClassDescriptor.createInstance();
                    Set elements = factory.getSupportedElements();

                    for (Iterator iter = elements.iterator(); iter.hasNext();) {
                        String elementName = (String) iter.next();

                       	if(LoaderExtensions.getInstance().containsKey(elementName) && ((Float)getPriorities().get(elementName)).floatValue() >= priority){
                        		logError(element, "A loader for this element with a higher or equal priority already exists, this loader will be ignored.");
                        } else {
                            //							PluginClassDescriptor classDescriptor = new PluginClassDescriptor(element, factory.getLoaderClassName(elementName));
                            //							
                            //							XMLFragmentLoader loader = (XMLFragmentLoader)classDescriptor.createInstance();
                            try {
                                XMLFragmentLoader loader = (XMLFragmentLoader) factory.getClass().getClassLoader().loadClass(factory.getLoaderClassName(elementName)).newInstance();

                                if (loader != null) {
                                    LoaderExtensions.getInstance().put(elementName, loader);
                                    getPriorities().put(elementName,new Float(priority));
                                }
                            } catch (Exception e) {
                                logError(element, e.getLocalizedMessage());
                            }
                        }
                    }
                } catch (Exception e) {
                    logError(element, e.getLocalizedMessage());
                }

                return true;
            }
        }

        return false;
    }
}
