/**********************************************************************
 * Copyright (c) 2003, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LoadersUtils.java,v 1.7 2010/09/02 17:07:35 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 * 
 * $Id: LoadersUtils.java,v 1.7 2010/09/02 17:07:35 jcayne Exp $
 **********************************************************************/
package org.eclipse.hyades.loaders.util;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.ParseException;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.InternalEObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.hierarchy.CorrelationContainer;
import org.eclipse.hyades.models.hierarchy.CorrelationContainerProxy;
import org.eclipse.hyades.models.hierarchy.CorrelationEngine;
import org.eclipse.hyades.models.hierarchy.HierarchyFactory;
import org.eclipse.hyades.models.hierarchy.HierarchyPackage;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.hierarchy.impl.CorrelationEngineImpl;
import org.eclipse.hyades.models.hierarchy.plugin.ModelsHierarchyPlugin;
import org.eclipse.hyades.models.hierarchy.util.HierarchyResourceSetImpl;
import org.eclipse.hyades.models.hierarchy.util.HyadesExtendedResourceFactory;
import org.eclipse.hyades.models.hierarchy.util.IHyadesResourceExtension;
import org.eclipse.hyades.models.hierarchy.util.PerfUtil;
import org.eclipse.hyades.models.hierarchy.util.SaveUtil;
import org.eclipse.hyades.models.hierarchy.util.StringUtil;
import org.eclipse.hyades.models.util.ModelDebugger;

import com.ibm.icu.text.DateFormat;
import com.ibm.icu.text.SimpleDateFormat;
import com.ibm.icu.util.TimeZone;
import com.ibm.icu.util.ULocale;

public class LoadersUtils {
    //~ Static fields/initializers -----------------------------------------------------------------

    protected final static Map threads = new HashMap();

    public static SimpleDateFormat partialCIMDateFormatDashDelimiter = new SimpleDateFormat("yyyy'-'MM'-'dd'T'HH:mm:ss",ULocale.ENGLISH);
    public static SimpleDateFormat partialCIMDateFormatPlusDelimiter = new SimpleDateFormat("yyyy'+'MM'+'dd'T'HH:mm:ss",ULocale.ENGLISH);
    public static SimpleDateFormat partialCIMDateFormatNoDelimiter = new SimpleDateFormat("yyyyMMdd'T'HH:mm:ss",ULocale.ENGLISH);
    public static long[] powerOf10;
    static{
		TimeZone gmtTimeZone = TimeZone.getTimeZone("GMT"); //$NON-NLS-1$
    	partialCIMDateFormatDashDelimiter.setTimeZone(gmtTimeZone);
    	partialCIMDateFormatPlusDelimiter.setTimeZone(gmtTimeZone);
    	partialCIMDateFormatNoDelimiter.setTimeZone(gmtTimeZone);
    	powerOf10=new long[19];
    	powerOf10[0]=1;
    	long power=1;
    	for (int i = 1; i < powerOf10.length; i++) {
    		power*=10;
			powerOf10[i]=power;
		}
    }
    public static String InternalCorrelationName = "Internal_";
//	private final static Map correlationMap = new HashMap();

    //~ Methods ------------------------------------------------------------------------------------

    public static void setDefaultContext(HierarchyContext context) {
        TRCMonitor monitor = context.getMonitor();
        TRCProcessProxy process = context.getProcessProxy();
        TRCNode node = context.getNode();

        /* If we don't know the node, place it in the default file */
        if (node == null) {
            /* If we already have a default node we use it */
            node = LoadersUtils.locateNodeFromMonitor(monitor, "Unknown", "0", "0");

            if (node == null) {
//                EList nodeExt = new BasicEList();
//                EList nExt = new BasicEList();
                String monitorFolder = getMonitorFolder(monitor);
                Resource nDoc = null;

                if (monitorFolder != null) {
                    String nName = StringUtil.change(monitor.getName().trim(), " ", "") + "_" + "Unknown";

                    String nodeFileName = monitorFolder + nName;

                    nDoc = Resource.Factory.Registry.INSTANCE.getFactory(SaveUtil.createURI(nodeFileName + ".trcnxmi")).createResource(SaveUtil.createURI(nodeFileName + ".trcnxmi"));
                    HierarchyResourceSetImpl.getInstance().getResources().add(nDoc);

//                    //*** adding support for multiple files
//                    SaveUtil.addDocument(nDoc);
                }

                node = HierarchyFactory.eINSTANCE.createTRCNode();
                node.setIpAddress("0");
                node.setRuntimeId("0");
                node.setName("Unknown");
                monitor.getNodes().add(node);
                node.setMonitor(monitor);
                context.setNode(node);

                if (nDoc != null) {
                    nDoc.getContents().add(node);
                    nDoc.setModified(true);
                }
            }

            context.setNode(node);
        }

        if (process == null) {
            /* Do we already have a default process? If not create it */
            process = LoadersUtils.locateProcessFromNode(node, "0");

            if (process != null) {
                context.setProcessProxy(process);

                return;
            }

            //create the process
//            EList procExt = new BasicEList();
            String monitorFolder = getMonitorFolder(monitor);
            Resource procDoc = null;

            if (monitorFolder != null) {
                String pName = StringUtil.change(monitor.getName().trim(), " ", "") + "_" + node.getName().trim() + "_" + "-1" /* The _pid */
                +"_" + "0"; /* The _processId */

                String pFileName = monitorFolder + pName;

                procDoc = Resource.Factory.Registry.INSTANCE.getFactory(SaveUtil.createURI(pFileName + ".trcpxmi")).createResource(SaveUtil.createURI(pFileName + ".trcpxmi"));
                HierarchyResourceSetImpl.getInstance().getResources().add(procDoc);

//                //*** adding support for multiple files
//                SaveUtil.addDocument(procDoc);
            }

            process = HierarchyFactory.eINSTANCE.createTRCProcessProxy();
            node.getProcessProxies().add(process);
            process.setNode(node);
            process.setPid(-1);
            process.setRuntimeId("0");
            process.setName("Unknown");

            context.setProcessProxy(process);

            if (procDoc != null) {
                procDoc.getContents().add(process);
                procDoc.setModified(true);
            }
        }
    }
    
    /**
     * Based on {@link LoadersUtils#setDefaultContext(HierarchyContext)} allowing for creating a new TRCNode and process (TRCProcessProxy) 
     * using the passed arguments.
     * 
     * Bug 322951
     * 
     * @param context The context to add the {@link TRCProcessProxy} to.
     * @param name The name of the process.
     * @param runtimeId The runtime ID of the process.
     * @param pid The PID of the process.
     * @param processId The process ID of the process.
     * @return The {@link TRCProcessProxy} created.
     */
    public static TRCProcessProxy createProcess(HierarchyContext context, String name, String runtimeId, String pid, String processId) {
        TRCMonitor monitor = context.getMonitor();
        TRCProcessProxy process = context.getProcessProxy();
        TRCNode node = context.getNode();

        if(name == null || name.length() == 0)
        	name = "Unknown";
        if(runtimeId == null || runtimeId.length() == 0)
        	runtimeId = "0";
        if(pid == null || pid.length() == 0)
        	pid = "-1";
        if(processId == null || processId.length() == 0)
        	processId = "0";
        
        // If we do not have a node, return the current process. If we have a process, we won't create one or it is not the right one.
        if (node != null && (process == null || process.getRuntimeId() != runtimeId)) {
            /* Do we already have a default process? If not create it */
            process = LoadersUtils.locateProcessFromNode(node, runtimeId);

            if (process != null) {
                context.setProcessProxy(process);

                return process;
            }

            //create the process
            String monitorFolder = getMonitorFolder(monitor);
            Resource procDoc = null;

            if (monitorFolder != null) {
                String pName = StringUtil.change(monitor.getName().trim(), " ", "") + "_" + node.getName().trim() + "_" + pid +"_" + processId;

                String pFileName = monitorFolder + pName;

                procDoc = Resource.Factory.Registry.INSTANCE.getFactory(SaveUtil.createURI(pFileName + ".trcpxmi")).createResource(SaveUtil.createURI(pFileName + ".trcpxmi"));
                HierarchyResourceSetImpl.getInstance().getResources().add(procDoc);
            }

            process = HierarchyFactory.eINSTANCE.createTRCProcessProxy();
            node.getProcessProxies().add(process);
            process.setNode(node);
            process.setPid(Integer.parseInt(pid));
            process.setRuntimeId(runtimeId);
            process.setName(name);

            context.setProcessProxy(process);

            if (procDoc != null) {
                procDoc.getContents().add(process);
                procDoc.setModified(true);
            }
        }
        return process;
    }
    
    /**
     * Creates a {@link TRCAgent} based on {@link LoadersUtils#createAgent(HierarchyContext)} using the passed arguments.
     * 
     * Bug 322951
     * 
     * @param context The context to add the {@link TRCAgent} to.
     * @param agentName The name of the agent.
     * @param startTime The start time of the agent.
     * @param stopTime The stop time of the agent.
     * @param runtimeId The runtime ID of the agent.
     * @param agentType The type of the agent.
     * @return The {@link TRCAgent} created.
     */
    public static TRCAgent createAgent(HierarchyContext context, String agentName, double startTime, double stopTime, String runtimeId, String agentType) {
        TRCAgent agent = context.getAgent();
        TRCAgentProxy agentProxy = context.getAgentProxy();
        TRCProcessProxy process = context.getProcessProxy();
        TRCNode node = context.getNode();
        TRCMonitor monitor = context.getMonitor();

        String monitorFolder = LoadersUtils.getMonitorFolder(monitor);
        Resource agDoc = null;

        agent = LoadersUtils.locateAgentFromProcess(process, agentName, runtimeId);
        if(agent != null) {
        	context.setAgent(agent);
        	context.setAgentProxy(agent.getAgentProxy());
        	return agent;
        }
        if (monitorFolder != null) {
            String aName = monitor.getName() + "_" + node.getName().trim() + "_" + process.getPid() + "_" + process.getRuntimeId() + "_" + StringUtil.change(agentName.trim(), " ", "");

            String pFileName = monitorFolder + aName;

            String rName = ModelDebugger.INSTANCE.debugDatabaseResourcePostfix!=null && ModelDebugger.INSTANCE.debugDatabaseResourcePostfix.length()>0 ? pFileName + ModelDebugger.INSTANCE.debugDatabaseResourcePostfix :	pFileName + ".trcaxmi";
            URI uri = SaveUtil.createURI(rName+"#").trimFragment();
            agDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
			agDoc.setModified(true);
			
            HierarchyResourceSetImpl.getInstance().getResources().add(agDoc);
        }

        agent = HierarchyFactory.eINSTANCE.createTRCAgent();
        // If we do not have an Agent Proxy or the Agent Proxy's process does not match the context's process.
        if (agentProxy == null || agentProxy.getProcessProxy() != process) {
            agentProxy = HierarchyFactory.eINSTANCE.createTRCAgentProxy();
            agentProxy.setRuntimeId(runtimeId); // "" + startTime
            agent.setRuntimeId(runtimeId);

            agent.setName(agentName);
            agentProxy.setName(agentName);
            
            agent.setType(agentType);
            agentProxy.setType(agentType);
            
            agent.setStartTime(startTime);
            agentProxy.setStartTime(startTime);
            
            agent.setStopTime(stopTime);
            agentProxy.setStopTime(stopTime);

            agent.setAgentProxy(agentProxy);

            agentProxy.setProcessProxy(process);

            agent.setCollectionMode(context.getCollectionMode());
            agentProxy.setCollectionMode(context.getCollectionMode());

            context.setAgent(agent);
            context.setAgentProxy(agentProxy);
        }
        else {
            agent.setRuntimeId(agentProxy.getRuntimeId());

            agent.setName(agentProxy.getName());
            if(agentProxy.getType()!=null)
	            agent.setType(agentProxy.getType());
            else
            {	
	            agent.setType(agentType);
	            agentProxy.setType(agentType);
            }
            agent.setStartTime(agentProxy.getStartTime());
            agent.setStopTime(agentProxy.getStopTime());

            agent.setAgentProxy(agentProxy);

            agent.setCollectionMode(context.getCollectionMode());

            context.setAgent(agent);
        }

        if (context.getContextURI() == null) {
            context.setContextURI(agDoc.getURI().toString());
            LookupServiceExtensions.getInstance().register(null, context);
        }

        if (agDoc != null) {
            agDoc.getContents().add(agent);
            agDoc.setModified(true);
        }

        return agent;
    }

    /**
     * Returns the signed time zone offset from GMT or Zulu time in minutes of the parameter XML schema dateTime datatype time stamp.
     * 
     * Assumption:  The caller is responsible for removing any leading and/or trailing whitespace.
     * 
     * @param creationTime XML schema dateTime datatype time stamp.
     * @return short Signed time zone offset from GMT in minutes.
     * @throws IllegalArgumentException The parameter time stamp is not a valid XML schema dateTime datatype. 
     */
    public synchronized static short getDeltaTimeZoneInMinutesFromCIMFormat(String creationTime) throws IllegalArgumentException {

        //XML schema dateTime datatype (http://www.w3.org/TR/xmlschema-2/#dateTime):
        //y...y[-/+]MM[-/+]ddTHH:mm:ss[.S...S][-/+HH:mm]/[Z]
        //For example: 2003-06-03T15:48:03.783455-04:00
        //             2003-06-03T19:48:03.12Z
        //             2003+06+03T19:48:03Z
        //             2000030603T19:48:03Z

        //Calculate the length of the creationTime time stamp:
        int creationTimeLength = creationTime.length();

        short timeZoneOffset = 0;

        //If the time stamp is in UTC (i.e. GMT), return a zero offset:
        if (creationTime.charAt(creationTimeLength - 1) != 'Z') {

            try {

                //Parse the time zone hour offset (i.e. [HH]):
                timeZoneOffset = Short.parseShort(creationTime.substring((creationTimeLength - 5), (creationTimeLength - 3)));

                //Convert the time zone hour offset to minutes:
                timeZoneOffset *= 60;

                //Parse the time zone minute offset (i.e. [mm]):
                timeZoneOffset += Short.parseShort(creationTime.substring(creationTimeLength - 2));

                //Change the sign of the time zone offset:
                if (creationTime.charAt(creationTimeLength - 6) == '-')
                    timeZoneOffset *= -1;
            }
            catch (Exception e) {
                throw new IllegalArgumentException("Invalid parameter time stamp format.");
            }
        }

        return timeZoneOffset;
    }

    /**
     * Returns the microsecond representation of the parameter XML schema dateTime datatype time stamp, 
     * adjusted for the local time zone.
     * 
     * Assumption:  The caller is responsible for removing any leading and/or trailing whitespace.   
     * 
     * @param creationTime XML schema dateTime datatype time stamp.
     * @return double Microsecond representation of the parameter XML schema dateTime datatype time stamp.
     * @throws IllegalArgumentException The parameter time stamp is not a valid XML schema dateTime datatype. 
     */
    public synchronized static double getMicrosecondsFromCIMFormat(String creationTime) throws IllegalArgumentException {

        //XML schema dateTime datatype (http://www.w3.org/TR/xmlschema-2/#dateTime):
        //y...y[-/+]MM[-/+]ddTHH:mm:ss[.S...S][-/+HH:mm]/[Z]
        //For example: 2003-06-03T15:48:03.783455-04:00
        //             2003-06-03T19:48:03.12Z
        //             2003+06+03T19:48:03Z
        //             2000030603T19:48:03Z

        //Microsecond representation of the creationTime time stamp:
        double creationTimeInMicros = 0.0;

        //'T' character is the reference index in the creationTime since this character is required according to the XML schema dateTime datatype specification:
        //NOTE: The reference character (i.e. 'T') is required due to variable length year and fractional second fields.
        int tCharIndex = creationTime.indexOf('T');

        //creationTime must have the reference character (i.e. 'T') according to the XML schema dateTime datatype specification:
        if (tCharIndex != -1) {
            try {
                //For performance reasons, attempt to parse the date and time (second granularity) part of the creationTime with the year/month/day delimiter as the '-' character (most common format):
                //NOTE:  1 millisecond = 1000 microseconds
                creationTimeInMicros = (partialCIMDateFormatDashDelimiter.parse(creationTime.substring(0, (tCharIndex + 9))).getTime() * 1000.0);
            }

            //The year/month/day delimiter is not the '+' character:
            catch (ParseException p) {

                try {

                    //If the year/month/day delimiter is not the '-' character, check if the year/month/day delimiter is the '+' character:
                    //NOTE:  1 millisecond = 1000 microseconds
                    if (creationTime.charAt(tCharIndex - 3) == '+')
                        creationTimeInMicros = (partialCIMDateFormatPlusDelimiter.parse(creationTime.substring(0, (tCharIndex + 9))).getTime() * 1000.0);

                    //If the year/month/day delimiter is not neither the '-' or '+' characters, no year/month/day delimiter is used:
                    //NOTE:  1 millisecond = 1000 microseconds
                    else
                        creationTimeInMicros = (partialCIMDateFormatNoDelimiter.parse(creationTime.substring(0, (tCharIndex + 9))).getTime() * 1000.0);
                }
                catch (ParseException pp) {
                    throw new IllegalArgumentException("Invalid parameter time stamp format.");
                }
            }

            //Convert the microsecond representation of the creationTime to UTC (i.e. GMT or Zulu time):
            //NOTE:  1 minute = 60000000 microseconds
            creationTimeInMicros -= (((double) (getDeltaTimeZoneInMinutesFromCIMFormat(creationTime)) * 60000000.0));

            //Retrieve, calculate and add the fractional second value of the creationTime if exists:
            try {

                //Retrieve the fractional second value of the creationTime if exists (i.e. [.S...S] and convert to microseconds:
                //NOTE:  1 second = 1000000 microseconds
                creationTimeInMicros += (Double.parseDouble("0.".concat(creationTime.substring((tCharIndex + 10), (creationTime.endsWith("Z") ? (creationTime.length() - 1) : (creationTime.length() - 6))))) * 1000000.0);
            }
            catch (Exception e) {
                //The fractional second is not included in this creationTime:
            }
        }
        else
            throw new IllegalArgumentException("Invalid parameter time stamp format.");

        return creationTimeInMicros;
    }

    // get the folder name of a monitor
    public static String getMonitorFolder(TRCMonitor monitor) {
        if (monitor == null) {
            return null;
        }

        Resource res = monitor.eResource();

        if (res == null) {
            return null;
        }

        String uri = res.getURI().toString();

        uri = uri.replace('\\', '/');

        int index = uri.lastIndexOf("/");

        if (index >= 0) {
            uri = uri.substring(0, index);
        }

        if (uri.startsWith("/")) {
            uri = uri.substring(1);
        }

        if (!uri.endsWith("/")) {
            uri += "/";
        }

        return uri;
    }

    /**
     * @deprecated Use getPartialCIMDateFormatDashDelimiter().
     */
    public static SimpleDateFormat getPartialCIMDateFormat() {
        return (getPartialCIMDateFormatDashDelimiter());
    }

    /**
     * @return SimpleDateFormat
     */
    public static SimpleDateFormat getPartialCIMDateFormatDashDelimiter() {
        return partialCIMDateFormatDashDelimiter;
    }

    /**
     * @return SimpleDateFormat
     */
    public static SimpleDateFormat getPartialCIMDateFormatPlusDelimiter() {
        return partialCIMDateFormatPlusDelimiter;
    }

    /**
     * @return SimpleDateFormat
     */
    public static SimpleDateFormat getPartialCIMDateFormatNoDelimiter() {
        return partialCIMDateFormatNoDelimiter;
    }

    /**
     * @param value
     * @return
     */
    public synchronized static double convertCreationTime(String value) {
        return getMicrosecondsFromCIMFormat(value);
    }

    /**
     * @return
     */
    public static TRCAgent getOrCreateAgent(HierarchyContext context) {
    	TRCAgent agent = context.getAgent();
    	if(agent==null)
    	{
    		if(context.getAgentProxy()!=null && context.getAgentProxy().getAgent()!=null)
    		{
    			agent = context.getAgentProxy().getAgent();
    		}
    		else
    		{
    			agent = LoadersUtils.createAgent(context);
    		}
    	}
    	return agent;
    }
    
    
    /**
     * @param context
     * @return
     */
    public static TRCAgent createAgent(HierarchyContext context) {
        setDefaultContext(context);

        TRCAgent agent = context.getAgent();
        TRCAgentProxy agentProxy = context.getAgentProxy();
        TRCProcessProxy process = context.getProcessProxy();
        TRCNode node = context.getNode();
        TRCMonitor monitor = context.getMonitor();

        String agentName = "Unknown-"+System.currentTimeMillis();
        String monitorFolder = LoadersUtils.getMonitorFolder(monitor);
        Resource agDoc = null;

        double startTime = ((double) (new Date()).getTime()) / 1000;

        if (monitorFolder != null) {
            String aName = monitor.getName() + "_" + node.getName().trim() + "_" + process.getPid() + "_" + process.getRuntimeId() + "_" + StringUtil.change(agentName.trim(), " ", "");

            String pFileName = monitorFolder + aName;

            String rName = ModelDebugger.INSTANCE.debugDatabaseResourcePostfix!=null && ModelDebugger.INSTANCE.debugDatabaseResourcePostfix.length()>0 ? pFileName + ModelDebugger.INSTANCE.debugDatabaseResourcePostfix :	pFileName + ".trcaxmi";
            URI uri = SaveUtil.createURI(rName+"#").trimFragment();
            agDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
			agDoc.setModified(true);
			
            HierarchyResourceSetImpl.getInstance().getResources().add(agDoc);
//            SaveUtil.addDocument(agDoc);
        }

        agent = HierarchyFactory.eINSTANCE.createTRCAgent();
        if (agentProxy == null) {
            agentProxy = HierarchyFactory.eINSTANCE.createTRCAgentProxy();
            agentProxy.setRuntimeId("" + startTime);
            agent.setRuntimeId("" + startTime);

            agent.setName(agentName);
            agentProxy.setName(agentName);
            
            agent.setType("Logging");
            agentProxy.setType("Logging");
            
            agent.setStartTime(startTime);
            agentProxy.setStartTime(startTime);

            agent.setAgentProxy(agentProxy);

            agentProxy.setProcessProxy(process);

            agent.setCollectionMode(context.getCollectionMode());
            agentProxy.setCollectionMode(context.getCollectionMode());

            context.setAgent(agent);
            context.setAgentProxy(agentProxy);
        }
        else {
            agent.setRuntimeId(agentProxy.getRuntimeId());

            agent.setName(agentProxy.getName());
            if(agentProxy.getType()!=null)
	            agent.setType(agentProxy.getType());
            else
            {	
	            agent.setType("Logging");
	            agentProxy.setType("Logging");
            }
            agent.setStartTime(agentProxy.getStartTime());

            agent.setAgentProxy(agentProxy);

            agent.setCollectionMode(context.getCollectionMode());

            context.setAgent(agent);
        }

        if (context.getContextURI() == null) {
            context.setContextURI(agDoc.getURI().toString());
            LookupServiceExtensions.getInstance().register(null, context);

        }

        if (agDoc != null) {
            agDoc.getContents().add(agent);
            agDoc.setModified(true);
        }

        return agent;
    }

    public static void deregisterAgentInstance(TRCAgentProxy agentProxy) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            c.setAgentInstance(null);
        }
    }

    public static void deregisterAgentListener(TRCAgentProxy agentProxy) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            c.setAgentListener(null);
        }
    }

    public static void deregisterAgentProxyContext(TRCAgentProxy agentProxy) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            LookupServiceExtensions.getInstance().deregister(null, c);
            c.cleanUp();
        }
    }

    public static void deregisterDataProcessor(TRCAgentProxy agentProxy) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            c.setDataProcessor(null);
        }
    }

    public static TRCAgent locateAgentFromMonitor(TRCMonitor monitor, String agentRId) {
        EList nodeList = monitor.getNodes();
        Iterator nodeIterator = nodeList.iterator();

        while (nodeIterator.hasNext()) {
            TRCNode node = (TRCNode) nodeIterator.next();

            if (!node.eIsProxy()) {
                EList processList = node.getProcessProxies();
                Iterator processIterator = processList.iterator();

                while (processIterator.hasNext()) {
                    TRCProcessProxy process = (TRCProcessProxy) processIterator.next();

                    if (!process.eIsProxy()) {
                        EList agents = process.getAgentProxies();
                        Iterator agentIterator = agents.iterator();

                        while (agentIterator.hasNext()) {
                            TRCAgentProxy agent = (TRCAgentProxy) agentIterator.next();

                            if (!agent.eIsProxy()) {
                                if ((agent.getRuntimeId() != null) && agent.getRuntimeId().equals(agentRId)) {
                                    return agent.getAgent();
                                }
                            }
                        }
                    }
                }
            }
        }

        return null;
    }

    public static TRCAgent locateAgentFromProcess(TRCProcessProxy process, String aName, String agentRId) {
        EList agents = process.getAgentProxies();
        Iterator i = agents.iterator();

        while (i.hasNext()) {
            TRCAgentProxy agent = (TRCAgentProxy) i.next();

            if (!agent.eIsProxy()) {
                if (agent.getName() != null && agent.getName().equals(aName)) {
                    if ((agent.getRuntimeId() == null) || agent.getRuntimeId().equals("") || agent.getRuntimeId().equals(agentRId)) {
                        return agent.getAgent();
                    }
                }
            }
        }

        return null;
    }

    public static Object locateAgentInstance(TRCAgentProxy agentProxy) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            return c.getAgentInstance();
        }
        else {
            return null;
        }
    }

    public static Object locateAgentListener(TRCAgentProxy agentProxy) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            return c.getAgentListener();
        }
        else {
            return null;
        }
    }

    public static XMLLoader locateDataProcessor(TRCAgentProxy agentProxy) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            return c.getDataProcessor();
        }
        else {
            return null;
        }
    }

    /**
     * @param agent
     * @return
     */
    public static HierarchyContext locateHierarchyContext(TRCAgent agent) {
        HierarchyContext context = null;

        if (agent.eResource() != null) {
            context = (HierarchyContext) LookupServiceExtensions.getInstance().locate(null, HierarchyContext.class, agent.eResource().getURI().toString());
        }
        if (context == null) {
            context = (HierarchyContext) LookupServiceExtensions.getInstance().locate(null, HierarchyContext.class, agent.getRuntimeId());
        }

        return context;
    }

    public static TRCNode locateNodeFromMonitor(TRCMonitor monitor, String hostName, String ipaddr, String nRId) {
        EList list = monitor.getNodes();
        int i=0;
        try {
            InetAddress addr = InetAddress.getByName(hostName);
            
            InetAddress localAddress;
            if ("localhost".equalsIgnoreCase(hostName))
            	localAddress = InetAddress.getLocalHost();
            else
            	localAddress = null;

            for (; i < list.size(); i++) {
				
                TRCNode node = (TRCNode) list.get(i);

                if (!node.eIsProxy()) {
                    String name = node.getName();

                    try {
                        InetAddress naddr = InetAddress.getByName(name);

                        if (addr.equals(naddr)) {
                            return node;
                        }
                        
                        if (localAddress != null && localAddress.equals(naddr)) {
                            return node;
                        }
                    }
                    catch (UnknownHostException exc) {
                        String ipaddress = node.getIpAddress();

                        if (name.equals(hostName) || ((ipaddress != null) && ipaddress.equals(ipaddr))) {
                            node.setIpAddress(ipaddr);
                            node.setRuntimeId(nRId);

                            return node;
                        }
                    }
                }
			}
        }
        catch (UnknownHostException exc) {
//        	i++;
            for (; i < list.size(); i++) {
                TRCNode node = (TRCNode) list.get(i);
                String name = node.getName();
                String ipaddress = node.getIpAddress();

                if (((name != null) && (hostName != null) && name.equals(hostName)) || ((ipaddress != null) && ipaddress.equals(ipaddr))) {
                    node.setIpAddress(ipaddr);
                    node.setRuntimeId(nRId);

                    return node;
                }
            }
        }

        return null;
    }

    public static TRCProcessProxy locateProcessFromNode(TRCNode node, String pRId) {
        EList processList = node.getProcessProxies();
        for (int i = 0; i < processList.size(); i++) {
            TRCProcessProxy process = (TRCProcessProxy) processList.get(i);

            if (!process.eIsProxy()) {
                if ((process.getRuntimeId() != null) && (process.getRuntimeId().compareTo(pRId) == 0)) {
                    return process;
                }
            }
        }

        return null;
    }

    public static TRCProcessProxy locateProcessFromNode(TRCNode node, int pId, String pRId) {
        EList processList = node.getProcessProxies();
        for (int i = 0; i < processList.size(); i++) {
			
            TRCProcessProxy process = (TRCProcessProxy) processList.get(i);

            if (!process.eIsProxy()) {
                int processId = process.getPid();

                if (processId == pId) {
                    String runtimeId = process.getRuntimeId();

                    if ((runtimeId == null) || runtimeId.equals("") || runtimeId.equals(pRId)) {
                        process.setRuntimeId(pRId);

                        return process;
                    }
                }
            }
        }

        return null;
    }

    /**
	 * @param e
	 * @deprecated Use {@link ModelDebugger#log(Throwable)} instead
	 */
	public static void log(Throwable e) {
		ModelDebugger.log(e);
	}

	/**
	 * @deprecated Use {@link ModelDebugger#log(Throwable,String)} instead
	 */
	public static void log(Throwable e, String msg) {
		ModelDebugger.log(e, msg);
	}

	/**
	 * @deprecated Use {@link ModelDebugger#logCallstackWithLimit(Throwable,int)} instead
	 */
	public static void logCallstackWithLimit(Throwable e, final int limit) {
		ModelDebugger.logCallstackWithLimit(e, limit,System.out);
	}

    /**
	 * @deprecated Use {@link ModelDebugger#log(String)} instead
	 */
	public static void log(String s) {
		ModelDebugger.log(s);
	}

    public static String makeString(byte[] data, int offset, int length) {
        if (data == null) {
            return null;
        }

        try {
            return new String(data, offset, length, "UTF8");
        }
        catch (Exception ex) {
            return null;
        }
    }

    public static void registerAgentInstance(TRCAgentProxy agentProxy, Object ai) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            c.setAgentInstance(ai);
        }
        else {
            c = new AgentProxyContext();
            c.setAgentProxy(agentProxy);
            c.setAgentInstance(ai);
            LookupServiceExtensions.getInstance().register(null, c);
        }
    }

    public static void registerAgentListener(TRCAgentProxy agentProxy, Object al) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            c.setAgentListener(al);
        }
        else {
            c = new AgentProxyContext();
            c.setAgentProxy(agentProxy);
            c.setAgentListener(al);
            LookupServiceExtensions.getInstance().register(null, c);
        }
    }

    public static void registerDataProcessor(TRCAgentProxy agentProxy, XMLLoader processor) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            c.setDataProcessor(processor);
        }
        else {
            c = new AgentProxyContext();
            c.setAgentProxy(agentProxy);
            c.setDataProcessor(processor);
            LookupServiceExtensions.getInstance().register(null, c);
        }
    }

    /**
     * @param processor
     */
    public static void loadRootEvent(XMLLoader processor) {
        processor.loadEvent("<TRACE>".getBytes(), "<TRACE>".getBytes().length);
    }

    public static TRCAgent locateAgent(String agentIdRef, TRCAgent agent) {
        AgentsContext agentsContext = (AgentsContext) LookupServiceExtensions.getInstance().locate(null, AgentsContext.class, LoadersUtils.getLookUpKey(agentIdRef));
        if (agentsContext != null) {
            return agentsContext.getActiveAgent(agent);
        }
        return null;
    }

    public static boolean registerAgent(String agentIdRef, TRCAgent agent) {
        AgentsContext agentsContext = (AgentsContext) LookupServiceExtensions.getInstance().locate(null, AgentsContext.class, LoadersUtils.getLookUpKey(agentIdRef));
        if (agentsContext == null) {
            agentsContext = new AgentsContext(agentIdRef);
            LookupServiceExtensions.getInstance().register(null, agentsContext);
        }

        return agentsContext.registerAgent(agent);
    }

    public static Object getLookUpKey(int key) {
        return new Integer(key);
    }

    public static Object getLookUpKey(long key) {
        return new Long(key);
    }

    public static Object getLookUpKey(short key) {
        return new Short(key);
    }
    public static Object getLookUpKey(String key) {
        if (key == null)
            return new Integer(0);
        return new Integer(key.hashCode());
    }

    public static GenericLookUpEntry getGenericLookUpEntry(Thread thread) {
        Integer key = new Integer(thread.hashCode());
        GenericLookUpEntry entry = (GenericLookUpEntry) threads.get(key);
        if (entry == null) {
            entry = new GenericLookUpEntry();
            threads.put(key, entry);
        }
        return entry;
    }

    public static void registerGenericLookUpEntry(HierarchyContext context, Object key, Object value) {
        GenericLookUpEntry entry = LoadersUtils.getGenericLookUpEntry(Thread.currentThread());
        entry.setKey(key);
        entry.setValue(value);
        LookupServiceExtensions.getInstance().register(context, entry);
        entry.reset();
    }

	public static int getHashCode(byte[] data, int offset, int length) {
		int h = 0;

		for (int i = 0; i < length; i++) {
			h = (31 * h) + data[offset + i];
		}

		return h;
	}

    public static int getHashCode(String value) {
    	if (value == null) return 0;
    	
        int h = 0;
        int len = value.length();
        for (int i = 0; i < len; i++) {
            h = 31 * h + value.charAt(i);
        }
        return h;
    }
    
	public static void setSessionStartTime(double time, TRCAgentProxy agentProxy) {
		if(agentProxy == null)
			return;

		if (!agentProxy.eIsSet(HierarchyPackage.eINSTANCE.getTRCAgentProxy_StartTime())) {
			agentProxy.setStartTime(time);

		}

		TRCAgent agent = (TRCAgent)agentProxy.eGet(HierarchyPackage.eINSTANCE.getTRCAgentProxy_Agent(),false);

		if ((agent != null) && !agent.eIsProxy() && (!agent.eIsSet(HierarchyPackage.eINSTANCE.getTRCAgent_StartTime()))) {
			agent.setStartTime(time);
		}
	}

	public static void setSessionStopTime(double time, TRCAgentProxy agentProxy) {
		if(agentProxy == null)
			return;

		if (!agentProxy.eIsSet(HierarchyPackage.eINSTANCE.getTRCAgentProxy_StopTime())) {
			agentProxy.setStopTime(time);

		}

		TRCAgent agent = (TRCAgent)agentProxy.eGet(HierarchyPackage.eINSTANCE.getTRCAgentProxy_Agent(),false);

		if ((agent != null) && !agent.eIsProxy() && (!agent.eIsSet(HierarchyPackage.eINSTANCE.getTRCAgent_StopTime()))) {
			agent.setStopTime(time);
		}
	}

	/**
	 * @param string
	 * @return
	 */
	public static CorrelationEngine findCorrelationEngine(String string) {
		return (CorrelationEngine)LookupServiceExtensions.getInstance().locate(null,CorrelationEngineImpl.class,string);
	}

	/**
	 * @param correlationEngine
	 */
	public static void addCorrelationEngine(CorrelationEngine correlationEngine) {
		Resource resource = HierarchyResourceSetImpl.getInstance().getResource(URI.createURI("http://www.eclipse.org/hyades/correlations.xmi"),false);
		resource.getContents().add(correlationEngine);
	}

	/**
	 * 
	 * @param agentProxy
	 * @param correlationEngine
	 * @return the correlation container proxy for an internal correlation container
	 */
	public static CorrelationContainerProxy getInternalCorrelationContainerProxy(TRCAgentProxy agentProxy, String correlationEngineID) {
		URI internalCorrelationURI = createCorrelationContainerURI(agentProxy, correlationEngineID, "internalcc");
		for (Iterator iter = agentProxy.getInternalCorrelations().iterator(); iter.hasNext();) {
			CorrelationContainerProxy correlationContainerProxy = (CorrelationContainerProxy) iter.next();
			CorrelationContainer correlationContainer = correlationContainerProxy.getCorrelationContainer();
			if(correlationContainer!=null)
			{	
				URI ccURI = correlationContainer.eIsProxy() ? ((InternalEObject)correlationContainer).eProxyURI() : correlationContainer.eResource().getURI();
				if(!correlationContainerProxy.eIsProxy() && ccURI.equals(internalCorrelationURI))
					return correlationContainerProxy;
			}
			else
			{
				correlationContainer = HierarchyFactory.eINSTANCE.createCorrelationContainer();
				correlationContainer.setCorrelationContainerProxy(correlationContainerProxy);
				createCorrelationContainerResource(agentProxy,correlationEngineID,correlationContainer,"internalcc");
				return correlationContainerProxy;
			}
		}
		CorrelationContainerProxy correlationContainerProxy = HierarchyFactory.eINSTANCE.createCorrelationContainerProxy();
		correlationContainerProxy.getCorrelatedAgents().add(agentProxy);
		agentProxy.getInternalCorrelations().add(correlationContainerProxy);
		correlationContainerProxy.setCreationTime(new Date().getTime());
		//correlationContainerProxy.setName("InternalCorrelation_"+correlationEngineID);
		correlationContainerProxy.setName(InternalCorrelationName);
		correlationContainerProxy.setMonitor(agentProxy.getProcessProxy().getNode().getMonitor());
		agentProxy.getProcessProxy().eResource().setModified(true);
		agentProxy.getProcessProxy().getNode().getMonitor().eResource().setModified(true);
		return correlationContainerProxy;
	}
	
	public static String getInternalCorrelationName(String correlationEngineName) {
		String name;
		if (ModelsHierarchyPlugin.getPlugin() == null) {
			DateFormat formatter = DateFormat.getDateTimeInstance(DateFormat.MEDIUM, DateFormat.MEDIUM, ULocale.getDefault());
			String formatDate = formatter.format(new Date());
			name = correlationEngineName != null ? correlationEngineName + "_Internal_" + formatDate : InternalCorrelationName + formatDate;
		} else {
			long i = ModelsHierarchyPlugin.getPlugin().getPluginPreferences().getLong("internalCorrelation") + 1;
			ModelsHierarchyPlugin.getPlugin().getPluginPreferences().setValue("internalCorrelation", i);
			name = correlationEngineName != null ? correlationEngineName + "_Internal_" + i : InternalCorrelationName + i;
		}
		return name;
	}

	public static CorrelationContainerProxy getCorrelationContainerProxy(TRCAgentProxy agentProxy, String correlationEngineID,String correlationTag, String sessionID) {
		CorrelationContainerProxy correlationContainerProxy = HierarchyFactory.eINSTANCE.createCorrelationContainerProxy();
		CorrelationContainer correlationContainer = HierarchyFactory.eINSTANCE.createCorrelationContainer();
		correlationContainer.setCorrelationContainerProxy(correlationContainerProxy);
		createCorrelationContainerResource(agentProxy,correlationEngineID,correlationContainer,correlationTag, sessionID);

		correlationContainerProxy.getCorrelatedAgents().add(agentProxy);
//		agentProxy.getInternalCorrelations().add(correlationContainerProxy);
		correlationContainerProxy.setCreationTime(new Date().getTime());
		correlationContainerProxy.setName(correlationTag+"_"+correlationEngineID);
		correlationContainerProxy.setMonitor(agentProxy.getProcessProxy().getNode().getMonitor());
//		agentProxy.getProcessProxy().eResource().setModified(true);
		agentProxy.getProcessProxy().getNode().getMonitor().eResource().setModified(true);
		return correlationContainerProxy;
	}

	public static CorrelationContainerProxy getSQLCorrelationContainerProxy(TRCAgentProxy agentProxy, String correlationEngineID, String correlationTag){
		return getSQLCorrelationContainerProxy(agentProxy, correlationEngineID, correlationTag, null);
	}
	
	public static CorrelationContainerProxy getSQLCorrelationContainerProxy(TRCAgentProxy agentProxy, String correlationEngineID, String correlationTag, String sessionID){
		
		URI uri = EcoreUtil.getURI(agentProxy.getAgent());
		Properties p = HyadesExtendedResourceFactory.getPropertiesFromURI(uri);
		String storeType =
		((IHyadesResourceExtension)HyadesResourceExtensions.getInstance().get(LoadersUtils.getPostfix(uri.toString()))).getStoreType(LoadersUtils.getPostfix(uri.toString()));
		HyadesResourceExtensions.getInstance().setProperties(storeType,p);


		CorrelationContainerProxy correlationContainerProxy = HierarchyFactory.eINSTANCE.createCorrelationContainerProxy();
		CorrelationContainer correlationContainer = HierarchyFactory.eINSTANCE.createCorrelationContainer();
		// create correlation container URI
		URI agentURI=EcoreUtil.getURI((TRCAgent)agentProxy.eGet(HierarchyPackage.eINSTANCE.getTRCAgentProxy_Agent(), false));
		String agentPostfix = LoadersUtils.getPostfix(agentURI);
		String ccPostfix = getCorrelationNamePostfix(storeType);
		String ccs = StringUtil.change(agentURI.toString(), agentPostfix, "_" + correlationTag + "-" +correlationEngineID+ccPostfix);
		
		URI correlationContainerResourceURI = SaveUtil.createURI(getContainerResourceURI(ccs));
		
		Resource ccr = HierarchyResourceSetImpl.getInstance(sessionID).createResource(correlationContainerResourceURI);
		ccr.getContents().add(correlationContainer);

		correlationContainerProxy.setCorrelationContainer(correlationContainer);
		correlationContainerProxy.getCorrelatedAgents().add(agentProxy);
		correlationContainerProxy.setCreationTime(new Date().getTime());
		correlationContainerProxy.setName(correlationTag + "_" + correlationEngineID);
		correlationContainerProxy.setMonitor(agentProxy.getProcessProxy().getNode().getMonitor());		
		agentProxy.getProcessProxy().getNode().getMonitor().eResource().setModified(true);
		correlationContainerProxy.setName(correlationTag);
		return correlationContainerProxy;
	}
	
	public static String getContainerResourceURI(String uri) {
		int i = uri.indexOf('#');
		if(i!=-1)
			uri=uri.substring(0,i);
		return uri;
	}
	
	public static String getCorrelationNamePostfix(String dbType) {
		String res=null;
//		Preferences store = ModelsPlugin.getPlugin().getPluginPreferences();
//		String activeDBType = store.getString(IHyadesResourceExtension.LARGE_RESOURCE_SUPPORT_CURRENT_DB_TYPE);
		for (Iterator iter = HyadesResourceExtensions.getInstance().entrySet().iterator(); iter.hasNext();) {
			Map.Entry entry = (Map.Entry) iter.next();
			if(entry.getKey() instanceof String)
			{
				IHyadesResourceExtension hyadesResourceFactory = (IHyadesResourceExtension)entry.getValue();
				String postfix = (String)entry.getKey();
				if(postfix.endsWith(".corrdb") && hyadesResourceFactory.getStoreType(postfix).equals(dbType))
				{
					return postfix;
				}
			}
		}

		if(res==null)
			res = ".corrxmi";
		return res;
	}
	

	public static URI createCorrelationContainerURI(TRCAgentProxy agentProxy, String correlationEngineID, String correlationTag) {
		URI internalCorrelationURI = agentProxy.getAgent().eResource().getURI();
		return createCorrelationContainerURI(internalCorrelationURI, correlationEngineID, correlationTag);
	}
	public static URI createCorrelationContainerURI(URI agentURI, String correlationEngineID, String correlationTag) {
		URI internalCorrelationURI = agentURI;
		//		String fileExt = internalCorrelationURI.fileExtension();
		internalCorrelationURI = internalCorrelationURI.trimFileExtension();
		String lastSegment = internalCorrelationURI.lastSegment() + "_" + (correlationTag + "-" + correlationEngineID + "-" + (new Date().getTime())).hashCode();
		internalCorrelationURI = internalCorrelationURI.trimSegments(1);
		internalCorrelationURI = internalCorrelationURI.appendSegment(lastSegment);
		internalCorrelationURI = internalCorrelationURI.appendFileExtension("corrxmi");
		return internalCorrelationURI;
	}

	/**
	 * @param container
	 */
	public static void createCorrelationContainerResource(TRCAgentProxy proxy, String correlationEngineID, CorrelationContainer container,String correlationTag) {
		createCorrelationContainerResource(proxy, correlationEngineID, container, correlationTag, null);
	}
	
	
	public static void createCorrelationContainerResource(TRCAgentProxy proxy, String correlationEngineID, CorrelationContainer container,String correlationTag, String sessionID) {
		URI internalCorrelationURI = createCorrelationContainerURI(proxy, correlationEngineID,correlationTag);
		Resource resource = HierarchyResourceSetImpl.getInstance(sessionID).createResource(internalCorrelationURI);
		resource.setModified(true);
		resource.getContents().add(container);
	}

	/**
	 * @param uri
	 * @return
	 */
	public static String getPostfix(URI uri) {
		String postfix = uri.trimFragment().toString();
		return getPostfix(postfix);
	}

	public static String getPostfix(String postfix) {
		int index = postfix.lastIndexOf("__");
		if (index > 0) {
			int fragmentIndex = postfix.indexOf("#", index);
			if (fragmentIndex > 0)
				postfix = postfix.substring(index, fragmentIndex);
			else
				postfix = postfix.substring(index);
		} else if(postfix.lastIndexOf('.') > -1){
				postfix = postfix.substring(postfix.lastIndexOf('.'));	
		}		
		return postfix;
	}

//	/**
//	 * @param correlationContainerProxy
//	 */
//	public static void addCorrelation(CorrelationContainerProxy correlationContainerProxy) {
//		String key="";
//		for (Iterator iter = correlationContainerProxy.getCorrelatedAgents().iterator(); iter.hasNext();) {
//			TRCAgentProxy element = (TRCAgentProxy) iter.next();
//			key = key+"_"+element.hashCode();
//		}
//		correlationMap.put(key,correlationContainerProxy.eResource().getURI());
//	}
//	/**
//	 * @param correlationContainerProxy
//	 */
//	public static void getCorrelation(List agents) {
//		String key=correlationContainerProxy.getCorrelationEngine().getId();
//		for (Iterator iter = correlationContainerProxy.getCorrelatedAgents().iterator(); iter.hasNext();) {
//			TRCAgentProxy element = (TRCAgentProxy) iter.next();
//			key = key+"_"+element.hashCode();
//		}
//		correlationMap.put(key,correlationContainerProxy.eResource().getURI());
//	}
	public static String scramblePassword(String password) throws InvalidPassword {
		if(password==null || password.length()==0)
			return "";
		int length = password.length();
		char[] out = new char[length];
		for (int i = 0; i < length; i++) {
			char value = password.charAt(i);
			if (value < 0 || value > 255)
				throw new InvalidPassword(password);
			out[i] = SCRAMBLING_TABLE[value];
		}
		return "A" + new String(out);//$NON-NLS-1$
	}

	protected static final char[] SCRAMBLING_TABLE = new char[]{0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 114, 120, 53, 79, 96, 109, 72, 108, 70, 64, 76, 67, 116, 74, 68, 87, 111, 52, 75, 119, 49, 34, 82, 81, 95, 65, 112, 86, 118, 110, 122, 105, 41, 57, 83, 43, 46, 102, 40, 89, 38, 103, 45, 50, 42, 123,
			91, 35, 125, 55, 54, 66, 124, 126, 59, 47, 92, 71, 115, 78, 88, 107, 106, 56, 36, 121, 117, 104, 101, 100, 69, 73, 99, 63, 94, 93, 39, 37, 61, 48, 58, 113, 32, 90, 44, 98, 60, 51, 33, 97, 62, 77, 84, 80, 85, 223, 225, 216, 187, 166, 229, 189, 222, 188, 141, 249, 148, 200, 184, 136, 248, 190, 199, 170, 181, 204, 138, 232, 218, 183, 255, 234, 220, 247, 213, 203, 226,
			193, 174, 172, 228, 252, 217, 201, 131, 230, 197, 211, 145, 238, 161, 179, 160, 212, 207, 221, 254, 173, 202, 146, 224, 151, 140, 196, 205, 130, 135, 133, 143, 246, 192, 159, 244, 239, 185, 168, 215, 144, 139, 165, 180, 157, 147, 186, 214, 176, 227, 231, 219, 169, 175, 156, 206, 198, 129, 164, 150, 210, 154, 177, 134, 127, 182, 128, 158, 208, 162, 132, 167, 209,
			149, 241, 153, 251, 237, 236, 171, 195, 243, 233, 253, 240, 194, 250, 191, 155, 142, 137, 245, 235, 163, 242, 178, 152};
	protected static char[] RESTORING_TABLE=null;

	protected final static void resetRESTORINGTALBE() {
		if(RESTORING_TABLE!=null)
		{
			return;
		}
		RESTORING_TABLE=new char[256];
		for (int i = 0; i < 256; i++) {
			int value = SCRAMBLING_TABLE[i];
			RESTORING_TABLE[value] = (char) i;
		}
	}
	public static String restorePassword(String password) throws InvalidPassword {
		if(password==null || password.length()==0)
			return "";
		int length = password.length();
		resetRESTORINGTALBE();
		char[] out = new char[length-1];
		for (int i = 1; i < length; i++) {
			char value = password.charAt(i);
			if (value < 0 || value > 255)
				throw new InvalidPassword(password);
			out[i-1] = RESTORING_TABLE[value];
		}
		return new String(out);
	}

	/**
	 * @param statements
	 * @return
	 */
	public static String toString(String[] statements) {
		StringBuffer res= new StringBuffer();
		if(statements!=null)
		{
			res.append('[');
			boolean first=true;
			for (int i = 0; i < statements.length; i++) {
				if(first)
				{
					first=false;
				}
				else
				{
					res.append(",\n");
				}
				res.append(statements[i]);
			}
			res.append(']');
		}
		return res.toString();
	}
	public static String getFirstLine(String name, int maxLength){
		if(name==null)
			return null;

		int idx = name.indexOf("\r\n", 0);
		if(idx != -1)
			name = name.substring(0, idx<maxLength ? idx:maxLength-1);
		else{
		 idx = name.indexOf("\n");
		 if(idx != -1)
		 	name = name.substring(0, idx<maxLength ? idx:maxLength-1);
		}
		name = name.replace('\t',' ');
		return name;
	}

	public static char[] getChars(StringBuffer sb2) {
		char[] c = new char[sb2.length()];
		sb2.getChars(0,sb2.length(),c,0);
		return c;
	}

	public static char[] makeCharArray(byte[] buffer, int offset, int length) {
		char[] res = new char[Math.min(buffer.length,length+offset)];
		for (int i = res.length-1; i <= 0; i--) {
			res[i]=(char)buffer[i+offset];
		}
		return res;
	}

	public static double parseTraceTime(String value) {
		double res=0;
		if(value!=null)
		{
			long temp =0;
			int power = 0;
			for (int i = value.length()-1; i-- > 0 ;) {
				char c = value.charAt(i);
				if(c=='.' || c==',')
				{
//					System.out.println("temp1="+temp);
					res = temp / (double)powerOf10[power];
//					System.out.println("res1="+res);
					temp = 0;
					power = 0;
				}
				else
				{
					temp = temp + (c-'0') * powerOf10[power++];
				}
			}
//			System.out.println("temp2="+temp);
			res += temp;
		}
		return res;
	}
	public static long parseLongTraceTime(String value) {
		long res=0;
		if(value!=null)
		{
			int power = 0;
			for (int i = value.length(); i-- > 0 ;) {
				char c = value.charAt(i);

				if(c=='.' || c==',')
				{
					if(power==9)
					{
						continue;
					}
					else if(power<9)
					{
						res = res * powerOf10[9-power]; // convert to nanoseconds
						power = 9;
					}
					else
					{
						res = res / powerOf10[power-9]; // trim to nanoseconds
						power = 9;
					}
						
				}
				else
				{
					res = res + (c-'0') * powerOf10[power++];
				}
			}
		}
		return res;
	}

	public static void main(String[] args) {
		System.out.println("1118198325.526252508 = " + parseLongTraceTime("1118198325.526252508"));
		System.out.println("1118198325.52625250 = " + parseLongTraceTime("1118198325.52625250"));
		System.out.println("1118198325.5262 = " + parseLongTraceTime("1118198325.5262"));
		
		//largest number covers about 295 years in nanoseconds starting with year 1970
		System.out.println("9199999999.999999998 = " + parseLongTraceTime("9199999999.999999998"));
		
		int iterations = 10000000;
		PerfUtil p = PerfUtil.createInstance("parseTraceTime iterations="+iterations,true);
		System.out.println("1118198325.526252508 = " + parseTraceTime("1118198325.526252508"));
		for (int i = iterations; i-- >0;) {
			parseTraceTime("1118198325.526252508");
		}
		p.stopAndPrintStatus();
//		p.setMessageAndStart("Double.parseDouble iterations="+iterations);
//	
//		System.out.println("1118198325.526252508 = " + Double.parseDouble("1118198325.526252508"));
//		for (int i = iterations; i-- >0;) {
//			
//			Double.parseDouble("1118198325.526252508");
//		}
//		p.stopAndPrintStatus();
		p.setMessageAndStart("Long.parseLong iterations="+iterations);
		
		System.out.println("1118198325526252508 = " + Long.parseLong("1118198325526252508"));
		for (int i = iterations; i-- >0;) {
			
			Long.parseLong("1118198325526252508");
		}
		
		p.stopAndPrintStatus();
		p.setMessageAndStart("parseLongTraceTime iterations="+iterations);

		System.out.println("1118198325.526252508 = " + parseLongTraceTime("1118198325.526252508"));
		for (int i = iterations; i-- >0;) {
			
			parseLongTraceTime("1118198325.526252508");
		}
		
		p.stopAndPrintStatus();
		p.setMessageAndStart("parseTraceTime iterations="+iterations);
		System.out.println("1118198325.526252508 = " + parseTraceTime("1118198325.526252508"));
		for (int i = iterations; i-- >0;) {
			parseTraceTime("1118198325.526252508");
		}
		p.stopAndPrintStatus();
		p.setMessageAndStart("parseLongTraceTime iterations="+iterations);

		System.out.println("1118198325.526252508 = " + parseLongTraceTime("1118198325.526252508"));
		for (int i = iterations; i-- >0;) {
			
			parseLongTraceTime("1118198325.526252508");
		}
		
		p.stopAndPrintStatus();
	}

	public static InputStream getFirstFileInputStream(String fileName) {
		try 
		{
			if(isZipFile(fileName))
			{
				ZipFile zf = new ZipFile(fileName);
				Enumeration entries = zf.entries();

				while (entries.hasMoreElements()) {
					ZipEntry zipEntry = (ZipEntry) entries.nextElement();
		            return zf.getInputStream(zipEntry);
		        }
				return new ZipInputStream(new BufferedInputStream(new FileInputStream(fileName)));
			}
			else
			{
				return new BufferedInputStream(new FileInputStream(fileName));
				
			}
		}catch (Exception e) {
			ModelDebugger.log(e);
			return null;
		}
	}
    public static boolean isZipFile(InputStream readStream) throws IOException {
        byte[] magic = new byte[2];

        if (readStream.read(magic) > 1) {
            if ((magic[0] == 'P') && (magic[1] == 'K')) {
                return true;
            }
        }

        return false;
    }

    public static boolean isZipFile(String fileName) throws IOException {
        FileInputStream readStream = new FileInputStream(fileName);
        boolean ret = isZipFile(readStream);

        readStream.close();
        readStream = null;

        return ret;
    }

}
