/**********************************************************************
 * Copyright (c) 2003, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LookupServiceExtensions.java,v 1.3 2008/01/24 02:28:17 apnan Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 * $Id: LookupServiceExtensions.java,v 1.3 2008/01/24 02:28:17 apnan Exp $
 **********************************************************************/
package org.eclipse.hyades.loaders.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.emf.common.util.BasicEMap;
import org.eclipse.emf.common.util.EMap;


/**
 * @author slavescu
 */
public class LookupServiceExtensions extends HashMap implements ExtendedLookupService {
    //~ Static fields/initializers -----------------------------------------------------------------

    /**
	 * 
	 */
	protected static final long serialVersionUID = 3978992054695442481L;

    protected static LookupServiceExtensions instance;

    //~ Instance fields ----------------------------------------------------------------------------

    protected EMap alreadyProcessed;
    protected Object result;

	protected EMap ignoredTypes;

    //~ Constructors -------------------------------------------------------------------------------

    /**
     *
     */
    protected LookupServiceExtensions() {
        super();
        alreadyProcessed = new BasicEMap() {
        	/**
			 * 
			 */
			protected static final long serialVersionUID = 3978992054695442481L;
        	protected boolean useEqualsForKey() {
        		return false;
        	}
        };
        ignoredTypes = new BasicEMap() {
        	/**
			 * 
			 */
			protected static final long serialVersionUID = 3978992054695442481L;
        	protected boolean useEqualsForKey() {
        		return false;
        	}
        };
    }

    //~ Methods ------------------------------------------------------------------------------------

    /**
     *
     */
    public static LookupServiceExtensions getInstance() {
        if (instance == null) {
            instance = new LookupServiceExtensions();
            new LookupServiceRegistryReader().readRegistry();
        }

        return instance;
    }

    public Set getSupportedTypes() {
        return keySet();
    }

    public boolean deregister(HierarchyContext context, Class type, Object key) {
        LookupService lookupService = (LookupService) get(type);

        if (lookupService != null) {
            return lookupService.deregister(context, type, key);
        }

        return false;
    }
    public boolean deregister(HierarchyContext context, Class type, int key) {
        ExtendedLookupService lookupService = (ExtendedLookupService) get(type);

        if (lookupService != null) {
            return lookupService.deregister(context, type, key);
        }

        return false;
    }
    public boolean deregister(HierarchyContext context, Class type, long key) {
    	ExtendedLookupService lookupService = (ExtendedLookupService) get(type);

        if (lookupService != null) {
            return lookupService.deregister(context, type, key);
        }

        return false;
    }


    public boolean deregister(HierarchyContext context, Object object) {
        LookupService lookupService = (LookupService) get(object.getClass());

        if (lookupService != null) {
            return lookupService.deregister(context, object);
        }

        return false;
    }

    public boolean deregister(HierarchyContext context, Class type) {
        LookupService lookupService = (LookupService) get(type);

        if (lookupService != null) {
            return lookupService.deregister(context, type);
        }

        return false;
    }

    public boolean deregister(HierarchyContext context) {
        boolean ret = false;
        List processed = new ArrayList();

        for (Iterator iter = entrySet().iterator(); iter.hasNext();) {
            Map.Entry entry = (Map.Entry) iter.next();
            LookupService lookupService = (LookupService) entry.getValue();

            if (!processed.contains(lookupService)) {
                if (!ret) {
                    ret = lookupService.deregister(context);
                } else {
                    lookupService.deregister(context);
                }

                processed.add(lookupService);
            }
        }

        return ret;
    }

    public synchronized Object locate(HierarchyContext context, Class type, Object key) {
        LookupService lookupService = (LookupService) get(type);

        if (lookupService != null) {
            return lookupService.locate(context, type, key);
        } else {
			alreadyProcessed.clear();
        	if(ignoredTypes.containsKey(type))
        		return null;

			for (Iterator iter = values().iterator(); iter.hasNext();) {
				LookupService element = (LookupService) iter.next();

				if (alreadyProcessed.containsKey(element)) {
					continue;
				}

				result = element.locate(context,type, key);

				if (result != null) {
					// associate type with the lookup service to speed up future calls
					put(type,element);

					alreadyProcessed.clear();

					return result;
				}

				alreadyProcessed.put(element,null);
			}
			alreadyProcessed.clear();
            return null;//locate(context, key);
        }
    }

    public synchronized Object locate(HierarchyContext context, Class type, int key) {
        ExtendedLookupService lookupService = (ExtendedLookupService) get(type);

        if (lookupService != null) {
        		return lookupService.locate(context, type, key);
        } else {
			alreadyProcessed.clear();
        	if(ignoredTypes.containsKey(type))
        		return null;

			for (Iterator iter = values().iterator(); iter.hasNext();) {
				Object element1 = iter.next();
					
				if(!(element1 instanceof ExtendedLookupService))
					continue;
				ExtendedLookupService element = (ExtendedLookupService)element1;
				if (alreadyProcessed.containsKey(element)) {
					continue;
				}

				result = element.locate(context,type, key);

				if (result != null) {
					// associate type with the lookup service to speed up future calls
					put(type,element);

					alreadyProcessed.clear();

					return result;
				}

				alreadyProcessed.put(element,null);
			}
			alreadyProcessed.clear();
            return null;//locate(context, key);
        }
    }

    public synchronized Object locate(HierarchyContext context, Class type, long key) {
        ExtendedLookupService lookupService = (ExtendedLookupService) get(type);

        if (lookupService != null) {
        		return lookupService.locate(context, type, key);
        } else {
			alreadyProcessed.clear();
        	if(ignoredTypes.containsKey(type))
        		return null;

			for (Iterator iter = values().iterator(); iter.hasNext();) {
				Object element1 = iter.next();
				
				if(!(element1 instanceof ExtendedLookupService))
					continue;
				ExtendedLookupService element = (ExtendedLookupService)element1;

				if (alreadyProcessed.containsKey(element)) {
					continue;
				}

				result = element.locate(context,type, key);

				if (result != null) {
					// associate type with the lookup service to speed up future calls
					put(type,element);

					alreadyProcessed.clear();

					return result;
				}

				alreadyProcessed.put(element,null);
			}
			alreadyProcessed.clear();
            return null;//locate(context, key);
        }
    }


    public Object locate(HierarchyContext context, Object key) {
        alreadyProcessed.clear();

        for (Iterator iter = values().iterator(); iter.hasNext();) {
            LookupService element = (LookupService) iter.next();

            if (alreadyProcessed.containsKey(element)) {
                continue;
            }

            result = element.locate(context, key);

            if (result != null) {
                alreadyProcessed.clear();

                return result;
            }

            alreadyProcessed.put(element,null);
        }

        alreadyProcessed.clear();

        return null;
    }

    public synchronized boolean register(HierarchyContext context, Object object) {
        LookupService lookupService;

        if (object.getClass() == GenericLookUpEntry.class) {
            lookupService = (LookupService) get(((GenericLookUpEntry) object).value.getClass());
        } else {
            lookupService = (LookupService) get(object.getClass());
        }

        if (lookupService != null) {
            return lookupService.register(context, object);
        } else {
        	Class type = object.getClass();
        	if(ignoredTypes.containsKey(type))
        		return false;
			alreadyProcessed.clear();

			for (Iterator iter = values().iterator(); iter.hasNext();) {
				LookupService element = (LookupService) iter.next();


				if (alreadyProcessed.containsKey(element)) {
					continue;
				}

				boolean ret = element.register(context,object);

				if (ret) {
					// associate type with the lookup service to speed up future calls
					put(type,element);

					alreadyProcessed.clear();

					return ret;
				}

				alreadyProcessed.put(element,null);
			}
			alreadyProcessed.clear();
			ignoredTypes.put(type,null);
        }

        return false;
    }
}


/**
 * A plugin extension reader that populates the
 * loaders registry.
 * Clients are not expected to use this class directly.
 */
class LookupServiceRegistryReader extends RegistryReader {
    //~ Static fields/initializers -----------------------------------------------------------------

    static final String TAG_LOADER = "lookupService";

    //~ Constructors -------------------------------------------------------------------------------

    public LookupServiceRegistryReader() {
        super(RegistryReader.LOOKUP_SERVICE_PPID);
    }

    //~ Methods ------------------------------------------------------------------------------------

    protected boolean readElement(HyadesConfigurationElement element) {
        if (element.getName().equals(TAG_LOADER)) {
            if (element.getAttribute(ATT_CLASS) == null) {
                logMissingAttribute(element, ATT_CLASS);
            } else {
                try {
                    IHyadesPluginClassDescriptor classDescriptor = createPluginClassDescriptor(element, ATT_CLASS);
                    LookupService lookupService = (LookupService) classDescriptor.createInstance();
                	
                    float priority = 0;
                    if (element.getAttribute(ATT_PRIORITY) != null) {
                    	priority = Float.parseFloat(element.getAttribute(ATT_PRIORITY));
                    }
                    if (lookupService != null) {
                        //reuse instances of the lookup service
                        for (Iterator iter = LookupServiceExtensions.getInstance().values().iterator();
                                 iter.hasNext();) {
                            LookupService l = (LookupService) iter.next();

                            if (element.getClass() == lookupService.getClass()) {
                                lookupService = l;

                                break;
                            }
                        }

                        for (Iterator iter = lookupService.getSupportedTypes().iterator();
                                 iter.hasNext();) {
                            Class type = (Class) iter.next();

                            if(LookupServiceExtensions.getInstance().containsKey(type) && ((Float)getPriorities().get(type)).floatValue() >= priority)
                            {
                            	logError(element, "A lookupService for this element with a higher or equal priority already exists, lookupService \""+lookupService.getClass().getName()+"\" will be ignored for type \""+type.getName()+"\".");
                            }
                            else
                            {
                            	LookupServiceExtensions.getInstance().put(type, lookupService);
                            	getPriorities().put(type, new Float(priority));
                            }
                        }
                    } else {
                        logError(element, "Error when creating instance of lookup service: " + element.getAttribute(ATT_CLASS));
                    }
                } catch (Exception e) {
                    logError(element, e.getLocalizedMessage());
                }

                return true;
            }
        }

        return false;
    }
}
