/**********************************************************************
 * Copyright (c) 2003, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EObjectsTraverser.java,v 1.2 2008/05/30 18:36:08 dmorris Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.models.hierarchy.util;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.util.ModelDebugger;

/**
 * @author slavescu
 */
public abstract class EObjectsTraverser {
	protected static final int AFTER_CHILDREN = 3;

	protected static final int BEFORE_CHILDREN = 1;

	protected Collection contents;

	protected List visitors;

	protected boolean pruneSubtree;

	public EObjectsTraverser() {
		visitors = new FastList();
	}

	public EObjectsTraverser(Collection contents) {
		this.contents = contents;
		visitors = new FastList();
	}

	public void dispose() {
		visitors.clear();
		contents = null;
	}

	protected boolean afterChildren(EObject element) {
		return callVisitors(AFTER_CHILDREN, element);
	}

	protected boolean beforeChildren(EObject element) {
		return callVisitors(BEFORE_CHILDREN, element);
	}

	public EObjectsTraverser registerVisitors(EObjectVisitor visitor) {
		if (visitor != null)
			visitors.add(visitor);
		return this;
	}

	public EObjectsTraverser deregisterVisitor(EObjectVisitor visitor) {
		if (visitor != null)
			visitors.remove(visitor);
		return this;
	}

	public EObjectsTraverser deregisterAllVisitors() {
		visitors.clear();
		return this;
	}

	protected boolean callVisitors(int flag, EObject element) {
		boolean ret = true;
		for (int i = visitors.size() - 1; i >= 0; i--) {
			try {
				switch (flag) {
				case BEFORE_CHILDREN:
					ret = ((EObjectVisitor) visitors.get(i)).beforeChildren(element) && ret;
					break;

				case AFTER_CHILDREN:
					ret = ((EObjectVisitor) visitors.get(i)).afterChildren(element) && ret;
					break;
				}
			} catch (RuntimeException re) {
				ModelDebugger.log(re);
			}
		}
		return ret;
	}

	public boolean traverse() {
		return traverse(contents.iterator());
	}

	protected boolean traverse(Iterator iter) {
		while (iter.hasNext()) {
			EObject traversedElement = (EObject) iter.next();
			EObject visitorElement = getElement(traversedElement);
			
			if (traversedElement != null  && visitorElement != null){
				
				if (!beforeChildren(visitorElement))
					return false;
	
				if (pruneSubtree)
					pruneSubtree = false;
				else
					traverseEObject(traversedElement);
	
				if (!afterChildren(visitorElement))
					return false;
	
				if (pruneSubtree) {
					pruneSubtree = false;
					break;
				}
			}
		}
		return true;
	}

	protected boolean traverse(List children) {
		for (Iterator iter = children.iterator(); iter.hasNext();) {
			EObject traversedElement = (EObject) iter.next();
			EObject visitorElement = getElement(traversedElement);
			if (!beforeChildren(visitorElement))
				return false;
			
			if (pruneSubtree)
				pruneSubtree = false;
			else
				traverseEObject(visitorElement);

			if (!afterChildren(visitorElement))
				return false;
			
			if (pruneSubtree) {
				pruneSubtree = false;
				break;
			}
		}
		return true;
	}

	protected abstract boolean traverseEObject(EObject object);

	protected abstract EObject getElement(EObject traversedElement);

	/**
	 * @return Returns the contents.
	 */
	public Collection getContents() {
		return contents;
	}

	/**
	 * @param contents
	 *            The contents to set.
	 */
	public void setContents(Collection contents) {
		this.contents = contents;
	}
}
