/**********************************************************************
 * Copyright (c) 2003, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HierarchyResourceSetImpl.java,v 1.3 2009/04/27 15:21:37 paules Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.models.hierarchy.util;

import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.common.util.WrappedException;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.URIConverter;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.models.hierarchy.util.internal.SessionManagerWrapper;


/**
 * This class allows the definition of
 * <p>
 * 1. The singleton to be used by the clients<p>
 * 2. The <code>URIConverter</code> - it must use ZipWorkbenchURIConverter<p>
 * 3. The use of the <code>PerftraceReosurceFactory</code> as the factory for the Agent resources.
 * This factory will handle the definition of keys to be used during the data load process.
 * 4. The Unresolved resource helper
 */
public class HierarchyResourceSetImpl extends ResourceSetImpl {
    //~ Static fields/initializers -----------------------------------------------------------------

    protected static HierarchyResourceSetImpl instance;

    //~ Instance fields ----------------------------------------------------------------------------

    protected IUnresolvedResourceHelper helper;

    //~ Constructors -------------------------------------------------------------------------------

    protected HierarchyResourceSetImpl() {
        super();
    }

    //~ Methods ------------------------------------------------------------------------------------

    public static ResourceSet getInstance(String sessionID) {
    	if (sessionID != null){
    		return SessionManagerWrapper.instance().getResourceSet(sessionID);
    	}
    	else {
    		return HierarchyResourceSetImpl.getInstance();
    	}
    }
    public static HierarchyResourceSetImpl getInstance() {
   		if (instance == null) {
            instance = new HierarchyResourceSetImpl();
            instance.setURIConverter(new HierarchyURIConverterImpl());

            //			Resource.Factory xmiResourceFactory = new XMIResourceFactoryImpl();
            Resource.Factory perftraceResourceFactory = new HierarchyResourceFactoryImpl();

            Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("trcmxmi", perftraceResourceFactory);
            Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("trcnxmi", perftraceResourceFactory);
            Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("trcpxmi", perftraceResourceFactory);
            Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("trcaxmi", perftraceResourceFactory);
			Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("corrxmi", perftraceResourceFactory);
			Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("trclxmi", perftraceResourceFactory);
			Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("trciovxmi", perftraceResourceFactory);
			Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("trcdbxmi", perftraceResourceFactory);
			
			Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("trcadb", new HyadesExtendedResourceFactory());
			Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("corrdb", new HyadesExtendedResourceFactory());
			Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("symptomdb", new HyadesExtendedResourceFactory());
		}
		return instance;  
    }

    /**
     * If an object is not found with the given uri, load the
     * resource that has it.
     */
    public EObject getObjectAndLoad(URI uri) throws WrappedException {
        EObject obj = null;

        try {
            obj = super.getEObject(uri, true);
        } catch (WrappedException e) {
            if (helper != null) {
                helper.uriException(uri, (Exception) e);
            }

            throw e;
        }

        return obj;
    }

    public IUnresolvedResourceHelper getUnresolvedException() {
        return helper;
    }

    public void setUnresolvedResourceHelper(IUnresolvedResourceHelper helper) {
        this.helper = helper;
    }

    /*
     * Javadoc copied from interface.
     * @provisional
     */
    public Resource getResource(URI uri, boolean loadOnDemand)
    {
      Map map = getURIResourceMap();
      if (map != null)
      {
        Resource resource = (Resource) map.get(uri);
        if (resource != null)
        {
          if (loadOnDemand && !resource.isLoaded())
          {
            demandLoadHelper(resource);
          }
          return resource;
        }
      }

      URIConverter theURIConverter = getURIConverter();
      URI normalizedURI = theURIConverter.normalize(uri);
      List res = getResources();
      for (Iterator iterator = res.iterator(); iterator.hasNext();) {
		Resource resource = (Resource) iterator.next();
        if (resource!=null && theURIConverter.normalize(resource.getURI()).equals(normalizedURI))
        {
          if (loadOnDemand && !resource.isLoaded())
          {
            demandLoadHelper(resource);
          }

          if (map != null)
          {
            map.put(uri, resource);
          }
          return resource;
        }
      }

      Resource delegatedResource = delegatedGetResource(uri, loadOnDemand);
      if (delegatedResource != null)
      {
        if (map != null)
        {
          map.put(uri, delegatedResource);
        }
        return delegatedResource;
      }

      if (loadOnDemand)
      {
        Resource resource = demandCreateResource(uri);
        if (resource == null)
        {
          throw new RuntimeException("Cannot create a resource for '" + uri + "'; a registered resource factory is needed");
        }

        demandLoadHelper(resource);

        if (map != null)
        {
          map.put(uri, resource);
        }
        return resource;
      }

      return null;
    }
    
    
}
