/**********************************************************************
 * Copyright (c) 2003, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HierarchyXMIResourceImpl.java,v 1.2 2009/10/19 15:54:37 paules Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.models.hierarchy.util;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.xmi.XMIResource;
import org.eclipse.emf.ecore.xmi.XMLLoad;
import org.eclipse.emf.ecore.xmi.XMLSave;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceImpl;
import org.eclipse.hyades.loaders.hierarchy.Constants;
import org.eclipse.hyades.loaders.util.AgentsContext;
import org.eclipse.hyades.loaders.util.HierarchyContext;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.loaders.util.LookupServiceExtensions;
import org.eclipse.hyades.loaders.util.ResourceExtensions;
import org.eclipse.hyades.loaders.util.XMLResourceLoader;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.UnresolvedCorrelation;
import org.eclipse.hyades.models.hierarchy.plugin.ModelsHierarchyPlugin;
import org.eclipse.hyades.models.hierarchy.util.internal.IntToObjectMapImpl;
import org.eclipse.hyades.models.hierarchy.util.internal.ObjectToIntMapImpl;
import org.eclipse.hyades.models.util.ModelDebugger;

/**
 * HierarchyXMIResourceImpl.java
 * 
 * 
 * @author  Paul Slauenwhite
 * @author  Alex Nan
 * @version October 19, 2009
 * @since   November 27, 2007
 */
public class HierarchyXMIResourceImpl extends XMIResourceImpl implements XMIResource, IHyadesExtendedResource {
	//~ Instance fields
	// -----------------------------------------------------

	public static final int INDEX_WATERMARK = 10000;

	//	private static final int INDEX_MAX = INDEX_WATERMARK;
	protected HierarchyContext context = null;

	protected boolean register = true;

	protected ArrayList intAndStrList = new ArrayList(INDEX_WATERMARK);

	//	private ArrayList featureList = new ArrayList();
	//	private ArrayList containerList = new ArrayList();
	protected InversedStringBuffer result = new InversedStringBuffer(2000);

	protected IntToObjectMap objectIndexes;

	public HierarchyXMIResourceImpl() {
		super();

		initHierarchyResource();
	}

	public HierarchyXMIResourceImpl(URI uri) {
		super(uri);

		initHierarchyResource();
	}

//	//~ Methods
//	// ------------------------------------------------------------------------------------
//
//	public EObject getEObject(String uriFragment) {
//		if (!isPath(uriFragment)) {
//			if (idToEObjectMap != null) {
//				EObject eObject = (EObject) idToEObjectMap.get(uriFragment);
//
//				return eObject;
//			} else {
//				return null;
//			}
//		}
//
//		return super.getEObject(uriFragment);
//	}

	public String getIndexStr(int index) {
		// index String slot size
		//   0 "0" 0 1
		//   1 "1" 1 2
		//   2 "2" 2 3
		int size = intAndStrList.size();

		if (index >= size) {
			intAndStrList.ensureCapacity(index + 1);

			for (int i = size; i < index; i++) {
				intAndStrList.add(i, null);
			}

			String indexStr = Integer.toString(index);

			intAndStrList.add(index, indexStr);

			return indexStr;
		}

		if (intAndStrList.get(index) == null) {
			String indexStr = Integer.toString(index);

			intAndStrList.set(index, indexStr);

			return indexStr;
		}

		return (String) intAndStrList.get(index);
	}

	public String getURIFragment(EObject eObject) {
		String id = EcoreUtil.getID(eObject);

		if (id != null) {
			return id;
		} else {
			result.setLength(0);

			for (EObject container = eObject.eContainer(); container != null; container = eObject.eContainer()) {
				EStructuralFeature eStructuralFeature = eObject.eContainmentFeature();

				if (eStructuralFeature.isMany()) {
					org.eclipse.emf.common.util.EList eList = (org.eclipse.emf.common.util.EList) container.eGet(eStructuralFeature, false);

					if (objectIndexes == null || objectIndexes.isEmpty()) {
						int index = eList.indexOf(eObject);

						if (index < INDEX_WATERMARK) {
							result.prepend( getIndexStr(index));
						} else {
							result.prepend( index);
						}
					} else {
						addIndex(eList, eObject,getListKey(container,eStructuralFeature));
					}

					result.prepend( '.');
					result.prepend( eStructuralFeature.getName());
					result.prepend( '@');
				} else {
					result.prepend( eStructuralFeature.getName());
					result.prepend( '@');
				}

				result.prepend( '/');
				eObject = container;
			}

			result.prepend( getURIFragmentRootSegment(eObject));
			result.prepend( '/');

			return result.toString();
		}
	}

	/**
	 * @param list
	 * @param object
	 * @return
	 */
	protected void addIndex(EList list, EObject object, int listKey) {
		if (list.size() < INDEX_WATERMARK) {
			int index = list.indexOf(object);
			result.prepend( getIndexStr(index));
			return;
		}

		ObjectToIntMap objects = (ObjectToIntMap) objectIndexes.get(listKey);
		if (objects == null) {
			objects = new ObjectToIntMapImpl(list.size());
			objectIndexes.put(list,objects);
			int index = list.indexOf(object);
			objects.put(object, index);
			if (index < INDEX_WATERMARK) {
				result.prepend( getIndexStr(index));
			} else {
				result.prepend( index);
			}
		} else {
			int index = objects.getInt(object);
			if (index == -1) {
				index = list.indexOf(object);
				objects.put(object, index);
				if (index < INDEX_WATERMARK) {
					result.prepend( getIndexStr(index));
				} else {
					result.prepend( index);
				}
			} else
				result.prepend( index);
		}

		return;
	}

	protected void putIndex(EList list, EObject object, int index, int key) {

		ObjectToIntMap objects = (ObjectToIntMap) objectIndexes.get(key);
		if (objects == null) {
			objects = new ObjectToIntMapImpl(list.size());
			objectIndexes.put(key,objects);
		}

		objects.put(object, index);
		return;
	}

//	protected Object getListKey(EObject container,EStructuralFeature structuralFeature) {
//		List key = new ArrayList(2);
//		key.add(container);
//		key.add(structuralFeature);
//		return key;
//	}
	protected int getListKey(EObject container,EStructuralFeature structuralFeature) {
		int hashCode = 31+container.hashCode();
		hashCode=31*hashCode+structuralFeature.hashCode();
		return hashCode;
	}

	/**
	 * @see org.eclipse.emf.ecore.resource.Resource.Internal#attached(org.eclipse.emf.ecore.EObject)
	 */
	public void attached(EObject value) {
		if (register) {
			LookupServiceExtensions.getInstance().register(getContext(value), value);
		}
	}

	/**
	 * @see org.eclipse.emf.ecore.resource.Resource.Internal#detached(org.eclipse.emf.ecore.EObject)
	 */
	public void detached(EObject value) {
		LookupServiceExtensions.getInstance().deregister(getContext(value), value);
	}

	public void doLoad(InputStream inputStream, Map options) throws IOException {
		PerfUtil p = PerfUtil.createInstance("HierarchyXMIResourceImpl.doLoad() uri=" + getURI(), true);

		try {
//			if(options == null || options == Collections.EMPTY_MAP)
//				options = new HashMap();
//			
//			options.put(XMLResource.OPTION_RECORD_UNKNOWN_FEATURE,Boolean.TRUE);
//			options.put(XMLResource.OPTION_USE_XML_NAME_TO_FEATURE_MAP,Boolean.TRUE);
//			options.put(XMLResource.OPTION_USE_DEPRECATED_METHODS,Boolean.FALSE);
						
			register = false;
			super.doLoad(inputStream, options);

			ContainmentTraverser containmentTraverser = new ContainmentTraverser(getContents());

			EObjectVisitor objectVisitor = new EObjectVisitor() {
				public boolean afterChildren(EObject element) {
					return true;
				}

				public boolean beforeChildren(EObject element) {
					LookupServiceExtensions.getInstance().register(getContext(element), element);

					return true;
				}
			};

			containmentTraverser.registerVisitors(objectVisitor);

			containmentTraverser.traverse();

			register = true;

		} catch (IOException e) {
			p.stopAndPrintStatus(e.getLocalizedMessage());
			throw e;
		}
		if(ModelDebugger.INSTANCE.debug)
			p.stopAndPrintStatus(Arrays.asList(new Throwable().getStackTrace()).toString());
		else
			p.stopAndPrintStatus();
	}

	/**
	 * This method deletes the resource and cleanup the memory.
	 * @deprecated Use delete() method instead.
	 *  
	 */
	public void doShallowDelete() {
		unloadLookupContext();
		for (Iterator iter = EMFUtil.delete(this).iterator(); iter.hasNext();) {
			Resource res = (Resource) iter.next();

			if (res != null) {
				res.setModified(true);
			}
		}

		if ((context != null) && (context.getAgentProxy() != null)) {
			context.getAgentProxy().setAgent(null);
		}
		if(context!=null)
		{
			context.cleanUp();
			context=null;
		}
		//        SaveUtil.removeDocument(this);
	}

	public void doShallowUnload() {
		unloadLookupContext();
		EMFUtil.unload(this);
		if(context!=null)
		{
			context.cleanUp();
			context=null;
		}
		//        SaveUtil.removeDocument(this);
	}

	/**
	 * @return
	 */
	protected HierarchyContext getContext(EObject object) {
		//        context = (HierarchyContext)
		// LookupServiceExtensions.getInstance().locate(null,
		// HierarchyContext.class, getURI().toString());
		if ((context == null) && object instanceof TRCAgent) {
			context = (HierarchyContext) LookupServiceExtensions.getInstance().locate(null, HierarchyContext.class, getURI().toString());

			if (context == null) {
				context = new HierarchyContext();
			}

			TRCAgent agent = (TRCAgent) object;

			AgentsContext agentsContext = (AgentsContext) LookupServiceExtensions.getInstance().locate(null, AgentsContext.class, LoadersUtils.getLookUpKey(agent.getRuntimeId()));

			if (agentsContext == null) {
				agentsContext = new AgentsContext(agent.getRuntimeId());
				LookupServiceExtensions.getInstance().register(null, agentsContext);
			}

			agentsContext.registerAgent(agent);

			context.setAgent(agent);
			context.setContextURI(getURI().toString());
			LookupServiceExtensions.getInstance().register(null, context);

			return null;
		} else if (object instanceof UnresolvedCorrelation) {
			return null; //this should be added to the global context
		}

		return context;
	}

	protected XMLLoad createXMLLoad() {
		XMLResourceLoader resourceLoader = null;

		try {
			resourceLoader = (XMLResourceLoader) ResourceExtensions.getInstance().get(getURI().fileExtension());

			if (resourceLoader != null) {
				resourceLoader = (XMLResourceLoader) resourceLoader.getClass().newInstance();
				resourceLoader.setXMLHelper(createXMLHelper());
			}
		} catch (Exception e) {
			ModelsHierarchyPlugin.log(e.getLocalizedMessage());
		}

		if (resourceLoader == null) {
			return super.createXMLLoad();
		}

		return resourceLoader;
	}

	/**
	 * @see org.eclipse.emf.ecore.xmi.impl.XMLResourceImpl#createXMLSave()
	 */
	protected XMLSave createXMLSave() {
		return new HierarchyXMISaveImpl(createXMLHelper());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.emf.ecore.resource.impl.ResourceImpl#doUnload()
	 */
	protected void doUnload() {
		PerfUtil p = PerfUtil.createInstance("HierarchyXMIResourceImpl.doUnload() uri=" + getURI(), true);
		doShallowUnload();
		p.stopAndPrintStatus();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.emf.ecore.xmi.impl.XMLResourceImpl#doSave(java.io.OutputStream, java.util.Map)
	 */
	public void doSave(OutputStream outputStream, Map options) throws IOException {
		PerfUtil p = PerfUtil.createInstance("HierarchyXMIResourceImpl.doSave() uri=" + getURI(), true);

		try {
//			if (options == null || options == Collections.EMPTY_MAP)
//				options = new HashMap();
			
			objectIndexes = new IntToObjectMapImpl();
			buildObjectsMap();
			if(objectIndexes.isEmpty())
				objectIndexes=null;

			super.doSave(outputStream, options);
			objectIndexes=null;
		} catch (IOException e) {
			p.stopAndPrintStatus(e.getLocalizedMessage());
			throw e;
		}
		p.stopAndPrintStatus();
	}

	protected void buildObjectsMap() {
		PerfUtil p = PerfUtil.createInstance("HierarchyXMIResourceImpl.buildObjectsMap() uri=" + getURI(), true);
		try {
			for (int i = getContents().size(); i-- > 0;) {
				EObject child = (EObject) getContents().get(i);
				traverseContaintment(child);
			}
		} catch (Exception e) {
			ModelDebugger.log(e,"HierarchyXMIResourceImpl.buildObjectsMap()");
		}

		p.stopAndPrintStatus("objectIndexes.size()="+objectIndexes.size());

	}

	protected void traverseContaintment(EObject object) {
		if(object==null)
			return;
		for (Iterator iter = object.eClass().getEAllContainments().iterator(); iter.hasNext();) {
			EReference ref = (EReference) iter.next();
			if(ref.isTransient())
				continue;
			if(ref.isMany())
			{
				EList list =  (EList)object.eGet(ref, true);
				if(list.size()<INDEX_WATERMARK)
				{
					for (int i = list.size(); i-- > 0;) {
						EObject child = (EObject) list.get(i);
						traverseContaintment(child);
					}
				}
				else
				{
					int lenght = list.size();
					for (int i = 0; i<lenght ;i++) {
						EObject child = (EObject) list.get(i);
//						if(i % 100 == 0)
//							LoadersUtils.log("list="+ref+", size="+lenght+", index="+i);
						putIndex(list, child, i, getListKey(object,ref));
						traverseContaintment(child);
					}
				}
			}
			else
			{
				EObject child =  (EObject)object.eGet(ref, true);
				traverseContaintment(child);
			}
		}
	}

	protected void unloadLookupContext() {
		if (context != null) {
			if (context.getAgent() != null) {
				//                LookupServiceExtensions.getInstance().deregister(null,
				// TRCAgentImpl.class, context.getAgent().getRuntimeId());
				TRCAgent agent = context.getAgent();
				AgentsContext agentsContext = (AgentsContext) LookupServiceExtensions.getInstance().locate(null, AgentsContext.class, LoadersUtils.getLookUpKey(agent.getRuntimeId()));

				if (agentsContext != null) {
					agentsContext.deregisterAgent(agent);
				}

				agentsContext = (AgentsContext) LookupServiceExtensions.getInstance().locate(null, AgentsContext.class, LoadersUtils.getLookUpKey(null));
				if (agentsContext != null) {
					agentsContext.deregisterAgent(agent);
				}

				deregisterContents(getContents());
			}

			LookupServiceExtensions.getInstance().deregister(context);
			LookupServiceExtensions.getInstance().deregister(null, HierarchyContext.class, getURI().toString());
//			context = null;
		}
	}

	protected void deregisterContents(List l) {
		ContainmentTraverser containmentTraverser = new ContainmentTraverser(l);

		EObjectVisitor objectVisitor = new EObjectVisitor() {
			public boolean afterChildren(EObject element) {
				LookupServiceExtensions.getInstance().deregister(context, element);

				return true;
			}

			public boolean beforeChildren(EObject element) {
				return true;
			}
		};

		containmentTraverser.registerVisitors(objectVisitor);

		containmentTraverser.traverse();
	}

	protected void initHierarchyResource() {
		for (int i = 0; i < INDEX_WATERMARK; i++) {
			intAndStrList.add(i, null);
		}
	}

	//	/* (non-Javadoc)
	//	 * @see org.eclipse.emf.ecore.resource.impl.ResourceImpl#doSave(java.io.OutputStream, java.util.Map)
	//	 */
	//	public void doSave(OutputStream outputStream, Map options)
	//		throws IOException {
	//		objectIndexes = new HashMap(); 
	//		super.doSave(outputStream, options);
	//		objectIndexes=null; 
	//	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.models.hierarchy.extensions.HyadesResource#delete()
	 */
	public boolean delete() {
		PerfUtil p = PerfUtil.createInstance("HierarchyXMIResourceImpl.delete() uri=" + getURI(), true);
		doShallowDelete();
		p.stopAndPrintStatus();
		return true;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.models.hierarchy.extensions.HyadesResource#deleteObjects(org.eclipse.emf.common.util.EList)
	 */
	public boolean deleteObjects(EList uris) {
		throw new UnsupportedOperationException(Constants.NOT_IMPLEMENTED_YET);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.models.hierarchy.extensions.HyadesResource#unloadObjects(org.eclipse.emf.common.util.EList)
	 */
	public boolean unloadObjects(EList uris) {
		throw new UnsupportedOperationException(Constants.NOT_IMPLEMENTED_YET);
	}
	//	/* (non-Javadoc)
	//	 * @see org.eclipse.emf.ecore.resource.impl.ResourceImpl#doSave(java.io.OutputStream, java.util.Map)
	//	 */
	//	public void doSave(OutputStream outputStream, Map options)
	//		throws IOException {
	//		objectIndexes = new HashMap(); 
	//		super.doSave(outputStream, options);
	//		objectIndexes=null; 
	//	}

	/* (non-Javadoc)
	 * @see org.eclipse.emf.ecore.resource.impl.ResourceImpl#isContentZipEntry(java.util.zip.ZipEntry)
	 */
	//	protected boolean isContentZipEntry(ZipEntry zipEntry) {
	//		if(zipEntry.getName().equals(super.newContentZipEntry().getName()))
	//			return super.isContentZipEntry(zipEntry);
	//		else 
	//			return false;
	//	}
	//	
	//	/* (non-Javadoc)
	//	 * @see org.eclipse.emf.ecore.resource.impl.ResourceImpl#newContentZipEntry()
	//	 */
	//	protected ZipEntry newContentZipEntry() {
	//		if(!getURI().hasQuery())
	//			return super.newContentZipEntry();
	//		else
	//			return null;
	//	}
	
	public static class InversedStringBuffer {
		int startOffset;
		int length;
		char[] data;
		
		public void setLength(int i) {
			startOffset=data.length;
			length=0;
		}

		public void prepend(String value) {
			if(value==null || value.length()==0)
				return;
			length+=value.length();
			startOffset-=value.length();
			value.getChars(0,value.length(),data,startOffset);
		}

		public void prepend(int value) {
			prepend(Integer.toString(value));
		}
		public void prepend(char value) {
			length++;
			startOffset--;
			data[startOffset]=value;
		}

		public InversedStringBuffer(int size) {
			data = new char[size];
		}
		
		public String toString() {
			if(length>0)
				return new String(data,startOffset,length);
			else
				return "";
		}
		
	}
}
