/**********************************************************************
 * Copyright (c) 2003, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HierarchyXMISaveImpl.java,v 1.3 2008/01/24 02:28:18 apnan Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.models.hierarchy.util;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EFactory;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.util.InternalEList;
import org.eclipse.emf.ecore.xmi.DanglingHREFException;
import org.eclipse.emf.ecore.xmi.XMIResource;
import org.eclipse.emf.ecore.xmi.XMLHelper;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.emf.ecore.xmi.impl.XMISaveImpl;


/**
 * This implements the XML serializer, possibly using an XMLMap
 * if one is provided as a save option.
 */
public class HierarchyXMISaveImpl extends XMISaveImpl {
    //~ Instance fields ----------------------------------------------------------------------------

    public static final int BUFFER_SIZE = 1024 * 256;
    protected List elementFeaturesPool = new ArrayList();
    protected StringBuffer ids = new StringBuffer(200);
    protected StringBuffer xsiSchemaLocation = new StringBuffer(100);
    protected char[] buffer = new char[BUFFER_SIZE];
    protected byte[] bytes = new byte[BUFFER_SIZE];
    protected int[] featureKinds;
    protected int currentDepth;
    protected int elementFeaturesCount = 0;
    protected byte[] byteArray;
    protected char[] charArray;
	protected List contentsSingleElementList;

//	protected Escape escape;
//	protected Lookup featureTable;

    //~ Constructors -------------------------------------------------------------------------------

    public HierarchyXMISaveImpl(XMLHelper helper) {
        super(helper);
        contentsSingleElementList = new ArrayList();
        contentsSingleElementList.add(null);
    }

    public HierarchyXMISaveImpl(Map options, XMLHelper helper, String encoding) {
        super(options, helper, encoding);
        contentsSingleElementList = new ArrayList();
        contentsSingleElementList.add(null);
    }

    //~ Methods ------------------------------------------------------------------------------------

    public void addNamespaceDeclarations() {
        doc.addAttribute(XMI_VER_NS, XMIResource.VERSION_VALUE);
        doc.addAttribute(XMI_XMLNS, XMIResource.XMI_URI);

        EPackage[] packages = helper.packages();

        xsiSchemaLocation.setLength(0);

        if (declareSchemaLocation) {
            for (int i = 0; i < packages.length; i++) {
                EPackage ePackage = packages[i];

                EObject root = EcoreUtil.getRootContainer(ePackage);

                if (root instanceof EPackage) {
                    EPackage rootEPackage = (EPackage) root;
                    Resource resource = rootEPackage.eResource();

                    if (resource != null) {
                        URI uri = resource.getURI();
                        String rootNsURI = rootEPackage.getNsURI();

                        if ((uri == null) ? (rootNsURI != null) : (!uri.toString().equals(rootNsURI))) {
                            declareXSI = true;

                            if (xsiSchemaLocation.length() > 0) {
                                xsiSchemaLocation.append(' ');
                            }

                            xsiSchemaLocation.append(rootNsURI);
                            xsiSchemaLocation.append(' ');
                            xsiSchemaLocation.append(helper.getHREF(rootEPackage));
                        }
                    }
                }
            }
        }

        //        if (declareXSI) {
        doc.addAttribute(XSI_XMLNS, XMIResource.XSI_URI);

        //        }
        for (int i = 0; i < packages.length; i++) {
            EPackage ePackage = packages[i];
            String nsURI = ePackage.getNsURI();
            String nsPrefix = ePackage.getNsPrefix();

            doc.addAttributeNS(XMIResource.XML_NS, nsPrefix, nsURI);
        }

        if ((xsiSchemaLocation != null) && (xsiSchemaLocation.length() > 0)) {
            doc.addAttribute(XSI_SCHEMA_LOCATION, xsiSchemaLocation.toString());
        }
    }

    public void save(XMLResource resource, OutputStream outputStream, Map options) throws IOException {
	    init(resource, options);
	    doc = new HierarchyXMLString(null);
	    
	
	    //   traverse(resource.getContents());
	    //    if (encoding.equals("US-ASCII") || encoding.equals("ASCII"))
	    //    {
	    //      writeAscii(outputStream);
	    //      outputStream.flush();
	    //    }
	    //    else
	    //    {
	    //      OutputStreamWriter outputStreamWriter = new OutputStreamWriter(outputStream, helper.getJavaEncoding(encoding));
	    //      write(outputStreamWriter);
	    //      outputStreamWriter.flush();
	    //    }
	    Writer writer = new BufferedWriter(new OutputStreamWriter(outputStream, helper.getJavaEncoding(encoding)), BUFFER_SIZE);
//	    Writer writer = new OutputStreamWriter(outputStream, helper.getJavaEncoding(encoding));
	    ((HierarchyXMLString) doc).setWriter(writer);
	    traverse(resource.getContents());
	    writer.flush();
	
	    featureTable = null;
	    doc = null;
	
	    if ((processDanglingHREF == null) || XMLResource.OPTION_PROCESS_DANGLING_HREF_THROW.equals(processDanglingHREF)) {
	        DanglingHREFException exception = helper.getDanglingHREFException();
	
	        if (exception != null) {
	            helper = null;
	            throw new Resource.IOWrappedException(exception);
	        }
	    }
	
	    helper = null;
	}

    public char[] toChar() {
        int size = doc.getLength();
        char[] output = allocCharArray(size);

        doc.getChars(output, 0);

        return output;
    }

    public void traverse(List contents) {
        if (declareXML) {
            doc.add("<?xml version=\"" + XML_VERSION + "\" encoding=\"" + encoding + "\"?>");
            doc.addLine();
        }

        int size = contents.size();

        // Reserve a place to insert xmlns declarations after we know what they all are.
        //
        Object mark;

        if (size == 1) {
            mark = writeTopObject((EObject) contents.get(0));
        } else {
            mark = writeTopObjects(contents);
        }

        // Go back and add all the XMLNS stuff.
        //
        doc.resetToMark(mark);

        //		addNamespaceDeclarations();
    }

    public void write(OutputStreamWriter os) throws IOException {
        int pos = 0;

        for (Iterator i = doc.iterator(); i.hasNext();) {
            String s = (String) i.next();
            int slen = s.length();

            if ((slen + pos) >= buffer.length) {
                os.write(buffer, 0, pos);
                pos = 0;

                if (slen > buffer.length) {
                    buffer = allocCharArray(slen);
                }
            }

            s.getChars(0, slen, buffer, pos);
            pos += slen;
        }

        os.write(buffer, 0, pos);
        os.flush();
    }

    public void writeAscii(OutputStream os) throws IOException {
        int pos = 0;

        for (Iterator i = doc.iterator(); i.hasNext();) {
            String s = (String) i.next();
            int slen = s.length();

            if ((slen + pos) >= buffer.length) {
                for (int x = 0; x < pos; x++) {
                    bytes[x] = (byte) (buffer[x] & 0xFF);
                }

                os.write(bytes, 0, pos);
                pos = 0;

                if (slen > buffer.length) {
                    buffer = allocCharArray(slen);
                    bytes = allocByteArray(slen);
                }
            }

            s.getChars(0, slen, buffer, pos);
            pos += slen;
        }

        for (int x = 0; x < pos; x++) {
            bytes[x] = (byte) (buffer[x] & 0xFF);
        }

        os.write(bytes, 0, pos);
        os.flush();
    }

    public Object writeTopObjects(List contents) {
        doc.startElement(XMI_TAG_NS);

        addExtraPackages(contents, Integer.MAX_VALUE);
        addNamespaceDeclarations();

        Object mark = doc.mark();

        for (int i = 0, size = contents.size(); i < size; i++) {
            EObject top = (EObject) contents.get(i);
            EClass eClass = top.eClass();
            String name = helper.getQName(eClass);

            doc.startElement(name);
            saveElementID(top);
        }

        doc.endElement();

        return mark;
    }

    protected String getContent(EObject o, EStructuralFeature[] features) {
        XMLResource.XMLMap map = ((XMLHelper) helper).getXMLMap();

        if (map == null) {
            return null;
        }

        for (int i = 0; i < features.length; i++) {
            XMLResource.XMLInfo info = map.getInfo(features[i]);

            if ((info != null) && (info.getXMLRepresentation() == XMLResource.XMLInfo.CONTENT)) {
                Object value = helper.getValue(o, features[i]);

                if (value == null) {
                    return null;
                } else {
                    EDataType d = (EDataType) features[i].getEType();
                    EPackage ePackage = d.getEPackage();
                    EFactory fac = ePackage.getEFactoryInstance();
                    String svalue = fac.convertToString(d, value);

                    if (escape != null) {
                        svalue = escape.convert(svalue);
                    }

                    return svalue;
                }
            }
        }

        return null;
    }

    /**
     * Method allocByteArray.
     * @param slen
     * @return byte[]
     */
    protected byte[] allocByteArray(int slen) {
        if (byteArray.length < slen) {
            byteArray = new byte[slen];
        }

        return byteArray;
    }

    protected char[] allocCharArray(int slen) {
        if (charArray.length < slen) {
            charArray = new char[slen];
        }

        return charArray;
    }

//    protected void init(XMLResource resource, Map options) {
//        this.xmiType = Boolean.TRUE.equals(options.get(XMIResource.OPTION_USE_XMI_TYPE));
//        declareXSI = false;
//        useEncodedAttributeStyle = Boolean.TRUE.equals(options.get(XMLResource.OPTION_USE_ENCODED_ATTRIBUTE_STYLE));
//        declareXML = !Boolean.FALSE.equals(options.get(XMLResource.OPTION_DECLARE_XML));
//        declareSchemaLocation = Boolean.TRUE.equals(options.get(XMLResource.OPTION_SCHEMA_LOCATION));
//
//        Integer lineWidth = (Integer) options.get(XMLResource.OPTION_LINE_WIDTH);
//
//        //    doc = new HierarchyXMLString(lineWidth == null ? Integer.MAX_VALUE : lineWidth.intValue());
//        doc = new HierarchyXMLString(null);
//        escape = Boolean.TRUE.equals(options.get(XMLResource.OPTION_SKIP_ESCAPE)) ? null : new Escape();
//
//        if (options.containsKey(XMLResource.OPTION_ENCODING)) {
//            encoding = (String) options.get(XMLResource.OPTION_ENCODING);
//        } else if (resource != null) {
//            encoding = resource.getEncoding();
//        }
//
//        processDanglingHREF = (String) options.get(XMLResource.OPTION_PROCESS_DANGLING_HREF);
//        helper.setProcessDanglingHREF(processDanglingHREF);
//
//        XMLResource.XMLMap map = (XMLResource.XMLMap) options.get(XMLResource.OPTION_XML_MAP);
//
//        if (map != null) {
//            helper.setXMLMap(map);
//
//            if (map.getIDAttributeName() != null) {
//                idAttributeName = map.getIDAttributeName();
//            }
//        }
//
//        featureTable = new Lookup(map);
//    }

    protected void saveDataTypeElementSingle(EObject o, EStructuralFeature f) {
        String name = helper.getQName(f);
        Object value = helper.getValue(o, f);

        if (value == null) {
            doc.startElement(name);
            doc.addAttribute(XSI_NIL, "true");
            doc.endEmptyElement();
            declareXSI = true;
        } else {
            EDataType d = (EDataType) f.getEType();
            EPackage ePackage = d.getEPackage();
            EFactory fac = ePackage.getEFactoryInstance();

            doc.startElement(name);

            String svalue = fac.convertToString(d, value);

            if (escape != null) {
                svalue = escape.convert(svalue);
            }

            doc.endContentElement(svalue);
        }
    }

    protected void saveDataTypeMany(EObject o, EStructuralFeature f) {
        EDataType d = (EDataType) f.getEType();
        EPackage ePackage = d.getEPackage();
        EFactory fac = ePackage.getEFactoryInstance();

        List values = (List) helper.getValue(o, f);
        int size = values.size();

        if (size > 0) {
            String name = helper.getQName(f);

            for (int i = 0; i < size; ++i) {
                Object value = values.get(i);

                if (value == null) {
                    doc.startElement(name);
                    doc.addAttribute(XSI_NIL, "true");
                    doc.endEmptyElement();
                    declareXSI = true;
                } else {
                    doc.startElement(name);

                    String svalue = fac.convertToString(d, value);

                    if (escape != null) {
                        svalue = escape.convert(svalue);
                    }

                    doc.endContentElement(svalue);
                }
            }
        }
    }

    protected void saveDataTypeSingle(EObject o, EStructuralFeature f) {
        EDataType d = (EDataType) f.getEType();
        EPackage ePackage = d.getEPackage();
        EFactory fac = ePackage.getEFactoryInstance();
        Object value = helper.getValue(o, f);

        if (value != null) {
            String svalue = fac.convertToString(d, value);

            if (escape != null) {
                svalue = escape.convert(svalue);
            }

            doc.addAttribute(helper.getQName(f), svalue);
        }
    }

    protected boolean saveFeatures(EObject o) {
        EClass eClass = o.eClass();
        EStructuralFeature[] features = featureTable.getFeatures(eClass);
        int[] featureKinds = featureTable.getKinds(eClass, features);
        int[] elementFeatures = null;
        int elementCount = 0;

        // Process XML attributes
        for (int i = 0; i < features.length; i++) {
            int kind = featureKinds[i];
            EStructuralFeature f = features[i];

            if ((kind != TRANSIENT) && o.eIsSet(f)) {
                switch (kind) {
                case DATATYPE_ELEMENT_SINGLE: {
                    if (elementFeatures == null) {
                        elementFeatures = allocElementFeatures(features.length);

                        //new int[features.length];
                    }

                    elementFeatures[elementCount++] = i;

                    break;
                }

                case DATATYPE_SINGLE: {
                    saveDataTypeSingle(o, f);

                    break;
                }

                case DATATYPE_SINGLE_NILLABLE: {
                    if (isNil(o, f)) {
                        if (elementFeatures == null) {
                            elementFeatures = allocElementFeatures(features.length);

                            //new int[features.length];
                        }

                        elementFeatures[elementCount++] = i;
                    } else {
                        saveDataTypeSingle(o, f);
                    }

                    break;
                }

                case OBJECT_HREF_SINGLE_UNSETTABLE: {
                    if (isNil(o, f)) {
                        if (elementFeatures == null) {
                            elementFeatures = allocElementFeatures(features.length);

                            //new int[features.length];
                        }

                        elementFeatures[elementCount++] = i;

                        break;
                    }

                    // it's intentional to keep going
                }

                case OBJECT_HREF_SINGLE: {
                    if (useEncodedAttributeStyle) {
                        saveEObjectSingle(o, f);
                    } else {
                        switch (sameDocSingle(o, f)) {
                        case SAME_DOC: {
                            saveIDRefSingle(o, f);

                            break;
                        }

                        case CROSS_DOC: {
                            if (elementFeatures == null) {
                                elementFeatures = allocElementFeatures(features.length);

                                //new int[features.length];
                            }

                            elementFeatures[elementCount++] = i;

                            break;
                        }
                        }
                    }

                    break;
                }

                case OBJECT_HREF_MANY_UNSETTABLE: {
                    if (isEmpty(o, f)) {
                        saveManyEmpty(f);

                        break;
                    }

                    // it's intentional to keep going
                }

                case OBJECT_HREF_MANY: {
                    if (useEncodedAttributeStyle) {
                        saveEObjectMany(o, f);
                    } else {
                        switch (sameDocMany(o, f)) {
                        case SAME_DOC: {
                            saveIDRefMany(o, f);

                            break;
                        }

                        case CROSS_DOC: {
                            if (elementFeatures == null) {
                                elementFeatures = allocElementFeatures(features.length);

                                //new int[features.length];
                            }

                            elementFeatures[elementCount++] = i;

                            break;
                        }
                        }
                    }

                    break;
                }

                case OBJECT_CONTAIN_MANY_UNSETTABLE:
                case DATATYPE_MANY: {
                    if (isEmpty(o, f)) {
                        saveManyEmpty(f);

                        break;
                    }

                    // it's intentional to keep going
                }

                case OBJECT_CONTAIN_SINGLE_UNSETTABLE:
                case OBJECT_CONTAIN_SINGLE:
                case OBJECT_CONTAIN_MANY:
                case OBJECT_ELEMENT_SINGLE:
                case OBJECT_ELEMENT_MANY: {
                    if (elementFeatures == null) {
                        elementFeatures = allocElementFeatures(features.length);

                        //new int[features.length];
                    }

                    elementFeatures[elementCount++] = i;

                    break;
                }
                }
            }
        }

        String content = getContent(o, features);

        if (elementFeatures == null) {
            if (content == null) {
                endSaveFeatures(o, EMPTY_ELEMENT, null);

                if (elementFeatures != null) {
                    dealocElementFeatures();
                }

                return false;
            } else {
                endSaveFeatures(o, CONTENT_ELEMENT, content);

                if (elementFeatures != null) {
                    dealocElementFeatures();
                }

                return true;
            }
        }

        // Process XML elements
        for (int i = 0; i < elementCount; i++) {
            int kind = featureKinds[elementFeatures[i]];
            EStructuralFeature f = features[elementFeatures[i]];

            switch (kind) {
            case DATATYPE_SINGLE_NILLABLE: {
                saveNil(f);

                break;
            }

            case DATATYPE_MANY: {
                saveDataTypeMany(o, f);

                break;
            }

            case DATATYPE_ELEMENT_SINGLE: {
                saveDataTypeElementSingle(o, f);

                break;
            }

            case OBJECT_CONTAIN_SINGLE_UNSETTABLE: {
                if (isNil(o, f)) {
                    saveNil(f);

                    break;
                }

                // it's intentional to keep going
            }

            case OBJECT_CONTAIN_SINGLE: {
                saveContainedSingle(o, f);

                break;
            }

            case OBJECT_CONTAIN_MANY_UNSETTABLE:
            case OBJECT_CONTAIN_MANY: {
                saveContainedMany(o, f);

                break;
            }

            case OBJECT_HREF_SINGLE_UNSETTABLE: {
                if (isNil(o, f)) {
                    saveNil(f);

                    break;
                }

                // it's intentional to keep going
            }

            case OBJECT_HREF_SINGLE: {
                saveHRefSingle(o, f);

                break;
            }

            case OBJECT_HREF_MANY_UNSETTABLE:
            case OBJECT_HREF_MANY: {
                saveHRefMany(o, f);

                break;
            }

            case OBJECT_ELEMENT_SINGLE: {
                saveElementReferenceSingle(o, f);

                break;
            }

            case OBJECT_ELEMENT_MANY: {
                saveElementReferenceMany(o, f);

                break;
            }
            }
        }

        endSaveFeatures(o, 0, null);

        if (elementFeatures != null) {
            dealocElementFeatures();
        }

        return true;
    }

    protected void saveIDRefMany(EObject o, EStructuralFeature f) {
        InternalEList values = (InternalEList) helper.getValue(o, f);

        if (!values.isEmpty()) {
            String name = helper.getQName(f);

            ids.setLength(0);

            for (Iterator i = values.basicIterator();;) {
                EObject value = (EObject) i.next();
                String id = helper.getIDREF(value);

                ids.append(id);

                if (i.hasNext()) {
                    ids.append(" ");
                } else {
                    break;
                }
            }

            doc.addAttribute(name, ids.toString());
        }
    }

    protected Object writeTopObject(EObject top) {
        EClass eClass = top.eClass();

        contentsSingleElementList.set(0,top);

        addExtraPackages(contentsSingleElementList, Integer.MAX_VALUE);

        String name = helper.getQName(eClass);

        doc.startElement(name);

        addNamespaceDeclarations();

        Object mark = doc.mark();

        saveElementID(top);

        return mark;
    }

    /**
     * @param top
     * @param i
     */
    protected void addExtraPackages(List objects, final int depth) {
        currentDepth = 0;

        ContainmentTraverser containmentTraverser = new ContainmentTraverser(objects);

        EObjectVisitor objectVisitor = new EObjectVisitor() {
            public boolean afterChildren(EObject element) {
                currentDepth--;

                //adds the package to the XMLHelper
                helper.getQName(element.eClass());

                return true;
            }

            public boolean beforeChildren(EObject element) {
                if (currentDepth > depth) {
                    return false;
                }

                currentDepth++;

                return true;
            }
        };

        containmentTraverser.registerVisitors(objectVisitor);
        containmentTraverser.traverse();
    }

    /**
     * Method allocElementFeatures.
     * @param i
     * @return int[]
     */
    protected int[] allocElementFeatures(int i) {
        //	return new int[i];
        int[] elementFeatures;

        if (elementFeaturesCount == elementFeaturesPool.size()) {
            elementFeatures = new int[i];
            elementFeaturesPool.add(elementFeatures);
        } else {
            elementFeatures = (int[]) elementFeaturesPool.get(elementFeaturesCount);

            if (elementFeatures.length < i) {
                elementFeatures = new int[i];
                elementFeaturesPool.set(elementFeaturesCount, elementFeatures);
            }
        }

        elementFeaturesCount++;

        return elementFeatures;
    }

    /**
     * Method dealocElementFeatures.
     */
    protected void dealocElementFeatures() {
        //	return;
        if (elementFeaturesCount > 0) {
            elementFeaturesCount--;

            //		System.out.println("hit dealoc = "+elementFeaturesCount);
        }

        //	else
        //	System.out.println("hit");
    }

    //~ Inner Classes ------------------------------------------------------------------------------

//    protected static class Escape {
//        protected final char[] AMP = { '&', 'a', 'm', 'p', ';' };
//        protected final char[] CR = { '&', '#', 'x', 'D', ';' };
//        protected final char[] LESS = { '&', 'l', 't', ';' };
//        protected final char[] LF = { '&', '#', 'x', 'A', ';' };
//        protected final char[] QUOTE = { '&', 'q', 'u', 'o', 't', ';' };
//        protected final char[] TAB = { '&', '#', 'x', '9', ';' };
//        protected char[] value;
//
//        Escape() {
//            value = new char[100];
//        }
//
//        /*
//         *  Convert:
//         *  & to &amp;
//         *  < to &lt;
//         *  " to &quot;
//         *  \t to &#x9;
//         *  \n to &#xA;
//         *  \r to &#xD;
//         */
//        public String convert(String input) {
//            boolean changed = false;
//            int inputLength = input.length();
//
//            grow(inputLength);
//            input.getChars(0, inputLength, value, 0);
//
//            int pos = 0;
//
//            while (inputLength-- > 0) {
//                switch (value[pos]) {
//                case '&':
//                    pos = replace(pos, AMP, inputLength);
//                    changed = true;
//
//                    break;
//
//                case '<':
//                    pos = replace(pos, LESS, inputLength);
//                    changed = true;
//
//                    break;
//
//                case '"':
//                    pos = replace(pos, QUOTE, inputLength);
//                    changed = true;
//
//                    break;
//
//                case '\n': {
//                    pos = replace(pos, LF, inputLength);
//                    changed = true;
//
//                    break;
//                }
//
//                case '\r': {
//                    pos = replace(pos, CR, inputLength);
//                    changed = true;
//
//                    break;
//                }
//
//                case '\t': {
//                    pos = replace(pos, TAB, inputLength);
//                    changed = true;
//
//                    break;
//                }
//
//                default:
//                    pos++;
//
//                    break;
//                }
//            }
//
//            return changed ? new String(value, 0, pos) : input;
//        }
//
//        protected void grow(int newSize) {
//            int vlen = value.length;
//
//            if (vlen < newSize) {
//                char[] newValue = new char[newSize + (newSize / 2)];
//
//                System.arraycopy(value, 0, newValue, 0, vlen);
//                value = newValue;
//            }
//        }
//
//        protected int replace(int pos, char[] replacement, int inputLength) {
//            int rlen = replacement.length;
//            int newPos = pos + rlen;
//
//            grow(newPos + inputLength);
//            System.arraycopy(value, pos + 1, value, newPos, inputLength);
//            System.arraycopy(replacement, 0, value, pos, rlen);
//
//            return newPos;
//        }
//    }

//    protected static class Lookup {
//        protected static final int SIZE = 4095; // 2^N-1
//        protected XMLResource.XMLMap map;
//        protected EClass[] classes;
//        protected int[][] featureKinds;
//        protected EStructuralFeature[][] features;
//
//        Lookup(XMLResource.XMLMap map) {
//            this.map = map;
//            classes = new EClass[SIZE + 1];
//            features = new EStructuralFeature[SIZE + 1][];
//            featureKinds = new int[SIZE + 1][];
//        }
//
//        public EStructuralFeature[] getFeatures(EClass cls) {
//            int index = cls.hashCode() & SIZE;
//            EClass c = classes[index];
//
//            if (c == cls) {
//                return features[index];
//            }
//
//            EStructuralFeature[] featureList = listFeatures(cls);
//
//            if (c == null) {
//                classes[index] = cls;
//                features[index] = featureList;
//                featureKinds[index] = listKinds(featureList);
//            }
//
//            return featureList;
//        }
//
//        public int[] getKinds(EClass cls, EStructuralFeature[] featureList) {
//            int index = cls.hashCode() & SIZE;
//            EClass c = classes[index];
//
//            if (c == cls) {
//                return featureKinds[index];
//            }
//
//            int[] kindsList = listKinds(featureList);
//
//            if (c == null) {
//                classes[index] = cls;
//                features[index] = featureList;
//                featureKinds[index] = kindsList;
//            }
//
//            return kindsList;
//        }
//
//        protected int featureKind(EStructuralFeature f) {
//            if (f.isTransient()) {
//                return TRANSIENT;
//            }
//
//            boolean isMany = f.isMany();
//            boolean isUnsettable = f.isUnsettable();
//
//            if (f instanceof EReference) {
//                EReference r = (EReference) f;
//
//                if (r.isContainment()) {
//                    return isMany ? (isUnsettable ? OBJECT_CONTAIN_MANY_UNSETTABLE : OBJECT_CONTAIN_MANY) : (isUnsettable ? OBJECT_CONTAIN_SINGLE_UNSETTABLE : OBJECT_CONTAIN_SINGLE);
//                }
//
//                EReference opposite = r.getEOpposite();
//
//                if ((opposite != null) && opposite.isContainment()) {
//                    return TRANSIENT;
//                }
//
//                if (map != null) {
//                    XMLResource.XMLInfo info = map.getInfo(f);
//
//                    if ((info != null) && (info.getXMLRepresentation() == XMLResource.XMLInfo.ELEMENT)) {
//                        return isMany ? OBJECT_ELEMENT_MANY : OBJECT_ELEMENT_SINGLE;
//                    }
//                }
//
//                return isMany ? (isUnsettable ? OBJECT_HREF_MANY_UNSETTABLE : OBJECT_HREF_MANY) : (isUnsettable ? OBJECT_HREF_SINGLE_UNSETTABLE : OBJECT_HREF_SINGLE);
//            } else {
//                // Attribute
//                EDataType d = (EDataType) f.getEType();
//
//                if (!d.isSerializable()) {
//                    return TRANSIENT;
//                }
//
//                if (isMany) {
//                    return DATATYPE_MANY;
//                }
//
//                if (isUnsettable && (map == null)) {
//                    return DATATYPE_SINGLE_NILLABLE;
//                }
//
//                if (map == null) {
//                    return DATATYPE_SINGLE;
//                } else {
//                    XMLResource.XMLInfo info = map.getInfo(f);
//
//                    if ((info != null) && (info.getXMLRepresentation() == XMLResource.XMLInfo.ELEMENT)) {
//                        return DATATYPE_ELEMENT_SINGLE;
//                    } else if ((info != null) && (info.getXMLRepresentation() == XMLResource.XMLInfo.CONTENT)) {
//                        return DATATYPE_CONTENT_SINGLE;
//                    } else {
//                        if (isUnsettable) {
//                            return DATATYPE_SINGLE_NILLABLE;
//                        } else {
//                            return DATATYPE_SINGLE;
//                        }
//                    }
//                }
//            }
//        }
//
//        protected EStructuralFeature[] listFeatures(EClass cls) {
//            List f = (map == null) ? cls.getEAllStructuralFeatures() : map.getFeatures(cls);
//
//            return (EStructuralFeature[]) f.toArray(new EStructuralFeature[f.size()]);
//        }
//
//        protected int[] listKinds(EStructuralFeature[] featureList) {
//            int[] kinds = new int[featureList.length];
//
//            for (int i = featureList.length - 1; i >= 0; i--) {
//                kinds[i] = featureKind(featureList[i]);
//            }
//
//            return kinds;
//        }
//    }
}
