/**********************************************************************
 * Copyright (c) 2003, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 *
 * $Id: HierarchyEObjectImpl.java,v 1.3 2008/01/24 02:28:17 apnan Exp $
 **********************************************************************/
package org.eclipse.hyades.models.hierarchy.util.internal;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.InternalEObject;
import org.eclipse.emf.ecore.EStructuralFeature.Internal.DynamicValueHolder;
import org.eclipse.emf.ecore.impl.BasicEObjectImpl;
import org.eclipse.emf.ecore.resource.Resource.Internal;
import org.eclipse.emf.ecore.util.EContentsEList;
import org.eclipse.emf.ecore.util.ECrossReferenceEList;

/**
 * EObject implementation that extends the virtual behavior to support primitive
 * types and allows to page in/out primitive types
 * 
 * @author slavescu
 * @since 4.2
 */
public abstract class HierarchyEObjectImpl extends BasicEObjectImpl {

	public static boolean useVirtualManager = false;//ModelDebugger.INSTANCE.debugUseVirtualImpl;
	public static boolean useStringCompression = true;

	public static final String STRING_ENCODING = "UTF8";

	public static final int DEFAULT_PROPERTIES_BASE = 0;

	// default properties - primitives
	public static final int CONTAINER_FEATURE_ID = DEFAULT_PROPERTIES_BASE - 1;

	public static final int DELIVER = DEFAULT_PROPERTIES_BASE - 2;

	public static final int DEFAULT_PRIMITIVES_PROPERTIES_COUNT = 2;

	// default properties - Objects

	// //int based proxy ID, if the bit == 1, the indexBitField0>>>1 == internal
	// proxy ID mantained by the TPTPVirtualMemoryManager
	// public static final int PROXY_ID = DEFAULT_PROPERTIES_BASE - 1;

	public static final int PRIMITIVES_BYTE_ARRAY = DEFAULT_PROPERTIES_BASE - 1;

	public static final int CONTAINER = DEFAULT_PROPERTIES_BASE - 2;

	public static final int ADAPTERS = DEFAULT_PROPERTIES_BASE - 3;

	public static final int DIRECT_RESOURCE = DEFAULT_PROPERTIES_BASE - 4;

	public static final int CONTENTS = DEFAULT_PROPERTIES_BASE - 5;

	public static final int CROSS_REFERENCES = DEFAULT_PROPERTIES_BASE - 6;

	public static final int ECLASS = DEFAULT_PROPERTIES_BASE - 7;

	public static final int DEFAULT_OBJECTS_PROPERTIES_COUNT = 7;

	protected Object data;

	protected int indexBits0;

	protected void cleanup() {
		// TODO add support for marking for removal this object
		if (useVirtualManager) {
			TPTPVirtualMemoryManager.INSTANCE.setData(this, null);
			TPTPVirtualMemoryManager.INSTANCE.setIndexBits(this, 0, 0); // it
																		// may
			// need
			// other
			// value to
			// mark
			// cleanup
		}
	}

	protected BasicEList eBasicAdapters() {
		return (BasicEList) eVirtualGet(ADAPTERS, (BasicEList) null);
	}

	public EList eAdapters() {
		EList eAdapters = null;
		if (eBasicAdapters() == null) {
			eAdapters = new EAdapterList(this);
			eVirtualSet(ADAPTERS, (EList) eAdapters);
		}
		return eAdapters;
	}

	protected void eBasicSetContainer(InternalEObject newContainer, int newContainerFeatureID) {
		eVirtualSet(CONTAINER, newContainer);
		eVirtualSet(CONTAINER_FEATURE_ID, newContainerFeatureID);
	}

	public EClass eClass() {
		return eStaticClass(); // we don't support dynamic models yet
	}

	protected int eComputeVirtualValuesCapacity(int minimumCapacity) {
		return minimumCapacity;
		// return minimumCapacity + (minimumCapacity >> 3) + 2;
	}

	public EList eContents() {
		EList o = (EList) eVirtualGet(CONTENTS, (EList) null);
		if (o == null) {
			o = new EContentsEList(this);
			eVirtualSet(CONTENTS, o);
		}
		return o;
	}

	public EList eCrossReferences() {
		EList o = (EList) eVirtualGet(CROSS_REFERENCES, (EList) null);
		if (o == null) {
			o = new ECrossReferenceEList(this);
			eVirtualSet(CROSS_REFERENCES, o);
		}
		return o;
	}

	public Internal eDirectResource() {
		return (Internal) eVirtualGet(DIRECT_RESOURCE, (Internal) null);
	}

	protected EClass eDynamicClass() {
		// return null; //Dynamic models not supported yet
		throw new UnsupportedOperationException("Dynamic models not supported yet");

	}

	/**
	 * Should convert the real primitive feature ID to a sequential value in the
	 * primitive features' ID list Must be overriten by any class that has
	 * primitive features.
	 * 
	 * @param eDerivedStructuralFeatureID
	 * @return
	 */
	public int eVirtualSequentialPrimitiveFeatureID(int eDerivedStructuralFeatureID) {
		return eDerivedStructuralFeatureID; // no primitive features defined;
	}

	/**
	 * Should convert the real Object feature ID to a sequential value in the
	 * Object features' ID list. Must be overriten by any class that has Object
	 * features.
	 * 
	 * @param eDerivedStructuralFeatureID
	 * @return
	 */
	public int eVirtualSequentialObjectFeatureID(int derivedStructuralFeatureID) {
		return derivedStructuralFeatureID; // no Object features defined;
	}

	protected boolean eHasSettings() {
		throw new UnsupportedOperationException("Dynamic models not supported yet");
	}

	public InternalEObject eInternalContainer() {
		return (InternalEObject) eVirtualGet(CONTAINER, (InternalEObject) null);
	}

	public int eContainerFeatureID() {
		return eVirtualGet(CONTAINER_FEATURE_ID, -1);
	}

	public boolean eDeliver() {
		return eVirtualGet(DELIVER, false);
	}

	public void eSetDeliver(boolean deliver) {
		eVirtualSet(DELIVER, deliver);
	}

	public boolean eIsProxy() {
		return eProxyURI() != null;
	}

	public URI eProxyURI() {
		Object data = getData();
		return data != null && data instanceof URI ? (URI) data : null;
	}

	protected Object getData() {
		if (useVirtualManager)
			return TPTPVirtualMemoryManager.INSTANCE.getData(this);
		else
			return data;
	}

	protected void setData(Object uri) {
		if (useVirtualManager)
			TPTPVirtualMemoryManager.INSTANCE.setData(this, uri);
		else
			data = uri;
	}

	public void eSetClass(EClass eClass) {
		// do nothing, we don't support dynamic models yet
		throw new UnsupportedOperationException("Dynamic models not supported yet");
	}

	protected void eSetDirectResource(Internal resource) {
		eVirtualSet(DIRECT_RESOURCE, (Internal) resource);
	}

	public void eSetProxyURI(URI uri) {
		cleanup();
		setData(uri); // the proxy object will not have any values, data will
		// be
		// just the proxy URI
	}

	protected DynamicValueHolder eSettings() {
		return null;
	}

	protected void eSetVirtualIndexBits(int offset, int newIndexBits) {
		switch (offset) {
		case 0:
			setIndexBits(0, newIndexBits);
			break;
		default:
			throw new IndexOutOfBoundsException("offset=" + offset);
		}
	}

	protected void setIndexBits(int i, int newIndexBits) {
		if(useVirtualManager)
		{
			// TPTPVirtualMemoryManager.INSTANCE.setIndexBits(this, i,
			// newIndexBits);
		}
		else
		{
			indexBits0 = newIndexBits;
		}
	}

	protected int getIndexBits(int i) {
		if(useVirtualManager)
		{
			return TPTPVirtualMemoryManager.INSTANCE.getIndexBits(this, i);
		}
		else {
			return indexBits0;
		}
	}

	protected void eSetVirtualValues(Object[] newValues) {
		setData(newValues);
	}

	protected boolean eVirtualGet(int eDerivedStructuralFeatureID, boolean defaultValue) {
		byte[] primitives = getPrimitivesValues();
		if (primitives != null) {
			int index = eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_IS_SET, primitives);
			return index < 0 ? defaultValue : !defaultValue;
		}
		return defaultValue;
	}

	protected byte eVirtualGet(int eDerivedStructuralFeatureID, byte defaultValue) {
		byte[] primitives = getPrimitivesValues();
		if (primitives != null) {
			int intialSize = ((eVirtualPrimitiveFeatureIDsCount()) >>> 3) + 1;
			int index = eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_GET, primitives);
			return index < 0 ? defaultValue : primitives[index + intialSize];

		}
		return defaultValue;

	}

	protected double eVirtualGet(int eDerivedStructuralFeatureID, double defaultValue) {
		byte[] primitives = getPrimitivesValues();
		if (primitives != null) {
			int intialSize = ((eVirtualPrimitiveFeatureIDsCount()) >>> 3) + 1;
			int index = eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_GET, primitives);
			return index < 0 ? defaultValue : Double.longBitsToDouble(getLong(primitives, index + intialSize));

		}
		return defaultValue;
	}

	protected float eVirtualGet(int eDerivedStructuralFeatureID, float defaultValue) {
		byte[] primitives = getPrimitivesValues();
		if (primitives != null) {
			int intialSize = ((eVirtualPrimitiveFeatureIDsCount()) >>> 3) + 1;
			int index = eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_GET, primitives);
			return index < 0 ? defaultValue : Float.intBitsToFloat(getInt(primitives, index + intialSize));

		}
		return defaultValue;
	}

	protected int eVirtualGet(int eDerivedStructuralFeatureID, int defaultValue) {
		byte[] primitives = getPrimitivesValues();
		if (primitives != null) {
			int intialSize = ((eVirtualPrimitiveFeatureIDsCount()) >>> 3) + 1;
			int index = eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_GET, primitives);
			return index < 0 ? defaultValue : getInt(primitives, index + intialSize);

		}
		return defaultValue;
	}

	protected long eVirtualGet(int eDerivedStructuralFeatureID, long defaultValue) {
		byte[] primitives = getPrimitivesValues();
		if (primitives != null) {
			int intialSize = ((eVirtualPrimitiveFeatureIDsCount()) >>> 3) + 1;
			int index = eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_GET, primitives);
			return index < 0 ? defaultValue : getLong(primitives, index + intialSize);
		}
		return defaultValue;
	}

	protected short eVirtualGet(int eDerivedStructuralFeatureID, short defaultValue) {
		byte[] primitives = getPrimitivesValues();
		if (primitives != null) {
			int intialSize = ((eVirtualPrimitiveFeatureIDsCount()) >>> 3) + 1;
			int index = eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_GET, primitives);
			return index < 0 ? defaultValue : (short) (primitives[index + intialSize + 1] | ((short) primitives[index + intialSize]) << 8);

		}
		return defaultValue;

	}

	public Object eVirtualGet(int eDerivedStructuralFeatureID, String defaultValue) {
		if (!useStringCompression) {
			return super.eVirtualGet(eDerivedStructuralFeatureID, (Object) defaultValue);
		} else {
			try {
				Object tValue = super.eVirtualGet(eDerivedStructuralFeatureID, defaultValue == null ? null : defaultValue);
				if (tValue == EVIRTUAL_NO_VALUE || defaultValue == tValue || tValue == null )
					return defaultValue;
				else
					return new String((byte[]) tValue, STRING_ENCODING);
			} catch (Exception e) {
				e.printStackTrace();
			}
			return defaultValue;
		}
	}

	public Object eVirtualSet(int eDerivedStructuralFeatureID, String value) {
		if (!useStringCompression) {
			return super.eVirtualSet(eDerivedStructuralFeatureID, (Object) value);
		} else {
			try {
				Object tValue = super.eVirtualSet(eDerivedStructuralFeatureID, value == null ? null : value.getBytes(STRING_ENCODING));
				if (tValue == EVIRTUAL_NO_VALUE || tValue == null)
					return null;
				else
					return new String((byte[]) tValue, STRING_ENCODING);
			} catch (Exception e) {
				e.printStackTrace();
			}
			return null;
		}
	}

	protected int eVirtualIndex(int eDerivedStructuralFeatureID, int action) {
		if (eDerivedStructuralFeatureID < 0)
			return super.eVirtualIndex(-eDerivedStructuralFeatureID - 1, action);
		else
			return super.eVirtualIndex(eVirtualSequentialObjectFeatureID(eDerivedStructuralFeatureID), action);
	}

	protected int eVirtualIndexBits(int offset) {
		switch (offset) {
		case 0:
			return getIndexBits(0);
		default:
			throw new IndexOutOfBoundsException("index=" + offset);
		}
	}

	/**
	 * Counts the bits in the value and adds for each bit set the size in bytes
	 * of the coresponding primitive feature. Must be overriten by any class
	 * that has primitive features. Could use a modified version of the
	 * Precompute_8bit method to compute this value. Needs to be implemented by
	 * subclases
	 * 
	 * @param value
	 * @return
	 */
	public int eVirtualDefaultPrimitiveBitCount(int sequentialPrimitiveFeatureID, byte value) {
		if (sequentialPrimitiveFeatureID == -1) {
			return 0;
		}

		// int count = 0;
		if ((value & 1) != 0) {
			return 4; // count += 4; // has CONTAINER_FEATURE_ID
		}
		// ignore DELIVER because it is boolean
		// if ((value & 2) != 0) // ignore
		// count += 0;
		// return count;
		return 0;
	}

	public int eVirtualPrimitiveFeatureIDsCount() {
		return DEFAULT_PRIMITIVES_PROPERTIES_COUNT;
	}

	public int eVirtualObjectFeatureIDsCount() {
		return DEFAULT_OBJECTS_PROPERTIES_COUNT;
	}

	protected int eVirtualPrimitiveIndex(int eDerivedStructuralFeatureID, int action, byte[] primitives) {
		// Compute the offset in the sequence of bit patterns for this feature
		// ID
		// and then get the bit pattern at that index.
		//
		if (eDerivedStructuralFeatureID < 0)
			eDerivedStructuralFeatureID = -eDerivedStructuralFeatureID - 1;
		else
			eDerivedStructuralFeatureID = eVirtualSequentialPrimitiveFeatureID(eDerivedStructuralFeatureID);

		int offset = eDerivedStructuralFeatureID >>> 3; // we will handle one
		// byte at a time

		byte bits = primitives[offset];

		// Compute the index within that bit pattern for this feature ID
		// and fetch that bit at that index.
		//
		int bitIndex = eDerivedStructuralFeatureID & 7;
		int bit = bits >>> bitIndex & 1;

		switch (action) {
		case EVIRTUAL_IS_SET: {
			// For isSet, we only need to check the bit and return -1 when the
			// bit is 0.
			//
			return bit - 1;
		}
		case EVIRTUAL_GET:
			if (bit == 0)
				return -1;
			// else
			// System.out.println("HierarchyEObjectImpl.eVirtualPrimitiveIndex()");
		case EVIRTUAL_UNSET: {
			if (bit == 0) {
				// If the value index set, there's no index to return.
				//
				return -1;
			}
			// Continue to compute the offset.
		}
		case EVIRTUAL_SET:
		default: {

			// Count just the bits up to this one.
			// Note that shifting 32 bits is a no op.
			//
			// int result =
			// eVirtualPrimitiveBitCount(eDerivedStructuralFeatureID, bitIndex
			// == 0 ? (byte) 0 : (byte) (bits << 8 - bitIndex));
			// int result =
			// eVirtualPrimitiveBitCount(eDerivedStructuralFeatureID, offset,
			// primitives[offset]);
			int result = eVirtualPrimitiveBitCount(eDerivedStructuralFeatureID - 1, primitives);
			// Count all the bits in the bit patterns up to this one in the
			// sequence of bit patterns.
			//
			// for (int i = offset; --i >= 0;) {
			// result += eVirtualPrimitiveBitCount(eDerivedStructuralFeatureID,
			// offset, primitives[i]);
			// }

			// Depending on the action and the current state, we'll toggle the
			// state.
			// i.e., for unset, we need to turn it off if it's on,
			// and for set we need to turn it on if it's off.
			//
			if (bit == action) {
				primitives[offset] = (byte) (bits ^ (1 << bitIndex));
			}

			// result+=offset+1;
			// If the index was previously assigned, return it.
			// Otherwise, return a negative result that encodes the newly
			// assigned index.
			//
			return bit != 0 ? result : ~result;
		}
		}

	}

	public int eVirtualPrimitiveBitCount(int derivedStructuralFeatureID, byte[] primitives) {
		return eVirtualDefaultPrimitiveBitCount(derivedStructuralFeatureID, primitives[0]);
	}

	public boolean eVirtualPrimitiveIsSet(int eDerivedStructuralFeatureID) {
		byte[] primitives = getPrimitivesValues();
		if (primitives != null) {
			return eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_IS_SET, primitives) >= 0;
		} else
			return false;
	}

	protected boolean eVirtualSet(int eDerivedStructuralFeatureID, boolean value) {
		byte[] primitives = getPrimitivesValues();
		if (primitives == null) {
			int intialSize = ((eVirtualPrimitiveFeatureIDsCount()) >>> 3) + 1;
			primitives = new byte[intialSize];
			eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, value ? EVIRTUAL_SET : EVIRTUAL_UNSET, primitives);
			eVirtualSet(PRIMITIVES_BYTE_ARRAY, primitives);
			return value;
		} else {
			int bitIndex = eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, value ? EVIRTUAL_SET : EVIRTUAL_UNSET, primitives);
			return bitIndex < 0 ? value : !value;

		}

		// return eVirtualGet(eDerivedStructuralFeatureID,
		// Boolean.valueOf(defaultValue));
	}

	protected byte eVirtualSet(int eDerivedStructuralFeatureID, byte value) {
		int intialSize = ((eVirtualPrimitiveFeatureIDsCount()) >>> 3) + 1;
		byte[] primitives = getPrimitivesValues();
		if (primitives == null) {
			primitives = new byte[intialSize + 1];
			eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_SET, primitives);
			primitives[intialSize] = value;
			eVirtualSet(PRIMITIVES_BYTE_ARRAY, primitives);
			return value;
		} else {
			int index = eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_SET, primitives);
			if (index < 0) {
				index = ~index;
				index += intialSize;
				primitives = resizePrimitiveArray(primitives, index, 1);
				primitives[index] = value;
				eVirtualSet(PRIMITIVES_BYTE_ARRAY, primitives);
				return value;
			}
			index += intialSize;
			byte previousValue = primitives[index];
			primitives[index] = value;
			return previousValue;

		}
	}

	protected byte[] getPrimitivesValues() {
		Object o = eVirtualGet(PRIMITIVES_BYTE_ARRAY, (byte[]) null);
		if (o == null || o == EVIRTUAL_NO_VALUE)
			return null;
		return (byte[]) o;
	}

	protected double eVirtualSet(int eDerivedStructuralFeatureID, double value) {
		int intialSize = ((eVirtualPrimitiveFeatureIDsCount()) >>> 3) + 1;
		byte[] primitives = getPrimitivesValues();
		long lValue = Double.doubleToLongBits(value);
		if (primitives == null) {
			primitives = new byte[intialSize + 8];
			eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_SET, primitives);
			setLong(lValue, primitives, intialSize);
			eVirtualSet(PRIMITIVES_BYTE_ARRAY, primitives);
			return value;
		} else {
			int index = eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_SET, primitives);
			if (index < 0) {
				index = ~index;
				index += intialSize;
				primitives = resizePrimitiveArray(primitives, index, 8);
				setLong(lValue, primitives, index);
				eVirtualSet(PRIMITIVES_BYTE_ARRAY, primitives);
				return value;

			}
			index += intialSize;
			long lPreviousValue = getLong(primitives, index);

			setLong(lValue, primitives, index);
			return Double.longBitsToDouble(lPreviousValue);
		}
	}

	protected float eVirtualSet(int eDerivedStructuralFeatureID, float value) {
		int intialSize = ((eVirtualPrimitiveFeatureIDsCount()) >>> 3) + 1;
		byte[] primitives = getPrimitivesValues();
		int iValue = Float.floatToIntBits(value);
		if (primitives == null) {
			primitives = new byte[intialSize + 4];
			eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_SET, primitives);
			setInt(iValue, primitives, intialSize);
			eVirtualSet(PRIMITIVES_BYTE_ARRAY, primitives);
			return value;
		} else {
			int index = eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_SET, primitives);
			if (index < 0) {
				index = ~index;
				index += intialSize;

				primitives = resizePrimitiveArray(primitives, index, 4);
				setInt(iValue, primitives, index);

				eVirtualSet(PRIMITIVES_BYTE_ARRAY, primitives);
				return value;

			}
			index += intialSize;
			int previousValue = getInt(primitives, index);
			setInt(iValue, primitives, index);
			return Float.intBitsToFloat(previousValue);

		}
	}

	protected int eVirtualSet(int eDerivedStructuralFeatureID, int value) {
		int intialSize = ((eVirtualPrimitiveFeatureIDsCount()) >>> 3) + 1;
		byte[] primitives = getPrimitivesValues();
		if (primitives == null) {
			primitives = new byte[intialSize + 4];
			eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_SET, primitives);
			setInt(value, primitives, intialSize);
			eVirtualSet(PRIMITIVES_BYTE_ARRAY, primitives);
			return value;
		} else {
			int index = eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_SET, primitives);
			if (index < 0) {
				index = ~index;
				index += intialSize;
				primitives = resizePrimitiveArray(primitives, index, 4);
				setInt(value, primitives, index);

				eVirtualSet(PRIMITIVES_BYTE_ARRAY, primitives);
				return value;

			}
			index += intialSize;
			byte previousValue = primitives[index];
			setInt(value, primitives, index);
			return previousValue;

		}
	}

	protected long eVirtualSet(int eDerivedStructuralFeatureID, long value) {
		int intialSize = ((eVirtualPrimitiveFeatureIDsCount()) >>> 3) + 1;
		byte[] primitives = getPrimitivesValues();
		if (primitives == null) {
			primitives = new byte[intialSize + 8];
			eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_SET, primitives);
			setLong(value, primitives, intialSize);
			eVirtualSet(PRIMITIVES_BYTE_ARRAY, primitives);
			return value;
		} else {
			int index = eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_SET, primitives);
			if (index < 0) {
				index = ~index;
				index += intialSize;
				primitives = resizePrimitiveArray(primitives, index, 8);
				setLong(value, primitives, index);

				eVirtualSet(PRIMITIVES_BYTE_ARRAY, primitives);
				return value;
			}
			index += intialSize;
			long previousValue = getLong(primitives, index);
			setLong(value, primitives, index);
			return previousValue;

		}
	}

	protected short eVirtualSet(int eDerivedStructuralFeatureID, short value) {
		int intialSize = ((eVirtualPrimitiveFeatureIDsCount()) >>> 3) + 1;
		byte[] primitives = getPrimitivesValues();
		if (primitives == null) {
			primitives = new byte[intialSize + 2];
			eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_SET, primitives);
			setShort(value, primitives, intialSize);
			eVirtualSet(PRIMITIVES_BYTE_ARRAY, primitives);
			return value;
		} else {
			int index = eVirtualPrimitiveIndex(eDerivedStructuralFeatureID, EVIRTUAL_SET, primitives);
			if (index < 0) {
				index = ~index;
				index += intialSize;
				primitives = resizePrimitiveArray(primitives, index, 2);
				setShort(value, primitives, index);

				eVirtualSet(PRIMITIVES_BYTE_ARRAY, primitives);
				return value;
			}
			index += intialSize;
			short previousValue = getShort(primitives, index);
			setShort(value, primitives, index);
			return previousValue;

		}

	}

	protected Object[] eVirtualValues() {
		Object data = getData();
		if (data != null && data instanceof Object[])
			return (Object[]) data;
		else
			return null;
	}

	protected int getInt(byte[] primitives, int index) {
		return (((int) primitives[index] << 24) + ((primitives[index + 1] & 255) << 16) + ((primitives[index + 2] & 255) << 8) + ((primitives[index + 3] & 255) << 0));
	}

	protected long getLong(byte[] primitives, int index) {
		return (((long) primitives[index] << 56) + ((long) (primitives[index + 1] & 255) << 48) + ((long) (primitives[index + 2] & 255) << 40)
				+ ((long) (primitives[index + 3] & 255) << 32) + ((long) (primitives[index + 4] & 255) << 24) + ((primitives[index + 5] & 255) << 16)
				+ ((primitives[index + 6] & 255) << 8) + ((primitives[index + 7] & 255) << 0));
	}

	protected short getShort(byte[] primitives, int index) {
		return (short) ((((int) primitives[index]) << 8) + (((int) primitives[index + 1]) << 0));
	}

	/**
	 * Resizes the primitives byte array to allow append of "deltaSize" bytes
	 * 
	 * @param primitives
	 * @param deltaSize
	 * @return
	 */
	protected byte[] resizePrimitiveArray(byte[] primitives, int index, int deltaSize) {
		byte[] temp = new byte[primitives.length + deltaSize];
		if (index > 0) {
			System.arraycopy(primitives, 0, temp, 0, index);
		}

		if (index < primitives.length) {
			System.arraycopy(primitives, index, temp, index + deltaSize, primitives.length - index);
		}

		return temp;
	}

	protected void setInt(int value, byte[] primitives, int index) {
		primitives[index] = (byte) (value >>> 24);
		primitives[index + 1] = (byte) (value >>> 16);
		primitives[index + 2] = (byte) (value >>> 8);
		primitives[index + 3] = (byte) (value >>> 0);
	}

	protected void setLong(long value, byte[] primitives, int index) {
		primitives[index] = (byte) (value >>> 56);
		primitives[index + 1] = (byte) (value >>> 48);
		primitives[index + 2] = (byte) (value >>> 40);
		primitives[index + 3] = (byte) (value >>> 32);
		primitives[index + 4] = (byte) (value >>> 24);
		primitives[index + 5] = (byte) (value >>> 16);
		primitives[index + 6] = (byte) (value >>> 8);
		primitives[index + 7] = (byte) (value >>> 0);
	}

	protected void setShort(short value, byte[] primitives, int index) {
		primitives[index] = (byte) ((((int) value) >>> 8) & 0xFF);
		primitives[index + 1] = (byte) ((((int) value) >>> 0) & 0xFF);
	}

	public String toString() {
		// Should use the following code to improve debuggability. Will need to
		// update testcase baselogs before this change can be made.

		StringBuffer result = new StringBuffer(getClass().getName());
		result.append('@');
		result.append(Integer.toHexString(hashCode()));

		if (eIsProxy()) {
			result.append(" (eProxyURI: ");
			result.append(eProxyURI());
			if (eDynamicClass() != null) {
				result.append(" eClass: ");
				result.append(eDynamicClass());
			}
			result.append(')');
		} else if (eStaticClass() != null) {
			result.append(" (eClass: ");
			result.append(eStaticClass());
			result.append(')');
		}

		return result.toString();
	}
}
