/**********************************************************************
 * Copyright (c) 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 *
 * $Id: 
 **********************************************************************/

package org.eclipse.hyades.models.hierarchy.util.internal;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.xml.type.util.XMLTypeUtil;
import org.eclipse.hyades.models.hierarchy.extensions.ArithmeticExpression;
import org.eclipse.hyades.models.hierarchy.extensions.ArithmeticOperators;
import org.eclipse.hyades.models.hierarchy.extensions.BinaryExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.NumericFunction;
import org.eclipse.hyades.models.hierarchy.extensions.Operand;
import org.eclipse.hyades.models.hierarchy.extensions.RelationalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleOperand;
import org.eclipse.hyades.models.hierarchy.extensions.WhereExpression;
import org.eclipse.hyades.models.hierarchy.extensions.impl.SimpleOperandImpl;

/** 
 * The purpose of this class is to process a given operand/query/expression, sometimes using partial results,
 * the result of which is a boolean value accessible through booleanValue();  
 * 
 * This class used to be an inner class of SSQE, now it is its own class but with limited ties
 * to SSQE through used of the _ssqe instance variable.
 * 
 * Also, while this class seems to support a wide variety of possible results, 
 * really only the boolean result is used externally through booleanValue()  
 * 
 *  - jgw*/
public class SSQEEvalResult {

	public static final int IS_COMPLETE = 0x01;

	public static final int IS_NOT_COMPLETE = 0x02;

	public static final int IS_TRUE = 0x04;

	public static final int IS_FALSE = 0x08;

	public static final int IS_APPLICABLE = 0x10;

	
	
	private static final int BOOLEAN = 4;

	private static final int NUMERIC = 2;

	private static final int OBJECT = 5;

	private static final int SIMPLE_OPERAND = 1;

	private static final int STRING = 3;

	private Boolean _booleanValue;

	private int _compareResult;

	private boolean _complete;

	private EObject _context;

	private EClass _contextType;

	private WhereExpression _expression;

	private Number _numericValue;

	private Object _objectValue;

	private Operand _operand;

	private List<SSQEEvalResult> _partialResults = new ArrayList<SSQEEvalResult>();

	private String _stringValue;

	private int _valueType = -1;

	private SimpleSearchQueryEngine _ssqe;

	public SSQEEvalResult(SimpleSearchQueryEngine ssqe) {
		this._ssqe = ssqe;
	}

	public SSQEEvalResult(Operand operand, SimpleSearchQueryEngine ssqe) {
		this._operand = operand;
		this._ssqe = ssqe;
		_contextType = getContextType();
	}

	public SSQEEvalResult(WhereExpression expression, SimpleSearchQueryEngine ssqe) {
		this._ssqe = ssqe;
		this._expression = expression;
	}

	public String toString() {
		StringBuffer result = new StringBuffer("\n" + getLevelSpaces().toString());
		// result.append(super.toString());
		result.append(" (complete: ");
		result.append(this._complete);
		// result.append(", objectValue: ");
		// result.append(this.objectValue);
		result.append(", valueType: ");
		result.append(this._valueType);
		result.append(", booleanValue: ");
		result.append(this._booleanValue);
		// result.append(", context: ");
		// result.append(this.context);
		result.append(", contextType: ");
		result.append(this._contextType);
		// result.append(", expression: ");
		// result.append(this.expression);
		result.append(", numericValue: ");
		result.append(this._numericValue);
		// result.append(", operand: ");
		// result.append(this.operand);
		result.append(", stringValue: ");
		result.append(this._stringValue);
		result.append(", partialResults: ");
		for (Iterator<SSQEEvalResult> iter = this._partialResults.iterator(); iter.hasNext();) {
			SSQEEvalResult element = iter.next();
			result.append(element.toString());
		}
		result.append(')');
		return result.toString();
	}

	private StringBuffer getLevelSpaces() {
		StringBuffer s = new StringBuffer();
		if (_operand != null)
			addLevelSpaces(_operand, s);
		else
			addLevelSpaces(_expression, s);
		return s;
	}

	private void addLevelSpaces(EObject o, StringBuffer s) {
		if (o == null)
			return;
		s.append(" ");
		addLevelSpaces(o.eContainer(), s);
	}

	/** Add together all of the numbers currently in the _partialResults list, and return the result */
	private Number add() {
		Number result = null;
		for (int i = 0; i < _partialResults.size() && i < 2; i++) {
			if (result == null)
				result = (_partialResults.get(i).numericValue());
			else {
				if (result instanceof Double) {
					result = new Double(result.doubleValue() + _partialResults.get(i).numericValue().doubleValue());
				} else if (result instanceof Integer) {
					result = new Integer(result.intValue() + _partialResults.get(i).numericValue().intValue());
				} else if (result instanceof Float) {
					result = new Float(result.floatValue() + _partialResults.get(i).numericValue().floatValue());
				} else if (result instanceof Short) {
					result = new Short((short) (result.shortValue() + _partialResults.get(i).numericValue().shortValue()));
				} else if (result instanceof Long) {
					result = new Long(result.longValue() + _partialResults.get(i).numericValue().longValue());
				} else if (result instanceof Byte) {
					result = new Long(result.byteValue() + _partialResults.get(i).numericValue().byteValue());
				}
			}
		}
		return result;
	}

	public void addPartialResult(SSQEEvalResult result) {
		_partialResults.add(result);
	}

	public Boolean booleanValue() {
		return _booleanValue;
	}

	/** Compares two results, returns value as per standard compare() method */
	private int compare(SSQEEvalResult lhs, SSQEEvalResult rhs) {
		
		if (lhs.isNumeric() && rhs.isNumeric() && lhs.numericValue() instanceof Comparable) {
			return ((Comparable<Number>) lhs.numericValue()).compareTo(rhs.numericValue());
			
		} else if (lhs.isBoolean() && lhs.isBoolean()) {
			if (lhs.booleanValue() == Boolean.TRUE) {
				if (rhs.booleanValue() == Boolean.TRUE)
					return 0;
				else
					return 1;
			} else {
				if (rhs.booleanValue() == Boolean.FALSE)
					return 0;
				else
					return -1;
			}
			
		} else if ((lhs.isObject() || lhs.isString()) && rhs.isObject()) {
			if (rhs.objectValue() instanceof List)
				try {
					if (lhs.objectValue() != null)
						return ((List) rhs.objectValue()).contains(lhs.objectValue()) ? 0 : -1;
					else
						return ((List) rhs.objectValue()).contains(lhs.stringValue()) ? 0 : -1;
				} catch (Exception e) {
					return -1;
				}
			else {
				if (rhs._objectValue.getClass().getName().endsWith(SimpleSearchQueryEngine.XMLCALENDAR) && lhs._objectValue.getClass().getName().endsWith(SimpleSearchQueryEngine.XMLCALENDAR))
					try {
						return XMLTypeUtil.compareCalendar(lhs.objectValue(), rhs.objectValue());

					} catch (Exception e) {
						// ignore and try String compare
					}
				// if(lhs.objectValue() instanceof Comparable)
				// return
				// ((Comparable)lhs.objectValue()).compareTo(rhs.objectValue());
				// else
				// else
				return lhs.objectValue().toString().compareTo(rhs.objectValue().toString());
			}
		} else if (((BinaryExpression) _expression).isCaseInsensitive())
			return lhs.stringValue().toLowerCase().compareTo(rhs.stringValue().toLowerCase());
		else
			return lhs.stringValue().compareTo(rhs.stringValue());
	}

	/** */
	private SSQEEvalResult computePartialResult() {
		if (isComplete())
			return this;
		
		if (_operand != null) {
			if (_operand instanceof NumericFunction)
				updateNumericFunctionValue();
			else if (_operand instanceof ArithmeticExpression)
				updateArithmeticExpressionValue();
			else
				updateSimpleOperand();
			
		} else if (_expression != null) {
			if (_expression instanceof LogicalExpression)
				updateLogicalExpressionValue();
			else
				updateBinaryExpressionValue();
		}
		return this;
	}

	/** Divides the first value in partial results by the second value, returns the result */
	private Number divide() {
		Number result = null;
		for (int i = 0; i < _partialResults.size() && i < 2; i++) {
			if (result == null)
				result = (_partialResults.get(i).numericValue());
			else {
				if (result instanceof Double) {
					result = new Double(result.doubleValue() / _partialResults.get(i).numericValue().doubleValue());
				} else if (result instanceof Integer) {
					result = new Integer(result.intValue() / _partialResults.get(i).numericValue().intValue());
				} else if (result instanceof Float) {
					result = new Float(result.floatValue() / _partialResults.get(i).numericValue().floatValue());
				} else if (result instanceof Short) {
					result = new Short((short) (result.shortValue() / _partialResults.get(i).numericValue().shortValue()));
				} else if (result instanceof Long) {
					result = new Long(result.longValue() / _partialResults.get(i).numericValue().longValue());
				} else if (result instanceof Byte) {
					result = new Long(result.byteValue() / _partialResults.get(i).numericValue().byteValue());
				}
			}
		}
		return result;
	}

	public SSQEEvalResult eval(EObject element) {
		
		// Call eval on all the partial results contained in this result
		// (e.g. evaluate all our children)
		for (int i = 0; i < _partialResults.size(); i++) {
			SSQEEvalResult r = _partialResults.get(i);
			r.eval(element);
		}
		
		
		if (_expression != null || !(_operand instanceof SimpleOperand)) {
			resetValues();
			_complete = false;
			
		} else if (isValidContext(element)) {
			_context = element;
			resetValues();
			_complete = false;
		}
		computePartialResult();
		return this;
	}

	private EClass getContextType() {
		if (_operand instanceof SimpleOperandImpl) {
			if (((SimpleOperand) _operand).getType() != null)
				return ((SimpleOperand) _operand).getType();
			if (((SimpleOperand) _operand).getFeature() != null)
				return ((SimpleOperand) _operand).getFeature().getEContainingClass();
		}
		return null;
	}

	/** Divide the first result in partialResults by the second result in partial Results, return the result */
	private Number integerDivide() {
		int result = 0;
		boolean first = true;
		for (int i = 0; i < _partialResults.size() && i < 2; i++) {
			if (first) {
				result = (_partialResults.get(i).numericValue()).intValue();
				first = false;
			} else {
				result = result / _partialResults.get(i).numericValue().intValue();
			}
		}
		return new Integer(result);
	}

	/** The expression is considered complete if all the partialResults are complete, and the number of partial
	 * results matches the number of arguments in ArithmeticExpression */
	private boolean isArithmeticExpressionComplete(ArithmeticExpression arithmeticExpression) {
		
		int i = 0;
		for (; i < _partialResults.size(); i++) {
			if (!_partialResults.get(i).isComplete()) {
				return false;
			}
		}
		return i == (arithmeticExpression.getArguments().size() - 1);
	}

	private boolean isBetweenComplete() {
		if (_partialResults.size() == 3) {
			SSQEEvalResult lhs = _partialResults.get(0);
			SSQEEvalResult rhs1 = _partialResults.get(1);
			SSQEEvalResult rhs2 = _partialResults.get(2);
			if (lhs.isComplete() && rhs1.isComplete() && rhs2.isComplete()) {
				int c1 = compare(rhs1, lhs);
				if (c1 > 0)
					_compareResult = -1;
				else {
					int c2 = compare(lhs, rhs2);
					if (c2 > 0)
						_compareResult = 1;
					else
						_compareResult = 0;
				}
				_complete = true;
				return true;
			}
		}
		return false;
	}

	private boolean isBinaryComplete() {
		if (_partialResults.size() == 2) {
			SSQEEvalResult lhs = _partialResults.get(0);
			SSQEEvalResult rhs = _partialResults.get(1);
			if (lhs.isComplete() && rhs.isComplete()) {
				_compareResult = compare(lhs, rhs);
				_complete = true;
				return true;
			}
		}
		return false;
	}

	private boolean isBoolean() {
		return _valueType == BOOLEAN;
	}

	public boolean isComplete() {
		if (_complete) {
			return true;
		}
		if (_objectValue != null || _stringValue != null || _booleanValue != null || _numericValue != null) {
			_complete = true;
			return true;
		} else
			return false;
	}

	private boolean isInComplete() {
		// BinaryExpression binaryExpression = (BinaryExpression)
		// expression;
		if (_partialResults.isEmpty())
			return false;
		
		SSQEEvalResult lhs = _partialResults.get(0);
		if (!lhs.isComplete())
			return false;
		
		int i = 1, j = 1;
		_compareResult = -1;
		
		
		for (; i < _partialResults.size(); i++) {
			
			// JGW: This is probably wrong (e.g. should be .get(i), but was 1 in the original code, so I'm leaving it)
			SSQEEvalResult rhs = _partialResults.get(1);
			if (rhs.isComplete()) {
				j++;
				_compareResult = compare(lhs, rhs);
				if (_compareResult == 0) {
					_complete = true;
					return true;
				}
			}
		}
		if (i == j) {
			_complete = true;
			_booleanValue = Boolean.FALSE;
		}
		return false;
	}

	private boolean isLikeComplete() {
		if (_partialResults.size() == 2) {
			SSQEEvalResult lhs = _partialResults.get(0);
			SSQEEvalResult rhs = _partialResults.get(1);
			if (lhs.isComplete() && rhs.isComplete()) {
				_compareResult = like(lhs, rhs);
				_complete = true;
				return true;
			}
		}
		return false;
	}

	private boolean isNumeric() {
		return _valueType == NUMERIC;
	}

	private boolean isObject() {
		return _valueType == OBJECT;
	}

	private boolean isString() {
		return _valueType == STRING;
	}

	private boolean isValidContext(EObject element) {
		if (_contextType != null) {
			if (element != null && (element.eClass() == _contextType || QueryUtils.getAllSubTypes(_contextType).contains(element.eClass())))
				return true;
		}
		return false;
	}

	private int like(SSQEEvalResult lhs, SSQEEvalResult rhs) {
		// TODO investigate support for like on list entries instead of
		// List.toString value
		// if (lhs.isObject() && lhs.objectValue() instanceof List)
		// {
		// List l = (List)lhs.objectValue();
		// String r = rhs.stringValue();
		// try {
		// for (int i = 0; i < l.size(); i++) {
		// if(like(l.get(i).toString(), r)==0)
		// return 0;
		// }
		// } catch (Exception e) {
		// // ignore exceptions
		// }
		// return -1;
		// }
		// else
		return like(lhs.stringValue(), rhs.stringValue());
	}

	private int like(String lhs, String rhs) {
		if (((BinaryExpression) _expression).isCaseInsensitive()) {
			lhs = lhs.toLowerCase();
			rhs = rhs.toLowerCase();
		}
		return QueryUtils.like(lhs, rhs);
	}

	private Number mod() {
		Number result = null;
		// will evaluate only the first two arguments like lsh % rhs,
		// returns the the lhs if rhs is missing
		for (int i = 0; i < _partialResults.size() && i < 2; i++) {
			if (result == null)
				result = (_partialResults.get(i).numericValue());
			else {
				if (result instanceof Double) {
					result = new Double(result.doubleValue() % _partialResults.get(i).numericValue().doubleValue());
				} else if (result instanceof Integer) {
					result = new Integer(result.intValue() % _partialResults.get(i).numericValue().intValue());
				} else if (result instanceof Float) {
					result = new Float(result.floatValue() % _partialResults.get(i).numericValue().floatValue());
				} else if (result instanceof Short) {
					result = new Short((short) (result.shortValue() % _partialResults.get(i).numericValue().shortValue()));
				} else if (result instanceof Long) {
					result = new Long(result.longValue() % _partialResults.get(i).numericValue().longValue());
				} else if (result instanceof Byte) {
					result = new Long(result.byteValue() % _partialResults.get(i).numericValue().byteValue());
				}
			}
		}
		return result;
	}

	private Number multiply() {
		Number result = null;
		for (int i = 0; i < _partialResults.size(); i++) {
			if (result == null)
				result = _partialResults.get(i).numericValue();
			else {
				if (result instanceof Double) {
					result = new Double(result.doubleValue() * _partialResults.get(i).numericValue().doubleValue());
				} else if (result instanceof Integer) {
					result = new Integer(result.intValue() * _partialResults.get(i).numericValue().intValue());
				} else if (result instanceof Float) {
					result = new Float(result.floatValue() * _partialResults.get(i).numericValue().floatValue());
				} else if (result instanceof Short) {
					result = new Short((short) (result.shortValue() * _partialResults.get(i).numericValue().shortValue()));
				} else if (result instanceof Long) {
					result = new Long(result.longValue() * _partialResults.get(i).numericValue().longValue());
				} else if (result instanceof Byte) {
					result = new Long(result.byteValue() * _partialResults.get(i).numericValue().byteValue());
				}
			}
		}
		return result;
	}

	private Number numericValue() {
		return _numericValue;
	}

	private Object objectValue() {
		return _objectValue;
	}

	public boolean reset(EObject element) {
		if (isValidContext(element)) {
			_context = null;
			_complete = false;
		}
		for (int i = 0; i < _partialResults.size(); i++) {
			SSQEEvalResult r = _partialResults.get(i);
			if (!r.reset(element)) {
				_complete = false;
			}
		}
		if (!_complete) {
			resetValues();
		}
		return isComplete();
	}

	private void resetValues() {
		_numericValue = null;
		_stringValue = null;
		_objectValue = null;
		_booleanValue = null;
	}

	private String stringValue() {
		if (_stringValue != null)
			return _stringValue;
		if (_numericValue != null)
			return _numericValue.toString();
		if (_booleanValue != null)
			return _booleanValue.toString();
		if (_objectValue != null)
			return _objectValue.toString();
		return "";
	}

	private Number substract() {
		Number result = null;
		for (int i = 0; i < _partialResults.size(); i++) {
			if (result == null)
				result = _partialResults.get(i).numericValue();
			else {
				if (result instanceof Double) {
					result = new Double(result.doubleValue() - _partialResults.get(i).numericValue().doubleValue());
				} else if (result instanceof Integer) {
					result = new Integer(result.intValue() - _partialResults.get(i).numericValue().intValue());
				} else if (result instanceof Float) {
					result = new Float(result.floatValue() - _partialResults.get(i).numericValue().floatValue());
				} else if (result instanceof Short) {
					result = new Short((short) (result.shortValue() - _partialResults.get(i).numericValue().shortValue()));
				} else if (result instanceof Long) {
					result = new Long(result.longValue() - _partialResults.get(i).numericValue().longValue());
				} else if (result instanceof Byte) {
					result = new Long(result.byteValue() - _partialResults.get(i).numericValue().byteValue());
				}
			}
		}
		return result;
	}

	private void updateArithmeticExpressionValue() {
		ArithmeticExpression arithmeticExpression = (ArithmeticExpression) _expression;
		
		if (isArithmeticExpressionComplete(arithmeticExpression)) {
			switch (arithmeticExpression.getOperator().getValue()) {
			case ArithmeticOperators.ADD:
				_numericValue = add();
				break;
			case ArithmeticOperators.DIVIDE:
				_numericValue = divide();
				break;
			case ArithmeticOperators.INTEGER_DIVIDE:
				_numericValue = integerDivide();
				break;
			case ArithmeticOperators.MOD:
				_numericValue = mod();
				break;
			case ArithmeticOperators.MULTIPLY:
				_numericValue = multiply();
				break;
			case ArithmeticOperators.SUBSTRACT:
				_numericValue = substract();
				break;
			}
		}
	}

	private void updateBinaryExpressionValue() {
		BinaryExpression binaryExpression = (BinaryExpression) _expression;
		
		if (binaryExpression.getOperator() == RelationalOperators.BETWEEN_LITERAL) {
			if (isBetweenComplete())
				_booleanValue = Boolean.valueOf(_compareResult == 0);
		
		} else if (binaryExpression.getOperator() == RelationalOperators.IN_LITERAL) {
			if (isInComplete())
				_booleanValue = Boolean.valueOf(_compareResult == 0);
		
		} else if (binaryExpression.getOperator() == RelationalOperators.LIKE_LITERAL) {
			if (isLikeComplete())
				_booleanValue = Boolean.valueOf(_compareResult == 0);
		
		} else if (isBinaryComplete()) {
			switch (binaryExpression.getOperator().getValue()) {
			case RelationalOperators.EQ:
				_booleanValue = Boolean.valueOf(_compareResult == 0);
				break;
			case RelationalOperators.GE:
				_booleanValue = Boolean.valueOf(_compareResult >= 0);
				break;
			case RelationalOperators.GT:
				_booleanValue = Boolean.valueOf(_compareResult > 0);
				break;
			case RelationalOperators.LE:
				_booleanValue = Boolean.valueOf(_compareResult <= 0);
				break;
			case RelationalOperators.LT:
				_booleanValue = Boolean.valueOf(_compareResult < 0);
				break;
			case RelationalOperators.NEQ:
				_booleanValue = Boolean.valueOf(_compareResult != 0);
				break;
			}
		}
	}

	private void updateLogicalExpressionValue() {
		LogicalExpression logicalExpression = (LogicalExpression) _expression;
		int processed = 0;
		
		for (int i = 0; i < _partialResults.size(); i++) {

			// For each of the results in the partial result list...
			SSQEEvalResult result = _partialResults.get(i);
			
			// If complete...
			if (result.isComplete()) {
				
				// If first, set complete, value, and break
				if (i == 0 && logicalExpression.getOperator().getValue() == LogicalOperators.NOT) {
					_complete = true;
					_booleanValue = Boolean.valueOf(!result.booleanValue().booleanValue());
					
				// If operator is and...
				} else if (logicalExpression.getOperator().getValue() == LogicalOperators.AND) {
					processed++;
					if (!result.booleanValue().booleanValue()) {
						_complete = true;
						_booleanValue = Boolean.FALSE;
					}
					
				// if operator is or
				} else if (logicalExpression.getOperator().getValue() == LogicalOperators.OR) {
					processed++;
					if (result.booleanValue().booleanValue()) {
						_complete = true;
						_booleanValue = Boolean.TRUE;
					}
				}
			}
			if (_complete)
				break;
		}
		
		if (!_complete) {
			if (logicalExpression.getArguments().size() == processed) {
				_complete = true;
				if (logicalExpression.getOperator().getValue() == LogicalOperators.AND)
					_booleanValue = Boolean.TRUE;
				else if (logicalExpression.getOperator().getValue() == LogicalOperators.OR) {
					if (logicalExpression.getArguments().size() == 0)
						_booleanValue = Boolean.TRUE;
					else
						_booleanValue = Boolean.FALSE;
				}
			}
		}
	}

	private void updateNumericFunctionValue() {
		// TODO implement COUNT, MAX and MIN
		// NumericFunction numericFunction = (NumericFunction) operand;
		// if (numericFunction.getFunction() ==
		// NumericFunctions.COUNT_LITERAL) {
		// }
		throw new RuntimeException("Not implemented yet !");
	}

	private void updateSimpleOperand() {
		if (isComplete())
			return;
		SimpleOperand simpleOperand = ((SimpleOperand) _operand);
		if (simpleOperand.getType() != null) {
			if (_context == null)
				return;
			_valueType = OBJECT;
			_objectValue = _context;
			_complete = true;
		} else if (simpleOperand.getFeature() != null) {
			if (_context == null)
				return;
			try {
				Object v = _context.eGet(simpleOperand.getFeature(), false);
				updateSimpleOperandValue(v);
			} catch (Exception e) {
				// ignore features that are not applicable to the current
				// context
			}
		} else {
			Object v = _ssqe.getNonModeledElementValue(this,simpleOperand);
			updateSimpleOperandValue(v);
		}
	}

	private void updateSimpleOperandValue(Object v) {
		if (v instanceof Number) {
			_numericValue = (Number) v;
			_valueType = NUMERIC;
		} else if (v instanceof Boolean) {
			_booleanValue = (Boolean) v;
			_valueType = BOOLEAN;
		} else if (v instanceof String) {
			_stringValue = v.toString();
			_valueType = STRING;
		} else {
			_objectValue = v;
			_valueType = OBJECT;
		}
		_complete = true;
	}

	public SSQEEvalResult evalOnly(EStructuralFeature sf, Object value) {
		EClass c = sf.getEContainingClass();
		if (!_ssqe._classPredicatesIndex.containsKey(c)) {
			_complete = true;
			_booleanValue = Boolean.TRUE;
			return this;
		}
		internalEvalOnly(sf, value);
		return this;
	}

	private SSQEEvalResult internalEvalOnly(EStructuralFeature sf, Object value) {
		if (_operand instanceof SimpleOperand && ((SimpleOperand) _operand).getFeature() == sf) {
			_objectValue = value;
			_complete = true;
		} else {
			for (int i = 0; i < _partialResults.size(); i++) {
				SSQEEvalResult r = _partialResults.get(i);
				r.internalEvalOnly(sf, value);
			}
			computePartialResult();
		}
		return this;
	}

	public int partialEvalOnly(EClass value) {
		if (!_ssqe._classPredicatesIndex.containsKey(value)) {
			return IS_COMPLETE & IS_TRUE;
		}
		return internalEvalOnly(value);
	}

	private int internalEvalOnly(EClass value) {
		if (_operand != null) {
			if (_operand instanceof SimpleOperand) {
				if (_contextType == null || _contextType == value) {
					return (isComplete() ? IS_COMPLETE : IS_NOT_COMPLETE) | IS_APPLICABLE;
				} else {
					return (isComplete() ? IS_COMPLETE : IS_NOT_COMPLETE);
				}
			}
		} else {
			// if (isComplete())
			// if (booleanValue().booleanValue())
			// return IS_COMPLETE | IS_TRUE | IS_APPLICABLE;
			// else if (partialResults.size() == 0)
			// return (IS_COMPLETE | IS_FALSE);
			int i = 0, k = 0, f = 0, t = 0;
			for (; i < _partialResults.size(); i++) {
				SSQEEvalResult r = _partialResults.get(i);
				int res = r.internalEvalOnly(value);
				if ((res & IS_APPLICABLE) != 0) {
					k++;
					if ((res & IS_COMPLETE) != 0) {
						if ((res & IS_TRUE) != 0)
							t++;
						else if ((res & IS_FALSE) != 0)
							f++;
					} else
						return IS_NOT_COMPLETE | IS_APPLICABLE;
				}
			}
			if (k == 0)
				return (isComplete() ? IS_COMPLETE : IS_NOT_COMPLETE);
			if (_expression instanceof BinaryExpression) {
				if (isComplete()) {
					if (booleanValue().booleanValue())
						return IS_COMPLETE | IS_APPLICABLE | IS_TRUE;
					else
						return IS_COMPLETE | IS_APPLICABLE | IS_FALSE;
				} else
					return IS_NOT_COMPLETE | IS_APPLICABLE;
			} else if (_expression instanceof LogicalExpression) {
				LogicalExpression le = (LogicalExpression) _expression;
				if (le.getOperator() == LogicalOperators.AND_LITERAL) {
					if (k == t)
						return IS_COMPLETE | IS_APPLICABLE | IS_TRUE;
					else
						return IS_COMPLETE | IS_APPLICABLE | IS_FALSE;
				}
				if (le.getOperator() == LogicalOperators.OR_LITERAL) {
					if (t > 0)
						return IS_COMPLETE | IS_APPLICABLE | IS_TRUE;
					else if (f > 0)
						return IS_COMPLETE | IS_APPLICABLE | IS_FALSE;
					else
						return IS_NOT_COMPLETE | IS_APPLICABLE;
				}
				if (le.getOperator() == LogicalOperators.NOT_LITERAL) {
					if (t > 0)
						return IS_COMPLETE | IS_APPLICABLE | IS_FALSE;
					else if (f > 0)
						return IS_COMPLETE | IS_APPLICABLE | IS_TRUE;
					else
						return IS_NOT_COMPLETE | IS_APPLICABLE;
				}
			}
		}
		return 0;
	}

	public boolean resetOnly(EStructuralFeature sf) {
		EClass c = sf.getEContainingClass();
		if (!_ssqe._classPredicatesIndex.containsKey(c)) {
			_complete = false;
			_booleanValue = null;
			return false;
		}
		return internalResetOnly(sf);
	}

	private boolean internalResetOnly(EStructuralFeature sf) {
		if (_operand instanceof SimpleOperand && ((SimpleOperand) _operand).getFeature() == sf) {
			_objectValue = null;
			_complete = false;
		} else {
			for (int i = 0; i < _partialResults.size(); i++) {
				SSQEEvalResult r = _partialResults.get(i);
				if (!r.internalResetOnly(sf)) {
					_complete = false;
				}
			}
			if (!_complete) {
				resetValues();
			}
		}
		return isComplete();
	}

	public EObject getContext() {
		return _context;
	}
}
