/**********************************************************************
 * Copyright (c) 2003, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 *
 * $Id: SimpleSearchQueryEngine.java,v 1.3 2010/08/11 13:49:54 bjerome Exp $
 **********************************************************************/
package org.eclipse.hyades.models.hierarchy.util.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.xml.type.util.XMLTypeUtil;
import org.eclipse.hyades.models.hierarchy.extensions.ArithmeticExpression;
import org.eclipse.hyades.models.hierarchy.extensions.BinaryExpression;
import org.eclipse.hyades.models.hierarchy.extensions.ExtensionsFactory;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalExpression;
import org.eclipse.hyades.models.hierarchy.extensions.NumericFunction;
import org.eclipse.hyades.models.hierarchy.extensions.Operand;
import org.eclipse.hyades.models.hierarchy.extensions.OrderByElement;
import org.eclipse.hyades.models.hierarchy.extensions.OrderByOperators;
import org.eclipse.hyades.models.hierarchy.extensions.QueryResult;
import org.eclipse.hyades.models.hierarchy.extensions.ResultEntry;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleOperand;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.hierarchy.extensions.WhereExpression;
import org.eclipse.hyades.models.hierarchy.util.ContainmentTraverser;
import org.eclipse.hyades.models.hierarchy.util.FastList;
import org.eclipse.hyades.models.hierarchy.util.PerfUtil;
import org.eclipse.hyades.models.hierarchy.util.SaveUtil;
import org.eclipse.hyades.models.util.ModelDebugger;
import org.eclipse.hyades.models.util.internal.IdentityHashSet;

/**
 * @author Marius Slavescu (slavescu@ca.ibm.com)
 * @author jerome bozier
 * @since 3.3
 * @version August 11, 2010
 */
public class SimpleSearchQueryEngine {
	public static final String XMLCALENDAR = "XMLCalendar";

	private static final boolean DEBUG_SSQE = false;
	
	// protected Set currentPath = new HashSet(20);
	private Set<EObject> _alreadyProcessed = new IdentityHashSet<EObject>();

	/** Generally speaking, this variable seems to contain only the EClasses that are relevant to the query engine,
	 * and the value of the map is not useful, only the key is useful (jgw) */
	Map<EClass, Collection<EObject>> _classPredicatesIndex = new IdentityHashMap<EClass, Collection<EObject>>();

	private Map<EClass, Collection<EObject>> _featurePredicatesIndex = new IdentityHashMap<EClass, Collection<EObject>>();

	// protected Map instancePredicatesIndex = new HashMap();
	
	/** Associates the name value of all elements seen by indexWhereExpression() with their element 
	 * (which apparently are either an Operand or a WhereExpression) */
	private Map<String, EObject> _namesIndex = new HashMap<String, EObject>();
	
	/** Generally speaking, a limited amount of processing will occur if a class is not in the following
	 * three listed of output classes (this will likely prevent their elements from being added 
	 * to the result, in the case of output classes, for instance) (jgw) */
	private Set<EClass> _outputClasses = new IdentityHashSet<EClass>();
	private Set<Object> _outputClassesSuperTypes = new IdentityHashSet<Object>();
	private Set<Object> _outputClassesSubTypes = new IdentityHashSet<Object>();

	SimpleSearchQuery _query;

	private QueryResult _queryResult;

	private Set<EStructuralFeature> _requiredPaths = new IdentityHashSet<EStructuralFeature>();

	private Collection<EClass> _rootClasses = new IdentityHashSet<EClass>();

	private Collection<EObject> _rootNodes = new IdentityHashSet<EObject>();

	private ResourceSet _targetResourceSet;

	private Map<EObject, SSQEEvalResult> _whereExpressionValuesIndex = new IdentityHashMap<EObject, SSQEEvalResult>();

	SimpleSearchQueryEvaluator _queryEvaluator = new SimpleSearchQueryEvaluator();

	private SSQEEvalResult _result;

	private List<EObject> _resetStack = new FastList<EObject>();

	private List<EObject> _resetParentsStack = new FastList<EObject>();

	/** For debug purposes */
	private PerfUtil _p;

	private ArrayList<EObject> _firstResList;

	// ----------------------------------------
	
//	public static String outputOperand(Operand e, int depth) {
//		String s = "";
//		s += r(depth) + e.g;
//		
//		return s;
//	}
	
	// ----------------------------------------
	
	public SimpleSearchQueryEngine(SimpleSearchQuery query, ResourceSet targetResourceSet) {
		this._query = query;
		this._targetResourceSet = targetResourceSet;
		for (Iterator<EStructuralFeature> iter = query.getRequiredPaths().iterator(); iter.hasNext();) {
			EStructuralFeature element = iter.next();
			_requiredPaths.add(element);
		}

	}

	private SSQEEvalResult createEvalResult(Operand operand) {
		return new SSQEEvalResult(operand, this);
	}

	protected SSQEEvalResult createEvalResult(WhereExpression expression) {
		return new SSQEEvalResult(expression, this);
	}

	private void addResultValue(EObject element) {
		// TODO add support for multiple output elements
		if (_queryResult.getResultEntries().size() == 1) {
			// List l = (List) ((ResultEntry)
			// queryResult.getResultEntries().get(0)).getValue();
			// if (l == null) {
			// l = new ArrayList();
			// ((ResultEntry)
			// queryResult.getResultEntries().get(0)).setValue(l);
			// }
			// l.add(element);
			if (_query.isDistinct() && _firstResList.lastIndexOf(element) >= 0)
				return;
			_firstResList.add(element);
			if(DEBUG_SSQE) System.out.println("Adding element: "+element);
			return;
		}
		throw new RuntimeException("Not implemented yet !");
	}

	protected void buildRequiredTraversalPaths() {
		_alreadyProcessed.clear();
		for (Iterator<EClass> iter = _rootClasses.iterator(); iter.hasNext();) {
			EClass element = iter.next();
			processContainmentPath(element);
		}
	}

	private void evalSelfAndParents(EObject element) {
		if (element == null || _alreadyProcessed.contains(element))
			return;
		if (_classPredicatesIndex.containsKey(element.eClass())) {
			_alreadyProcessed.add(element);
			_result.eval(element);
			_resetParentsStack.add(element);
			// traverseFeatures(element, resetParentsStack);
		}
		if (element.eContainer() != null)
			evalSelfAndParents(element.eContainer());
	}

//	private void resetSelfAndParents(EObject element, SSQEEvalResult result) {
//		if (element == null || _alreadyProcessed.contains(element))
//			return;
//		if (_classPredicatesIndex.containsKey(element.eClass()))
//			result.reset(element);
//		_alreadyProcessed.add(element);
//		if (element.eContainer() != null)
//			resetSelfAndParents(element.eContainer(), result);
//	}

	public QueryResult execute() {
		if (ModelDebugger.INSTANCE.debugPerfUtil) {
			_p = PerfUtil.createInstance(null, false);
			// p.setDebug(true);
			_p.setMessageAndStart("SimpleSearchQueryEngine.execute()");
		}
		prepareResult();
		populateRootNodesAndClasses();
		buildRequiredTraversalPaths();
		indexWhereExpression();
		populateResult();
		sortResult();
		limitResult();
		if (ModelDebugger.INSTANCE.debugPerfUtil) {
			_p.stopAndPrintStatus("firstResList.size=" + (_firstResList == null ? "null" : "" + _firstResList.size()));
		}
		return _queryResult;
	}

	protected void indexWhereExpression() {
		List<WhereExpression> roots = new ArrayList<WhereExpression>();
		if (_query.getWhereExpression() == null)
			return;
		roots.add(_query.getWhereExpression());
		
		ContainmentTraverser containmentTraverser = new ContainmentTraverser(roots) {
			protected boolean beforeChildren(EObject element) {
				// - Affects _namesIndex, _featurePredicatesIndex, _classPredicatesIndex, _whereExpressionValuesIndex				
				if(DEBUG_SSQE) System.out.println("indexWhereExpression - after -  " + element);
				
				String name = QueryUtils.getElementName(element);
				if (name != null && name.startsWith(QueryUtils.DOLLAR_SIGN))
					return super.beforeChildren(element);
				if (_whereExpressionValuesIndex.containsKey(element))
					return super.beforeChildren(element);
				
				// initialize evaluation result
				_whereExpressionValuesIndex.put(element, null);
				if (element instanceof SimpleOperand) {
					SimpleOperand o = (SimpleOperand) element;
					EClass c = o.getType();
					if (c != null) {
						Collection<EObject> l = _classPredicatesIndex.get(c);
						if (l == null) {
							l = new HashSet<EObject>();
							_classPredicatesIndex.put(c, l);
							QueryUtils.addSubTypes(c, _classPredicatesIndex, l);
						}
						if (!l.contains(element))
							l.add(element);
					} else if (o.getFeature() != null) {
						if (!_requiredPaths.contains(o.getFeature())) {
							c = o.getFeature().getEContainingClass();
							// if(o.getFeature().getName().equals("isA"))
							// System.out.println("SimpleSearchQueryEngine.indexWhereExpression()");
							// if(o.getFeature().getName().equals("refTarget"))
							// System.out.println("SimpleSearchQueryEngine.indexWhereExpression()");
							// if(c.getName().equals("TRCHeapObject"))
							// System.out.println("SimpleSearchQueryEngine.indexWhereExpression()");
							Collection<EObject> l = _classPredicatesIndex.get(c);
							if (l == null) {
								l = new HashSet<EObject>();
								_classPredicatesIndex.put(c, l);
								QueryUtils.addSubTypes(c, _classPredicatesIndex, l);
							}
							if (!l.contains(element)) {
								l.add(element);
								QueryUtils.addSubTypes(c, _classPredicatesIndex, l);
							}
							if (o.getFeature() instanceof EReference) { // ignore
								// attributes)
								l = _featurePredicatesIndex.get(c);
								if (l == null) {
									l = new HashSet<EObject>();
									_featurePredicatesIndex.put(c, l);
									QueryUtils.addSubTypes(c, _featurePredicatesIndex, l);
								}
								if (!l.contains(o.getFeature())) {
									l.add(o.getFeature());
									QueryUtils.addSubTypes(c, _featurePredicatesIndex, l);
								}
							}
						}
					}
				}
				if (name != null)
					_namesIndex.put(name, element);
				return super.beforeChildren(element);
			}
		};
		containmentTraverser.traverse();
	}

	private boolean isOutputElement(EObject element) {
		EClass c = element.eClass();
		return isOutputElement(c);
	}

	private boolean isOutputElement(EClass eClass) {
		return _outputClasses.contains(eClass) || _outputClassesSubTypes.contains(eClass);
	}

	private void limitResult() {
		if (_query.getMaxElements() == 0)
			return;

		// support only for one output element
		ResultEntry tempResult = (ResultEntry) _queryResult.getResultEntries().get(0);
		List initialList = ((List) ((ResultEntry) _queryResult.getResultEntries().get(0)).getValue());
		List resultList = new ArrayList();
		int start, end = 0;
		if (_query.getStartWith() >= 0) {
			start = _query.getStartWith();
			end = Math.min(start + _query.getMaxElements() - 1, initialList.size() - 1);
		} else {
			end = Math.max(0, initialList.size() + _query.getStartWith());
			start = Math.max(0, end - _query.getMaxElements());
		}
		for (int i = start; i <= end; i++) {
			resultList.add(initialList.get(i));
		}
		tempResult.setValue(resultList);
	}

	protected void populateResult() {
		_result = _queryEvaluator.prepareEvalResult();
		_alreadyProcessed.clear();
		if (_firstResList != null)
			_firstResList.clear();
		
//		SSQEDebug.outputQuery(_query, 0);

		ContainmentTraverser containmentTraverser = new ContainmentTraverser(_rootNodes) {
			protected int depth = 0;

			protected boolean afterChildren(EObject element) {
				
				if(DEBUG_SSQE) System.out.println("populateResult - after -  " + element);
				depth--;
				EClass eClass = element.eClass();
				
				// If the name of the class is not contains in classPredicates Index, then return				
				if (_classPredicatesIndex.containsKey(eClass)) {
					if (depth == 0) {
						// alreadyProcessed.clear();
						// resetSelfAndParents(element.eContainer(), result);
						if (!_resetParentsStack.isEmpty()) {
							processResetList(_resetParentsStack, 0);
						}
					}

					if (isOutputElement(eClass)) {
						int ei = _resetStack.lastIndexOf(element);
						if (ei >= 0) {
							processResetList(_resetStack, ei);
						}
					}
					pruneSubtree = shouldPrune(element, depth, false);
				}
				return true;
			}

			protected boolean beforeChildren(EObject element) {
				depth++;
				EClass eClass = element.eClass();
				
				if(DEBUG_SSQE) System.out.println("populateResult - before -  " + element);
				
				// If the name of the class is not contained in classPredicates Index, then return
				if (_classPredicatesIndex.containsKey(element.eClass())) {
					if (depth == 1) {
						_alreadyProcessed.clear();
						evalSelfAndParents(element.eContainer());
					}
					_result.eval(element);
					_resetStack.add(element);
					
					// Whether or not _outputClasses or _outputClassesSubTypes contains the eClass
					if (isOutputElement(eClass)) {
						if(DEBUG_SSQE) System.out.println("populateResult - before - isOutputElement true, result complete " + _result.isComplete());
						if (_result.isComplete()) {
							if (_result.booleanValue().booleanValue()) {
								addResultValue(element);
								if(DEBUG_SSQE) System.out.println("populateResult - before - addResultValue1 ");
								pruneSubtree = shouldPrune(element, depth, true);
							}
						} else {
							if (traverseFeatures(element, _resetStack)) {
								if (_result.isComplete())
									if (_result.booleanValue().booleanValue()) {
										addResultValue(element);
										if(DEBUG_SSQE) System.out.println("populateResult - before - addResultValue2 ");
										pruneSubtree = shouldPrune(element, depth, true);
									}
							}
						}
						// pruneSubtree = true;
					} else {
						if(DEBUG_SSQE) System.out.println("populateResult - before - isOutputElement false, result complete " + _result.isComplete());
						if (_result.isComplete()) {
							if (_result.booleanValue().booleanValue()) {
								EObject outputElement = findOutputElementInParents(element);
								if (outputElement != null) {
									addResultValue(outputElement);
									if(DEBUG_SSQE) System.out.println("populateResult - before - addResultValue3 ");
									pruneSubtree = shouldPrune(element, depth, true);
								}
							}
						} else
							traverseFeatures(element, _resetStack);
					}
				} else {
					// Class was not in class Predicates index
					if(DEBUG_SSQE) System.out.println("beforeChildren - Class not in class predicates index ("+element.eClass()+")");
				}
				return true;
			}

			protected boolean isRequiredFeature(EStructuralFeature sf) {
				return _requiredPaths.contains(sf);
			}
		};
		try {
			containmentTraverser.traverse();
		} catch (ConcurrentModificationException e) {
			// ignore this exception and return;
		}
	}

	/**
	 * 
	 * @param element
	 * @param depth
	 * @param children
	 *            true when the the request is for element's children , false
	 *            for the remaining element's siblings
	 * @return
	 */
	private boolean shouldPrune(EObject element, int depth, boolean children) {
		return false;
	}

	/**
	 * Searches for a parent of element which is outputElement, up to a root
	 * node
	 * 
	 * @param element
	 * @return
	 */
	protected EObject findOutputElementInParents(EObject element) {
		if (_rootNodes.contains(element))
			return null; // stop on rootNodes elemements
		EObject eContainer = element.eContainer();
		if (eContainer != null) {
			if (isOutputElement(eContainer))
				return eContainer;
			else {
				if (_rootNodes.contains(eContainer))
					return null; // stop on rootNodes elemements
					//Bug 199307 - We have to recurse through each of the containers
				return findOutputElementInParents(eContainer);
			}
		}
		return null;
	}

	protected boolean traverseFeatures(EObject element, List<EObject> resetList) {
		Collection<EObject> fl = _featurePredicatesIndex.get(element.eClass());
		boolean ret = false;
		if (fl != null) {
			for (Iterator<EObject> iter = fl.iterator(); iter.hasNext();) {
				if (ret)
					break;
				EReference sf = (EReference) iter.next();
				if (_requiredPaths.contains(sf))
					continue;
				if (sf.isMany()) {
					// try {
					List<EObject> rl = (List<EObject>) element.eGet(sf);
					for (Iterator<EObject> iterator = rl.iterator(); iterator.hasNext();) {
						if (ret)
							break;
						EObject le = iterator.next();
						if (_classPredicatesIndex.get(le.eClass()) != null) {
							resetList.add(le);
							int pos = resetList.size() - 1;
							ret = processEObject(resetList, le, pos);
							if (!ret && processResetListIsRequired(le)) {
								processResetList(resetList, pos);
							}
						}
					}
					// } catch (Exception e) {
					// // TODO: handle exception
					// if(ModelDebugger.INSTANCE.debug)
					// LoadersUtils.log(e);
					// }
				} else {
					// if (sf instanceof EReference) {
					// try {
					EObject ro = (EObject) element.eGet(sf);
					if (_classPredicatesIndex.get(ro.eClass()) != null) {
						resetList.add(ro);
						int pos = resetList.size() - 1;
						ret = processEObject(resetList, ro, pos);
						if (!ret && processResetListIsRequired(ro)) {
							processResetList(resetList, pos);
						}
					}
					// } catch (Exception e) {
					// // TODO: handle exception
					// if(ModelDebugger.INSTANCE.debug)
					// LoadersUtils.log(e);
					// }
					// }
				}
			}
		}
		return ret;
	}

	private boolean processResetListIsRequired(EObject eObject) {
		if ((_result.partialEvalOnly(eObject.eClass()) & (SSQEEvalResult.IS_COMPLETE | SSQEEvalResult.IS_TRUE | SSQEEvalResult.IS_APPLICABLE)) == (SSQEEvalResult.IS_COMPLETE | SSQEEvalResult.IS_TRUE | SSQEEvalResult.IS_APPLICABLE)) {
			return false;
		}
		return true;
	}

	private boolean processEObject(List<EObject> resetList, EObject eObject, int pos) {
		boolean ret;
		_result.eval(eObject);
		if (_result.isComplete()) {
			ret = addResultValueIfRequired(eObject, pos);
		} else {
			ret = traverseFeatures(eObject, resetList);
			if (_result.isComplete()) {
				ret = addResultValueIfRequired(eObject, pos);
			}
		}
		return ret;
	}

	private boolean addResultValueIfRequired(EObject eObject, int pos) {
		if (_result.booleanValue().booleanValue()) {
			if (isOutputElement(eObject)) {
				addResultValue(eObject);
				return false;
			} else
				return true;
		}
		return false;
	}

	protected void populateRootNodesAndClasses() {
		_rootNodes.clear();
		_rootClasses.clear();
		for (Iterator<String> iter = _query.getSources().iterator(); iter.hasNext();) {
			String element = iter.next();
			URI uri = SaveUtil.createURI(element);
			EObject n = _targetResourceSet.getEObject(uri, false);
			if (n != null) {
				EClass c = n.eClass();
				if (!_rootNodes.contains(n))
					_rootNodes.add(n);
				if (!_rootClasses.contains(c))
					_rootClasses.add(c);
			}
		}
	}

	protected void prepareResult() {
		_queryResult = ExtensionsFactory.eINSTANCE.createQueryResult();
		_queryResult.setQuery(_query);
		for (Iterator<Operand> iter = _query.getOutputElements().iterator(); iter.hasNext();) {
			// support only simple operand
			SimpleOperand outputElement = (SimpleOperand) iter.next();
			EClass c = outputElement.getType();
			if (c == null && outputElement.getFeature() != null) {
				c = outputElement.getFeature().getEContainingClass();
			}
			if (c != null) {
				_outputClasses.add(c);
				QueryUtils.addSubTypes(c, _outputClassesSubTypes);
				QueryUtils.addSuperTypes(c, _outputClassesSuperTypes);
				_classPredicatesIndex.put(c, null);
				QueryUtils.addSubTypes(c, _classPredicatesIndex, null);
			}
			ResultEntry resultEntry = ExtensionsFactory.eINSTANCE.createResultEntry();
			_queryResult.getResultEntries().add(resultEntry);
			_firstResList = new ArrayList<EObject>();
			resultEntry.setValue(_firstResList);
		}
	}

	private boolean processContainmentPath(EClass element) {
		if (_alreadyProcessed.contains(element))
			return _outputClasses.contains(element) || _outputClassesSuperTypes.contains(element);
		_alreadyProcessed.add(element);
		boolean result = false;
		boolean ret = false;
		if (_outputClasses.contains(element) || _outputClassesSuperTypes.contains(element))
			result = true;
		for (Iterator<EReference> iter = element.getEAllContainments().iterator(); iter.hasNext();) {
			EReference ref = iter.next();
			ret = processContainmentPath(ref.getEReferenceType());
			result = result || ret;
			if (ret)
				_requiredPaths.add(ref);
		}
		return result;
	}

	private void sortResult() {
		// support only one order by element
		if (_query.getOrderByExpresions().size() != 1)
			return;
		final EAttribute compareAttribute = (EAttribute) ((SimpleOperand) ((OrderByElement) _query.getOrderByExpresions().get(0)).getOperand()).getFeature();
		final OrderByOperators compareOperator = ((OrderByElement) _query.getOrderByExpresions().get(0)).getOperator();
		final boolean reversOrder = compareOperator.getValue() == OrderByOperators.DESC;
		
		Comparator<EObject> comparator = new Comparator<EObject>() {
			public int compare(EObject o1, EObject o2) {
				try {
					Object o1Value = ((EObject) o1).eGet(compareAttribute);
					Object o2Value = ((EObject) o2).eGet(compareAttribute);

					try {
						if (o2Value.getClass().getName().endsWith(XMLCALENDAR))
							return XMLTypeUtil.compareCalendar(o1Value, o2Value);
						else
							return reversOrder ? -((Comparable<Object>) o1Value).compareTo(o2Value) : ((Comparable<Object>) o1Value).compareTo(o2Value);
					} catch (Exception e) {
						// ignore and compare the toString values
					}
					return reversOrder ? -o1Value.toString().compareTo(o2Value.toString()) : o1Value.toString().compareTo(o2Value.toString());
				} catch (Exception e) {
					// if feature doesn't exist, don't change the order
					return 0;
				}
			}
		};
		ResultEntry tempResult = (ResultEntry) _queryResult.getResultEntries().get(0);
		List<EObject> resultList = (List<EObject>) tempResult.getValue();
		Collections.sort(resultList, comparator);
		tempResult.setValue(resultList);
	}

	private void processResetList(List<EObject> list, int startPosition) {
		_alreadyProcessed.clear();
		for (int i = list.size() - 1; i >= startPosition; i--) {
			EObject resetElement = list.get(i);
			if (_alreadyProcessed.contains(resetElement.eClass()))
				continue;
			_result.reset(resetElement);
			_alreadyProcessed.add(resetElement.eClass());
			list.remove(i);
		}
	}

	/**
	 * 
	 * @return the required paths Set, can be used to add required paths, the
	 *         paths that needs to be followed
	 */
	public Set<EStructuralFeature> getRequiredPaths() {
		return _requiredPaths;
	}

	protected Object getNonModeledElementValue(SSQEEvalResult evalResult, SimpleOperand simpleOperand) {
		return simpleOperand.getValue();
	}

	

	/**
	 * @author Marius Slavescu (slavescu@ca.ibm.com)
	 * @since 3.3
	 * 
	 * I moved SimpleSearchQueryEvaluator from the top of the file, to the bottom, but otherwise
	 * did not alter the contents. - jgw
	 * 
	 */
	public class SimpleSearchQueryEvaluator {
		public SimpleSearchQueryEvaluator() {
		}

		protected WhereExpression getExpression(String string) {
			return (WhereExpression) _namesIndex.get(string);
		}

		protected Operand getOperand(String string) {
			return (Operand) _namesIndex.get(string);
		}

		private SSQEEvalResult internalPrepareEvalResult() {
			
			if (_query.getWhereExpression() == null)
				return new SSQEEvalResult(SimpleSearchQueryEngine.this);
			
			if (_whereExpressionValuesIndex.get(_query.getWhereExpression()) != null)
				return _whereExpressionValuesIndex.get(_query.getWhereExpression());
			
			return parseWhereExpression(_query.getWhereExpression());
		}

		private SSQEEvalResult parseWhereExpression(WhereExpression expression) {
			if (expression instanceof LogicalExpression)
				return parseLogicalExpression((LogicalExpression) expression);
			else
				return parseBinaryExpression((BinaryExpression) expression);
		}

		public SSQEEvalResult prepareEvalResult() {
			return internalPrepareEvalResult();
		}

		
		private SSQEEvalResult parseArithmeticExpression(ArithmeticExpression expression) {
			
			if (_whereExpressionValuesIndex.get(expression) != null)
				return _whereExpressionValuesIndex.get(expression);
			
			if (expression.getName() != null && expression.getName().startsWith(QueryUtils.DOLLAR_SIGN))
				return parseOperand(getOperand(expression.getName().substring(1)));
			
			SSQEEvalResult result = createEvalResult(expression);
			_whereExpressionValuesIndex.put(expression, result);
			for (Iterator<Operand> iter = expression.getArguments().iterator(); iter.hasNext();) {
				Operand element = iter.next();
				result.addPartialResult(parseOperand(element));
			}
			return result;
		}

		private SSQEEvalResult parseBinaryExpression(BinaryExpression expression) {
			
			if (_whereExpressionValuesIndex.get(expression) != null)
				return _whereExpressionValuesIndex.get(expression);
			
			if (expression.getName() != null && expression.getName().startsWith(QueryUtils.DOLLAR_SIGN))
				return parseWhereExpression(getExpression(expression.getName().substring(1)));
			
			SSQEEvalResult result = createEvalResult(expression);
			_whereExpressionValuesIndex.put(expression, result);
			Operand operand = expression.getLeftOperand();
			result.addPartialResult(parseOperand(operand));
			for (Iterator<Operand> iter = expression.getRightOperands().iterator(); iter.hasNext();) {
				Operand element = iter.next();
				result.addPartialResult(parseOperand(element));
			}
			return result;
		}

		private SSQEEvalResult parseLogicalExpression(LogicalExpression expression) {
			
			if (_whereExpressionValuesIndex.get(expression) != null)
				return _whereExpressionValuesIndex.get(expression);
			
			if (expression.getName() != null && expression.getName().startsWith(QueryUtils.DOLLAR_SIGN))
				return parseWhereExpression(getExpression(expression.getName().substring(1)));
			
			SSQEEvalResult result = createEvalResult(expression);
			_whereExpressionValuesIndex.put(expression, result);
			for (Iterator<WhereExpression> iter = expression.getArguments().iterator(); iter.hasNext();) {
				WhereExpression element = iter.next();
				result.addPartialResult(parseWhereExpression(element));
			}
			return result;
		}

		private SSQEEvalResult parseNumericFunction(NumericFunction function) {
			
			if (_whereExpressionValuesIndex.get(function) != null)
				return _whereExpressionValuesIndex.get(function);
			
			if (function.getName() != null && function.getName().startsWith(QueryUtils.DOLLAR_SIGN))
				return parseOperand(getOperand(function.getName().substring(1)));
			
			SSQEEvalResult result = createEvalResult(function);
			_whereExpressionValuesIndex.put(function, result);
			for (Iterator<Operand> iter = function.getArguments().iterator(); iter.hasNext();) {
				Operand element = iter.next();
				result.addPartialResult(parseOperand(element));
			}
			return result;
		}

		private SSQEEvalResult parseOperand(Operand operand) {
			if (operand instanceof ArithmeticExpression) {
				return parseArithmeticExpression((ArithmeticExpression) operand);
			} else if (operand instanceof NumericFunction) {
				return parseNumericFunction((NumericFunction) operand);
			}
			return parseSimpleOperand(operand);
		}

		private SSQEEvalResult parseSimpleOperand(Operand operand) {
			
			if (_whereExpressionValuesIndex.get(operand) != null)
				return _whereExpressionValuesIndex.get(operand);
			
			if (operand.getName() != null && operand.getName().startsWith(QueryUtils.DOLLAR_SIGN))
				return parseSimpleOperand(getOperand(operand.getName().substring(1)));
			
			SSQEEvalResult result = createEvalResult(operand);
			_whereExpressionValuesIndex.put(operand, result);
			return result;
		}

	}

}