/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RecorderControlView.java,v 1.22 2009/12/15 14:54:20 paules Exp $
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.internal.execution.recorder.ui.views;

import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.execution.recorder.IRecorderListenerFullFeedback;
import org.eclipse.hyades.internal.execution.recorder.ui.CopyAction;
import org.eclipse.hyades.internal.execution.recorder.ui.RecorderUIUtility;
import org.eclipse.hyades.internal.execution.recorder.ui.actions.NewGenericRecordingAction;
import org.eclipse.hyades.internal.execution.recorder.ui.actions.StopRecordingAction;
import org.eclipse.hyades.test.core.internal.resources.TestCorePluginResourceBundle;
import org.eclipse.hyades.test.core.testgen.ITestgenListener;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.internal.util.ContextIds;
import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.ToolBarManager;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.FontMetrics;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchPartSite;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.part.ViewPart;

/**
 * <p>Recorder Control View for receiving feedback and controlling recorders during recording.</p>
 * 
 * 
 * @author  Ernest Jessee
 * @author  Paul E. Slauenwhite
 * @author  Paul Klicnik
 * @version December 15, 2009
 * @since   February 1, 2005
 */
public class RecorderControlView extends ViewPart implements ITestgenListener, IRecorderListenerFullFeedback 
{
	/**
	 * kbytes received text box
	 */
	private Text kBytesReceived;

	/** 
	 * Recorder status text box
	 */
	private Text recStatus;

	/**
	 * list for feedback items
	 */
	private List statusList;
	
	/**
	 * Singleton instance of the Recorder Control View.
	 */
	public static RecorderControlView instance = null;
	
	/**
	 * Setup the clipboard   
	 */
	private Clipboard clipboard = null;
	
	/**
	 * Create the action for copying
	 */
	private CopyAction copyAction = null;
	
	/**
	 * the view's ID
	 */
	public static final String ID = "org.eclipse.hyades.execution.recorder.ui.views.RecorderControlView"; //$NON-NLS-1$
	
	/**
	 * toolbar group that goes at the front of the toolbar.
	 */
	public static final String STARTGROUP = "start";	
	/**
	 * @see ViewPart#createPartControl
	 */
	public void createPartControl(final Composite grandParent)  
	{		
		FontMetrics fm = new GC(grandParent).getFontMetrics();
		int heightHint = fm.getHeight() + fm.getAscent() + fm.getDescent();
		int widthHint = fm.getAverageCharWidth() * 50;
		Composite parent = new Composite(grandParent,SWT.NULL);

		//Set the context sensitive help:
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, UiPlugin.getID() + ContextIds.RECORDER_CONTROL_VIEW);
		
		RecorderUIUtility.assignGridLayout(parent,1, false);
		Canvas bytesReceivedCanvas = new Canvas(parent,SWT.NULL);
		bytesReceivedCanvas.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		RowLayout layout = new RowLayout(SWT.HORIZONTAL);
		layout.pack = true;
		layout.wrap = true;
		layout.spacing = 0;
		bytesReceivedCanvas.setLayout(layout);
		
		
		Canvas recvCanvas = new Canvas(bytesReceivedCanvas, SWT.NULL);
		recvCanvas.setLayout(new GridLayout(2,false));
		recvCanvas.setLayoutData(new RowData(widthHint, heightHint));
		Label bytesReceivedLabel = RecorderUIUtility.createLabel(recvCanvas,UiPluginResourceBundle.RecorderControlView_KBYTES_RECEIVED_CONTROL_LABEL,null); 
		bytesReceivedLabel.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));
		kBytesReceived = new Text(recvCanvas,SWT.READ_ONLY|SWT.BORDER);
		kBytesReceived.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		//add status text box
		Canvas statusCanvas = new Canvas(bytesReceivedCanvas,SWT.NULL);
		statusCanvas.setLayout(new GridLayout(2, false));
		statusCanvas.setLayoutData(new RowData(widthHint, heightHint));
		
		Label statusLabel = RecorderUIUtility.createLabel(statusCanvas,UiPluginResourceBundle.RecorderControlView_STATUS_LABEL,null); 
		statusLabel.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));
		recStatus = new Text(statusCanvas,SWT.READ_ONLY|SWT.BORDER);
		recStatus.setText(TestCorePluginResourceBundle.RecorderClient_STATUS_STOPPED);
		recStatus.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		
		statusList = new List(parent,SWT.V_SCROLL|SWT.H_SCROLL|SWT.READ_ONLY|SWT.BORDER|SWT.MULTI);
		statusList.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		clipboard = new Clipboard(grandParent.getDisplay());
		copyAction = new CopyAction( clipboard );
		copyAction.setEnabled( false );
		statusList.addSelectionListener( new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				/* A new selection has been made in the Recorder Control ...
				 * Update the selected lines */
				setSelectedLines();
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});

		/* Add the support for the context menu */
		statusList.addListener( SWT.MenuDetect, new Listener () {
			public void handleEvent (Event event) {
				Menu menu = new Menu( grandParent.getShell(), SWT.POP_UP );
				
				MenuItem copyItem = new MenuItem( menu, SWT.PUSH );
				copyItem.setText( UiPluginResourceBundle.command_Copy );
				copyItem.addListener( SWT.Selection, new Listener () {
					public void handleEvent (Event e) {
						setSelectedLines();
						copyAction.run();
					}
				});
				menu.setLocation( event.x, event.y );
				MenuItem selectAllItem = new MenuItem( menu, SWT.PUSH );
				selectAllItem.setText( UiPluginResourceBundle.command_SelectAll );
				selectAllItem.addListener( SWT.Selection, new Listener() {
					public void handleEvent (Event e) {
						statusList.selectAll();
						setSelectedLines();
						copyAction.setEnabled( true );
					}
				});
				if ( statusList.getSelection().length == 0 ) {
					copyAction.setEnabled( false );
					copyItem.setEnabled( false );
				}
				menu.setVisible( true );
			}
		});
		
		getViewSite().getActionBars().setGlobalActionHandler(ActionFactory.COPY.getId(), copyAction);
		ToolBarManager toolbarManager = (ToolBarManager)getViewSite().getActionBars().getToolBarManager();
		
		//add new items group
		toolbarManager.add(new GroupMarker(STARTGROUP));
		toolbarManager.add(new NewGenericRecordingAction());							
		toolbarManager.add(new StopRecordingAction());							
	}
	
	private void setSelectedLines() {
		
		if ((statusList != null) && (copyAction != null)){
			
			String[] statusLines = statusList.getSelection();

			copyAction.setSelectedStatusItems(statusLines);
			copyAction.setEnabled(statusLines.length > 0);
		}
	}
	
	/**
	 * <p>Resets the contents of the Recorder Control View.</p>
	 * 
	 * <p>Resetting includes:</p>
	 * 
	 * <ul>
	 * <li>Removing all recording events.</li>
	 * <li>Resetting the KBytes Recorded field to <code>0</code>.</li>
	 * </ul>
	 *  
	 * @deprecated As of TPTP V.4.6.2, use {@link #controlMessage(String, String)} with {@link IRecorderListenerFullFeedback#KB_RECEIVED} and/or {@link #getStatusList()} with {@link List#removeAll()}.
	 */
	public void reset()
	{
		statusList.removeAll();
		kBytesReceived.setText("0"); //$NON-NLS-1$
	}
	
	/**
	 * Resolves the singleton instance of the Recorder Control View.
	 * <p/>
	 * When the singleton instance is instantiated, the Recorder Control View
	 * is opened (if closed) and given focus in the active page of the active 
	 * workbench window.
	 * <p/> 
	 * The Recorder Control View is given focus on subsequent calls after the 
	 * singleton instance has been instantiated.
	 * <p/> 
	 * 
	 * @return The singleton instance of the Recorder Control View.
	 */
	public static RecorderControlView getInstance(){

		if(instance == null){
			
			Display.getDefault().syncExec(new Runnable() {

				public void run() {
					
					try{
						instance = ((RecorderControlView)(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().showView(ID)));
					}
					catch(Exception e){

						UiPlugin.logError(e);

						ErrorDialog.openError(Display.getDefault().getActiveShell(),
								UiPluginResourceBundle.W_ERROR,
								UiPluginResourceBundle.RecorderControlView_UNABLE_TO_CREATE_RECORDER_CONTROL_VIEW_ERROR_MSG, 
								new Status(Status.WARNING, UiPlugin.PLUGIN_ID, IResourceStatus.INTERNAL_ERROR, e.toString(), null));
					}
				}
			});
		}

		return instance;		
	}

	/**
	 * asynchronously adds a message to the status box
	 * @param String message
	 * void
	 */
	public void addMessage(final String message)
	{
		addMessage(message, false);
	}
	
	/**
	 * adds a message to the status box
	 * @param String message
	 * @param boolean bSync - whether this method is synchronous
	 * void
	 */
	public void addMessage(final String message, boolean bSync)
	{
		Runnable a = new Runnable(){

			public void run(){

				if(!statusList.isDisposed()){					

					//Attempt to activate the singleton instance of the Recorder Control View:					
					if(instance != null){

						IWorkbenchPartSite workbenchPartSite = instance.getSite();

						if(workbenchPartSite != null){
							workbenchPartSite.getPage().activate(instance);
						}
					}

					statusList.add(message);
					statusList.setSelection(statusList.getItems().length-1);
				}
			}
		};
 		if (bSync)
			Display.getDefault().syncExec(a);
		else
			Display.getDefault().asyncExec(a);
	}
	
	
	/**
	 * sets the number in the kByes received control
	 * @param int receivedKBytes
	 */
	public void setKBytesReceived(final int receivedKBytes)
	{
		Display.getDefault().asyncExec(new Runnable(){

			public void run(){

				if(!kBytesReceived.isDisposed()){
					kBytesReceived.setText(String.valueOf(receivedKBytes));
				}
			}
		});
	}
	
	/**
	 * Sets text of the status text field
	 * @param status
	 */
	public void setStatus(String status)
	{
		recStatus.setText(status);
	}
	/**
	 * @see org.eclipse.ui.IWorkbenchPart#setFocus()
	 */
	public void setFocus(){
		
		//Set the focus of the status list:
		if(statusList != null){
			statusList.setFocus();
		}
		
		//Set the selected status item(s) and enablement on the copy action:
		setSelectedLines();
	}

	/**
	 * @see org.eclipse.ui.IWorkbenchPart#dispose()
	 */
	public void dispose()
	{
		super.dispose();
		instance=null;
	}
	/**
	 * @return
	 */
	public List getStatusList() {
		return statusList;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.testgen.ui.ITestgenListener#notifyEnd(boolean)
	 */
	public void notifyEnd(boolean bSuccess) {
		if (bSuccess)
			addMessage(TestCorePluginResourceBundle.RecorderClient_TEST_GENERATION_COMPLETE); 
		else
			addMessage(TestCorePluginResourceBundle.RecorderClient_TEST_GENERATION_ERRORS); 
	
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.internal.execution.recorder.IRecorderListener#updateRecorderActive(boolean)
	 */
	public void updateRecorderActive(boolean active) {	
	}
	
	/* (non-Javadoc)
     * @see org.eclipse.hyades.internal.execution.recorder.IRecorderListenerFullFeedback#controlMessage(java.lang.String, java.lang.String)
     */
    public void controlMessage(String key, String msg) {
		if (key.equals(IRecorderListenerFullFeedback.KB_RECEIVED))
        {
		    try {
		        setKBytesReceived(Integer.parseInt(msg));
		    }
		    catch (NumberFormatException e) {} //bad data, don't update
        }
		else if (key.equals(IRecorderListenerFullFeedback.VIEW_MESSAGE))
		{
		    addMessage(msg);
		}
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.internal.execution.recorder.IRecorderListener#updateStatus(java.lang.String)
     */
    public void updateStatus(final String str) {
        
    	Display.getDefault().asyncExec(new Runnable() {

    		public void run() {

    			if(!recStatus.isDisposed()){
    				recStatus.setText(str);
    			}
    		}
    	});
    }    
}
