/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: UiPlugin.java,v 1.50 2010/03/03 21:19:40 paules Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui;

import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.equinox.events.MemoryEventConstants;
import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.hyades.test.core.TestCorePreferences;
import org.eclipse.hyades.test.ui.adapter.IResourceSetFileAdaptable;
import org.eclipse.hyades.test.ui.adapter.TestAdapterFactory;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.FileProxyNodeCacheMemoryEventHandler;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.IExtendedProxyNode;
import org.eclipse.hyades.test.ui.navigator.ITypedElementProxyNode;
import org.eclipse.hyades.ui.internal.logicalfolder.LogicalFolder;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialogWithToggle;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.forms.FormColors;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.service.event.EventConstants;
import org.osgi.service.event.EventHandler;

/**
 * <p>Test UI plug-in class.</p>
 * 
 * <p><b>Note:</b> This class should not be used outside 
 * the context of Eclipse.</p>
 * 
 * 
 * @author      Paul E. Slauenwhite
 * @version     March 3, 2010
 * @since       January 27, 2005
 */
public class UiPlugin extends AbstractUIPlugin {
	
	public static final String PLUGIN_ID = "org.eclipse.hyades.test.ui"; //$NON-NLS-1$
	private static UiPlugin instance;
	// resource bundle of plugin.properties
	private ResourceBundle resourceBundle;
	private FormColors formColors;
	
	//added for bugzilla #95329 - george.sebastian@ca.com
	private Map editorMappings;
	private static final String HYADES_EDITOR_EXTENSION_ID = "hyadesEditors"; //$NON-NLS-1$
	private static final String HYADES_EDITOR_ID_PREFIX = "org.eclipse.hyades."; //$NON-NLS-1$
    
	/**
	 * Extension point id used to register the test case creation wizards.
	 */
	public static final String EP_TEST_CASE_WIZARD = "newTestCaseWizards"; //$NON-NLS-1$
	
	/**
	 * Extension point id used to register the generate wizards.
	 */
	public static final String EP_GENERATE_WIZARDS = "generateWizards"; //$NON-NLS-1$

	/**
	 * @deprecated As of TPTP 4.5.0, this preference key is no longer supported.  
	 */
	public static final String TEST_GENERATOR = "default_test_generator"; //$NON-NLS-1$
	
	/**
	 * The constructor.
	 */
	/**
	 * 
	 */
	public UiPlugin() {
		instance = this;
		//added for bugzilla #95329 - george.sebastian@ca.com
		setEditorMappings();
	}

	/**
	 * Returns the shared instance.
	 */
	public static UiPlugin getInstance() {
		return instance;
	}
	

	/**
	 * Returns this plugin's id.
	 * @return String
	 */
	public static String getID()
	{
		return PLUGIN_ID;
	}

	/**
     * Resolves the plugin's resource bundle.
     * <p>
     * If the plugin's resource bundle can not be resolved, <code>null</code> is returned.
     * <p>
     * IMPORTANT: Do not use this method to retrieve values from the resource bundle.  This method 
     * is provided so this plugin's resource bundle can be used as the parent of another plugin's 
     * resource bundle.
     * <p>
     * 
     * @return The plugin's resource bundle, otherwise <code>null</code>.
     * @deprecated As of TPTP V4.4, use {@link org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle} for resolving resources.  The plugin.properties property file ONLY contains messages for the this plug-in's MANIFEST.MF and plugin.xml files.
     */
    public ResourceBundle getResourceBundle() {
        try {
            if (resourceBundle == null)
                resourceBundle   = Platform.getResourceBundle(Platform.getBundle(PLUGIN_ID));
        } catch (MissingResourceException x) {
            resourceBundle = null;
        }
        return resourceBundle;
    }
	
    /**
     * Resolves the plugin's resource bundles.
     * <p>
     * If the plugin's resource bundles can not be resolved, <code>null</code> is returned.
     * <p>
     * IMPORTANT: Do not use this method to retrieve values from the resource bundles.  This method 
     * is provided so this plugin's resource bundles can be used as the parent of another plugin's 
     * resource bundles.
     * <p>
     * 
     * @return The plugin's resource bundles, otherwise <code>null</code>.
     * @deprecated As of TPTP V4.4, use {@link org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle} for resolving resources.  The plugin.properties property file ONLY contains messages for the this plug-in's MANIFEST.MF and plugin.xml files.
     */
    public ResourceBundle[] getResourceBundles()
	{
		
		if(getResourceBundle() != null)
		{
			return (new ResourceBundle[]{getResourceBundle()});
		}
		
		return null;
	}

    /**
     * Resolves the localized message associated with the parameter <code>key</code>
     * based on the default locale from the plugin's resource bundle.
     * <p>
     * If the <code>key</code> does not exist in the plugin's resource bundle, the 
     * <code>key</code> is returned.
     * <p>
     * Noteworthy, the resultant message is not formatted (e.g. no message parameter substitution). 
     * <p>
     * 
     * @param key The <code>key</code> of the message in the plugin's resource bundle.
     * @return The localized message associated with the parameter <code>key</code> from the plugin's resource bundle, otherwise the <code>key</code>.
     * @deprecated As of TPTP V4.4, use {@link org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle} for resolving resources.  The plugin.properties property file ONLY contains messages for the this plug-in's MANIFEST.MF and plugin.xml files.
     */
    public static String getResourceString(String key) {
        ResourceBundle bundle = UiPlugin.getDefault().getResourceBundle();
        try {
            return (bundle != null) ? bundle.getString(key) : key;
        } catch (MissingResourceException e) {
            return key;
        }
    }

    /**
     * Resolves the localized message associated with the parameter <code>key</code>
     * based on the default locale from the plugin's resource bundle.
     * <p>
     * If the <code>key</code> does not exist in the plugin's resource bundle, the 
     * <code>key</code> is returned.
     * <p>
     * Noteworthy, the resultant message is not formatted (e.g. no message parameter substitution). 
     * <p>
     * 
     * @param key The <code>key</code> of the message in the plugin's resource bundle.
     * @return The localized message associated with the parameter <code>key</code> from the plugin's resource bundle, otherwise the <code>key</code>.
     * @deprecated As of TPTP V4.4, use {@link org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle} for resolving resources.  The plugin.properties property file ONLY contains messages for the this plug-in's MANIFEST.MF and plugin.xml files.
     */
    public static String getString(String key)
    {
        return getResourceString(key);
    }

    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * based on the default locale from the plugin's resource bundle.
     * <p>
     * Message formatting is done by the {@link org.eclipse.osgi.util.NLS} class using the 
     * default locale. 
     * <p>
     * If the <code>key</code> does not exist in the plugin's resource bundle, the 
     * <code>key</code> is returned.
     * <p>
     * 
     * @param key The <code>key</code> of the message in the plugin's resource bundle.
     * @param argument The first message parameter for formatting in the localized and formatted message.
     * @return The localized message associated with the parameter <code>key</code> from the plugin's resource bundle, otherwise the <code>key</code>.
     * @deprecated As of TPTP V4.4, use {@link org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle} for resolving resources.  The plugin.properties property file ONLY contains messages for the this plug-in's MANIFEST.MF and plugin.xml files.
     */
    public static String getString(String key, String argument)
    {
        return getString(key, new String[] {argument});
    }   

    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * based on the default locale from the plugin's resource bundle.
     * <p>
     * Message formatting is done by the {@link org.eclipse.osgi.util.NLS} class using the 
     * default locale. 
     * <p>
     * If the <code>key</code> does not exist in the plugin's resource bundle, the 
     * <code>key</code> is returned.
     * <p>
     * 
     * @param key The <code>key</code> of the message in the plugin's resource bundle.
     * @param arguments The array of message parameters for formatting in the localized and formatted message.
     * @return The localized message associated with the parameter <code>key</code> from the plugin's resource bundle, otherwise the <code>key</code>.
     * @deprecated As of TPTP V4.4, use {@link org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle} for resolving resources.  The plugin.properties property file ONLY contains messages for the this plug-in's MANIFEST.MF and plugin.xml files.
     */
    public static String getString(String key, String[] arguments)
    {
        return NLS.bind(getString(key), ((Object[])(arguments)));
    }
    
	public FormColors getFormColors(Display display)
	{
		if(formColors == null)
		{
			formColors = new FormColors(display);
			formColors.markShared();
		}
		return formColors;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.plugin.AbstractUIPlugin#start(org.osgi.framework.BundleContext)
	 */
	public void start(BundleContext context) throws Exception {

		//Force the org.eclipse.hyades.test.core plug-in to be loaded thereby causing the org.eclipse.hyades.test.core.TestCorePlugin.start(BundleContext) 
		//method to be invoked to register the resource factories:
		TestCorePlugin.getDefault();
		
		Platform.getAdapterManager().registerAdapters(TestAdapterFactory.INSTANCE, IResourceSetFileAdaptable.class);
		Platform.getAdapterManager().registerAdapters(TestAdapterFactory.INSTANCE, IResource.class);
		Platform.getAdapterManager().registerAdapters(TestAdapterFactory.INSTANCE, LogicalFolder.class);
		Platform.getAdapterManager().registerAdapters(TestAdapterFactory.INSTANCE, CMNNamedElement.class);
		Platform.getAdapterManager().registerAdapters(TestAdapterFactory.INSTANCE, TPFExecutionEvent.class);
		Platform.getAdapterManager().registerAdapters(TestAdapterFactory.INSTANCE, ITypedElementProxyNode.class);
		Platform.getAdapterManager().registerAdapters(TestAdapterFactory.INSTANCE, Resource.class);
		
		Platform.getAdapterManager().registerAdapters(TestAdapterFactory.INSTANCE, IExtendedProxyNode.class);

		//Register memory event handler(s):
		Hashtable<String, String> memoryEventHandlerProperties = new Hashtable<String, String>();
		memoryEventHandlerProperties.put(EventConstants.EVENT_TOPIC, MemoryEventConstants.TOPIC_CRITICAL);

		context.registerService(EventHandler.class.getName(), new FileProxyNodeCacheMemoryEventHandler(), memoryEventHandlerProperties);
		
		super.start(context);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.plugin.AbstractUIPlugin#stop(org.osgi.framework.BundleContext)
	 */
	public void stop(BundleContext context) throws Exception {

		TestUIExtension.dispose();
		TestUI.dispose();

		savePluginPreferences();
		Platform.getAdapterManager().unregisterAdapters(TestAdapterFactory.INSTANCE);
		TestAdapterFactory.INSTANCE.dispose();
		
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().remove(ICommonConstants.TEST_SUITE_FILE_EXTENSION);
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().remove(ICommonConstants.DEPLOYMENT_FILE_EXTENSION);
		/**
		 * @author psun
		 */
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().remove(ICommonConstants.DATAPOOL_FILE_EXTENSION);
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().remove(ICommonConstants.LOCATION_FILE_EXTENSION);
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().remove(ICommonConstants.ARTIFACT_FILE_EXTENSION);
		
		instance = null;		
		super.stop(context);
	}
	
	/**
	 * Logs an error described by a throwable.
	 * 
	 * <p>This method should be used whenever a class in this instance
	 * has to log an error since it adheres to the global logging
	 * strategy.
	 * 
	 * @param throwable
	 */
	public static void logError(Throwable throwable)
	{
		logError(throwable.toString(), throwable);
	}
	
	/**
	 * Logs an error described by a text.
	 * 
	 * <p>This method should be whenever a class in this instance
	 * has to log an error since it adheres to the global logging
	 * strategy.
	 * 
	 * @param text
	 */
	public static void logError(String text)
	{
		logError(new Throwable(text));
	}
	
	/**
	 * Logs an error described by a a text and a throwable.
	 * 
	 * <p>This method should be used whenever a class in this instance
	 * has to log an error since it adheres to the global logging
	 * strategy.
	 * @param text
	 * @param throwable
	 */
	public static void logError(String text, Throwable throwable) {
		Status status = new Status(IStatus.ERROR, getID(), 0, text, throwable);
		instance.getLog().log(status);
	}
	
	/**
	 * Logs an information described by a text.
	 * 
	 * <p>This method should be whenever a class in this instance
	 * has to log an information since it adheres to the global logging
	 * strategy.
	 * 
	 * @param text
	 */
	public static void logInfo(String text)
	{
		Status status = new Status(IStatus.INFO, getID(), 0, text, new Throwable(text));
		instance.getLog().log(status);
	}
	
	public static void logWarning(String text)
	{
		Status status = new Status(IStatus.WARNING, getID(), 0, text, new Throwable(text));
		instance.getLog().log(status);
	}
	
	public static void logWarning(String text, Throwable throwable)
	{
		Status status = new Status(IStatus.WARNING, getID(), 0, text, throwable);
		instance.getLog().log(status);
	}
	
	// TestGen info
	public static String getPreference(String key)
	{
		return getDefault().getPreferenceStore().getString(key);
	}	
	
	/**
	 * @see org.eclipse.ui.plugin.AbstractUIPlugin#initializeDefaultPreferences(org.eclipse.jface.preference.IPreferenceStore)
	 */
	protected void initializeDefaultPreferences(IPreferenceStore store)
	{		
		store.setDefault(TestUI.OPEN_EDITOR, MessageDialogWithToggle.PROMPT);
		store.setDefault(TestUI.EXECUTION_FILTER, 3);
        store.setDefault(TestUI.HIDE_LAUNCH_VALIDATION_WARNING, false);

		store.setDefault(TestCorePreferences.SAVE_INTERVAL, 5000);			
		store.setDefault(TestCorePreferences.DEFAULT_PORT, "10002"); //$NON-NLS-1$
		store.setDefault(TestCorePreferences.DEFAULT_HOST_NAME, "localhost"); //$NON-NLS-1$

		
		//bugzilla_95064 - copy ui preferences to core
		TestCorePlugin.getDefault().getPluginPreferences().setValue(TestCorePreferences.DEFAULT_HOST_NAME,
				store.getString(TestCorePreferences.DEFAULT_HOST_NAME));
		TestCorePlugin.getDefault().getPluginPreferences().setValue(TestCorePreferences.DEFAULT_PORT,
				store.getString(TestCorePreferences.DEFAULT_PORT));
		TestCorePlugin.getDefault().getPluginPreferences().setValue(TestCorePreferences.SAVE_INTERVAL,
				store.getInt(TestCorePreferences.SAVE_INTERVAL));
	}

	/**
	 * This method always returns <code>null</code> for API compatability.
	 * <p/>
	 * 
	 * @return Always returns <code>null</code>.
	 * @deprecated As of TPTP 4.5.0, internalized in the <code>org.eclipse.tptp.test.recorders.url</code> plug-in.
	 */
	public String getDefaultGeneratorID(){

		//Return null since only for API compatability.
		return null;
	}

	/**
	 * Returns the workspace instance.
	 */
	public static IWorkspace getWorkspace() {
		return ResourcesPlugin.getWorkspace();
	}
	public static UiPlugin getDefault() {
		return instance;
	}

	/**Method to obtain extensions handled by different type of editors.
	 * added for bugzilla #95329 - george.sebastian@ca.com
	 */
	public  Map getEditorMappings() {
		
		return editorMappings;
	}

	/**To retrieve the extensions handled by different editors
	 * Add different editors extensions with ids and names to the map as key-value pair of
	 * extension handled by the editor as key  and editor group id as value.     
	 * added for bugzilla #95329 - george.sebastian@ca.com
	 */
	protected void setEditorMappings() {
		editorMappings = new HashMap();
		IExtension editorExtension = Platform.getExtensionRegistry().getExtension(PLUGIN_ID + "." + HYADES_EDITOR_EXTENSION_ID); //$NON-NLS-1$
		IConfigurationElement [] configurationElements = editorExtension.getConfigurationElements();
		
		for (int i = 0; i < configurationElements.length; i++) {
			editorMappings.put(configurationElements[i].getAttribute("extensions"),HYADES_EDITOR_ID_PREFIX); //$NON-NLS-1$
		}
		
	}
	public String getRAServerPort()
	{
		return TestCorePlugin.getDefault().getPluginPreferences().getString(TestCorePreferences.DEFAULT_PORT);
	}
	
	/**
	 * returns the host specified for the Agent Controller as used by Hyades Test
	 * The value returend by this method is specified using Window->Preferences->Test
	 * @return String
	 */
	public String getRAServerHost()
	{
		return TestCorePlugin.getDefault().getPluginPreferences().getString(TestCorePreferences.DEFAULT_HOST_NAME);
		
	}
	
	/**
	 * This method causes a dialog box to be displayed warning the user of an exception.
	 * @param Exception e1 - the exception thrown 
	 * @param consequenceDescription - a description of the consequence (i.e. Recording session aborted)
	 * @param cause - a description of the cause. (i.e. Another recorder is active or Agent Controller not available on host/port)
	 * @param dlgTitle - the Title to be displayed on the error dialog
	 */
	public static void reportExceptionToUser(final Exception e1,
										     final String consequenceDescription,
										     final String cause,
										     final String dlgTitle)
	{
		Display.getDefault().asyncExec
		(
			new Runnable()
			{
				public void run()
				{
					Status errorStatus = new Status(IStatus.ERROR,UiPlugin.PLUGIN_ID,IStatus.ERROR,cause,e1);
					new ErrorDialog(Display.getDefault().getActiveShell(),dlgTitle,consequenceDescription,errorStatus,IStatus.ERROR).open();
				
	
				}
	
			}
		);
	}

    /**
     * Returns the image associated to the given key from the local image registry.
     * @param imageKey the key of the image
     * @return the image associated to the given key or <code>null</code> if unable to retrieve the image file.
     */
    public Image getImage(String imageKey) {
        return getImageRegistry().get(imageKey);
    }
    
    /**
     * Adds an image located in a specified bundle in the image registery
     * @param bundle the bundle in which the image file is located.
     * @param imageKey the image key
     */
    public void putImage(Bundle bundle, String imageKey) {
        if(getImageRegistry().get(imageKey) == null) {
            ImageDescriptor imageDescriptor = UIUtil.getImageDescriptorFromPlugin(bundle, imageKey);
            if(imageDescriptor != null) {
                UiPlugin.getInstance().getImageRegistry().put(imageKey, imageDescriptor);
            }
        }
    }
	
}
