/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: NamedElementPropertySource.java,v 1.5 2007/03/22 14:42:08 paules Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.adapter;

import org.eclipse.hyades.edit.datapool.INamedElement;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.ListenerList;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.ui.views.properties.IPropertyDescriptor;
import org.eclipse.ui.views.properties.IPropertySource2;
import org.eclipse.ui.views.properties.PropertyDescriptor;
import org.eclipse.ui.views.properties.TextPropertyDescriptor;

/**
 * This class is a {@link org.eclipse.ui.views.properties.IPropertySource2 IPropertySource2}
 * implemented for {@link org.eclipse.hyades.edit.datapool.INamedElement INamedElement}.
 * 
 * @author bjiang
 * @since  3.3
 */
public class NamedElementPropertySource implements IPropertySource2 {

	public static final String PROPERTY_SOURCE_NAME = "datapool.propertySource.name"; //$NON-NLS-1$
	public static final String PROPERTY_SOURCE_DESCRIPTION = "datapool.propertySource.description"; //$NON-NLS-1$
	public static final String PROPERTY_SOURCE_ID = "datapool.propertySource.id"; //$NON-NLS-1$
	
	private INamedElement namedElement;
	private IPropertyDescriptor[] descriptors;
	
	private ListenerList propertyChangeListeners = new ListenerList();
	
	/**
	 * 
	 */
	public NamedElementPropertySource(INamedElement namedElement) {
		if(namedElement == null)
			throw new IllegalArgumentException();
		
		this.namedElement = namedElement;
	}
	
	public INamedElement getNamedElement()
	{
		return this.namedElement;
	}
	
	public void addPropertyChangeListener(IPropertyChangeListener listener)
	{
		if(listener != null)
			propertyChangeListeners.add(listener);
	}
	
	public void removePropertyChangeListener(IPropertyChangeListener listener)
	{
		if(listener != null)
			propertyChangeListeners.remove(listener);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.views.properties.IPropertySource2#isPropertyResettable(java.lang.Object)
	 */
	public boolean isPropertyResettable(Object id) {
		if(id != null && id.equals(PROPERTY_SOURCE_ID))
			return false;
		return true;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.views.properties.IPropertySource#getEditableValue()
	 */
	public Object getEditableValue() {
		return namedElement.getName();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.views.properties.IPropertySource#getPropertyDescriptors()
	 */
	public IPropertyDescriptor[] getPropertyDescriptors() {		
		if(descriptors == null)
		{
			 descriptors = new IPropertyDescriptor[3];
			 descriptors[0] = new TextPropertyDescriptor(PROPERTY_SOURCE_NAME, UiPluginResourceBundle.Word_Name); 
			 ((PropertyDescriptor)descriptors[0]).setCategory(UiPluginResourceBundle.Word_Attributes); 
			 descriptors[1] = new TextPropertyDescriptor(PROPERTY_SOURCE_DESCRIPTION, UiPluginResourceBundle.Word_Description); 
			 ((PropertyDescriptor)descriptors[1]).setCategory(UiPluginResourceBundle.Word_Attributes); 
			 descriptors[2] = new PropertyDescriptor(PROPERTY_SOURCE_ID, UiPluginResourceBundle.Word_ID); 
			 ((PropertyDescriptor)descriptors[2]).setCategory(UiPluginResourceBundle.Word_Attributes); 
		}
		return this.descriptors;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.views.properties.IPropertySource#getPropertyValue(java.lang.Object)
	 */
	public Object getPropertyValue(Object id) {
		if(id != null && id.equals(PROPERTY_SOURCE_NAME))
			return namedElement.getName();
		else if(id != null && id.equals(PROPERTY_SOURCE_DESCRIPTION))
			return namedElement.getDescription() == null ? "" : namedElement.getDescription(); //$NON-NLS-1$
		else if(id != null && id.equals(PROPERTY_SOURCE_ID))
			return namedElement.getId();
		else
			return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.views.properties.IPropertySource#isPropertySet(java.lang.Object)
	 */
	public boolean isPropertySet(Object id) {
		if(id != null && id.equals(PROPERTY_SOURCE_NAME))
			return true;
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.views.properties.IPropertySource#resetPropertyValue(java.lang.Object)
	 */
	public void resetPropertyValue(Object id) {
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.views.properties.IPropertySource#setPropertyValue(java.lang.Object, java.lang.Object)
	 */
	public void setPropertyValue(Object id, Object value) {
		Object oldValue;
		if(id != null && id.equals(PROPERTY_SOURCE_NAME))
		{
			oldValue = getPropertyValue(id);
			namedElement.setName(value.toString());
			firePropertyChangeEvent(id.toString(), oldValue, value);
		}
		else if(id != null && id.equals(PROPERTY_SOURCE_DESCRIPTION))
		{
			oldValue = getPropertyValue(id);
			namedElement.setDescription(value.toString());
			firePropertyChangeEvent(id.toString(), oldValue, value);
		}
		else if(id != null && id.equals(PROPERTY_SOURCE_ID))
		{
			oldValue = getPropertyValue(id);
			namedElement.setId(value.toString());
			firePropertyChangeEvent(id.toString(), oldValue, value);
		}
	}

	protected void firePropertyChangeEvent(String propertyName, Object oldValue, Object newValue)
	{
		PropertyChangeEvent event = new PropertyChangeEvent(this, propertyName, oldValue, newValue);
		Object[] listeners = propertyChangeListeners.getListeners();
		for(int i = 0; i < listeners.length; i++)
		{
			((IPropertyChangeListener)listeners[i]).propertyChange(event);
		}
	}
}
