/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestAdapterFactory.java,v 1.9 2009/04/08 10:15:53 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.adapter;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IAdapterFactory;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.TestUI;
import org.eclipse.hyades.test.ui.internal.PropertySourceExtensionManager;
import org.eclipse.hyades.test.ui.internal.model.CMNNamedElementUtil;
import org.eclipse.hyades.test.ui.internal.model.ResourceUtil;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.IExtendedProxyNode;
import org.eclipse.hyades.test.ui.navigator.CMNNamedElementProxyNode;
import org.eclipse.hyades.test.ui.navigator.FileProxyNode;
import org.eclipse.hyades.test.ui.navigator.ITestCaseProxyNode;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer;
import org.eclipse.hyades.ui.adapter.IHyadesWorkbenchAdapter;
import org.eclipse.hyades.ui.internal.logicalfolder.LogicalFolder;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.hyades.ui.util.INamedElement;
import org.eclipse.ui.IPersistableElement;
import org.eclipse.ui.model.IWorkbenchAdapter;
import org.eclipse.ui.views.properties.IPropertySource;

/**
 * <p>Test adapter factory.</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul E. Slauenwhite
 * @author  Jerome Bozier
 * @version April 7, 2009
 * @since   February 1, 2005
 */
public class TestAdapterFactory 
implements IAdapterFactory, IDisposable
{
	/**
	 * Singleton for TestAdapterFactory
	 */
	public static final TestAdapterFactory INSTANCE = new TestAdapterFactory();
	
	private TestWorkbenchAdapter testWorkbenchAdapter;
	
	/**
	 * Constructor for TestAdapterFactory
	 */
	protected TestAdapterFactory()
	{
		testWorkbenchAdapter = new TestWorkbenchAdapter();
	}
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		testWorkbenchAdapter = null;
	}

	/**
	 * Returns the <code>IHyadesWorkbenchAdapter</code> used by this factory 
	 * @return IHyadesWorkbenchAdapter
	 */
	public IHyadesWorkbenchAdapter getHyadesWorkbenchAdapter()
	{
		return testWorkbenchAdapter;
	}

	/**
	 * @see org.eclipse.core.runtime.IAdapterFactory#getAdapterList()
	 */
	public Class[] getAdapterList()
	{
		return new Class[] {
			CMNNamedElement.class,
			IWorkbenchAdapter.class,
			TestWorkbenchAdapter.class,
			IPropertySource.class,
			IPersistableElement.class,
			INamedElement.class,
			IResource.class,
			IFile.class,
			Resource.class,
			IProxyNodeRenamer.class,
		};
	}

	/**
	 * @see org.eclipse.core.runtime.IAdapterFactory#getAdapter(java.lang.Object, java.lang.Class)
	 */
	public Object getAdapter(Object adaptableObject, Class adapterType)
	{
		if(adapterType == CMNNamedElement.class)
			return getCMNNamedElement(adaptableObject);

		if((adapterType == TestWorkbenchAdapter.class) || (adapterType == IWorkbenchAdapter.class))
			return getTestWorkbenchAdapter(adaptableObject);
			
		if(adapterType == IPropertySource.class)
			return getPropertySource(adaptableObject);

		if(adapterType == INamedElement.class)
			return getNamedElement(adaptableObject);

		if(adapterType == IPersistableElement.class)
			return getPersistableElement(adaptableObject);

		if((adapterType == IFile.class) || (adapterType == IResource.class)) {
			try {
				IResource res = getResource(adaptableObject);
				if (res == null) {
					// no resource => return null
					return null;
				}
				if (adapterType.isAssignableFrom(res.getClass())) return res;
				return null;
			} catch (Exception e) {
				return null;
			}
		}
		if(adapterType == IProxyNodeRenamer.class) {
			getProxyNodeRenamer(adaptableObject);
		}
		if(adapterType == Resource.class) {
			getEMFResource(adaptableObject);
		}
		return null;
	}

	
	/**
	 * Returns the <code>CMNNamedElement</code> associated with a given 
	 * adaptable object.
	 * @param adaptableObject
	 * @return CMNNamedElement
	 */
	protected CMNNamedElement getCMNNamedElement(Object adaptableObject)
	{
		ResourceSet resourceSet = null;
		IFile file = null;
		
		if(adaptableObject instanceof IResourceSetFileAdaptable)
		{
			IResourceSetFileAdaptable adaptable = (IResourceSetFileAdaptable)adaptableObject;
			resourceSet = adaptable.getResourceSet();
			file = adaptable.getFile();
		}
		else if(adaptableObject instanceof IFile)
		{
			file = (IFile)adaptableObject;
		}
		
		if(file != null)
		{
			if(TestUI.isSupportedFileExtension(file.getFileExtension()))
			{
				EObject[] eObjects = EMFUtil.load(resourceSet, file);
				if((eObjects.length == 1) && (eObjects[0] instanceof CMNNamedElement))
					return (CMNNamedElement)eObjects[0];
			}
		}
			
		return null;
	}
	
	/**
	 * Returns the Hyades workbench adapter associated with a given 
	 * adaptable object.
	 * @param adaptableObject
	 * @return IHyadesWorkbenchAdapter
	 */
	protected IHyadesWorkbenchAdapter getTestWorkbenchAdapter(Object adaptableObject)
	{
		if(adaptableObject instanceof CMNNamedElement)
			return getHyadesWorkbenchAdapter();
			
		if(adaptableObject instanceof TPFExecutionEvent)
			return getHyadesWorkbenchAdapter();

		if(adaptableObject instanceof IResource)
			return getHyadesWorkbenchAdapter();

		if(adaptableObject instanceof LogicalFolder)
			return getHyadesWorkbenchAdapter();

		if(adaptableObject instanceof Resource)
			return getHyadesWorkbenchAdapter();

		return null;
	}
	
	/**
	 * Returns the property source associated with a given adaptable object.
	 * @param adaptableObject
	 * @return IPropertySource
	 */
	protected IPropertySource getPropertySource(Object adaptableObject){
		
		if(PropertySourceExtensionManager.getInstance().isExtendableObject(adaptableObject)) {
			
			IPropertySource propertySource = PropertySourceExtensionManager.getInstance().getPropertySource(adaptableObject);
			
			if(propertySource != null) {
				return propertySource;
			}
		}
		
		if(adaptableObject instanceof CMNNamedElementProxyNode){
			return (new CMNNamedElementPropertySource(((CMNNamedElementProxyNode)(adaptableObject))));			
		}
		else 
			if(adaptableObject instanceof CMNNamedElement){
			return (new CMNNamedElementPropertySource(((CMNNamedElement)(adaptableObject))));
		}
		else if(adaptableObject instanceof Resource){
			return ((IPropertySource)(getResource(adaptableObject).getAdapter(IPropertySource.class)));
		}
			
		return null;
	}

	/**
	 * Returns the persisted element associated with a given adaptable object.
	 * @param adaptableObject
	 * @return IPersistableElement
	 */
	protected IPersistableElement getPersistableElement(Object adaptableObject)
	{
		if(adaptableObject instanceof CMNNamedElement)
			return new CMNNamedElementUtil.PersistableElement((CMNNamedElement)adaptableObject);

		if(adaptableObject instanceof Resource)
			return new ResourceUtil.PersistableElement((Resource)adaptableObject);

//		if(adaptableObject instanceof IAdaptable)
//			return (IPersistableElement)((IAdaptable)adaptableObject).getAdapter(IPersistableElement.class);

		return null;
	}

	/**
	 * Returns the named element associated with a given adaptable object.
	 * @param adaptableObject
	 * @return IPropertySource
	 */
	protected INamedElement getNamedElement(Object adaptableObject)
	{
		if(adaptableObject instanceof CMNNamedElement)
			return new CMNNamedElementUtil.UINamedElement((CMNNamedElement)adaptableObject);
		
		return null;
	}

	/**
	 * Returns the file associated with a given adaptable object.
	 * @param adaptableObject
	 * @return IPropertySource
	 * @deprecated
	 */
	protected IFile getFile(Object adaptableObject)
	{
		IResource res = getResource(adaptableObject);
		return res instanceof IFile ? (IFile) res : null;
	}	
	
	/**
	 * Returns the resource associated with a given adaptable object.
	 * @param adaptableObject
	 * @return resource
	 */
	protected IResource getResource(Object adaptableObject)
	{
		if(adaptableObject instanceof Resource)
			return EMFUtil.getWorkspaceFile((Resource)adaptableObject);
		
		if (adaptableObject instanceof ITestCaseProxyNode) {
			// special case : no real resource under
			return null;
		}

		if(adaptableObject instanceof EObject){
			return (EMFUtil.getWorkspaceFile(((EObject)(adaptableObject)).eResource()));
		}
		
		if(adaptableObject instanceof IExtendedProxyNode) {
			return ((IExtendedProxyNode)adaptableObject).getCorrespondingResource();
		}
		return null;
	}	
	

	private Resource getEMFResource(Object adaptableObject) {
		if(adaptableObject instanceof FileProxyNode) {
			return EMFUtil.getResource(null, (IFile) ((FileProxyNode)adaptableObject).getUnderlyingResource());
		}
		return null;
	}

	private IProxyNodeRenamer getProxyNodeRenamer(Object adaptableObject) {
		if(adaptableObject instanceof CMNNamedElementProxyNode) {
			return (IProxyNodeRenamer) adaptableObject;
		}
		return null;
	}
}
