/********************************************************************** 
 * Copyright (c) 2007, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 *  
 * 
 * Contributors: 
 * 		IBM Corporation - initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.datapool.internal.control;

import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.datapool.internal.action.CopyAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.CutAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.DeleteCellAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.FindReplaceAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.PasteAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.SelectAllAction;
import org.eclipse.hyades.test.ui.internal.editor.DatapoolEditorPart;
import org.eclipse.hyades.ui.internal.editor.action.ActionHandlerListener;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.custom.TableCursor;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.actions.ActionFactory;

/** 
 * <p>Datapool action handler for the following datapool actions:</p>
 * 
 * <ul>
 * <li>Cut</li>
 * <li>Copy</li>
 * <li>Paste</li>
 * <li>Delete</li>
 * <li>Select All</li>
 * <li>Find/Replace...</li>
 * </ul>
 * 
 * 
 * @author  Patrick Nedelec
 * @author  Paul Slauenwhite
 * @version March 30, 2010
 * @since   February 27, 2007
 */
public class DatapoolActionHandlerListener extends ActionHandlerListener {
   
	public static final DatapoolActionHandlerListener INSTANCE = new DatapoolActionHandlerListener();

    private FindReplaceAction findReplaceAction = null;

    public DatapoolActionHandlerListener() {
        super();
    }

    /** 
     * @see org.eclipse.hyades.ui.internal.editor.action.ActionHandlerListener#createEditActions()
     */
    protected void createEditActions() {
        
    	super.createEditActions();

    	delete = new DeleteCellAction();
        selectAll = new SelectAllAction();

        cut = new CutAction();
        copy = new CopyAction();
        paste = new PasteAction();
    }

    /** 
     * @see org.eclipse.hyades.ui.internal.editor.action.ActionHandlerListener#filter(org.eclipse.ui.IWorkbenchPart)
     */
    protected boolean filter(IWorkbenchPart part) {
        if (part instanceof DatapoolEditorPart) {
            return false;
        }
        return true;
    }

    /** 
     * @see org.eclipse.hyades.ui.internal.editor.action.ActionHandlerListener#partActivated(org.eclipse.ui.IWorkbenchPart)
     */
    public void partActivated(IWorkbenchPart part) {
        super.partActivated(part);
        connectFindReplaceAction(part);
    }

    /** 
     * @see org.eclipse.hyades.ui.internal.editor.action.ActionHandlerListener#partBroughtToTop(org.eclipse.ui.IWorkbenchPart)
     */
    public void partBroughtToTop(IWorkbenchPart part) {
        super.partBroughtToTop(part);
        connectFindReplaceAction(part);
    }

    /** 
     * @see org.eclipse.hyades.ui.internal.editor.action.ActionHandlerListener#partOpened(org.eclipse.ui.IWorkbenchPart)
     */
    public void partOpened(IWorkbenchPart part) {
        super.partOpened(part);
        connectFindReplaceAction(part);
    }

    private void connectFindReplaceAction(IWorkbenchPart part) {
        if (filter(part)) {
            return;
        }
        IActionBars actionBars = getActionBars(part);
        findReplaceAction = new FindReplaceAction(UiPlugin.getInstance().getResourceBundle(), "find_replace_action.", //$NON-NLS-1$
                part);
        actionBars.setGlobalActionHandler(ActionFactory.FIND.getId(), findReplaceAction);
        actionBars.updateActionBars();
    }

    /** 
     * @see org.eclipse.hyades.ui.internal.editor.action.ActionHandlerListener#updateActions()
     */
    protected void updateActions() {
        IWorkbenchPart part = getWorkbenchPart();
        if (part != null && isRegisteredPart(part)) {
            revert.setWorkbenchPart(part);
            IActionBars actionBars = getActionBars(part);
            Control focusControl = cut.getFocusControl();
            if (isValidFocusControl()) {
                //- cut/copy/paste on a text, styled text or combo
                String clipboardText = paste.getClipboardText();
                String selectionText = copy.getSelectionText();
                boolean isReadOnly = paste.isReadOnlyFocusControl();
                boolean clipboardNotEmpty = clipboardText != null && !clipboardText.equals(""); //$NON-NLS-1$
                boolean textSelected = selectionText != null && !selectionText.equals(""); //$NON-NLS-1$

                cut.setEnabled(!isReadOnly && textSelected);
                copy.setEnabled(textSelected && !(focusControl instanceof Combo && isReadOnly));
                paste.setEnabled(!isReadOnly && clipboardNotEmpty);
                delete.setEnabled(!isReadOnly && (focusControl instanceof StyledText));
                selectAll.setEnabled(!(focusControl instanceof Combo));
                findReplaceAction.setEnabled(false);
            } else if (!isOnTableCursor()) {
                //- Focus on an 'unknown' widget
                cut.setEnabled(false);
                copy.setEnabled(false);
                paste.setEnabled(false);
                delete.setEnabled(false);
                selectAll.setEnabled(false);
                findReplaceAction.setEnabled(false);
            } 
            
            actionBars.updateActionBars();
        }
    }

    private boolean isOnTableCursor() {
        Control control = cut.getFocusControl();
        return (control instanceof TableCursor);
    }

    public FindReplaceAction getFindReplaceAction() {
        return findReplaceAction;
    }
}
