/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DatapoolClipboard.java,v 1.13 2010/04/01 17:23:36 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.datapool.internal.control;

import java.util.StringTokenizer;

import org.eclipse.hyades.edit.datapool.IDatapoolCell;
import org.eclipse.hyades.edit.datapool.IDatapoolRecord;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;

/**
 * <p>Lightweight wrapper around the {@link Clipboard} for datapool cell, record, and string data.</p>
 * 
 * 
 * @author  Peter Sun
 * @author  Paul Slauenwhite
 * @version April 1, 2010
 * @since   January 27, 2005
 * @see     Clipboard
 */
public class DatapoolClipboard {
	
	private Clipboard clipboard = null;

	//Assumption: Datapool cells cannot contain the cell and/or record delimiters.
	private static final String CELL_DELIMITER = "\t"; //$NON-NLS-1$
	
	//Assumption: Datapool cells cannot contain the cell and/or record delimiters.
	private static final String RECORD_DELIMITER = "\n"; //$NON-NLS-1$

	public DatapoolClipboard(Clipboard clipboard){		
		this.clipboard = clipboard;
	}

	public void setStringContents(String contents){
		
		if((clipboard != null) && (!clipboard.isDisposed()) && (contents != null) && (contents.length() > 0)){
			clipboard.setContents(new Object[]{contents}, new Transfer[]{TextTransfer.getInstance()});
		}
	}	
	
	public void setCellContents(IDatapoolCell cell){

		if(cell != null){
			setStringContents(cell.getPersistedRepresentation());
		}	
	}

	//Null and no cells check handled by serialzie method
	public void setRecordContents(IDatapoolRecord record){
		setStringContents(serialize(record));
	}

	//Null, no records, and no cells check handled by serialzie method
	public void setRecordContents(IDatapoolRecord[] records){
		setStringContents(serialize(records));
	}
	
	/**
	 * <p>Resolves the most recently copied string data from the system clipboard.<p/>
	 * 
	 * <p>This method returns <code>null</code> if the clipboard is empty or the 
	 * clipboard does not contain string data.</p>
	 * 
	 * @return Most recently copied string data from the system clipboard, otherwise <code>null</code>.
	 */
	public String getStringContents(){
		
		if((clipboard != null) && (!clipboard.isDisposed())){
			return ((String)(clipboard.getContents(TextTransfer.getInstance())));
		}
		
		return null;
	}
	
	/**
	 * Returns the most recently copied cell data or <code>null</code> if there has
	 * been no copy or previous copy was not a cell.
	 */
	public String getCellContents(){
		
		String stringContents = getStringContents();
		
		if(stringContents != null){

			StringTokenizer cellTokenizer = new StringTokenizer(stringContents.trim(), (CELL_DELIMITER + RECORD_DELIMITER));
			
			if(cellTokenizer.countTokens() == 1){
				return (cellTokenizer.nextToken());
			}
		}
		
		return null;
	}
	
	/**
	 * Returns the most recently copied record data set or <code>null</code> if there has
	 * been no copy or previous copy was not record data.
	 */
	public String[][] getRecordContents(){
		
		String[][] recordContents = null;
		String clipboardContents = getStringContents();
		
		if(clipboardContents != null){

			StringTokenizer recordsTokenizer = new StringTokenizer(clipboardContents.trim(), RECORD_DELIMITER);
			int recordCount = recordsTokenizer.countTokens();

			for (int recordIndex = 0; recordIndex < recordCount; recordIndex++) {

				StringTokenizer cellTokenizer = new StringTokenizer(recordsTokenizer.nextToken().trim(), CELL_DELIMITER);
				int cellCount = cellTokenizer.countTokens();

				//Assumption: All records have the same number of cells.
				if(recordContents == null){
					recordContents = new String[recordCount][cellCount];
				}

				for (int cellIndex = 0; cellIndex < cellCount; cellIndex++) {
					recordContents[recordIndex][cellIndex] = cellTokenizer.nextToken();
				}
			}
		}
		
		return recordContents;
	}
	
	public boolean isEmpty(){
		return (getStringContents() == null);
	}
	
	private String serialize(IDatapoolRecord[] records){

		StringBuffer recordsStringBuffer = new StringBuffer();

		if(records != null){

			for (int counter = 0; counter < records.length; counter++) {

				if(recordsStringBuffer.length() > 0){
					recordsStringBuffer.append(RECORD_DELIMITER);
				}

				recordsStringBuffer.append(serialize(records[counter]));
			}
		}

		return (recordsStringBuffer.toString());
	}
	
	private String serialize(IDatapoolRecord record){

		StringBuffer recordStringBuffer = new StringBuffer();

		if(record != null){

			for (int counter = 0; counter < record.getCellCount(); counter++) {

				IDatapoolCell cell = ((IDatapoolCell)(record.getCell(counter)));

				if(cell != null){

					if(recordStringBuffer.length() > 0){
						recordStringBuffer.append(CELL_DELIMITER);
					}

					recordStringBuffer.append(cell.getPersistedRepresentation());
				}
			}
		}

		return (recordStringBuffer.toString());
	}
}
