/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DatapoolMenuManager.java,v 1.20 2010/04/01 17:23:36 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.datapool.internal.control;

import java.util.Vector;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.datapool.internal.action.CopyAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.CutAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.DeleteCellAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.DeleteColumnAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.DeleteRowAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.EditColumnAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.EditRowAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.FindReplaceAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.InsertColumnAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.InsertRowAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.PasteAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.SelectAllAction;
import org.eclipse.hyades.test.ui.datapool.internal.interfaces.IDatapoolAction;
import org.eclipse.hyades.ui.internal.editor.action.EditAction;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ISelectionProvider;

/**
 * <p>DatapoolMenuManager.java</p>
 * 
 * 
 * @author  Peter Sun
 * @author  Paul E. Slauenwhite
 * @version April 1, 2010
 * @since   January 27, 2005
 */
public class DatapoolMenuManager {
    
	private InsertRowAction insertRowAction = null;
    private DeleteRowAction deleteRowAction = null;
    private EditRowAction editRowAction = null;
    private InsertColumnAction insertColumnAction = null;
    private DeleteColumnAction deleteColumnAction = null;
    private EditColumnAction editColumnAction = null;
    private EditAction deleteCellAction = null;
    private EditAction selectAllAction = null;
    private EditAction cutAction = null;
    private EditAction copyAction = null;
    private EditAction pasteAction = null;

    private DatapoolTable table = null;
    private Vector customActions = null;
    private String vendorID = null;
    private boolean showColumnActions = true;
    private boolean showRowActions = true;
    private boolean showEditActions = true;
    
	/**
	 * Display mode Display mode constant for enabling the {@link InsertRowAction} (value is 1&lt;&lt;1).
	 */
	public static final int INSERT_ROW_ACTION_ENABLED = 1 << 1;

	/**
	 * Display mode constant for enabling the {@link DeleteRowAction} (value is 1&lt;&lt;2).
	 */
	public static final int DELETE_ROW_ACTION_ENABLED = 1 << 2;

	/**
	 * Display mode constant for enabling the {@link EditRowAction} (value is 1&lt;&lt;3).
	 */
	public static final int EDIT_ROW_ACTION_ENABLED = 1 << 3;

	/**
	 * Display mode constant for enabling the {@link InsertColumnAction} (value is 1&lt;&lt;4).
	 */
	public static final int INSERT_COLUMN_ACTION_ENABLED = 1 << 4;

	/**
	 * Display mode constant for enabling the {@link DeleteColumnAction} (value is 1&lt;&lt;5).
	 */
	public static final int DELETE_COLUMN_ACTION_ENABLED = 1 << 5;

	/**
	 * Display mode constant for enabling the {@link EditColumnAction} (value is 1&lt;&lt6).
	 */
	public static final int EDIT_COLUMN_ACTION_ENABLED = 1 << 6;

	/**
	 * Display mode constant for enabling the {@link CutAction} (value is 1&lt;&lt;7).
	 */
	public static final int CUT_ACTION_ENABLED = 1 << 7;

	/**
	 * Display mode constant for enabling the {@link CopyAction} (value is 1&lt;&lt;8).
	 */
	public static final int COPY_ACTION_ENABLED = 1 << 8;

	/**
	 * Display mode constant for enabling the {@link PasteAction} (value is 1&lt;&lt9).
	 */
	public static final int PASTE_ACTION_ENABLED = 1 << 9;

	/**
	 * Display mode constant for enabling the{@link FindReplaceAction} (value is 1&lt;&lt10).
	 */
	public static final int FIND_REPLACE_ACTION_ENABLED = 1 << 10;

	/**
	 * Display mode constant for enabling the {@link SelectAllAction} (value is 1&lt;&lt11).
	 */
	public static final int SELECT_ALL_ACTION_ENABLED = 1 << 11;

	/**
	 * Display mode constant for enabling the {@link DeleteCellAction} (value is 1&lt;&lt12).
	 */
	public static final int DELETE_CELL_ACTION_ENABLED = 1 << 12;

    public DatapoolMenuManager(DatapoolTable table, String vendorID, boolean showEquivalenceClassActions, boolean showVariableActions, boolean showRecordActions) {
        if (table == null) return;
        this.customActions = new Vector();
        this.vendorID = vendorID;
        this.showColumnActions = showVariableActions;
        this.showRowActions = showRecordActions;
        setTable(table);
        makeActions();
    }

    /**
     * Sets the datapool table for the datapool menu manager and 
     * its cut/copy/paste menus.
     * <p/>
     * 
     * @param table The new datapool table.
     */
    public void setTable(DatapoolTable table){
    	
    	this.table = table;
    	
    	//Set the table on the delete action:
    	if(deleteCellAction instanceof DeleteCellAction){
    		((DeleteCellAction)(deleteCellAction)).setTable(table);
    	}
    	
    	//Set the table on the select all action:
    	if(selectAllAction instanceof SelectAllAction){
    		((SelectAllAction)(selectAllAction)).setTable(table);
    	}
    	
    	//Set the table on the cut action:
    	if(cutAction instanceof CutAction){
    		((CutAction)(cutAction)).setTable(table);
    	}

    	//Set the table on the copy action:
    	if(copyAction instanceof CopyAction){
    		((CopyAction)(copyAction)).setTable(table);
    	}

    	//Set the table on the paste action:
    	if(pasteAction instanceof PasteAction){
    		((PasteAction)(pasteAction)).setTable(table);
    	}
   }
    
    /*
     * Creates all the actions
     */
    public void makeActions() {
        ISelectionProvider provider = table.getViewer();

        insertRowAction = new InsertRowAction(provider, table);
        deleteRowAction = new DeleteRowAction(provider, table);
        editRowAction = new EditRowAction(provider, table);
        insertColumnAction = new InsertColumnAction(provider, table);
        deleteColumnAction = new DeleteColumnAction(provider, table);
        editColumnAction = new EditColumnAction(provider, table);

        DatapoolActionHandlerListener datapoolActionHandlerListener = new DatapoolActionHandlerListener();
        
        deleteCellAction = datapoolActionHandlerListener.getDeleteAction();
        selectAllAction = datapoolActionHandlerListener.getSelectAllAction();
        cutAction = datapoolActionHandlerListener.getCutAction();
        copyAction = datapoolActionHandlerListener.getCopyAction();
        pasteAction = datapoolActionHandlerListener.getPasteAction();
        
        if (this.vendorID == null) return;
        IExtensionPoint extensionPoint = Platform.getExtensionRegistry().getExtensionPoint(UiPlugin.getID() + ".menuExtension"); //$NON-NLS-1$

        if (extensionPoint != null) {
            try {

                IConfigurationElement[] extensionPoints = extensionPoint.getConfigurationElements();
                for (int i = 0; i < extensionPoints.length; i++) {
                    String extVendorID = extensionPoints[i].getAttribute("vendorID"); //$NON-NLS-1$
                    if (this.vendorID.equals(extVendorID)) {
                        //String displayName = extensionPoints[i].getAttribute("displayName"); //$NON-NLS-1$
                        IDatapoolAction action = (IDatapoolAction) extensionPoints[i].createExecutableExtension("actionClass"); //$NON-NLS-1$
                        customActions.add(action);
                        action.setDatapoolTable(this.table);
                    }
                }
            } catch (Exception e) {
            }
        }
    }

    public void addActions(IMenuManager imenuMgr) {
        MenuManager menuMgr = (MenuManager) imenuMgr;

        // Add actions to menu.
        if (showRowActions) {
            menuMgr.add(insertRowAction);
            menuMgr.add(deleteRowAction);
            menuMgr.add(editRowAction);
            menuMgr.add(new Separator());
        }
        if (showColumnActions) {
            menuMgr.add(insertColumnAction);
            menuMgr.add(deleteColumnAction);
            menuMgr.add(editColumnAction);
            menuMgr.add(new Separator());
        }
        
        if (showEditActions) {

            menuMgr.add(deleteCellAction);
            menuMgr.add(selectAllAction);
            menuMgr.add(new Separator());

            menuMgr.add(cutAction);
            menuMgr.add(copyAction);
            menuMgr.add(pasteAction);
            menuMgr.add(new Separator());
            
            FindReplaceAction findReplaceAction = DatapoolActionHandlerListener.INSTANCE.getFindReplaceAction();
        	
        	if (findReplaceAction != null) {
        		
                menuMgr.add(findReplaceAction);
                menuMgr.add(new Separator());        		
        	}
        }
        if (customActions.size() > 0) {
            for (int i = 0; i < customActions.size(); i++) {
                IAction action = (IAction) customActions.get(i);
                menuMgr.add(action);
            }
        }
    }

    public void setDisplayMode(int displayMode) {
            	
    	//Set the display modes for the row actions:
    	insertRowAction.aboutToShow((displayMode & INSERT_ROW_ACTION_ENABLED) != 0);
        deleteRowAction.aboutToShow((displayMode & DELETE_ROW_ACTION_ENABLED) != 0);
		editRowAction.aboutToShow((displayMode & EDIT_ROW_ACTION_ENABLED) != 0);
        
    	//Set the display modes for the columns actions:
		insertColumnAction.aboutToShow((displayMode & INSERT_COLUMN_ACTION_ENABLED) != 0);
        deleteColumnAction.aboutToShow((displayMode & DELETE_COLUMN_ACTION_ENABLED) != 0);
        editColumnAction.aboutToShow((displayMode & EDIT_COLUMN_ACTION_ENABLED) != 0);
        
    	//Set the display modes for the edit actions:
        deleteCellAction.setEnabled((displayMode & DELETE_CELL_ACTION_ENABLED) != 0);
        selectAllAction.setEnabled((displayMode & SELECT_ALL_ACTION_ENABLED) != 0);
       
        cutAction.setEnabled((displayMode & CUT_ACTION_ENABLED) != 0);
        copyAction.setEnabled((displayMode & COPY_ACTION_ENABLED) != 0);
        pasteAction.setEnabled((displayMode & PASTE_ACTION_ENABLED) != 0);
        
    	FindReplaceAction findReplaceAction = DatapoolActionHandlerListener.INSTANCE.getFindReplaceAction();
    	
    	if (findReplaceAction != null) {
    		findReplaceAction.setEnabled((displayMode & FIND_REPLACE_ACTION_ENABLED) != 0);
    	}
    }
}
