/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DatapoolTable.java,v 1.85 2010/05/20 13:53:44 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.datapool.internal.control;

import java.lang.reflect.Constructor;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.edit.datapool.IDatapoolCell;
import org.eclipse.hyades.edit.datapool.IDatapoolEquivalenceClass;
import org.eclipse.hyades.edit.datapool.IDatapoolFactory;
import org.eclipse.hyades.edit.datapool.IDatapoolListener;
import org.eclipse.hyades.edit.datapool.IDatapoolRecord;
import org.eclipse.hyades.edit.datapool.IDatapoolSuggestedType;
import org.eclipse.hyades.edit.datapool.IDatapoolVariable;
import org.eclipse.hyades.models.common.datapool.DPLDatapool;
import org.eclipse.hyades.models.common.datapool.DPLVariable;
import org.eclipse.hyades.models.common.datapool.internal.util.DatapoolSuggestedTypeChecker;
import org.eclipse.hyades.models.common.datapool.util.DatapoolEncryptManager;
import org.eclipse.hyades.models.common.util.EncryptionManager;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.datapool.internal.dialog.DatapoolColumnDialog;
import org.eclipse.hyades.test.ui.datapool.internal.dialog.DatapoolConstants;
import org.eclipse.hyades.test.ui.datapool.internal.dialog.DatapoolDeleteColumnDialog;
import org.eclipse.hyades.test.ui.datapool.internal.dialog.DatapoolInputKeyDialog;
import org.eclipse.hyades.test.ui.datapool.internal.dialog.DatapoolRowDialog;
import org.eclipse.hyades.test.ui.datapool.internal.interfaces.IDatapoolPart;
import org.eclipse.hyades.test.ui.datapool.internal.interfaces.IValidateValueClass;
import org.eclipse.hyades.test.ui.datapool.internal.interfaces.IValueClassFactory;
import org.eclipse.hyades.test.ui.datapool.internal.util.DatapoolUtil;
import org.eclipse.hyades.test.ui.datapool.internal.util.EncryptedValueObject;
import org.eclipse.hyades.test.ui.datapool.internal.util.ValueClassMap;
import org.eclipse.hyades.test.ui.datapool.internal.util.ValueObject;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.ICellEditorListener;
import org.eclipse.jface.viewers.ICellEditorValidator;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.accessibility.ACC;
import org.eclipse.swt.accessibility.AccessibleAdapter;
import org.eclipse.swt.accessibility.AccessibleEvent;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.custom.ControlEditor;
import org.eclipse.swt.custom.TableCursor;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MenuDetectEvent;
import org.eclipse.swt.events.MenuDetectListener;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.events.TraverseEvent;
import org.eclipse.swt.events.TraverseListener;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Widget;

/**
 * <p>A datapool table is a spreadsheet-like table editor (extension of {@link TableViewer}
 * and {@link Table}) for editing {@link DPLDatapool}s.</p>  
 * 
 * <p>A variable is represented as a column in the table.  A record is represented 
 * as a row in the table.  An equivalence class is a grouping of variables and one 
 * or more records with the same name.</p> 
 * 
 * <p>Equivalence classes, variables, and records can be inserted, deleted, and 
 * edited in the datapool table.</p>
 * 
 * 
 * @author  Peter Sun
 * @author  Paul E. Slauenwhite
 * @version May 20, 2010
 * @since   January 27, 2005
 */
public class DatapoolTable implements SelectionListener, IDatapoolListener {
		
	private IDatapoolEquivalenceClass equivalenceClass = null;
	private IDatapool datapool = null;	
	private Table table = null;
	private DatapoolTableUtil tableUtil = null;
	private TableViewer viewer = null;
	private Composite parent = null;
	private Menu contextMenu = null;
	private TableCursor tableCursor = null;
	private ControlEditor controlEditor = null;
	private IDatapoolPart datapoolEditorPart = null;
	private IDatapoolFactory datapoolFactory = null;
	private CellEditor cellEditor = null;
	private ValueObject theValueObject = null;
	private DatapoolMenuManager datapoolMenuManager = null;
	private boolean altKeyDown = false;
	private boolean isInputValid = true;
	private boolean showVariables = true;
	private boolean showRecords = true;
	private boolean isF2Mode = false;
	private int headerSelectionIndex = -1;
	private String cellkey = null;
	private String password = null;
	private String vendorID = null;
	private DatapoolClipboard datapoolClipboard = null;
	
	/**
	 * <p>Flag to denote when the {@link #refreshRows()} method is scheduled to be invoked.</p>
	 * 
	 * <p>This flag may be used to increase the performance of certain operations (for example, removing
	 * multiple records) by eliminating duplicate operations (for example, re-populate the row number labels).</p>
	 */
	private boolean refreshRowsScheduled = false;
	
	private static final int MAX_TABLE_WIDTH = 60;
	private static final int MAX_TABLE_HEIGHT = 60;
	private static final int HEADER_COLUMN_DEFAULT_WIDTH = 20;
	private static final int COLUMN_DEFAULT_WIDTH = 110;

	/**
	 * Handles the selection of the header area.
	 * <p>Clicking on the header area bring up the dialog to change and/or 
	 * move the selected variable/column.
	 */
	protected SelectionListener headerListener = new SelectionAdapter()
	{
		public void widgetSelected(SelectionEvent event)
		{
			datapoolEditorPart.notifyEdit();
			if(datapoolEditorPart.isReadOnly())
				return;
			clearCellEditor();
			TableColumn tableColumn = (TableColumn) event.widget;
			editColumnAux(tableColumn);
		}		
	};

	/**
	 * Handles the resizing of scroll bars when columns in the table are 
	 * resized.  
	 */
	protected ControlListener resizeColumnListener = new ControlAdapter()
	{
		public void controlMoved(ControlEvent e) 
		{
		}

		public void controlResized(ControlEvent e) 
		{
			if(tableCursor != null && !tableCursor.isDisposed())
			{
				if(table.getSelectionIndex() >= 0 && tableCursor.getColumn() >= 0)
					tableCursor.setSelection(table.getSelectionIndex(), tableCursor.getColumn());
			}
		}				
	};

	class TableCellEditorAdapter implements ICellEditorListener
	{
		private CellEditor editor = null;
		
		public TableCellEditorAdapter(CellEditor cellEditor, ValueObject valueObject, TableItem row, int column)
		{
			this.editor = cellEditor;
		}
		
		/**
		 * @see org.eclipse.jface.viewers.ICellEditorListener#applyEditorValue()
		 */
		public void applyEditorValue() 
		{
			isF2Mode = false;
			if(!isInputValid)
			{
				return;
			}
			applyEditingValue(true);
		}

		/**
		 * @see org.eclipse.jface.viewers.ICellEditorListener#cancelEditor()
		 */
		public void cancelEditor() 
		{
			isF2Mode = false;
		}

		/**
		 * @see org.eclipse.jface.viewers.ICellEditorListener#editorValueChanged(boolean, boolean)
		 */
		public void editorValueChanged(boolean oldValidState, boolean newValidState) 
		{
			if(newValidState != true)
				isInputValid = false;
			else 
				isInputValid = true;
			editor.getControl().redraw();
		}
	}
	
	class TableCellPaintAdapter implements PaintListener
	{
		/**
		 * @see org.eclipse.swt.events.PaintListener#paintControl(PaintEvent)
		 */
		public void paintControl(PaintEvent event)
		{
			if(!isInputValid){
				drawPolyline( event.gc, (Widget)event.getSource() );
			}
		}
		
		private void drawPolyline(GC gc, Widget widget ) 
		{
			if (gc == null) 
				return;
					
			Rectangle rect = gc.getClipping();
			String str = ""; //$NON-NLS-1$
			if( widget instanceof Text )
			{
				str = ((Text)widget).getText();
			}
			else if( widget instanceof CCombo )
			{
				str = ((CCombo)widget).getText();
			}
			else if( widget instanceof Combo )
			{
				str = ((Combo)widget).getText();
			}
			else
				return;

			int offset = 2;
			char c;
			for( int i = 0; i < str.length(); i++ )
			{
				c = str.charAt( i );
				if( Character.isWhitespace( c ) )
					offset += gc.getAdvanceWidth( c );
				else
					break;
			}

			int length = gc.textExtent( str.trim() ).x;
			
			Point left = new Point( offset, 0 );				
			Point right = new Point( (length + offset + 2), 0 );
			
			gc.setForeground(Display.getCurrent().getSystemColor( SWT.COLOR_RED ));
			int[] polyline= computePolyline(left, right, rect.height );//gc.getFontMetrics().getHeight());
			gc.drawPolyline(polyline);
		}
		
		private int[] computePolyline(Point left, Point right, int height) 
		{		
			final int WIDTH= 4; // must be even
			final int HEIGHT= 2; // can be any number
			
			int leftX= left.x;
			int peeks= (right.x - left.x) / WIDTH;
					
			// compute (number of point) * 2
			int length= ((2 * peeks) + 1) * 2;
			if (length < 0)
				return new int[0];
				
			int[] coordinates= new int[length];
			
			// cache peeks' y-coordinates
			int bottom= left.y + height - 1;
			int top= bottom - HEIGHT;
			
			// populate array with peek coordinates
			for (int i= 0; i < peeks; i++) {
				int index= 4 * i;
				coordinates[index]= leftX + (WIDTH * i);
				coordinates[index+1]= bottom;
				coordinates[index+2]= coordinates[index] + WIDTH/2;
				coordinates[index+3]= top;
			}
			
			// the last down flank is missing
			coordinates[length-2]= left.x + (WIDTH * peeks);
			coordinates[length-1]= bottom;
			
			return coordinates;
		}

	}
	
	/*
	 * Handles the keystrokes of the cell editor when it is in edit mode.
	 */
	class TableCellKeyAdapter implements KeyListener
	{
		private int rowIndex = -1;
		private int column = -1;
		
		public TableCellKeyAdapter(CellEditor cellEditor, ValueObject valueObject, TableItem row, int column)
		{
			this.column = column;
			if(row != null)
				this.rowIndex = table.indexOf(row);
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.KeyListener#keyPressed(org.eclipse.swt.events.KeyEvent)
		 */
		public void keyPressed(KeyEvent e) 
		{
			if(equivalenceClass == null)
				return;

			if((e.character == SWT.CR) || (e.character == SWT.KEYPAD_CR))
			{
				if(!isInputValid)
				{
					e.doit = false;
					return;
				}
				return;
			}

			if((e.keyCode == SWT.F10) && ((e.stateMask & SWT.SHIFT) != 0)){
				
				//Delegate to the widget's context menu:
				return;
			}

			//Arrow keys pressed during cell editing:
			if (e.keyCode == SWT.ARROW_DOWN ||
				 e.keyCode == SWT.ARROW_UP ||
				 e.keyCode == SWT.ARROW_LEFT ||
				 e.keyCode == SWT.ARROW_RIGHT){
				
				if(!isInputValid)
				{
					e.doit = false;
					return;
				}				
				applyEditingValue(true);
				// Move the cursor if any of the arrow keys are hit.  This is to
				// simulate excel behavior.
				if(tableCursor != null && !tableCursor.isDisposed())
				{
					if(e.keyCode == SWT.ARROW_DOWN)
					{
						if(rowIndex + 1 < table.getItemCount())
						{
							table.setSelection(rowIndex + 1);
							tableCursor.setSelection(rowIndex + 1, column);	
						}
					}
					if(e.keyCode == SWT.ARROW_UP)
					{
						if(rowIndex > 0)
						{
							table.setSelection(rowIndex - 1);
							tableCursor.setSelection(rowIndex - 1, column);	
						}
					}					
					if(e.keyCode == SWT.ARROW_RIGHT)
					{
						if(column + 1 < tableUtil.getColumnCount())
						{
							tableCursor.setSelection(rowIndex, column + 1);
						}
					}
					if(e.keyCode == SWT.ARROW_LEFT)
					{
						if(column > 0)
						{
							tableCursor.setSelection(rowIndex, column - 1);
						}
					}
					tableCursor.setFocus();
				}
				return;				
			}
		}	

		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
		 */
		public void keyReleased(KeyEvent e) 
		{			
			if (e.character == SWT.ESC)
			{
				clearCellEditor();
				return;
			}	
			if (e.keyCode == SWT.ARROW_DOWN ||
				e.keyCode == SWT.ARROW_UP ||
				e.keyCode == SWT.ARROW_LEFT ||
				e.keyCode == SWT.ARROW_RIGHT) 
			{
				if(!isInputValid)
				{
					e.doit = false;
					return;
				}
			}
		}			
	}

	class TableCellValidator implements ICellEditorValidator
	{

		IValidateValueClass validator = null;
		
		public TableCellValidator(IValidateValueClass validator)
		{
			this.validator = validator;
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ICellEditorValidator#isValid(java.lang.Object)
		 */
		public String isValid(Object value) 
		{
			if(this.validator == null)
				return null;
			if(this.validator.checkValue(value))
				return null;
			else
			{
				Error error = this.validator.getError();
				if(error != null)
					return error.toString();
				else 
					return "Error"; //$NON-NLS-1$
			}
		}		
	}
	
	/**
	 * Abstract base class used for traverse adapters, simple use of the template
	 * method pattern using a few abstract and defaulted methods to be overridden
	 * by subclasses.
	 * 
	 * @author Scott E. Schneider
	 */
	abstract class AbstractTraverseAdapter implements TraverseListener {
		
		private AbstractTraverseAdapter() {}
		
		/**
		 * This will be invoked before a traverse is processed and the selection is changed
		 */
		protected void preTraverse() {}
		
		/**
		 * The table cursor to be used is returned
		 * 
		 * @return the table cursor to be used
		 */
		abstract protected TableCursor getCursor();
		
		/**
		 * The column to be used for operations requiring current column
		 * 
		 * @return the current column
		 */
		abstract protected int getColumn();
		
		/**
		 * The row to be used for operations requiring current row
		 * 
		 * @return the current row
		 */
		abstract protected int getRow();
		
		/**
		 * @see org.eclipse.swt.events.TraverseListener#keyTraversed(org.eclipse.swt.events.TraverseEvent)
		 */
		public void keyTraversed(TraverseEvent e) {
			if(!DatapoolTable.this.isInputValid ||
				DatapoolTable.this.tableCursor == null ||
				DatapoolTable.this.tableCursor.isDisposed()) { 
				e.doit = false;
				return;
			}
            if ((e.stateMask & SWT.CTRL) != 0) {
                //- If the key modifier is Ctrl, then do nothing but let the others handling it.
                //- Specially the Ctrl+PgUp/PgDown, Ctrl+Tab or Ctrl+Shift+Tab that are no more eaten here 
                e.doit = true;
                return;
            }
	
			if(e.detail == SWT.TRAVERSE_TAB_NEXT ||
				e.detail == SWT.TRAVERSE_TAB_PREVIOUS ||
				e.detail == SWT.TRAVERSE_RETURN)
			{
				e.doit = false;
				this.preTraverse();
				if(e.detail == SWT.TRAVERSE_TAB_NEXT) {
					if(this.getColumn() + 1 == DatapoolTable.this.tableUtil.getColumnCount())
						insertColumn();
					else
						traverseRight();
				}
				if(e.detail == SWT.TRAVERSE_TAB_PREVIOUS) {
					traverseLeft();
				}			
				if(e.detail == SWT.TRAVERSE_RETURN){
					if(this.getRow() + 1 == DatapoolTable.this.table.getItemCount()) {
						DatapoolTable.this.addRow();
					}
					traverseDown();	
				}	
			}
		}
	
		protected void traverseRight()
		{
			if(this.getColumn() + 1 < DatapoolTable.this.tableUtil.getColumnCount()) {
				DatapoolTable.this.tableCursor.setSelection(this.getRow(), this.getColumn() + 1);
			}
		}
		
		protected void traverseLeft()
		{			
			if(this.getColumn() > 0) {
				DatapoolTable.this.tableCursor.setSelection(this.getRow(), this.getColumn() - 1);		
			}
		}
		
		protected void traverseUp()
		{	
			if(this.getRow() > 0) {
				DatapoolTable.this.table.setSelection(this.getRow() - 1);
				DatapoolTable.this.tableCursor.setSelection(this.getRow() - 1, this.getColumn());
			}
		}
		
		protected void traverseDown()
		{		
			int newRowIndex = this.getRow() + 1; 
			if(newRowIndex < DatapoolTable.this.table.getItemCount()) {
				DatapoolTable.this.table.setSelection(newRowIndex);
				DatapoolTable.this.tableCursor.setSelection(newRowIndex, this.getColumn());
			}
		}
	}
	
	/**
	 * Traverse adapter to be used for listening to cursor traverse events.
	 * 
	 * @author Scott E. Schneider
	 */
	private class CursorTraverseAdapter extends AbstractTraverseAdapter {
		
		/**
		 * To be constructed internally only, no parameters defined, uses state from the current outer instance 
		 * associated with this inner class instance
		 */
		private CursorTraverseAdapter() {}
		
		/* (non-Javadoc)
		 * @see org.eclipse.hyades.test.ui.datapool.internal.control.DatapoolTable.AbstractTraverseAdapter#getColumn()
		 */
		protected int getColumn() { 
			return DatapoolTable.this.tableCursor.getColumn();
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.hyades.test.ui.datapool.internal.control.DatapoolTable.AbstractTraverseAdapter#getRow()
		 */
		protected int getRow() {
			return DatapoolTable.this.table.indexOf(DatapoolTable.this.tableCursor.getRow());
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.hyades.test.ui.datapool.internal.control.DatapoolTable.AbstractTraverseAdapter#getCursor()
		 */
		protected TableCursor getCursor() {
			return DatapoolTable.this.tableCursor;
		}
	
	}
	
	/**
	 * Traverse adapter to be used for listening to table cell traverse events
	 * 
	 * @author Scott E. Schneider
	 */
	class TableCellTraverseAdapter extends AbstractTraverseAdapter {
		
		/**
		 * The associated value object
		 */
		private ValueObject valueObject;
		
		/**
		 * The current row to be used for operation
		 */
		private int row;
		
		/**
		 * The current column to be used for operation
		 */
		private int column;
		
		/**
		 * To be constructed internally only, given the cell editor, value object and current row and column
		 * 
		 * @param editor the editor to be used for operations
		 * @param valueObject the value object to be used for operations
		 * @param row the current row
		 * @param column the current column
		 */
		private TableCellTraverseAdapter(CellEditor editor, ValueObject valueObject, TableItem row, int column) {
			this.valueObject = valueObject;
			this.row = DatapoolTable.this.table.indexOf(row);
			this.column = column;
		}
		
		public void keyTraversed(TraverseEvent e) {
			super.keyTraversed(e);
			
			if(e.detail == SWT.TRAVERSE_ARROW_NEXT || e.detail == SWT.TRAVERSE_ARROW_PREVIOUS)
			{
				if(isF2Mode)
					return;
				
				e.doit = false;
				this.preTraverse();
				
				if(e.keyCode == SWT.ARROW_RIGHT)
					traverseRight();
				if(e.keyCode == SWT.ARROW_LEFT)
					traverseLeft();
				if(e.keyCode == SWT.ARROW_UP)
					traverseUp();
				if(e.keyCode == SWT.ARROW_DOWN)
					traverseDown();
			}
		}
		/* (non-Javadoc)
		 * @see org.eclipse.hyades.test.ui.datapool.internal.control.DatapoolTable.AbstractTraverseAdapter#preTraverse()
		 */
		protected void preTraverse() {
			if(this.valueObject != null) {
				DatapoolTable.this.applyEditingValue(true);
			}
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.hyades.test.ui.datapool.internal.control.DatapoolTable.AbstractTraverseAdapter#getColumn()
		 */
		protected int getColumn() { 
			return this.column;
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.hyades.test.ui.datapool.internal.control.DatapoolTable.AbstractTraverseAdapter#getRow()
		 */
		protected int getRow() {
			return this.row;
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.hyades.test.ui.datapool.internal.control.DatapoolTable.AbstractTraverseAdapter#getCursor()
		 */
		protected TableCursor getCursor() {
			return DatapoolTable.this.tableCursor;
		}
		
	}
	
	/**
	 * Menu detect listener for open the context menu when:
	 * <p/>
	 * <ul>
	 * <li>Windows menu (between the right Alt (or right Windows key) and Ctrl keys) key press.</li>
	 * <li>Shift+F10 key press.</li>
	 * <li>Right-button (button 3) mouse click.</li>
	 * </ul>
	 */
	class CursorMenuDetectAdapter implements MenuDetectListener {

		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.MenuDetectListener#menuDetected(org.eclipse.swt.events.MenuDetectEvent)
		 */
		public void menuDetected(MenuDetectEvent e) {

			if(equivalenceClass != null){

				//Create the context menu:
				if((contextMenu == null) || (contextMenu.isDisposed())){
					createMenu();
				}
				
				//Open the context menu:
				contextMenu.setEnabled(true);
				contextMenu.setVisible(true);
				
				headerSelectionIndex = -1;
			}
		}
	}
	
	class CursorKeyAdapter implements KeyListener
	{
		private TableCursor cursor = null;
		private IDatapoolPart datapoolPart = null;
		
		public CursorKeyAdapter(TableCursor cursor, CellEditor cellEditor, IDatapoolPart datapoolPart)
		{
			this.cursor = cursor;
			this.datapoolPart = datapoolPart;
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.KeyListener#keyPressed(org.eclipse.swt.events.KeyEvent)
		 */
		public void keyPressed(KeyEvent e){
			
			if(equivalenceClass == null){
				return;
			}
						
			if(e.keyCode == SWT.ALT){
				
				altKeyDown = true;
				
				return;
			}

			if ((e.keyCode == SWT.SHIFT) || 
					((e.stateMask & SWT.SHIFT) != 0)){

				this.cursor.setVisible(false);

				return;
			}

			//Note: The CursorMenuDetectAdapter handles opening the context menu (Shift + F10).
			if((e.keyCode == SWT.ESC) || 
					(e.keyCode == SWT.CTRL) ||
					(e.keyCode == SWT.F1) ||
					(e.keyCode == SWT.TAB) ||
					((e.keyCode >= SWT.F3) && (e.keyCode <= SWT.F12)) ||
					((e.keyCode == SWT.F10) && ((e.stateMask & SWT.SHIFT) != 0))){				
				return;
			}

			//Resolve the new column selection index:
			int newColumnSelectionIndex = this.cursor.getColumn();

			//Resolve the new row selection table item:
			TableItem newRowSelection = this.cursor.getRow();
            
			//Return if a new row is not selected:
			if (newRowSelection == null){
            	return;
            }
				
			//Row navigation keys:
			if((e.keyCode == SWT.ARROW_DOWN) ||
					(e.keyCode == SWT.ARROW_UP) ||
					(e.keyCode == SWT.PAGE_DOWN) ||
					(e.keyCode == SWT.PAGE_UP) ||
					(e.keyCode == SWT.HOME) ||
					(e.keyCode == SWT.END) ||
					(e.keyCode == SWT.CR) ||
					(e.keyCode == SWT.KEYPAD_CR)){

				//Resolve the new row selection index:
				int newRowSelectionIndex = table.indexOf(newRowSelection);

				//De-select multiple rows if an unselected row is selected:
				if ((newRowSelectionIndex > -1) && (!table.isSelected(newRowSelectionIndex))){
					table.setSelection(newRowSelectionIndex);
				}

				return;
			}

			//Column navigation keys:
			if((e.keyCode == SWT.ARROW_LEFT) ||
					(e.keyCode == SWT.ARROW_RIGHT)){

				//Resolve the new row selection index:
				int newRowSelectionIndex = table.indexOf(newRowSelection);
				
				//De-select multiple rows:
				if (newRowSelectionIndex > -1){
					table.setSelection(newRowSelectionIndex);
				}
				
				return;
			}
			
			if(this.datapoolPart.isReadOnly())
				return;
			
			if(e.keyCode == SWT.INSERT && (e.stateMask & SWT.CTRL) != 0){
				
				if(newColumnSelectionIndex >= 0){
					insertColumn();
				}
				
				return;
			}
			
			if(e.keyCode == SWT.DEL){

				if((e.stateMask & SWT.CTRL) != 0){

					//Delete the record when the Ctrl key is pressed and the record number column is selected:
					if(newColumnSelectionIndex == 0){
						deleteRow();
					}
					
					//Delete the variable when the Ctrl key is pressed, a variable column (a single cell) is selected, and there are two or more variable columns:
					else if((newColumnSelectionIndex > 0) && (table.getColumnCount() > 2)){
						deleteColumn();
					}
				}

				return;
			}
			
			if(((e.stateMask & SWT.CTRL) != 0) ||
				((e.stateMask & SWT.ALT) != 0))
			{
				return;
			}
			
			if(e.keyCode == SWT.INSERT){
				
				if(newColumnSelectionIndex >= 0){
					insertRow();
				}
				
				return;
			}
			
			//Only edit variable columns and not the first (record number) column:
			else if(newColumnSelectionIndex != 0){
				
				String inputString = String.valueOf(e.character);

				//Prepend the current cell text (text cells editors and editable combo box cell editors) to the 
				//input string when the cell editor is not disposed to handle multiple key events on the table 
				//cursor (see defect #247972):
				if((cellEditor != null) && (cellEditor.getControl() != null)){
					
					if(cellEditor instanceof TextCellEditor){
						inputString = ((((Text)(cellEditor.getControl())).getText()) + inputString);
					}
					else if((cellEditor instanceof ComboBoxCellEditor) && ((cellEditor.getStyle() & SWT.READ_ONLY) == 0)){
						inputString = ((((CCombo)(cellEditor.getControl())).getText()) + inputString);
					}									
				}
					
				datapoolPart.notifyEdit();
								
				startCellEditing(newRowSelection, newColumnSelectionIndex);

				if((cellEditor != null) && (cellEditor.getControl() != null)){

					isF2Mode = true;
					
					if (e.keyCode == SWT.BS) {

						if(cellEditor instanceof TextCellEditor){
	                         
							//Remove the text and move the cursor to the beginning:
							Text cellEditorText = ((Text)(cellEditor.getControl()));
							cellEditorText.setText(new String());
							cellEditorText.setSelection(0);	
						}
						else if(cellEditor instanceof ComboBoxCellEditor){

							//Remove the text and move the cursor to the beginning:
							CCombo cellEditorCombo = ((CCombo)(cellEditor.getControl()));
							cellEditorCombo.setText(new String());
							cellEditorCombo.setSelection(new Point(0, 0));	
						}				
                    } 
					else if (e.keyCode == SWT.F2) {

                                     
						if(cellEditor instanceof TextCellEditor){
                         						
							//Set the text and move the cursor to the end:
							Text cellEditorText = ((Text)(cellEditor.getControl()));
							cellEditorText.setSelection(cellEditorText.getText().length());	
						}
						else if(cellEditor instanceof ComboBoxCellEditor){

							//Set the text and move the cursor to the end:
							CCombo cellEditorCombo = ((CCombo)(cellEditor.getControl()));
							
							int textLength = cellEditorCombo.getText().length();
							
							cellEditorCombo.setSelection(new Point(textLength, textLength));	
						}
                    } 
					else {
                       
						if ((cellEditor.getValidator() != null) && (cellEditor.getValidator().isValid(inputString) != null)) {
                            isInputValid = false;
                        }
						
						if(cellEditor instanceof TextCellEditor){
	                         
							//Set the text and move the cursor to the end:
							Text cellEditorText = ((Text)(cellEditor.getControl()));
							cellEditorText.setText(inputString);
							cellEditorText.setSelection(inputString.length());	
						}
						
						//Set the text and selection on only text cells editors and editable combo box cell editors:
						else if((cellEditor instanceof ComboBoxCellEditor) && ((cellEditor.getStyle() & SWT.READ_ONLY) == 0)){

							//Set the text and move the cursor to the end:
							CCombo cellEditorCombo = ((CCombo)(cellEditor.getControl()));
							cellEditorCombo.setText(inputString);
							cellEditorCombo.setSelection(new Point(inputString.length(), inputString.length()));	
						}		
                    }
				}
			}
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
		 */
		public void keyReleased(KeyEvent e) {
			
			if ((e.keyCode == SWT.CTRL) || ((e.stateMask & SWT.CTRL) != 0)){ 
				terminateMultipleSelection();
			}
			
			if(altKeyDown)
			{
				altKeyDown = false;
				return;				
			}
	
			//Note: Special handling on Linux where keyPressed() doesn't get this event.
			if((e.keyCode == SWT.INSERT) && ((e.stateMask & SWT.CTRL) != 0)){
				
				int selectedColumnIndex = this.cursor.getColumn();
				
				if(selectedColumnIndex >= 0){
					insertColumn();
				}
				
				return;
			}
		}		
	}
	
	class CursorMouseAdapter extends MouseAdapter{
		
		private Table table = null;
		private TableCursor cursor = null;
		private IDatapoolPart datapoolPart = null;
		
		public CursorMouseAdapter(Table table, TableCursor cursor, CellEditor cellEditor, IDatapoolPart datapoolPart){
			
			this.table = table;
			this.cursor = cursor;
			this.datapoolPart = datapoolPart;
		}
			
		public void mouseDown(MouseEvent event) {	
			
			if(event.button == 1){				
				
				clearCellEditor();
				
				if(equivalenceClass != null){

					datapoolPart.notifyEdit();
					
					if(!datapoolPart.isReadOnly()){

						TableItem row = this.cursor.getRow();
						int column = this.cursor.getColumn();
			
						if(column == 0){
							editRowAux(row);
						}
						else{
			
							startCellEditing(row, column);
							
							if((cellEditor != null) && (cellEditor.getControl() != null)){

								isF2Mode = true;

								if(cellEditor instanceof TextCellEditor){

									//Set the text and move the cursor to the end:
									Text cellEditorText = ((Text)(cellEditor.getControl()));
									cellEditorText.setSelection(cellEditorText.getText().length());	
								}
								else if(cellEditor instanceof ComboBoxCellEditor){

									//Set the text and move the cursor to the end:
									CCombo cellEditorCombo = ((CCombo)(cellEditor.getControl()));

									int textLength = cellEditorCombo.getText().length();
									
									cellEditorCombo.setSelection(new Point(textLength, textLength));	
								}
							}
						}
					}
				}
			}
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.MouseAdapter#mouseUp(org.eclipse.swt.events.MouseEvent)
		 */
		public void mouseUp(MouseEvent event) {

			//De-select multiple rows when either the second/third mouse buttons are released on any cell in the 1...n columns:
			if((event.button > 1) && (this.cursor.getColumn() > 0)){	
				this.table.setSelection(this.cursor.getRow());
			}
		}
	}
	
	class CursorSelectionAdapter implements SelectionListener
	{
		private IDatapoolPart datapoolPart = null;
		
		public CursorSelectionAdapter(TableCursor cursor, CellEditor cellEditor, IDatapoolPart datapoolPart)
		{
			this.datapoolPart = datapoolPart;
		}
		
		// When the TableEditor is over a cell, select the corresponding row in 
		// the table.
		public void widgetSelected(SelectionEvent event) 
		{
			if(datapoolPart.isReadOnly())
				return;
			clearCellEditor();
			if(!isInputValid)
				return;

			//table.setSelection(new TableItem[] {cursor.getRow()});
			if(datapoolMenuManager != null)
				setMenuMode(datapoolMenuManager);
		}
		
		public void widgetDefaultSelected(SelectionEvent event)
		{
		}
	}
	
	/**
	 * Default constructor for the Datapool Table.  Registers various key, 
	 * menu, and selection listeners.  Creates context menu and populates 
	 * the table.
	 * 
	 * @param parent
	 * @param datapoolPart
	 * @param datapool
	 * @param datapoolFactory
	 * @param vendorID
	 * @deprecated use #DatapoolTable(Composite parent, IDatapoolPart datapoolPart, IDatapoolEquivalenceClass ec, IDatapoolFactory datapoolFactory, String vendorID) instead.
	 */
	public DatapoolTable(Composite parent, 
			 IDatapoolPart datapoolPart, 
			 IDatapool datapool, 
			 IDatapoolFactory datapoolFactory, 
			 String vendorID)
	{
		this.vendorID = vendorID;
		ValueClassMap.setVendorID(vendorID);
		// backword compatibility - use this table to show the default or first equivalence class.
	    IDatapoolEquivalenceClass ec = getDefaultEquivalenceClass(datapool);
		DatapoolTableConstructor(parent, datapoolPart, ec, datapoolFactory);
	}
	
	public DatapoolTable(Composite parent, 
			 IDatapoolPart datapoolPart, 
			 IDatapoolEquivalenceClass ec, 
			 IDatapoolFactory datapoolFactory, 
			 String vendorID)
	{
		this.vendorID = vendorID;
		ValueClassMap.setVendorID(vendorID);
		DatapoolTableConstructor(parent, datapoolPart, ec, datapoolFactory);
	}

	/**
	 * Default constructor for the Datapool Table.  Registers various key, 
	 * menu, and selection listeners.  Creates context menu and populates 
	 * the table.
	 * 
	 * @param parent
	 * @param datapoolPart
	 * @param datapool
	 * @param datapoolFactory
	 * @deprecated use #DatapoolTable(Composite, IDatapoolPart, IDatapoolEquivalenceClass, IDatapoolFactory) instead.
	 */
	public DatapoolTable(Composite parent, 
						 IDatapoolPart datapoolPart, 
						 IDatapool datapool, 
						 IDatapoolFactory datapoolFactory)
	{
	    // backword compatibility - use this table to show the default or first equivalence class.
	    IDatapoolEquivalenceClass ec = getDefaultEquivalenceClass(datapool);
		DatapoolTableConstructor(parent, datapoolPart, ec, datapoolFactory);
	}	
	
	public DatapoolTable(Composite parent, 
			 IDatapoolPart datapoolPart, 
			 IDatapoolEquivalenceClass ec, 
			 IDatapoolFactory datapoolFactory)
	{
	    DatapoolTableConstructor(parent, datapoolPart, ec, datapoolFactory);
	}

	/**
	 * @param parent
	 * @param datapoolPart
	 * @param datapool
	 * @param datapoolFactory
	 */
	private void DatapoolTableConstructor(Composite parent, IDatapoolPart datapoolPart, IDatapoolEquivalenceClass ec, IDatapoolFactory datapoolFactory) {
		if(parent == null || datapoolPart == null || datapoolFactory == null)
			return;
		setEquivalenceClass(ec);
		this.datapoolFactory = datapoolFactory;
		
		this.parent = new Composite(parent, SWT.NONE);
		this.parent.setLayoutData(new GridData(GridData.FILL_BOTH));
		GridLayout layout = new GridLayout(1, true);
		layout.marginHeight = layout.marginWidth = 0;
		this.parent.setLayout(layout);
		
		try{
			
			setWaitCursor();
			setVendorConfiguration();
			
			this.datapoolEditorPart = datapoolPart;
			
			datapoolClipboard = new DatapoolClipboard(datapoolPart.getClipboard());
			
			createTable(this.parent);
			tableUtil = new DatapoolTableUtil(this);
	
			createMenu();
			table.addSelectionListener(this);
			if(equivalenceClass != null)
				getDatapool().addDatapoolListener(this);
			this.datapoolMenuManager = new DatapoolMenuManager(this, vendorID, false, showVariables, showRecords);
			setMenuMode(datapoolMenuManager);
		} 
        finally {
        	unsetWaitCursor();
        }
	}
	
	/**
	 * Registers Find/Replace global action. 
	 * This requires datapool table to be instantiated thus cannot be done inside the constructor.
	 */
	public void init()
	{
		// https://bugs.eclipse.org/bugs/show_bug.cgi?id=57612
		if(this.datapoolMenuManager == null)
			this.datapoolMenuManager = new DatapoolMenuManager(this, vendorID, false, showVariables, showRecords);
	}

	/**
	 * Creates the context menu.
	 */
	private void createMenu() {
		
		MenuManager menuManager = new MenuManager("#DatapoolEditor"); //$NON-NLS-1$
		menuManager.setRemoveAllWhenShown(true);
		menuManager.addMenuListener(new IMenuListener(){
		
			/* (non-Javadoc)
			 * @see org.eclipse.jface.action.IMenuListener#menuAboutToShow(org.eclipse.jface.action.IMenuManager)
			 */
			public void menuAboutToShow(IMenuManager menuManager) {
				
				datapoolMenuManager.addActions(menuManager);
				
				if(headerSelectionIndex != -1){
					setHeaderMenuMode(datapoolMenuManager);					
				}
				else{
					setMenuMode(datapoolMenuManager);
				}
			}
		});

		contextMenu = menuManager.createContextMenu(table);
		
		table.setMenu(contextMenu);
		table.addListener(SWT.MenuDetect, new Listener(){
			
			/* (non-Javadoc)
			 * @see org.eclipse.swt.widgets.Listener#handleEvent(org.eclipse.swt.widgets.Event)
			 */
			public void handleEvent(Event event) {
				
				Point point = Display.getCurrent().map(null, table, new Point(event.x, event.y));				
				Rectangle clientArea = table.getClientArea();

				//Header context menu:
				if((point.y >= clientArea.y) && (point.y < (clientArea.y + table.getHeaderHeight()))){
					headerSelectionIndex = getColumnIndex(point);					
				}
				
				//Cell context menu:
				else{
					headerSelectionIndex = -1;
				}
			}
		});
	}

	/**
	 * Resolves the column index of the selection{@link Point}.
	 * 
	 * @param point The section point.
	 * @return The column index of the selection{@link Point}.
	 */
	private int getColumnIndex(Point point){
		
		//Resolve the first row:
		TableItem tableItem = table.getItem(0);
		
		for (int counter = 0; counter < table.getColumnCount(); counter++) {
			
			Rectangle clientArea = tableItem.getBounds(counter);

			if((point.x >= clientArea.x) && (point.x <= (clientArea.x + clientArea.width))){
				return counter;
			}
		}
		
		return -1;
	}

	/**
	 * Sets the mode for the context menu.
	 * 
	 * @param datapoolMenuManager Datapool menu manager.
	 */
	private void setMenuMode(DatapoolMenuManager datapoolMenuManager){
		
		if(datapoolMenuManager != null){
						
			int displayMode = 0;

			if((equivalenceClass != null)){

				boolean isReadOnly = datapoolEditorPart.isReadOnly();
				int columnCount = -1;
				int columnSelectionIndex = -1;
				int rowCount = -1;
				int rowSelectionCount = -1;

				if(!table.isDisposed()){

					columnCount = table.getColumnCount();
					rowCount = table.getItemCount();
					rowSelectionCount = getRowSelectionCount();
				}

				if ((tableCursor != null) && (!tableCursor.isDisposed())) {				
					columnSelectionIndex = tableCursor.getColumn();
				}

				//Enable the insert/edit/delete row/variable actions when the editor is opened in read/write mode:
				if(!isReadOnly){

					//Enable the insert row action when the editor is opened in read/write mode and zero/one row is selected:
					if(rowSelectionCount <= 1){
						displayMode |= DatapoolMenuManager.INSERT_ROW_ACTION_ENABLED;
					}
					
					//Enable the edit row action when the editor is opened in read/write mode and one row is selected:
					if(rowSelectionCount == 1){						
						displayMode |= DatapoolMenuManager.EDIT_ROW_ACTION_ENABLED;
					}
					
					//Enable the delete row(s) action when the editor is opened in read/write mode and one or more rows are selected:
					if(rowSelectionCount >= 1){
						displayMode |= DatapoolMenuManager.DELETE_ROW_ACTION_ENABLED;
					}

					//Enable the insert variable action when the editor is opened in read/write mode:
					displayMode |= DatapoolMenuManager.INSERT_COLUMN_ACTION_ENABLED;

					//Enable the edit/delete variable actions when the editor is opened in read/write mode and a variable column (a single cell) is selected:
					if (columnSelectionIndex > 0) {

						//Enable the edit variable action when the editor is opened in read/write mode and a variable column (a single cell) is selected:
						displayMode |= DatapoolMenuManager.EDIT_COLUMN_ACTION_ENABLED;

						//Enable the delete cell action when the editor is opened in read/write mode and a variable column (a single cell) is selected:
						displayMode |= DatapoolMenuManager.DELETE_CELL_ACTION_ENABLED;

						//Enable the delete variable action when the editor is opened in read/write mode, a variable column (a single cell) is selected, and there are two or more variable columns:
						if(columnCount > 2){							
							displayMode |= DatapoolMenuManager.DELETE_COLUMN_ACTION_ENABLED;
						}						
					}
				}

				//Enable the cut/copy/paste actions when a column (one or more rows or a single cell) is selected:
				if(columnSelectionIndex > -1){

					//Disable the cut/copy menus when the selected rows/cell are encrypted since password fields 
					//(see SWT.PASSWORD) also disable the cut/copy menus: 
					boolean enableCutCopy = true;

					//One or more rows are selected so check if the datapool contains any encrypted variables:
					if(columnSelectionIndex == 0){
						enableCutCopy = (!DatapoolEncryptManager.containsEncryptedVariable(getDatapool()));
					}

					//A single cell is selected so check if the variable is encrypted:
					else {						
						enableCutCopy = (!DatapoolEncryptManager.isVariableEncrypted(((IDatapoolVariable)(table.getColumn(columnSelectionIndex).getData(DatapoolTableUtil.TAG_VARIABLE)))));
					}

					//Enable the cut/copy actions when an unencrypted column (one or more rows or a single cell) is selected:
					if(enableCutCopy){

						//Enable the cut action when an unencrypted column (one or more rows or a single cell) is selected and the editor is opened in read/write mode:
						if(!isReadOnly){							
							displayMode |= DatapoolMenuManager.CUT_ACTION_ENABLED;
						}

						//Enable the copy action when an unencrypted column (one or more rows or a single cell) is selected:
						displayMode |= DatapoolMenuManager.COPY_ACTION_ENABLED;
					}

					//Enable the paste action when a column (one or more rows or a single cell) is selected, the editor is opened in read/write mode, and the datapool clipboard is not empty:
					if((!isReadOnly) && (!datapoolClipboard.isEmpty())){
						displayMode |= DatapoolMenuManager.PASTE_ACTION_ENABLED;					
					}
				}
				
				//Enable the find and replace/select all actions when there are one or more rows:
				if(rowCount > 0){
					
					//Enable the find and replace action when there are one or more rows:
					displayMode |= DatapoolMenuManager.FIND_REPLACE_ACTION_ENABLED;

					//Enable the find and replace action when there are one or more rows and not all rows are selected:
					if(rowSelectionCount < rowCount){
						displayMode |= DatapoolMenuManager.SELECT_ALL_ACTION_ENABLED;
					}
				}
			}

			datapoolMenuManager.setDisplayMode(displayMode);			
		}
	}

	/**
	 * Sets the mode for the header context menu.
	 * 
	 * @param datapoolMenuManager Datapool menu manager.
	 */
	private void setHeaderMenuMode(DatapoolMenuManager datapoolMenuManager){
		
		if(datapoolMenuManager != null){
						
			int displayMode = 0;

			//Enable the insert/edit/delete variable actions when the editor is opened in read/write mode:
			if(equivalenceClass != null){

				boolean isReadOnly = datapoolEditorPart.isReadOnly();
				int rowCount = -1;
				int rowSelectionCount = -1;

				if(!table.isDisposed()){

					rowCount = table.getItemCount();
					rowSelectionCount = getRowSelectionCount();
				}

				//Enable the insert/edit/delete variable actions when the editor is opened in read/write mode:
				if(!isReadOnly){

					//Enable the insert variable action when the editor is opened in read/write mode:
					displayMode |= DatapoolMenuManager.INSERT_COLUMN_ACTION_ENABLED;

					//Enable the edit/delete variable actions when the editor is opened in read/write mode and a variable column (a single cell) is selected:
					if(headerSelectionIndex > 0){

						displayMode |= DatapoolMenuManager.EDIT_COLUMN_ACTION_ENABLED;

						//Enable the delete variable action when the editor is opened in read/write mode, a variable column (a single cell) is selected, and there are two or more variable columns:
						if((!table.isDisposed()) && (table.getColumnCount() > 2)){
							displayMode |= DatapoolMenuManager.DELETE_COLUMN_ACTION_ENABLED;
						}						
					}
				}

				//Enable the find and replace/select all actions when there are one or more rows:
				if(rowCount > 0){

					//Enable the find and replace action when there are one or more rows:
					displayMode |= DatapoolMenuManager.FIND_REPLACE_ACTION_ENABLED;

					//Enable the find and replace action when there are one or more rows and not all rows are selected:
					if(rowSelectionCount < rowCount){
						displayMode |= DatapoolMenuManager.SELECT_ALL_ACTION_ENABLED;
					}
				}
			}

			datapoolMenuManager.setDisplayMode(displayMode);			
		}
	}
	
	/* 
	 * Creates the table widget and viewer.  Populates the table from the datapool.
	 */
	private void createTable(Composite parent)
	{					
		table = new Table(parent, SWT.MULTI | SWT.BORDER | SWT.FULL_SELECTION | SWT.VIRTUAL);
		viewer = new TableViewer(table);

		table.setLinesVisible(true);		
		table.setHeaderVisible(true);
		GridData gd = new GridData();
		gd.horizontalAlignment = GridData.FILL;
		gd.grabExcessHorizontalSpace = true;
		gd.verticalAlignment = GridData.FILL;
		gd.grabExcessVerticalSpace = true;
		gd.widthHint = MAX_TABLE_WIDTH;
		gd.heightHint = MAX_TABLE_HEIGHT;
		table.setLayoutData(gd);
		
		
		/*TableLayout layout = new TableLayout();
		layout.addColumnData(new ColumnPixelData(100, false));
		table.setLayout(layout);*/
				
		// Populate table		
		//if(isWithoutColumns())
			//makeHeaderColumn();	
		//else
			makeColumns();
		
		if(!isWithoutRows())
			makeRows();

		createCursor();
			
		// Show the TableCursor when the user releases the "SHIFT" or "CTRL" key.
		// This signals the end of the multiple selection task.
		table.addKeyListener(new KeyAdapter() {
			
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.KeyAdapter#keyReleased(org.eclipse.swt.events.KeyEvent)
			 */
			public void keyReleased(KeyEvent e) {
				
				if (((e.keyCode == SWT.CTRL) && ((e.stateMask & SWT.SHIFT) != 0)) || 
						((e.keyCode == SWT.SHIFT) && ((e.stateMask & SWT.CTRL) != 0)) ||
						((e.keyCode != SWT.CTRL) && ((e.stateMask & SWT.CTRL) != 0)) ||
						((e.keyCode != SWT.SHIFT) && ((e.stateMask & SWT.SHIFT) != 0))){
					return;
				}

				if ((e.keyCode == SWT.SHIFT) || ((e.stateMask & SWT.SHIFT) != 0)){ 
					terminateMultipleSelection();
				}
			}
		});
		
		table.addMouseListener(new MouseAdapter() {
			
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.MouseAdapter#mouseDown(org.eclipse.swt.events.MouseEvent)
			 */
			public void mouseDown(MouseEvent e) {
				
				//Required to terminate the multiple selection when the mouse is clicked on the able when the Shift key is pressed:
				if ((e.stateMask & SWT.SHIFT) != 0){ 
					terminateMultipleSelection();
				}
			}
		});
		
		table.getAccessible().addAccessibleListener(new AccessibleAdapter() {
			public void getName(AccessibleEvent e) {
				if(e.childID == ACC.CHILDID_MULTIPLE)
				{
					TableItem[] selection = table.getSelection();
					if(selection != null && selection.length > 0 && 
					   tableCursor != null && !tableCursor.isDisposed())
					{
						try
						{
							e.result = selection[0].getText(tableCursor.getColumn());
						}
						catch (Exception ex) {}
					}
				}
				else
					e.result = UiPluginResourceBundle.W_DATATABLE; 
			}
		});		
		
		/*if(equivalenceClass == null)
			hideTable();*/
	}

	/*
	 * Creates the table cursor and its associated event listeners.
	 */
	private void createCursor() 
	{
		if(equivalenceClass == null || equivalenceClass.getRecordCount() == 0)
			return;
		tableCursor = new TableCursor(table, SWT.NONE);
		controlEditor = new ControlEditor(tableCursor);
		controlEditor.grabHorizontal = true;
		controlEditor.grabVertical = true;
		
		tableCursor.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_LIST_SELECTION));
		tableCursor.setForeground(Display.getCurrent().getSystemColor(SWT.COLOR_LIST_SELECTION_TEXT));		
		tableCursor.setMenu(contextMenu);
		tableCursor.addSelectionListener(new CursorSelectionAdapter(tableCursor, cellEditor, datapoolEditorPart));
		tableCursor.addMenuDetectListener(new CursorMenuDetectAdapter());
		tableCursor.addKeyListener(new CursorKeyAdapter(tableCursor, cellEditor, datapoolEditorPart));
		tableCursor.addTraverseListener(new CursorTraverseAdapter());
		tableCursor.addMouseListener(new CursorMouseAdapter(table, tableCursor, cellEditor, datapoolEditorPart));
		tableCursor.getAccessible().addAccessibleListener(new AccessibleAdapter() {
			public void getName(AccessibleEvent e) {
				TableItem[] selection = table.getSelection();
				if(selection != null && selection.length > 0 && 
				   tableCursor != null && !tableCursor.isDisposed())
				{
					try
					{
						e.result = selection[0].getText(tableCursor.getColumn());
					}
					catch (Exception ex) {}
				}
			}
		});
		
		if(table.getItemCount() > 0)
		{
			table.select(0);
			if(table.getColumnCount() > 1)
			    tableCursor.setSelection(0,1);
			else
			    tableCursor.setSelection(0, 0);
		}
	}
	
	private void resetCursor()
	{
		if(tableCursor == null || tableCursor.isDisposed())
			createCursor();
		else
		{
			if(table.getItemCount() > 0)
			{
				table.select(0);
				tableCursor.setSelection(0,0);
				tableCursor.redraw();
			}
		}
	}
	
	/*
	 * Creates and displays a wait cursor.
	 */
	private void setWaitCursor()
	{
		if(parent == null)
			return;
		Display display = parent.getShell().getDisplay();
		Cursor waitCursor = new Cursor(display, SWT.CURSOR_WAIT);
		setDisplayCursor(waitCursor);		
	}

	/*
	 * Cancels the display of the wait cursor.
	 */
	private void unsetWaitCursor()
	{
		setDisplayCursor(null);
	}

	/*
	 * Creates and sets the cursor to c.
	 */
	private void setDisplayCursor(Cursor c) {
		if(parent == null)
			return;
		Display display = parent.getShell().getDisplay();
		Shell[] shells = display.getShells();
		for (int i = 0; i < shells.length; i++)
			shells[i].setCursor(c);
	}
	
	/**
	 * Returns the 0 based row and column index of the cursor's position as a point, 
	 * or [0, 0] as the default if cursor is not available.
	 */
	public Point getCursorPosition()
	{
		if(tableCursor != null && !tableCursor.isDisposed())
		{
			TableItem row = tableCursor.getRow();
			TableItem[] items = table.getItems();
			for(int i = 0; i < items.length; i++)
			{
				if(items[i] == row)
					return new Point(i, tableCursor.getColumn());
			}
		}	
		
		return new Point(0,0);
	}
	
	/**
	 * @param row
	 * @param column
	 */
	private void startCellEditing(TableItem row, int column) {
		IDatapoolCell[] cells = (IDatapoolCell[])row.getData(DatapoolTableUtil.TAG_DATA);
		if(column < 1 || column > cells.length)
			return;
		IDatapoolCell cell = cells[column - 1];

        if (cell == null) return;
		IDatapoolVariable variable = (IDatapoolVariable)cell.getCellVariable();
		IDatapoolSuggestedType suggestedType = (IDatapoolSuggestedType)variable.getSuggestedType();
		Object rawValue = cell.getCellValue();
		rawValue = (rawValue == null) ? "" : rawValue; //$NON-NLS-1$
	
		//Create a combo box cell editor for the enumeration suggested type:
		if(suggestedType.getSuggestedType() == IDatapoolSuggestedType.TYPE_ENUMERATION){			
			
			String[] enumerationLiterals = suggestedType.getEnumerationLiterals();
			
			if((enumerationLiterals == null) || (enumerationLiterals.length == 0)){
				enumerationLiterals = new String[]{};
			}
				
			String rawValueString = String.valueOf(rawValue);
			
			cellEditor = new ComboBoxCellEditor(tableCursor, enumerationLiterals);				
			
			//Set the text and move the cursor to the end:
			CCombo cellEditorCombo = ((CCombo)(cellEditor.getControl()));
			cellEditorCombo.setText(rawValueString);
			cellEditorCombo.setSelection(new Point(rawValueString.length(), rawValueString.length()));				
			
			theValueObject = new ValueObject(rawValue);
		}

		//Create a combo box cell editor for the boolean suggested type:
		else if(suggestedType.getSuggestedType() == IDatapoolSuggestedType.TYPE_BOOLEAN){
			
			String rawValueString = String.valueOf(rawValue);
			
			cellEditor = new ComboBoxCellEditor(tableCursor, new String[]{"true", "false"}, SWT.READ_ONLY); //$NON-NLS-1$  //$NON-NLS-2$		
			
			//Set the text and move the cursor to the end:
			CCombo cellEditorCombo = ((CCombo)(cellEditor.getControl()));
			cellEditorCombo.setText(rawValueString);
			cellEditorCombo.setSelection(new Point(rawValueString.length(), rawValueString.length()));				
			
			theValueObject = new ValueObject(rawValue);			
		}
		
		if(cellEditor == null){
			
			if((rawValue == null) || (rawValue == "")){	//$NON-NLS-1$
			
				String typeName = suggestedType.getSuggestedClassName();					
				
				if ((typeName == null) || (typeName.trim().length() == 0)){
					rawValue = new String();
				} 
				else {
					rawValue = createEmptyCellObject(typeName);
				}
			}
			if(DatapoolEncryptManager.isVariableEncrypted(variable))
			{
				String key = null;
				if(password != null){
					key = password;
					cellkey = key;
				}
				else{
					
					DatapoolInputKeyDialog datapoolInputKeyDialog = new DatapoolInputKeyDialog(
							Display.getCurrent().getActiveShell(),
							datapool);
	
					if(datapoolInputKeyDialog.open() == IDialogConstants.OK_ID) {				
						
						key = datapoolInputKeyDialog.getKey();
						cellkey = key;
						password = key;
					}
					else{
						return;
					}
				}
				rawValue = EncryptionManager.decrypt(rawValue.toString(), key);
				theValueObject = new EncryptedValueObject(rawValue);
				cellEditor = (CellEditor) theValueObject.getEncryptedDisplay(tableCursor);
			}else{
				theValueObject = new ValueObject(rawValue);
				cellEditor = (CellEditor) theValueObject.getPropertyDisplay(tableCursor);
			}
		}
		
		if(cellEditor != null && cellEditor.getControl() != null)
		{
			cellEditor.getControl().addKeyListener(new TableCellKeyAdapter(cellEditor, theValueObject, row, column));
			cellEditor.getControl().addTraverseListener(new TableCellTraverseAdapter(cellEditor, theValueObject, row, column));
			cellEditor.getControl().addPaintListener(new TableCellPaintAdapter());
			cellEditor.getControl().setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_LIST_BACKGROUND));
			cellEditor.getControl().setForeground(Display.getCurrent().getSystemColor(SWT.COLOR_LIST_FOREGROUND));
			cellEditor.addListener(new TableCellEditorAdapter(cellEditor, theValueObject, row, column));				
			controlEditor.setEditor(cellEditor.getControl());
			cellEditor.setValidator(new TableCellValidator(ValueClassMap.getValueValidatorClass(rawValue)));
			cellEditor.setFocus();
			
			if(cellEditor instanceof TextCellEditor)
			{
				String currentText = row.getText(column);
				((Text)(cellEditor.getControl())).setSelection(currentText.length());												
			}
			
			this.datapoolEditorPart.markDirty();
		}
	}

	/**
	 * @param typeName
	 * @return
	 */
	private Object createEmptyCellObject(String typeName) 
	{
		Object value;
		IValueClassFactory valueFactory = ValueClassMap.getValueClassFactory(typeName);
		if(valueFactory != null)
			value = valueFactory.createEmptyObject();
		else
		{						
			try
			{
				Class objectClass  = Class.forName(typeName);
				Constructor objectConstructor = objectClass.getConstructor(new Class[0]);
		        value = objectConstructor.newInstance(new Object[0]);		  		    					
			}
			catch(Exception exception)
			{
				value = new String();
			}
		}
		return value;
	}

	/**
	 * Takes the value text from the editor and sets it to the table item as
	 * well as the datapool cell.
	 */ 
	private void applyEditingValue(boolean dispose)	
	{
		if(cellEditor != null)
		{
			if (tableCursor != null && tableCursor.getColumn() > 0)
			{					
				Object updatedValue = null;
				String newDescription = null;
				boolean update = true;
				
				TableItem selectedTableItem = tableCursor.getRow();
				int selectedColumnIndex = tableCursor.getColumn();
				if(selectedTableItem == null || selectedTableItem.isDisposed())
					return;
				IDatapoolCell[] rowData = (IDatapoolCell[])selectedTableItem.getData(DatapoolTableUtil.TAG_DATA); 
				IDatapoolCell cell = rowData[tableCursor.getColumn() - 1];
				
				// bugzilla70477/RATLC458847
				// check suggested type first
				IDatapoolSuggestedType suggestedType = (IDatapoolSuggestedType)cell.getCellVariable().getSuggestedType();
				if(suggestedType.getSuggestedType() == IDatapoolSuggestedType.TYPE_ENUMERATION ||
					suggestedType.getSuggestedType() == IDatapoolSuggestedType.TYPE_BOOLEAN)
				{
					updatedValue = ((CCombo)cellEditor.getControl()).getText();
					newDescription = updatedValue.toString();
				}
				else
				{
				
					if(theValueObject == null)
						update = false;
					else
					{
						String oldDescription = theValueObject.getDescription();
						updatedValue = theValueObject.updateObject();
						newDescription = theValueObject.getDescription();
						
						// only make changes when the value has changed.		
						if((newDescription == null && oldDescription == null) ||
						   (newDescription != null && newDescription.equals(oldDescription)))
						{
							update = false;
						}
					}
				}
				
				if (update) {
					if(DatapoolEncryptManager.isVariableEncrypted(cell.getCellVariable()))
					{
						String value = String.valueOf(updatedValue);
						String encryptValue = DatapoolEncryptManager.encrypt(value, cellkey);
						updatedValue = (Object)encryptValue;
						
					}
					cell.setCellValue(updatedValue);
					selectedTableItem.setText(selectedColumnIndex, newDescription);
					tableCursor.setSelection(table.getSelectionIndex(), selectedColumnIndex);
					datapoolEditorPart.markDirty();
				}
			}
			
			if(dispose)
				refreshRows();
				clearCellEditor();
		}						
	}

	/**
	 * Creates TableColumn objects from the DPLVariable objects in the 
	 * datapool.  The indexes are shifted by one because there is a 
	 * column for the row headers.
	 */
	private void makeColumns(){
		
		makeHeaderColumn();
		
		IDatapool datapool = getDatapool();
		
		if (datapool != null){
			
			for(int counter = 0; counter < datapool.getVariableCount(); counter++){
				
				IDatapoolVariable variable = ((IDatapoolVariable)(datapool.getVariable(counter)));
	            String type = DatapoolUtil.getInstance().getVariableTypeInfo(variable);
				String tableColumnText = variable.getName();
	    		
	    		if((type != null) && (type.trim().length() > 0)){	
	    			tableColumnText = (variable.getName() + DatapoolConstants.VARIABLE_NAME_TYPE_DELIMITER + type).trim();
	    		}
	
				TableColumn tableColumn = new TableColumn(table, SWT.LEFT, counter + 1);
				tableColumn.setResizable(true);
	    		tableColumn.setText(tableColumnText);  
				tableColumn.setData(DatapoolTableUtil.TAG_VARIABLE, variable); 
				tableColumn.setWidth(COLUMN_DEFAULT_WIDTH);
				tableColumn.addSelectionListener(headerListener);	
				tableColumn.addControlListener(resizeColumnListener);
				
				if(DatapoolEncryptManager.isVariableEncrypted(variable)){
					tableColumn.setImage(TestUIImages.INSTANCE.getImage(TestUIImages.IMG_DATAPOOL_ENCRYPTED_VARIABLE));
				}
			}
		}
    }

	/**
	 * Creates the column header.
	 */
	private void makeHeaderColumn() {
		TableColumn recordTableColumn = new TableColumn(table, SWT.CENTER, 0);
		recordTableColumn.setResizable(true);
		recordTableColumn.setWidth(HEADER_COLUMN_DEFAULT_WIDTH);
		recordTableColumn.setData(DatapoolTableUtil.TAG_VARIABLE, null);
		recordTableColumn.addControlListener(resizeColumnListener); 
	}

	/**
	 * Creates TableItem objects from the DPLEquivalenceClass and the IDatapoolRecord
	 * objects in the datapool.  Records do not need to have cells for every
	 * variable.
	 */
	private void makeRows(){

		if(equivalenceClass != null){

			int variableCount = getDatapool().getVariableCount();

			for(int recordIndex = 0; recordIndex < equivalenceClass.getRecordCount(); recordIndex++){

				IDatapoolCell[] rowData = new IDatapoolCell[variableCount];
				String rowContents[] = new String[variableCount + 1];
				rowContents[0] = String.valueOf(recordIndex);

				IDatapoolRecord record = ((IDatapoolRecord)(equivalenceClass.getRecord(recordIndex)));

				for(int cellIndex = 0; cellIndex < record.getCellCount(); cellIndex++){

					IDatapoolCell cell = (IDatapoolCell)(record.getCell(cellIndex));
					
					String cellDescription = new ValueObject(cell.getCellValue()).getDescription();
					
					//Replace the encrypted text (hex characters) with asterisks:
					//Note: Each decrypted character is encrypted as two hex characters so every two characters are replaced with one asterisk.
					if((DatapoolEncryptManager.isVariableEncrypted(cell.getCellVariable())) && (cellDescription != null) && (cellDescription.length() > 0)){
						cellDescription = cellDescription.replaceAll(".{2}", "*"); //$NON-NLS-1$ //$NON-NLS-2$
					}
					
					//Assumption: There is a cell for each variable of the record making the cell index the variable index.
					rowData[cellIndex] = cell;

					//Increment the cell index to account for the first column:
					rowContents[cellIndex + 1] = cellDescription;
				}				

				TableItem item = new TableItem(table, SWT.NULL);
				item.setText(rowContents);
				item.setData(DatapoolTableUtil.TAG_DATA, rowData);
				item.setData(DatapoolTableUtil.TAG_RECORD, record); 
				item.setData(DatapoolTableUtil.TAG_EQUIVALENCE_CLASS, equivalenceClass); 
			}
		}
	}

	/**
	 * Checks to see if the datapool is empty of equivalence classes causing
	 * the table to be without rows.
	 */
	private boolean isWithoutRows()
	{
		if(equivalenceClass == null || equivalenceClass.getRecordCount() == 0)
			return true;
		else
			return false;
	}

//	/**
//	 * Checks to see if the datapool is empty of variables causing the table 
//	 * to be without columns.
//	 */
//	private boolean isWithoutColumns()
//	{
//		if(equivalenceClass == null)
//			return true;
//		
//		IDatapool datapool = getDatapool();
//		if(datapool.getVariableCount() == 0)
//			return true;
//		else
//			return false;
//	}
	
	/**
	 * Finds the column index from the varible's ID.
	 */
	private int findColumnIndex(String name)
	{
		//Start at second column to account for the first empty column:
		for(int i = 1; i < table.getColumnCount(); i++)
		{
			TableColumn tableColumn = table.getColumn(i);
			IDatapoolVariable variable = (IDatapoolVariable)tableColumn.getData(DatapoolTableUtil.TAG_VARIABLE); 
			if(variable != null && name.equals(variable.getName()))
				return i;	
		}	
		return -1;
	}	

	// SelectionListener
	
	/** 
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e) 
	{
		int selectedRowIndex = table.getSelectionIndex();
		if(tableCursor != null && !tableCursor.isDisposed())
		{
			int selectedColumnIndex = tableCursor.getColumn();
            int columnCount = table.getColumnCount();
            int maxColumnIndex =  columnCount == 0 ? 0 : columnCount - 1;
			if (selectedColumnIndex >= 0 && !(selectedRowIndex < 0 || selectedRowIndex >= table.getItemCount() || selectedColumnIndex < 0 || selectedColumnIndex > maxColumnIndex))
				tableCursor.setSelection(selectedRowIndex, selectedColumnIndex);
		}		
	}

	/** 
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e){
		//No-operation.
	}
				
	/**
	 * Inserts a new record/row to the table before the current record.
	 */
	public void insertRow()
	{
		insertRowAfter(0);
	}
    
    /**
     * Adds a new record/row to the table after the current record.
     */
    public void addRow() {
        insertRowAfter(1);
    }
    
    private void insertRowAfter(int offset) {
        if (equivalenceClass == null) return;
        int newRowIndex = table.getSelectionIndex() + offset;
        if (newRowIndex > equivalenceClass.getRecordCount() + 1) {
            return;
        }
        if (newRowIndex < 0) {
            newRowIndex = 0;
        }
        
        try{

        	setWaitCursor();
	        IDatapoolRecord record = equivalenceClass.constructRecord();
	        equivalenceClass.insertRecord(record, newRowIndex);
	        refreshRows();
	        tableCursor.redraw();
        } 
        finally {
        	unsetWaitCursor();
        }
    }
	
	/**
	 * Deletes the currently selected record(s)/row(s).  Removes the record(s) from the 
	 * datapool object.  Removes the row(s) from the table.
	 */
	public void deleteRow()
	{
		if(showRecords == false)
			return;	
		
		int[] selectionIndices = table.getSelectionIndices();
		if(selectionIndices.length == 0)
			return;
		
		try {
			
			setWaitCursor();
			
			//Assumption: The refreshRows() method is always called after the record(s) are removed.
			//Note: When each record is removed, the row number labels are re-populated, which is a 
			//time consuming for large datapools).  However, the row number labels are already 
			//re-populated by the refreshRows() method. 
			//Set the flag denoting that the refreshRows() method is scheduled to be invoked:  
			refreshRowsScheduled = true;
			
			//Sort the cells in order since the order of the indices is unspecified:
			Arrays.sort(selectionIndices);
	
			//Remove the records from the datapool object and rows from the table in order (accounting for the previously removed record(s)/row(s)) to ensure the table cursor is updated correctly (see defect #252958):
			for (int counter = 0; counter < selectionIndices.length; counter++){
				equivalenceClass.removeRecord(selectionIndices[counter] - counter);
			}
			
			refreshRows();
			if (tableCursor != null && !tableCursor.isDisposed()) {
	            tableCursor.redraw();
	        }
		} 
        finally {
        	unsetWaitCursor();
        }
	}

	/*
	 * Edits the currently selected record/row.  Moves the record in the 
	 * datapool object.  Updates the table to reflect move.
	 */
	public void editRow()
	{
		if((tableCursor != null) && (!tableCursor.isDisposed())){
			editRowAux(tableCursor.getRow());
		}
	}

	/*
	 * Function that actually does the editing of the record/row.
	 */
	private void editRowAux(TableItem tableItem){

		if((showRecords) && (getEquivalenceClass().getRecordCount() > 0)){

			IDatapoolRecord previousRecord = null;			
			int rowIndex = table.getSelectionIndex();
			
			if(rowIndex != 0){
				
				TableItem previousTableItem = table.getItem(rowIndex - 1);
				
				if(previousTableItem != null){
					previousRecord = ((IDatapoolRecord)(previousTableItem.getData(DatapoolTableUtil.TAG_RECORD)));
				}
			}
			
			DatapoolRowDialog dialog = new DatapoolRowDialog(Display.getCurrent().getActiveShell(), ((IDatapoolRecord)(tableItem.getData(DatapoolTableUtil.TAG_RECORD))), previousRecord);    
	
			if (dialog.open() == IDialogConstants.OK_ID){
	
				try{
					
					setWaitCursor();
					
					IDatapoolRecord record = ((IDatapoolRecord)(tableItem.getData(DatapoolTableUtil.TAG_RECORD)));
					int recordIndex = getRecordIndex(equivalenceClass, record);
					int insertionRecordIndex = dialog.getInsertionRecordIndex();

					if(recordIndex != insertionRecordIndex){

						if (insertionRecordIndex == -1){
							equivalenceClass.moveRecord(recordIndex, 0);
						}
						else if (insertionRecordIndex > recordIndex){
							equivalenceClass.moveRecord(recordIndex, insertionRecordIndex);
						}
						else{
							equivalenceClass.moveRecord(recordIndex, insertionRecordIndex + 1);	
						}
					}			
				} 
				catch(Exception e){
					e.printStackTrace();
				}
				finally {
					unsetWaitCursor();
				}
			}
		}
	}

	/*
	 * Inserts a new variable/column to the table.  Brings up a dialog to 
	 * gather information about the variable/column.  Adds the variable to the 
	 * datapool object.  Inserts a column in the table.
	 */
	public void insertColumn()
	{
		if(showVariables == false){
			return;
		}
			
		if(table.getColumnCount() >= DatapoolConstants.MAXIMUM_VARIABLE_LIMIT){

			MessageDialog.openWarning(Display.getCurrent().getActiveShell(), UiPluginResourceBundle.DatapoolEditor_Variables, NLS.bind(UiPluginResourceBundle.DATA_COL_DLG_ERROR_VARIABLE_LIMIT_EXCEEDED, DatapoolConstants.MAXIMUM_VARIABLE_LIMIT));
			
			return;
		}       
		
		int selectedColumnIndex = -1;
		
		if(headerSelectionIndex != -1){
			
			selectedColumnIndex = headerSelectionIndex;
			
			headerSelectionIndex = -1;
		}
		else if(tableCursor != null && !tableCursor.isDisposed()){
			selectedColumnIndex = tableCursor.getColumn();
		}
		
		TableColumn previousTableColumn = null;
		if(selectedColumnIndex > 0)
			previousTableColumn = table.getColumn(selectedColumnIndex - 1);
		IDatapoolVariable previousVariable = null;
		if(previousTableColumn != null)
			previousVariable = (IDatapoolVariable)previousTableColumn.getData(DatapoolTableUtil.TAG_VARIABLE);

		IDatapool datapool = getDatapool();
		
		DatapoolColumnDialog datapoolColumnDialog = new DatapoolColumnDialog(Display.getCurrent().getActiveShell(), 
				datapool,
				password,
				null, 
				previousVariable);
		
		if (datapoolColumnDialog.open() == IDialogConstants.OK_ID) {

			String newKey = datapoolColumnDialog.getNewKey();
			
			if(newKey != null){
				
				password = newKey;
				
				String oldKey = datapoolColumnDialog.getOldKey();
				
				if(oldKey != null){	
					DatapoolEncryptManager.changeKeyOfEncryptedCell(datapool, oldKey, password);
				}
				
				DatapoolEncryptManager.changeKey(password, datapool);
			}
						
			try{
			
				setWaitCursor();
				IDatapoolVariable variable = datapool.constructVariable();
				variable.setName(datapoolColumnDialog.getName());
				IDatapoolSuggestedType suggestedType = (IDatapoolSuggestedType)variable.getSuggestedType();
				DatapoolSuggestedTypeChecker.getInstance().setVariableType(suggestedType, datapoolColumnDialog.getType());
				variable.setSuggestedType(suggestedType);
				
				if(variable instanceof DPLVariable){
					((DPLVariable)(variable)).setEncrypted(datapoolColumnDialog.isEncrypted());
				}
				
				int insertionIndex = findColumnIndex(datapoolColumnDialog
						.getInsertionVariableName());
				if(insertionIndex == -1)
					insertionIndex = 0;
	
				datapool.insertVariable(variable, insertionIndex);
	            int row = table.getSelectionIndex() > -1 ? table.getSelectionIndex() : 0;
	            int columnCount = table.getColumnCount();
	            int maxColumnIndex =  columnCount == 0 ? 0 : columnCount - 1;
				if (tableCursor != null && !tableCursor.isDisposed() && !(row < 0 || row >= table.getItemCount() || insertionIndex < 0 || insertionIndex > maxColumnIndex)) {
	                tableCursor.setSelection(row, insertionIndex + 1);			
	            }
				refreshRows();
			} 
	        finally {
	        	unsetWaitCursor();
	        }
		}
	}

	/*
	 * Deletes the currently selected variable/column.  Removes the column and 
	 * all associated cells from the datapool object.  Removes the column from 
	 * the table.
	 */
	public void deleteColumn()
	{
		if(showVariables == false)
			return;	
		
		IDatapool datapool = getDatapool();
		if(tableCursor == null || tableCursor.isDisposed())
		{
			if(table.getItemCount() == 0 && tableUtil.getColumnCount() > 0)
			{
				DatapoolDeleteColumnDialog dialog = new DatapoolDeleteColumnDialog(Display.getCurrent().getActiveShell(), datapool);
				if ( dialog.open() == IDialogConstants.OK_ID)
				{
					
					try{

						setWaitCursor();
						String variableID = dialog.getDeletedVariableID();
						int variableIndex = datapool.getVariableIndexById(variableID);
						datapool.removeVariable(variableIndex);
					} 
			        finally {
			        	unsetWaitCursor();
			        }
				}			
			}
			else
				return;
		}
		else{
			
			int selectedColumnIndex = -1;
			
			if(headerSelectionIndex != -1){
				
				selectedColumnIndex = headerSelectionIndex;
				
				headerSelectionIndex = -1;
			}
			else if(tableCursor != null && !tableCursor.isDisposed()){
				selectedColumnIndex = tableCursor.getColumn();
			}
	
			// Don't delete the row name column.
			if (selectedColumnIndex < 1)	
				return;
					
			try{
			
				setWaitCursor();
				TableColumn tableColumn = table.getColumn(selectedColumnIndex);
				IDatapoolVariable variable = (IDatapoolVariable)tableColumn.getData(DatapoolTableUtil.TAG_VARIABLE);
				int variableIndex = datapool.getVariableIndexById(variable.getId());
				datapool.removeVariable(variableIndex);
			} 
	        finally {
	        	unsetWaitCursor();
	        }
		}
		
		//Remove the datapool's challenge if there are no encrypted variables:
		if(!DatapoolEncryptManager.containsEncryptedVariable(datapool)){
			
			DatapoolEncryptManager.removeKey(datapool);
			
			password = null;
		}
	}
	
	/*
	 * Edits the currently selected variable/column.  Renames and/or moves the 
	 * variable in the datapool object.  Updates the table to reflect the edit 
	 * and/or move.
	 */
	public void editColumn(){
		
		int selectedColumnIndex = -1;
		
		if(headerSelectionIndex != -1){
			
			selectedColumnIndex = headerSelectionIndex;
			
			headerSelectionIndex = -1;
		}
		else if(tableCursor != null && !tableCursor.isDisposed()){
			selectedColumnIndex = tableCursor.getColumn();
		}
		
		editColumnAux(((TableColumn)(table.getColumn(selectedColumnIndex))));		
	}

	/*
	 * Function that actually does the editing of the variable/column.
	 */	
	private void editColumnAux(TableColumn tableColumn) {

		if (showVariables == false) return;

		IDatapool datapool = getDatapool();
		int columnIndex = table.indexOf(tableColumn);
		TableColumn previousTableColumn = null;
		if (columnIndex != 0) previousTableColumn = table.getColumn(columnIndex - 1);
		IDatapoolVariable previousVariable = null;
		if (previousTableColumn != null) previousVariable = (IDatapoolVariable) previousTableColumn.getData(DatapoolTableUtil.TAG_VARIABLE);
		IDatapoolVariable variable = (IDatapoolVariable) tableColumn.getData(DatapoolTableUtil.TAG_VARIABLE);

		DatapoolColumnDialog datapoolColumnDialog = new DatapoolColumnDialog(Display.getCurrent().getActiveShell(), 
				datapool, 
				password,
				variable, 
				previousVariable); 

		if (datapoolColumnDialog.open() == IDialogConstants.OK_ID) {

			String newKey = datapoolColumnDialog.getNewKey();
			
			if(newKey != null){
				
				password = newKey;
				
				String oldKey = datapoolColumnDialog.getOldKey();
				
				if(oldKey != null){	
					DatapoolEncryptManager.changeKeyOfEncryptedCell(datapool, oldKey, password);
				}
				
				DatapoolEncryptManager.changeKey(password, datapool);
			}
			
			if((datapoolColumnDialog.isEncrypted()) && (!DatapoolEncryptManager.isVariableEncrypted(variable))) {
				
				DatapoolEncryptManager.encryptedCellInVarible(variable, password, datapool);
				
				if(variable instanceof DPLVariable){
					((DPLVariable)(variable)).setEncrypted(true);
				}
			}
			else if((!datapoolColumnDialog.isEncrypted()) && (DatapoolEncryptManager.isVariableEncrypted(variable))){

				DatapoolEncryptManager.decryptedCellInVarible(variable, password, datapool);

				if(variable instanceof DPLVariable){
					((DPLVariable)(variable)).setEncrypted(false);
				}
				
				//Remove the datapool's challenge if there are no encrypted variables:
				if(!DatapoolEncryptManager.containsEncryptedVariable(datapool)){

					DatapoolEncryptManager.removeKey(datapool);

					password = null;
				}
			}
			
			try{

				setWaitCursor();

				String name = datapoolColumnDialog.getName();
				String insertionVariableID = datapoolColumnDialog.getInsertionVariableID();

				IDatapoolSuggestedType suggestedType = (IDatapoolSuggestedType) variable
				.getSuggestedType();
				if (name.equals(variable.getName()) && insertionVariableID.equals(variable.getId())) {
					return;
				}
				variable.setName(datapoolColumnDialog.getName());
				DatapoolSuggestedTypeChecker.getInstance().setVariableType(suggestedType, datapoolColumnDialog.getType());
				variable.setSuggestedType(suggestedType);

				int insertionIndex = findColumnIndex(datapoolColumnDialog.getInsertionVariableName());
				if (insertionIndex == columnIndex - 1) {
					refresh(true);
					return;
				}
				if (insertionIndex == -1)
					datapool.moveVariable(columnIndex - 1, 0);
				else if (insertionIndex > columnIndex)
					datapool.moveVariable(columnIndex - 1, insertionIndex - 1);
				else
					datapool.moveVariable(columnIndex - 1, insertionIndex);
				refreshRows();//needs to refresh row
				tableCursor.setSelection(0, 0);
			}
			finally{
				unsetWaitCursor();
			}
		}
	}

	/* 
	 * Clears the value in cell.
	 */
	public void clearCell()
	{
		clearCellEditor();
		if (tableCursor != null && !tableCursor.isDisposed() && tableCursor.getColumn() > 0)
		{					
			TableItem selectedTableItem = tableCursor.getRow();
			int selectedColumnIndex = tableCursor.getColumn();
			if(selectedTableItem != null && !selectedTableItem.isDisposed())
			{
				IDatapoolCell[] rowData = (IDatapoolCell[])selectedTableItem.getData(DatapoolTableUtil.TAG_DATA); 
				IDatapoolCell cell = rowData[tableCursor.getColumn() - 1];
				Object value = cell.getCellValue();
				if(value == null)
					return;
				
				if(value instanceof String)
				{
					if(((String)value).length() == 0)
						return;
					else
						value = new String();
				}
				else
					value = createEmptyCellObject(value.getClass().getName());
				
				cell.setCellValue(value);
				selectedTableItem.setText(selectedColumnIndex, (new ValueObject(value)).getDescription());
				tableCursor.setSelection(table.getSelectionIndex(), selectedColumnIndex);
				tableCursor.redraw();
				datapoolEditorPart.markDirty();
			}
		}
	}
	
	private void clearRecords(){

		if(getRowSelectionCount() > 0){

			TableItem[] tableItems = table.getSelection();

			if(tableItems != null){

				for (int recordIndex = 0; recordIndex < tableItems.length; recordIndex++) {		

					IDatapoolRecord record = ((IDatapoolRecord)(tableItems[recordIndex].getData(DatapoolTableUtil.TAG_RECORD)));

					if((record != null) && (record.getCellCount() > 0)){

						for(int cellIndex = 0; cellIndex < record.getCellCount(); cellIndex++){

							((IDatapoolCell)(record.getCell(cellIndex))).setCellValue(null);
							tableItems[recordIndex].setText((cellIndex + 1), new String());
						}

						datapoolEditorPart.markDirty();
					}
				}
			}
		}
	}
	
	public IDatapoolCell getSelectedCell(){		
		
		IDatapoolCell selectedCell = null;
		
		if ((tableCursor != null) && (tableCursor.getColumn() > 0)){					
			
			TableItem selectedTableItem = tableCursor.getRow();
			
			if((selectedTableItem != null) && (!selectedTableItem.isDisposed())){
				selectedCell = ((IDatapoolCell[])(selectedTableItem.getData(DatapoolTableUtil.TAG_DATA)))[tableCursor.getColumn() - 1];
			}
		}

		return selectedCell;
	}
	
	public int getRowSelectionCount(){

		if((table != null) && (!table.isDisposed())){
			return (table.getSelectionCount());
		}

		return -1;
	}
	
	// IDatapoolListener 
	// for equivalence class related events, this table should only handles the ones from its own equivalence clas.
	
	/*
	 * The following methods follow a spec that will eventually correspond to a
	 * listener interface that will make callbacks to the Datapool table.  
	 * Based on the callback, the table will update accordingly.  
	 */

	public void variableAdded(IDatapool datapool, int newVariableIndex){
	
		IDatapoolVariable variable = ((IDatapoolVariable)(datapool.getVariable(newVariableIndex)));
        String type = DatapoolUtil.getInstance().getVariableTypeInfo(variable);
		String tableColumnText = variable.getName();
		
		if((type != null) && (type.trim().length() > 0)){	
			tableColumnText = (variable.getName() + DatapoolConstants.VARIABLE_NAME_TYPE_DELIMITER + type).trim();
		}
		
		TableColumn tableColumn = new TableColumn(table, SWT.LEFT, (newVariableIndex + 1));
		tableColumn.setResizable(true);
		tableColumn.setText(tableColumnText);  
		tableColumn.setData(DatapoolTableUtil.TAG_VARIABLE, variable);
		tableColumn.setWidth(COLUMN_DEFAULT_WIDTH);
		tableColumn.addSelectionListener(headerListener);
		tableColumn.addControlListener(resizeColumnListener);		
		
		if(DatapoolEncryptManager.isVariableEncrypted(variable)){
			tableColumn.setImage(TestUIImages.INSTANCE.getImage(TestUIImages.IMG_DATAPOOL_ENCRYPTED_VARIABLE));
		}
		
		tableUtil.insertColumn(tableColumn, newVariableIndex);	

		if(tableCursor != null && !tableCursor.isDisposed())
		{
			int selectedRowIndex = table.getSelectionIndex();
			if(selectedRowIndex >= 0)
			{
				table.setSelection(selectedRowIndex);
				tableCursor.setSelection(selectedRowIndex, newVariableIndex + 1);
				tableCursor.setFocus();
			}

		}
		datapoolEditorPart.markDirty();		
	}	
	
	public void variableRemoved(IDatapool datapool, int variableIndex)
	{
		tableUtil.deleteColumn(variableIndex + 1);

		if(tableCursor != null && !tableCursor.isDisposed())
		{
			//bugzillar#85035: when variableIndex == 0 table selection is lost
			if(variableIndex >= 0)
			{
				int selectedRowIndex = table.getSelectionIndex();
				if(selectedRowIndex >= 0)
				{
					table.select(selectedRowIndex);
					tableCursor.setSelection(selectedRowIndex, variableIndex);
				}
			}
			else
				table.deselectAll();

		}
		datapoolEditorPart.markDirty();		
	}
	
	public void variableMoved(IDatapool datapool, int sourceVariableIndex, int targetVariableIndex)
	{
		tableUtil.moveColumn(sourceVariableIndex + 1, targetVariableIndex + 1);
		if(tableCursor != null && !tableCursor.isDisposed())
			if(targetVariableIndex > 0)
			{
				int selectedIndex = table.getSelectionIndex();
				if(selectedIndex >= 0)
				{
					table.select(selectedIndex);
					tableCursor.setSelection(selectedIndex, targetVariableIndex);
				}
			}
		datapoolEditorPart.markDirty();		
	}
	
	public void variableChanged(IDatapool datapool, int variableIndex){
		
		IDatapoolVariable variable = ((IDatapoolVariable)(datapool.getVariable(variableIndex)));
        String type = DatapoolUtil.getInstance().getVariableTypeInfo(variable);
		String tableColumnText = variable.getName();
		
		if((type != null) && (type.trim().length() > 0)){	
			tableColumnText = (variable.getName() + DatapoolConstants.VARIABLE_NAME_TYPE_DELIMITER + type).trim();
		}
		
		TableColumn tableColumn = table.getColumn(findColumnIndex(variable.getName()));		
		tableColumn.setText(tableColumnText);  
		tableColumn.setData(DatapoolTableUtil.TAG_VARIABLE, variable);
		
		if(DatapoolEncryptManager.isVariableEncrypted(variable)){
			tableColumn.setImage(TestUIImages.INSTANCE.getImage(TestUIImages.IMG_DATAPOOL_ENCRYPTED_VARIABLE));
		}
		else{
			tableColumn.setImage(null);
		}
		
		datapoolEditorPart.markDirty();		
	}
	
	public void variableChanged(IDatapool datapool, int variableIndex, String oldName)
	{
		variableChanged(datapool, variableIndex);
	}
	
	public void equivalenceClassChanged(IDatapool datapool, int equivalenceClassIndex)
	{
		if(getEquivalenceClassIndex() == equivalenceClassIndex)
	        refreshRows();
	}

	public void equivalenceClassChanged(IDatapool datapool, int equivalenceClassIndex, String oldName)
	{
	}
	
	public void equivalenceClassAdded(IDatapool datapool, int newEquivalenceClassIndex)
	{
	}

	public void equivalenceClassRemoved(IDatapool datapool, int equivalenceClassIndex)
	{
	}
	
	public void equivalenceClassMoved(IDatapool datapool, int sourceEquivalenceClassIndex, int targetEquivalenceClassIndex)
	{
		if(getEquivalenceClassIndex() == sourceEquivalenceClassIndex)
	    {
	        setEquivalenceClass((IDatapoolEquivalenceClass)datapool.getEquivalenceClass(targetEquivalenceClassIndex));
	        refreshRows();
	    }
	}

	public void recordAdded(IDatapool datapool, int equivalenceClassIndex, int newRecordIndex)
	{
	    if(getEquivalenceClassIndex() == equivalenceClassIndex)
	        recordAdded(newRecordIndex);
	}
	
	protected void recordAdded(int newRecordIndex)
	{
	    if(equivalenceClass == null)
	        return;

		IDatapoolRecord record = (IDatapoolRecord)equivalenceClass.getRecord(newRecordIndex);
		String rowContents[] = new String[table.getColumnCount()];
		IDatapoolCell[] rowData = new IDatapoolCell[table.getColumnCount() - 1];
		rowContents[0] = String.valueOf(newRecordIndex);
		for(int i = 0; i < record.getCellCount(); i++)
		{
			IDatapoolCell cell = (IDatapoolCell)record.getCell(i);
			org.eclipse.hyades.execution.runtime.datapool.IDatapoolVariable cellVariable = cell.getCellVariable();
			int index = findColumnIndex(cellVariable.getName());			
			rowContents[index] = new ValueObject(cell.getCellValue()).getDescription();;	
			rowData[index - 1] = cell;			
		}
		//int insertionIndex = findRowIndex(equivalenceClassIndex, newRecordIndex);
		TableItem tableItem = new TableItem(table, SWT.NULL, newRecordIndex);
		tableItem.setText(rowContents);
		tableItem.setData(DatapoolTableUtil.TAG_DATA, rowData);
		tableItem.setData(DatapoolTableUtil.TAG_RECORD, record);
		tableItem.setData(DatapoolTableUtil.TAG_EQUIVALENCE_CLASS, equivalenceClass);
		repopulateRowLabels();
		if(table.getItemCount() == 1)
			createCursor();		
		if(tableCursor != null && !tableCursor.isDisposed() && table != null)
		{
			table.setSelection(newRecordIndex);
			tableCursor.setSelection(newRecordIndex, tableCursor.getColumn());
			tableCursor.setFocus();
		}		
		datapoolEditorPart.markDirty();		
	}
	
	public void recordRemoved(IDatapool datapool, int equivalenceClassIndex, int recordIndex)
	{
	    if(getEquivalenceClassIndex() == equivalenceClassIndex)
	        recordRemoved(recordIndex);
	}
	
	protected void recordRemoved(int recordIndex)
	{
		//int tableRowIndex = findRowIndex(recordIndex);
		table.remove(recordIndex);

		//For performance reasons, only re-populate the row number labels if the rows are not scheduled to be refreshed: 
		if(!refreshRowsScheduled){
			repopulateRowLabels();
		}

		datapoolEditorPart.markDirty();				
		if(tableCursor != null && !tableCursor.isDisposed() && table != null)
		{
			int tableSize = table.getItemCount();
			if(tableSize == 0)
			{
				table.deselectAll();
				//tableCursor.dispose(); //bugzilla 76200, remove last record cause widget disposed exception
                tableCursor.setVisible(false); //- PN: Bugzilla #137400
				return;								
			}
			if(tableSize > recordIndex)
			{
				table.setSelection(recordIndex);
				tableCursor.setSelection(recordIndex, 0);
				return;
			}
			if(tableSize > recordIndex - 1)	
			{
				table.setSelection(recordIndex - 1);
				tableCursor.setSelection(recordIndex - 1, 0);
			}
		}		
	}
	
	public void recordMoved(IDatapool datapool, int equivalenceClassIndex, int sourceRecordIndex, int targetRecordIndex)
	{
	    if(getEquivalenceClassIndex() == equivalenceClassIndex)
	        recordMoved(sourceRecordIndex, targetRecordIndex);
	}
	
	protected void recordMoved(int sourceRecordIndex, int targetRecordIndex)
	{
		tableUtil.moveRow(sourceRecordIndex, targetRecordIndex);
		repopulateRowLabels();
		tableCursor.redraw();
		datapoolEditorPart.markDirty();		
	}
	
	public void cellChanged(IDatapool datapool, int equivalenceClassIndex, int recordIndex, int variableIndex)
	{
	    if(getEquivalenceClassIndex() == equivalenceClassIndex)
	        cellChanged(recordIndex, variableIndex);
	}
	
	protected void cellChanged(int recordIndex, int variableIndex)
	{
		IDatapoolRecord record = (IDatapoolRecord)equivalenceClass.getRecord(recordIndex);
		IDatapoolVariable variable = (IDatapoolVariable)getDatapool().getVariable(variableIndex);

		IDatapoolCell cell = null;		
		for(int i = 0; i < record.getCellCount(); i++)
		{
			cell = (IDatapoolCell)record.getCell(i);
			if(cell.getCellVariable().equals(variable))
				break;
			else
				cell = null;
		}

		if(cell != null)
		{
			TableItem tableItem = table.getItem(recordIndex);
			tableItem.setText(variableIndex + 1, new ValueObject(cell.getCellValue()).getDescription());
			tableCursor.setSelection(recordIndex, variableIndex + 1);
			datapoolEditorPart.markDirty();
		}
		datapoolEditorPart.markDirty();		
	}

	public void equivalenceClassReordered(IDatapool datapool, int equivalenceClassIndex)
	{
	    if(getEquivalenceClassIndex() == equivalenceClassIndex)
	        equivalenceClassReordered();
	}
	
	protected void equivalenceClassReordered()
	{
	    boolean changed = false;		
		for(int i = 0; i < equivalenceClass.getRecordCount(); i++)
		{
			IDatapoolRecord record = (IDatapoolRecord)equivalenceClass.getRecord(i);
			int oldTableRowIndex = findRowIndex(record);
			if(oldTableRowIndex != i)
			{
			    tableUtil.swapRow(oldTableRowIndex, i);
			    changed = true;
			}
		}
		if(changed == true)
		    datapoolEditorPart.markDirty();		
	}
	
	public void recordChanged(IDatapool datapool, int EquivClassIndex, int recordIndex) 
	{
		return;
	}

	public void save(IDatapool datapool) 
	{
	}

	// End IDatapoolListener

	/* 
	 * Finds the index of the row in the table that corresponds to the record 
	 * given by the record object.  Returns -1 if the record is not found.
	 */
	private int findRowIndex(IDatapoolRecord record)
	{
		for(int i = 0; i < equivalenceClass.getRecordCount(); i++)
		{
			IDatapoolRecord currentRecord = (IDatapoolRecord)equivalenceClass.getRecord(i);
			if(currentRecord.equals(record))
				return i;
		}
		return -1;
	}

	/**
	 * Recreates the row labels for a row group when a member changes or
	 * moves.
	 */	
	private void repopulateRowLabels()
	{
		for(int i = 0; i < equivalenceClass.getRecordCount(); i++)
		{
			TableItem tableItem = table.getItem(i);
			tableItem.setText(0, String.valueOf(i));
		}
	}
	
	private int getRecordIndex(IDatapoolEquivalenceClass equivalenceClass, IDatapoolRecord record)
	{
		for(int i = 0; i < equivalenceClass.getRecordCount(); i++)
		{
			org.eclipse.hyades.execution.runtime.datapool.IDatapoolRecord compRecord = equivalenceClass.getRecord(i);
			if(compRecord.equals(record))
				return i;
		}
		return -1;
	}
	
	protected Object[] createCells(IDatapool datapool)
	{
	 	Object[] cells = new Object[datapool.getVariableCount()];
	 	for(int i = 0; i < datapool.getVariableCount(); i++)
	 		cells[i] = new String();
	 	return cells;
	}

	private void setVendorConfiguration()
	{
		if(this.vendorID == null)
			return;
		IExtensionPoint extensionPoint = Platform.getExtensionRegistry().getExtensionPoint(UiPlugin.getID() + ".vendorConfigurationExtension"); //$NON-NLS-1$
		if (extensionPoint != null)
		{
			try {
	  
				IConfigurationElement[] extensionPoints = extensionPoint.getConfigurationElements();
				for(int i = 0; i < extensionPoints.length; i++)
				{
					String extVendorID = extensionPoints[i].getAttribute("vendorID"); //$NON-NLS-1$
					if(this.vendorID.equals(extVendorID))
					{
						String strShowVar = extensionPoints[i].getAttribute("showVariableCommands"); //$NON-NLS-1$
						Boolean boolShowVar = Boolean.valueOf(strShowVar);
						showVariables = boolShowVar.booleanValue();
						showRecords = boolShowVar.booleanValue();
					}
				}
			}
			catch(Exception e)
			{
			}
		}	
		
	}	
	
	public TableViewer getViewer()
	{
		return viewer;
	}
	
	public TableCursor getTableCursor()
	{
		return tableCursor;
	}
	
	public void save()
	{
	    IDatapool datapool = getDatapool();
		datapoolFactory.save(datapool);
	}
	
	public void dispose()
	{
		clearCellEditor();
		if(table != null && !table.isDisposed())
		    table.dispose();
		if(tableCursor != null && !tableCursor.isDisposed())
			tableCursor.dispose();
		if(contextMenu != null && !contextMenu.isDisposed())
			contextMenu.dispose();
		if(viewer != null)
			viewer = null;
		
		if(getDatapool() != null)
		    getDatapool().removeDatapoolListener(this);
		equivalenceClass = null;
		datapool = null;
		tableUtil = null;
		datapoolEditorPart = null;
		datapoolFactory = null;
		vendorID = null;
		controlEditor = null;
	}
	
	public IDatapool getDatapool()
	{
	    return this.datapool;
	}
		
	public void cut(){

		if (this.isF2Mode) {
			this.cellEditor.performCut();
		} 
		else {

			if((tableCursor != null) && (!tableCursor.isDisposed())){

				int columnIndex = tableCursor.getColumn();
				
				if(columnIndex == 0){
					cutRecords();
				}
				else if(columnIndex > 0){
					cutCell();
				}
			}
		}
	}
	
	public void copy(){
		
		if (this.isF2Mode) {
			this.cellEditor.performCopy();
		} 
		else {

			if((tableCursor != null) && (!tableCursor.isDisposed())){

				int columnIndex = tableCursor.getColumn();
				
				if(columnIndex == 0){
					copyRecords();
				}
				else if(columnIndex > 0){
					copyCell();
				}
			}
		}
	}
	
	public void paste(){
		
		if (this.isF2Mode) {
			this.cellEditor.performPaste();
		} 
		else {

			if((tableCursor != null) && (!tableCursor.isDisposed())){

				int columnIndex = tableCursor.getColumn();
				
				if(columnIndex == 0){
					pasteRecords();
				}
				else if(columnIndex > 0){
					pasteCell();
				}
			}
		}
	}
		
	private void cutCell(){
		
		copyCell();
		clearCell();
	}
	
	private void copyCell(){

		if((tableCursor != null) && (!tableCursor.isDisposed())){

			int rowIndex = -1;
			TableItem row = tableCursor.getRow();

			if(row != null){
				rowIndex = table.indexOf(row);
			}

			int columnIndex = tableCursor.getColumn();

			if((columnIndex > 0) && (rowIndex != -1)){
				datapoolClipboard.setCellContents(((IDatapoolCell)(((IDatapoolRecord)(row.getData(DatapoolTableUtil.TAG_RECORD))).getCell(columnIndex - 1))));
			}
		}
	}
	
	private void pasteCell(){
		
		if((tableCursor != null) && (!tableCursor.isDisposed())){

			int columnIndex = tableCursor.getColumn();
			TableItem row = tableCursor.getRow();

			if((columnIndex != -1) && (row != null)){

				int rowIndex = table.indexOf(row);

				if(rowIndex != -1){

					IDatapoolCell cell = ((IDatapoolCell)(((IDatapoolRecord)(row.getData(DatapoolTableUtil.TAG_RECORD))).getCell(columnIndex - 1)));
					String cellContents = datapoolClipboard.getCellContents();

					//Previous copy was cell data:
					if(cellContents != null){

						String cellText = setCellValue(cellContents, cell);
						
						if(cellText != null){

							row.setText(columnIndex, cellText);
	
							tableCursor.setSelection(table.getSelectionIndex(), columnIndex);
	
							datapoolEditorPart.markDirty();
	
							tableCursor.redraw();
						}
					}

					//Previous copy was records or string data:
					else{

						String[][] recordContents = datapoolClipboard.getRecordContents();						

						//Previous copy was records:
						if(recordContents != null){

							//Paste the whole record if this is the first cell in the row:
							if(columnIndex == 1){

								pasteRecords(new TableItem[]{row});

								tableCursor.redraw();
							}

							//Otherwise, warn the user of the different size:
							else if(columnIndex > 1){

								MessageDialog messageDialog = new MessageDialog(parent.getShell(),
										UiPluginResourceBundle.PASTE_TEXT, 
										null,
										UiPluginResourceBundle.DATA_PASTE_DIFF_SIZE, 
										MessageDialog.WARNING, 
										new String[] {IDialogConstants.OK_LABEL}, 
										0);
								messageDialog.open();
							}
						}

						//Previous copy was string data:
						else{

							String stringContents = datapoolClipboard.getStringContents();

							if(stringContents != null){

								String cellText = setCellValue(stringContents, cell);
								
								if(cellText != null){

									row.setText(columnIndex, cellText);

									tableCursor.setSelection(table.getSelectionIndex(), columnIndex);

									datapoolEditorPart.markDirty();

									tableCursor.redraw();
								}
							}
						}
					}
				}
			}
		}
	}

	private void cutRecords(){
		
		copyRecords();
		clearRecords();
	}
	
	private void copyRecords(){

		if(getRowSelectionCount() > 0){

			TableItem[] tableItems = table.getSelection();

			if(tableItems != null){

				IDatapoolRecord[] records = new IDatapoolRecord[tableItems.length];

				for (int counter = 0; counter < tableItems.length; counter++) {				
					records[counter] = ((IDatapoolRecord)(tableItems[counter].getData(DatapoolTableUtil.TAG_RECORD)));
				}

				datapoolClipboard.setRecordContents(records);
			}
		}
	}

	private void pasteRecords(){
		
		if(getRowSelectionCount() > 0){
			pasteRecords(table.getSelection());
		}
	}
	
	private void pasteRecords(TableItem[] tableItems)
	{
		// no record to paste
		if(table.getColumnCount() <= 1)
			return;

		//tableItems = table.getSelection();
		if(tableItems == null || tableItems.length == 0)
			return;		
		
		// check if the pasting area has any values
		boolean isEmpty = true;
		for(int i = 0; i < tableItems.length; i++)
		{
			for(int j = 1; j < table.getColumnCount(); j++)
			{
				if(tableItems[i].getText(j).length() > 0)
				{
					isEmpty = false;
					break;
				}				
			}
			if(isEmpty == false)
				break;
		}
		
		if(isEmpty == false)
		{
			Object[] messageElements = {UiPluginResourceBundle.DATA_PASTE_OVERWRITE}; 
			String message = NLS.bind("{0}", messageElements); //$NON-NLS-1$
			MessageDialog overwriteDialog = new MessageDialog(parent.getShell(),
															  UiPluginResourceBundle.PASTE_TEXT, 
															  null,
															  message, 
															  MessageDialog.WARNING, 
															  new String[] {IDialogConstants.YES_LABEL, IDialogConstants.NO_LABEL}, 
															  0);
			
			if (overwriteDialog.open() != 0)
				return;
		}
	
		TableItem firstItem = tableItems[0];
		int firstItemIndex = table.indexOf(firstItem);
		String[][] recordContents = datapoolClipboard.getRecordContents();
		if(recordContents == null || recordContents.length == 0)
		{
			// if previous copy was a cell, paste it to the first cell.
			String cellContents = datapoolClipboard.getCellContents();
			
			//Previous copy was cell data:
			if(cellContents != null){
				pasteDataToCell(cellContents, firstItem, 1);
			}
			else{
				
				String stringConents = datapoolClipboard.getStringContents();
				
				//Previous copy was string data:
				if(stringConents != null){
					pasteDataToCell(stringConents, firstItem, 1);
				}
			}
		
			return;
		}
		
		//Previous copy was records or string data:
		for(int i = 0; i < recordContents.length; i++)
		{
			if(firstItemIndex + i < table.getItemCount())
			{
				String[] recordData = recordContents[i];
				if(recordData != null)
				{
					TableItem tableItem = table.getItem(firstItemIndex + i);
					IDatapoolRecord record = (IDatapoolRecord)tableItem.getData(DatapoolTableUtil.TAG_RECORD);
					for(int j = 0; j < recordData.length; j++)
					{
						if(j >= record.getCellCount())
							continue;
						
						pasteDataToCell(recordData[j], tableItem, (j + 1));
					}
				}
			}
			else 
				continue;
		}
	}
	
	private void pasteDataToCell(String data, TableItem tableItem, int index){
		
		IDatapoolRecord record = ((IDatapoolRecord)(tableItem.getData(DatapoolTableUtil.TAG_RECORD)));
		
		if((index - 1) < record.getCellCount()){
			
			String cellText = setCellValue(data, ((IDatapoolCell)(record.getCell(index - 1))));
			
			if(cellText != null){
			
				tableItem.setText(index, cellText);
				
				datapoolEditorPart.markDirty();
			}
		}
	}
	
	private String setCellValue(String cellValue, IDatapoolCell cell){

		if(DatapoolEncryptManager.isVariableEncrypted(cell.getCellVariable())){

			if(password == null){

				DatapoolInputKeyDialog datapoolInputKeyDialog = new DatapoolInputKeyDialog(
						Display.getCurrent().getActiveShell(),
						datapool);

				if(datapoolInputKeyDialog.open() == IDialogConstants.OK_ID) {				
					password = datapoolInputKeyDialog.getKey();
				}
				else{
					return null;
				}
			}
			
			cell.setCellValue(DatapoolEncryptManager.encrypt(cellValue, password));
			
			//Replace the encrypted text (hex characters) with asterisks:
			//Note: Each decrypted character is encrypted as two hex characters so every two characters are replaced with one asterisk.
			return (new ValueObject(cell.getCellValue()).getDescription().replaceAll(".{2}", "*")); //$NON-NLS-1$ //$NON-NLS-2$
		}
		
		cell.setCellValue(cellValue);
			
		return (new ValueObject(cell.getCellValue()).getDescription());
	}
		
	public IDatapoolPart getIDatapoolPart()
	{
		return datapoolEditorPart;
	}

	/**
	 * <p>Refreshes and reloads the datapool table.</p>
	 * 
	 * <p>If the current cell is in editing mode, this method
	 * first closes the cell editor.  If the datapool associated with 
	 * this datapool table is <code>null</code> or does not contain 
	 * an equivalence class, this method will hide the datapool 
	 * table and return.  Otherwise, this method will refresh the 
	 * datapool table, including the following:</p>
	 * 
	 * <ul>
	 * <li>Restores the table (e.g. enable/visible).</li>
	 * <li>Reloads the rows and columns.</li>
	 * <li>Resets the table cursor.</li>
	 * </ul>
	 *  
	 * <p><b>Caution:</b> This method rebuilds the datapool table, which
	 * is a costly operation for larger datapools.</p>
	 */
	public void refresh(){
		refresh(true);
	}
	
	/**
	 * <p>Refreshes the datapool table and reloads the rows
	 * and columns, if the <code>reload</code> parameter 
	 * is <code>true</code>.</p>
	 * 
	 * <p>If the current cell is in editing mode, this method
	 * first closes the cell editor.  If the datapool associated with 
	 * this datapool table is <code>null</code> or does not contain 
	 * an equivalence class, this method will hide the datapool 
	 * table and return.  Otherwise, this method will refresh the 
	 * datapool table, including the following:</p>
	 * 
	 * <ul>
	 * <li>Restores the table (e.g. enable/visible).</li>
	 * <li>Reloads the rows and columns, if the <code>reload</code> parameter is <code>true</code>.</li>
	 * <li>Resets the table cursor.</li>
	 * </ul>
	 *  
	 * <p><b>Caution:</b> This method rebuilds the datapool table if the 
	 * <code>reload</code> parameter is <code>true</code>, which is a 
	 * costly operation for larger datapools.</p>
	 * 
	 * @param reload Reload the rows and columns.
	 */
	public void refresh(boolean reload)
	{
		try{
			
			setWaitCursor();

			clearCellEditor(); // end editing if it's not.

			if(equivalenceClass == null)
			{
				hideTable();
			}
			else
			{
				restoreTable();

				if(reload){

					refreshColumns();
					refreshRows();
				}

				resetCursor();  // bugzilla74984
			}
		} 
		finally {
        	unsetWaitCursor();
        }	
	}
	
	/**
	 * <p>Refreshes the datapool table with the default 
	 * equivalence class from the <code>datapool</code> parameter 
	 * and reloads the rows and columns, if the new equivalence class
	 * is different than the current equivalence class associated with
	 * the datapool table.</p>
	 * 
	 * <p>If the new equivalence class is different than the current 
	 * equivalence class associated with the datapool table, the datapool 
	 * table is associated with the new equivalence class.  Then, the datapool
	 * table is refreshed.</p>
	 * 
	 * <p>When refreshing the datapool table, if the current cell is in editing 
	 * mode, the cell editor is first closed.  If the datapool associated with 
	 * this datapool table is <code>null</code> or does not contain 
	 * an equivalence class, this method will hide the datapool 
	 * table and return.  Otherwise, this method will refresh the 
	 * datapool table, including the following:</p>
	 * 
	 * <ul>
	 * <li>Restores the table (e.g. enable/visible).</li>
	 * <li>Reloads the rows and columns, if the new equivalence class is different than the current equivalence class associated with the datapool table.</li>
	 * <li>Resets the table cursor.</li>
	 * </ul>
	 *  
	 * <p><b>Caution:</b> This method rebuilds the datapool table if the 
	 * new equivalence class is different than the current equivalence 
	 * class associated with the datapool table, which is a costly 
	 * operation for larger datapools.</p>
	 * 
	 * @param      datapool The datapool containing the new default equivalence class.
	 * @see        #refresh(IDatapoolEquivalenceClass)
	 * @deprecated A datapool table is associated with a single equivalence class. Use {@link #refresh(IDatapoolEquivalenceClass)} instead.
	 */
	public void refresh(IDatapool datapool){
		refresh(getDefaultEquivalenceClass(datapool));
	}
	
	/**
	 * <p>Refreshes the datapool table with the <code>ec</code> parameter 
	 * and reloads the rows and columns, if the new equivalence class
	 * is different than the current equivalence class associated with
	 * the datapool table.</p>
	 * 
	 * <p>If the new equivalence class is different than the current 
	 * equivalence class associated with the datapool table, the datapool 
	 * table is associated with the new equivalence class.  Then, the datapool
	 * table is refreshed.</p>
	 * 
	 * <p>When refreshing the datapool table, if the current cell is in editing 
	 * mode, the cell editor is first closed.  If the datapool associated with 
	 * this datapool table is <code>null</code> or does not contain 
	 * an equivalence class, this method will hide the datapool 
	 * table and return.  Otherwise, this method will refresh the 
	 * datapool table, including the following:</p>
	 * 
	 * <ul>
	 * <li>Restores the table (e.g. enable/visible).</li>
	 * <li>Reloads the rows and columns, if the new equivalence class is different than the current equivalence class associated with the datapool table.</li>
	 * <li>Resets the table cursor.</li>
	 * </ul>
	 *  
	 * <p><b>Caution:</b> This method rebuilds the datapool table if the 
	 * new equivalence class is different than the current equivalence 
	 * class associated with the datapool table, which is a costly 
	 * operation for larger datapools.</p>
	 * 
	 * @param ec The new equivalence class.
	 */
	public void refresh(IDatapoolEquivalenceClass ec)
	{	
	    if(this.equivalenceClass != ec)
	    {	    
			IDatapool dp = (IDatapool)ec.getDatapool();

			if(this.datapool != null && this.datapool != dp)
			    datapool.removeDatapoolListener(this);
			
			if(dp != null && dp != this.datapool)
				dp.addDatapoolListener(this);
					
		    setEquivalenceClass(ec);
		    
			refresh(true);
	    }
	    else{
	    	refresh(false);
	    }
	}
	
	private void hideTable()
	{	
		table.setVisible(false);
		table.setEnabled(false);
		table.setHeaderVisible(false);
		
		datapoolMenuManager.setTable(null);
	}
	
	private void restoreTable()
	{
		if(table != null && !table.isDisposed())
		{
			table.setEnabled(true);
			table.setVisible(true);
			table.setHeaderVisible(true);	
			
			datapoolMenuManager.setTable(this);
		}
	}
	
	private void refreshColumns()
	{
	    IDatapool datapool = getDatapool();
		int newColumnCount = datapool.getVariableCount();
		int oldColumnCount = tableUtil.getColumnCount() - 1;
		if(tableCursor != null && !tableCursor.isDisposed())
		{
			TableItem selectedItem = tableCursor.getRow();
			if(selectedItem != null)
			{
				int selectedColumnIndex = tableCursor.getColumn();
				if(selectedColumnIndex > newColumnCount)
					tableCursor.setSelection(selectedItem, newColumnCount);
				else
					tableCursor.setSelection(selectedItem, selectedColumnIndex);
			}
		}

		if(oldColumnCount > newColumnCount)
		{
			int numberToDelete = oldColumnCount - newColumnCount;
			for(int i = 0; i < numberToDelete; i++)
			{
				tableUtil.deleteColumn(newColumnCount + 1);
			}
		}

		for(int counter = 0; counter < newColumnCount; counter++){
			
			TableColumn tableColumn = null;
			
			if((counter + 1) < tableUtil.getColumnCount()){
				tableColumn = table.getColumn(counter + 1);
			}
			else{
				
				tableColumn = new TableColumn(table, SWT.LEFT, (counter + 1));
				tableColumn.setWidth(COLUMN_DEFAULT_WIDTH);
				tableColumn.addSelectionListener(headerListener);	
				tableColumn.addControlListener(resizeColumnListener);
			}
			
			IDatapoolVariable variable = ((IDatapoolVariable)(datapool.getVariable(counter)));
            String type = DatapoolUtil.getInstance().getVariableTypeInfo(variable);
			String tableColumnText = variable.getName();
    		
    		if((type != null) && (type.trim().length() > 0)){	
    			tableColumnText = (variable.getName() + DatapoolConstants.VARIABLE_NAME_TYPE_DELIMITER + type).trim();
    		}
    		
    		tableColumn.setResizable(true);			
			tableColumn.setText(tableColumnText);  
			tableColumn.setData(DatapoolTableUtil.TAG_VARIABLE, variable);
			
			if(DatapoolEncryptManager.isVariableEncrypted(variable)){
				tableColumn.setImage(TestUIImages.INSTANCE.getImage(TestUIImages.IMG_DATAPOOL_ENCRYPTED_VARIABLE));
			}
			else{
				tableColumn.setImage(null);
			}
		}		
	}
	
	private void refreshRows(){
		
		//Reset the flag denoting when this method is scheduled to be invoked:
		refreshRowsScheduled = false;
		
		int rowIndex = 0;
		int oldRowCount = table.getItemCount();
		int columnCount = table.getColumnCount();
		int variableCount = getDatapool().getVariableCount();
		Map<String, Integer> columnIndexes = new HashMap<String, Integer>(columnCount - 1);
		
		//Map the variable names to column indexes, starting at second column to account for the first empty column:
		for(int columnIndex = 1; columnIndex < columnCount; columnIndex++){
			
			IDatapoolVariable variable = ((IDatapoolVariable)(table.getColumn(columnIndex).getData(DatapoolTableUtil.TAG_VARIABLE))); 
			
			if(variable != null){
				columnIndexes.put(variable.getId(), new Integer(columnIndex));
			}
		}	
		
		for(int i = 0; i < equivalenceClass.getRecordCount(); i++)
		{
			IDatapoolRecord record = (IDatapoolRecord)equivalenceClass.getRecord(i);
							
			String rowContents[] = new String[variableCount + 1];
			rowContents[0] = String.valueOf(i);
			IDatapoolCell[] rowData = new IDatapoolCell[variableCount];

			for(int k = 0; k < record.getCellCount(); k++)
			{
				IDatapoolCell cell = (IDatapoolCell)record.getCell(k);
				
				String cellDescription = new ValueObject(cell.getCellValue()).getDescription();
				
				//Replace the encrypted text (hex characters) with asterisks:
				//Note: Each decrypted character is encrypted as two hex characters so every two characters are replaced with one asterisk.
				if((DatapoolEncryptManager.isVariableEncrypted(cell.getCellVariable())) && (cellDescription != null) && (cellDescription.length() > 0)){
					cellDescription = cellDescription.replaceAll(".{2}", "*"); //$NON-NLS-1$ //$NON-NLS-2$
				}
				
				int columnIndex = columnIndexes.get(cell.getCellVariable().getId()).intValue();

				//Assumption: There is a cell for each variable of the record making the cell index the variable index.
				rowData[columnIndex - 1] = cell;

				//Increment the cell index to account for the first column:
				rowContents[columnIndex] = cellDescription;
			}				
			
			TableItem item = null;
			if(rowIndex < table.getItemCount())
				item = table.getItem(rowIndex);
			else
				item = new TableItem(table, SWT.NULL);
			item.setText(rowContents);
			item.setData(DatapoolTableUtil.TAG_DATA, rowData);
			item.setData(DatapoolTableUtil.TAG_RECORD, record); 
			item.setData(DatapoolTableUtil.TAG_EQUIVALENCE_CLASS, equivalenceClass); 
			rowIndex++;
		}	
		
		try
		{
			if(tableCursor == null || tableCursor.isDisposed())
				createCursor();
		}
		catch(Exception e)
		{
		}
		
		int newRowCount = rowIndex;
		if(tableCursor != null && !tableCursor.isDisposed())
		{
			TableItem selectedItem = tableCursor.getRow();
			if(selectedItem != null)
			{
				int selectedColumnIndex = tableCursor.getColumn();
				int selectedRowIndex = table.indexOf(selectedItem);
				if(selectedRowIndex > newRowCount - 1)
				{
					if(newRowCount - 1 < 0)
					{
						table.deselectAll();
						if(tableCursor != null && !tableCursor.isDisposed())
							tableCursor.dispose();
					}
					else
					{
						table.setSelection(newRowCount - 1);
						tableCursor.setSelection(newRowCount - 1, selectedColumnIndex);
					}
				}
				else
					tableCursor.setSelection(selectedRowIndex, selectedColumnIndex);
			}
		}
		
		if(oldRowCount > newRowCount)
		{
			int numberToDelete = oldRowCount - newRowCount;
			for(int i = 0; i < numberToDelete; i++)
			{
				table.remove(newRowCount);
			}
		}
	}	
	
	private IDatapoolEquivalenceClass getDefaultEquivalenceClass(IDatapool datapool)
	{
	    if(datapool == null)
	        return null;
	    
	    int newECIndex = datapool.getDefaultEquivalenceClassIndex();
		if(newECIndex == -1)
		{
		    if(datapool.getEquivalenceClassCount() == 0)
		        datapool.appendEquivalenceClass(datapool.constructEquivalenceClass());
		    newECIndex = 0;
		}			
		
		return (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(newECIndex);
	}
	
	private void setEquivalenceClass(IDatapoolEquivalenceClass ec)
	{
	    // cache datapool whenever ec is changed for this table.
	    if(this.equivalenceClass == ec)
	        return;
	    this.equivalenceClass = ec;
	    
	    if(ec == null)
	    {
	        this.datapool = null;
	    }
	    else
	    {
	        this.datapool = (IDatapool)ec.getDatapool();
	    }
	}
	
	/**
	 * <p>Resolves the zero-based index of the equivalence class associated 
	 * with this datapool table.</p>
	 * 
	 * <p>This method returns <code>-1</code> if:</p>
	 * 
	 * <ul>
	 * <li>no equivalence class is associated with this datapool table, or</li>
	 * <li>the equivalence class associated with this datapool table is not assocaited with a datapool.</li>
	 * </ul>
	 * 
	 * @return The zero-based index of the equivalence class associated with this datapool table.
	 */
	private int getEquivalenceClassIndex(){
		
		if((datapool != null) && (equivalenceClass != null)){
			return (datapool.getEquivalenceClassIndexById(getEquivalenceClass().getId()));
		}
		
	    return -1;
	}
	
	public IDatapoolEquivalenceClass getEquivalenceClass()
	{
	    return this.equivalenceClass;
	}

	public void setPassword(String password){
		this.password = password;
	}

	private void clearCellEditor()
	{
        if (cellEditor == null) return;
        this.isF2Mode = false;
        if (cellEditor != null) {
            cellEditor.dispose();
            cellEditor = null;
        }
        if (theValueObject != null) {
            theValueObject = null;
        }
        if (tableCursor != null && !tableCursor.isDisposed()) {
            tableCursor.setFocus();
        }
    }

	private void terminateMultipleSelection(){

		if(table.getItemCount() > 0){

			int[] selectionIndices = table.getSelectionIndices();

			if(selectionIndices.length > 1){

				//Sort the cells in order since the order of the indices is unspecified:
				Arrays.sort(selectionIndices);

				TableItem row = table.getItem(selectionIndices[0]);
				table.showItem(row);

				if((tableCursor != null) && (!tableCursor.isDisposed())){
					tableCursor.setSelection(row, 0);
				}
			}

			if((tableCursor != null) && (!tableCursor.isDisposed())){

				if(!tableCursor.isVisible()){
					tableCursor.setVisible(true);
				}

				tableCursor.setFocus();
			}
		}
	}

	public void selectAll(){

		table.selectAll();

		tableCursor.setSelection(0, 0);
		tableCursor.setFocus();
	}
}
