/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DatapoolColumnDialog.java,v 1.29 2010/02/03 13:58:57 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.datapool.internal.dialog;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.edit.datapool.IDatapoolSuggestedType;
import org.eclipse.hyades.edit.datapool.IDatapoolVariable;
import org.eclipse.hyades.models.common.datapool.DPLVariable;
import org.eclipse.hyades.models.common.datapool.util.DatapoolEncryptManager;
import org.eclipse.hyades.test.ui.datapool.internal.util.DatapoolUtil;
import org.eclipse.hyades.test.ui.datapool.internal.util.GridDataUtil;
import org.eclipse.hyades.test.ui.datapool.util.DatapoolUtilities;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.ibm.icu.text.NumberFormat;

/**
 * <p>Dialog to collect information about a variable/column that is being
 * inserted or edited by the datapool table.<p/>
 * 
 * 
 * @author  Peter Sun
 * @author  Paul E. Slauenwhite
 * @author  Jerome Bozier
 * @version February 3, 2010
 * @since   January 27, 2005
 */
public class DatapoolColumnDialog extends DatapoolBaseDialog{

	private IDatapool datapool = null;
	private String password = null;
	private IDatapoolVariable selectedVariable = null;
	private IDatapoolVariable previousVariable = null;
	
	private String name = null;
	private String type = null;
	private String insertionVariableName = null;
	private String insertionVariableID = null;
	private boolean isEncrypted = false;
	private String oldKey = null;
	private String newKey = null;

	private Label errorLabel = null;
	private int defaultNameCounter = 1;	

	private static final String TAG_VARIABLES = "variables"; //$NON-NLS-1$
	
	public DatapoolColumnDialog(Shell parentShell, 
									     IDatapool datapool,
									     String password,
									     IDatapoolVariable selectedVariable,
									     IDatapoolVariable previousVariable){
		
		super(parentShell);
		
		this.datapool = datapool;
		this.password = password;
		this.selectedVariable = selectedVariable;
		this.previousVariable = previousVariable;		
	}

	public String getName(){
		return (name == null ? "" : name); //$NON-NLS-1$
	}

	public String getType(){
		return (type == null ? "" : type); //$NON-NLS-1$
	}
	
	public String getInsertionVariableName(){
		return insertionVariableName;
	}

	public String getInsertionVariableID(){
		return insertionVariableID;
	}

	public boolean isEncrypted(){
		return isEncrypted;
	}

	public String getOldKey() {
		return oldKey;
	}  

	public String getNewKey() {
		return newKey;
	}  
	
	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#createButtonsForButtonBar(org.eclipse.swt.widgets.Composite)
	 */
	protected void createButtonsForButtonBar(Composite parent) {
		
		super.createButtonsForButtonBar(parent);

		if(selectedVariable == null){
			enableOK();
		}
		else{
			getButton(IDialogConstants.OK_ID).setEnabled(false);
		}
	}	

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.datapool.internal.dialog.DatapoolBaseDialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createDialogArea(Composite parent){

		if(selectedVariable == null){
			getShell().setText(UiPluginResourceBundle.DATA_COL_DLG_TITLE_INS);
		}
		else{
			getShell().setText(UiPluginResourceBundle.DATA_COL_DLG_TITLE_EDIT);
		}
		
		GridData gridData = GridDataUtil.createFill();
        gridData.minimumWidth = 300;
        
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;

		Composite parentComposite = ((Composite)(super.createDialogArea(parent)));
		parentComposite.setLayoutData(gridData);		
		parentComposite.setLayout(gridLayout);

		Label nameLabel = new Label(parentComposite, SWT.NONE);
		nameLabel.setText(UiPluginResourceBundle.DATA_COL_DLG_NAME); 

		final Text nameField = new Text(parentComposite, SWT.SINGLE | SWT.BORDER);
		nameField.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		nameField.addKeyListener(new KeyAdapter() {
		
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.KeyAdapter#keyReleased(org.eclipse.swt.events.KeyEvent)
			 */
			public void keyReleased(KeyEvent e){
				
				name = nameField.getText();
				
				enableOK();
			}
		});
		
		Label typeLabel = new Label(parentComposite, SWT.NONE);
		typeLabel.setText(UiPluginResourceBundle.DATA_COL_DLG_TYPE); 

		final Text typeField = new Text(parentComposite, SWT.SINGLE | SWT.BORDER);
		typeField.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		typeField.addKeyListener(new KeyAdapter() {
		
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.KeyAdapter#keyReleased(org.eclipse.swt.events.KeyEvent)
			 */
			public void keyReleased(KeyEvent e){
				
				type = typeField.getText();
				
				enableOK();
			}		
		});
		
		IDatapoolVariable[] variables = DatapoolUtil.getInstance().getVariables(datapool, null);
		
		if(selectedVariable != null){
			
			name = selectedVariable.getName();
			
			nameField.setText(name);
			nameField.selectAll();
			
			type = DatapoolUtil.getInstance().getVariableTypeInfo(selectedVariable);

			if(type != null){
				
				typeField.setText(type);
				typeField.setSelection(type.length() + 1);
			}
		}

		Label insertionLabel = new Label(parentComposite, SWT.NONE);
		insertionLabel.setText(UiPluginResourceBundle.DATA_DLG_POSITION); 
		
		if(selectedVariable == null){
			
			name = (NLS.bind(UiPluginResourceBundle.DATA_VARIABLE_NAME, NumberFormat.getInstance().format(++defaultNameCounter)));
			
			while(!DatapoolUtilities.getInstance().isVariableNameUnique(datapool, name, null)){
				name = (NLS.bind(UiPluginResourceBundle.DATA_VARIABLE_NAME, NumberFormat.getInstance().format(++defaultNameCounter)));
			}
			
			nameField.setText(name);
			nameField.selectAll();
			
			type = IDatapoolSuggestedType.TYPE_DISPLAY_STRING;
			typeField.setText(type);
		}
		
		String[] variableLabels = getVariableLabels(variables);
		
		final Combo insertionVariables = new Combo(parentComposite, SWT.DROP_DOWN | SWT.READ_ONLY);
		insertionVariables.setItems(variableLabels);
		insertionVariables.setData(TAG_VARIABLES, variables);
	    insertionVariables.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		insertionVariables.addSelectionListener(new SelectionAdapter(){
		
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetSelected(SelectionEvent selectionEvent) {

				int selectionIndex = insertionVariables.getSelectionIndex();
				
				if(selectionIndex > 0){

					String selectionText = insertionVariables.getItem(selectionIndex);
					IDatapoolVariable[] variables = ((IDatapoolVariable[])(insertionVariables.getData(TAG_VARIABLES)));
					String variableName = null;

					for (int counter = 0; counter < variables.length; counter++) {

						variableName = variables[counter].getName();

						if(selectionText.endsWith(variableName)){

							insertionVariableName = variableName;
							insertionVariableID = variables[counter].getId();

							break;
						}
					}
				}
				else{
					insertionVariableName = new String();
					insertionVariableID = new String();
				}		

				enableOK();
			}
		});

		int previousVariableLastIndex = getPreviousVariableLastIndex(variableLabels);

		if(previousVariableLastIndex == -1){

			insertionVariables.select(0);
			insertionVariableName = new String();
			insertionVariableID = new String();			
		}
		else{

			insertionVariables.select(previousVariableLastIndex);

			String variableName = null;

			for (int counter = 0; counter < variables.length; counter++) {

				variableName = variables[counter].getName();

				if(variableLabels[previousVariableLastIndex].endsWith(variableName)){

					insertionVariableName = variableName;
					insertionVariableID = variables[counter].getId();

					break;
				}
			}
		}
        
        final Button encryptCheckBox = new Button(parentComposite,SWT.CHECK);
        encryptCheckBox.setText(UiPluginResourceBundle.DatapoolDialog_ENCRYPTED);

        final Button changeEncryptionKeyButton = new Button(parentComposite,SWT.NONE);
        changeEncryptionKeyButton.setText(UiPluginResourceBundle.DatapoolDialog_CHANGEKEY);
        changeEncryptionKeyButton.addSelectionListener(new SelectionAdapter(){
		
        	/* (non-Javadoc)
        	 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
        	 */
        	public void widgetSelected(SelectionEvent selectionEvent) {
        		
				DatapoolChangeKeyDialog datapoolChangeKeyDialog = new DatapoolChangeKeyDialog(getShell(), 
        				datapool, 
        				true);

        		if(datapoolChangeKeyDialog.open() == IDialogConstants.OK_ID) {				
        			
        			oldKey = datapoolChangeKeyDialog.getOldKey();
        			newKey = datapoolChangeKeyDialog.getNewKey();
        		}
        		
        		enableOK();
        	}		
		});
        
        encryptCheckBox.addSelectionListener(new SelectionAdapter(){
    		
        	/* (non-Javadoc)
        	 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
        	 */
        	public void widgetSelected(SelectionEvent selectionEvent) {

        		isEncrypted = encryptCheckBox.getSelection();

        		if((changeEncryptionKeyButton != null) && (DatapoolEncryptManager.isDatapoolEncrypted(datapool))){
        			changeEncryptionKeyButton.setEnabled(encryptCheckBox.getSelection());
        		}		

        		enableOK();
        	}
		});
        
        if((selectedVariable instanceof DPLVariable) && ((DPLVariable)(selectedVariable)).isEncrypted()){
			
        	if(DatapoolEncryptManager.isDatapoolEncrypted(datapool)){
				changeEncryptionKeyButton.setEnabled(true);
			}
			
        	encryptCheckBox.setSelection(true);
			
        	isEncrypted = true;
		}
		else{
			
			changeEncryptionKeyButton.setEnabled(false);
			
			encryptCheckBox.setSelection(false);
			
			isEncrypted = false;
		}	

		gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.horizontalSpan = 2;	
		
		//Note: Set the text of the error label to the longest possible string for layout purposes.
		errorLabel = new Label(parentComposite, SWT.NONE);
		errorLabel.setText(UiPluginResourceBundle.DATA_COL_DLG_ERROR_NAME_NOT_VALID); 		
		errorLabel.setLayoutData(gridData);
		errorLabel.setForeground(Display.getDefault().getSystemColor(SWT.COLOR_RED));
		errorLabel.setVisible(false);
		
		parentComposite.pack();
		
		return parentComposite;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#okPressed()
	 */
	protected void okPressed() {
		
		//The user has not entered the correct password for the datapool or in the modify password dialog:
		if((password == null) && (oldKey == null)){	

			//The datapool is encrypted so the user must enter the existing password for the datapool:
			if(DatapoolEncryptManager.isDatapoolEncrypted(datapool)){

				if((isEncrypted()) || ((selectedVariable instanceof DPLVariable) && ((DPLVariable)(selectedVariable)).isEncrypted())){
					
					DatapoolInputKeyDialog datapoolInputKeyDialog = new DatapoolInputKeyDialog(
							getShell(),
							datapool);
	
					if(datapoolInputKeyDialog.open() == IDialogConstants.OK_ID) {				
						newKey = datapoolInputKeyDialog.getKey();
					}
					else{
						return;
					}
				}
			}

			//The datapool is not encrypted so the user must enter a new password (twice) for the datapool:
			else if(isEncrypted()){

				DatapoolChangeKeyDialog datapoolChangeKeyDialog = new DatapoolChangeKeyDialog(
						getShell(), 
						datapool,
						false);

				if(datapoolChangeKeyDialog.open() == IDialogConstants.OK_ID) {				
					newKey = datapoolChangeKeyDialog.getNewKey();
				}
				else{
					return;
				}
			}
		}
		
		super.okPressed();
	}

	private void enableOK(){

		boolean enableOK = true;

		//Validate if the name exists:
		if((name == null) || (name.length() == 0)){			
			enableOK = false;			
		}

		//Validate if the name is unique:
		else if(!DatapoolUtilities.getInstance().isVariableNameUnique(datapool, name, selectedVariable)){

			enableOK = false;				

			errorLabel.setText(UiPluginResourceBundle.DATA_COL_DLG_ERROR_NAME_NOT_UNIQUE); 
		}

		//Validate if the name is valid:
		else if(!DatapoolUtilities.getInstance().isVariableNameValid(name)){

			enableOK = false;				

			errorLabel.setText(UiPluginResourceBundle.DATA_COL_DLG_ERROR_NAME_NOT_VALID); 
		}
		
		//Validate if the name exists and is valid:
		else if((type != null) && (type.length() > 0) && (!DatapoolUtilities.getInstance().isVariableTypeValid(type))){

			enableOK = false;				

			errorLabel.setText(UiPluginResourceBundle.DATA_COL_DLG_ERROR_TYPE_NOT_VALID); 
		}
		
		errorLabel.setVisible(!enableOK);

		getButton(IDialogConstants.OK_ID).setEnabled(enableOK);
	}
	
	private String[] getVariableLabels(IDatapoolVariable[] variables){

		String[] variableLabels = null;

		if(variables != null){

			List<String> variableLabelsList = new ArrayList<String>();

			for(int counter = 0; counter < variables.length; counter++){

				if(variables[counter] != null){

					String variableName = variables[counter].getName();			

					//Insert mode:
					if(selectedVariable == null){

						//Add the before label for the first variable:
						if(counter == 0){
							variableLabelsList.add(NLS.bind(UiPluginResourceBundle.DATA_DLG_BEFORE, variableName)); 
						}

						variableLabelsList.add(NLS.bind(UiPluginResourceBundle.DATA_DLG_AFTER, variableName));
					}

					//Edit mode (do not add before/after labels for the selected variable):
					else if(variables[counter] != selectedVariable){

						//Add the before label for the first non-selected variable:
						if(variableLabelsList.size() == 0){
							variableLabelsList.add(NLS.bind(UiPluginResourceBundle.DATA_DLG_BEFORE, variableName)); 
						}

						variableLabelsList.add(NLS.bind(UiPluginResourceBundle.DATA_DLG_AFTER, variableName));
					}
				}
			}

			if(variableLabelsList.size() == 0){
				variableLabels = new String[]{UiPluginResourceBundle.DATA_DLG_FIRST_VARIABLE}; 
			}
			else{
				variableLabels = ((String[])(variableLabelsList.toArray(new String[variableLabelsList.size()])));			 
			}
		}

		return variableLabels;
	}

	private int getPreviousVariableLastIndex(String[] variableLabels){

		int index = -1;

		if((variableLabels != null) && (previousVariable != null)){

			String previousVariableName = previousVariable.getName();		

			for(int counter = 0; counter < variableLabels.length; counter++){

				if(variableLabels[counter].endsWith(previousVariableName)){										
					
					if(index != -1){
						
						index = counter;

						break;
					}
					else{
						index = counter;
					}
				}
			}
		}

		return index;
	}
}
