/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DatapoolRowDialog.java,v 1.17 2009/08/13 12:40:47 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.datapool.internal.dialog;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.hyades.edit.datapool.IDatapoolRecord;
import org.eclipse.hyades.execution.runtime.datapool.IDatapoolEquivalenceClass;
import org.eclipse.hyades.test.ui.datapool.internal.util.GridDataUtil;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

/**
 * <p>Dialog to collect information about a record/row that is being
 * inserted or edited by the datapool table.</p>
 * 
 * 
 * @author  Peter Sun
 * @author  Paul Slauenwhite
 * @version August 13, 2009
 * @since   January 27, 2005
 */
public class DatapoolRowDialog extends DatapoolBaseDialog{

	private IDatapoolRecord selectedRecord = null;
	private IDatapoolRecord previousRecord = null;

	private int insertionRecordIndex = -1;

	public DatapoolRowDialog(Shell parentShell, 								
								IDatapoolRecord selectedRecord,
								IDatapoolRecord previousRecord){
		
		super(parentShell);

		this.selectedRecord = selectedRecord;
		this.previousRecord = previousRecord;
	}

	/**
	 * @see org.eclipse.jface.window.Window#createContents(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createContents(Composite parent){
		
		Control control = super.createContents(parent);
		
		getOKButton().setEnabled(false);
		
		return control;
	}
	
	/**
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createDialogArea(Composite parent){
		
		getShell().setText(UiPluginResourceBundle.DATA_ROW_DLG_TITLE_EDIT);

		GridData gridData = GridDataUtil.createFill();
		gridData.minimumWidth = 200;

		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		
		Composite superComposite = (Composite)super.createDialogArea(parent);
		superComposite.setLayoutData(gridData);		
		superComposite.setLayout(gridLayout);

		Label recordIndexLabel = new Label(superComposite, SWT.NONE);
		recordIndexLabel.setText(UiPluginResourceBundle.DATA_DLG_POSITION); 

		String[] recordLabels = getRecordLabels();

		final Combo recordIndexCombo = new Combo(superComposite, SWT.READ_ONLY | SWT.DROP_DOWN);
		recordIndexCombo.setItems(recordLabels);
		recordIndexCombo.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		recordIndexCombo.addSelectionListener(new SelectionAdapter(){
			
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetSelected(SelectionEvent e){
				
				insertionRecordIndex = -1;		

				int selectionIndex = recordIndexCombo.getSelectionIndex();
				
				if(selectionIndex > 0){

					String selectionText = recordIndexCombo.getItem(selectionIndex);
					
					if(selectionText.startsWith(NLS.bind(UiPluginResourceBundle.DATA_DLG_BEFORE, ""))){
						insertionRecordIndex = Integer.parseInt(selectionText.substring(NLS.bind(UiPluginResourceBundle.DATA_DLG_BEFORE, "").length()).trim());				
					}
					else if(selectionText.startsWith(NLS.bind(UiPluginResourceBundle.DATA_DLG_AFTER, ""))){
						insertionRecordIndex = Integer.parseInt(selectionText.substring(NLS.bind(UiPluginResourceBundle.DATA_DLG_AFTER, "").length()).trim());				
					}
				}
				
				getOKButton().setEnabled(true);
			}
		});
		
		insertionRecordIndex = -1;		

		int previousRecordLastIndex = getPreviousRecordLastIndex(recordLabels);

		if(previousRecordLastIndex == -1){
			recordIndexCombo.select(0);
		}
		else{

			recordIndexCombo.select(previousRecordLastIndex);

			if(recordLabels[previousRecordLastIndex].startsWith(NLS.bind(UiPluginResourceBundle.DATA_DLG_BEFORE, ""))){
				insertionRecordIndex = Integer.parseInt(recordLabels[previousRecordLastIndex].substring(NLS.bind(UiPluginResourceBundle.DATA_DLG_BEFORE, "").length()).trim());				
			}
			else if(recordLabels[previousRecordLastIndex].startsWith(NLS.bind(UiPluginResourceBundle.DATA_DLG_AFTER, ""))){
				insertionRecordIndex = Integer.parseInt(recordLabels[previousRecordLastIndex].substring(NLS.bind(UiPluginResourceBundle.DATA_DLG_AFTER, "").length()).trim());				
			}
		}
		
		superComposite.pack();
		
		return superComposite;
	}

	private String[] getRecordLabels(){
		
		String[] recordLabels = null;
		IDatapoolEquivalenceClass equivalenceClass = selectedRecord.getEquivalenceClass();
		
		if(equivalenceClass != null){

			List recordLabelsList = new ArrayList();
			int selectedRecordIndex = getRecordIndex(equivalenceClass, selectedRecord);
			
			for(int counter = 0; counter < equivalenceClass.getRecordCount(); counter++){

				//Do not add before/after labels for the selected :
				if(counter != selectedRecordIndex){
				
					//Add the before label for the first non-selected record:
					if(recordLabelsList.size() == 0){
						recordLabelsList.add(NLS.bind(UiPluginResourceBundle.DATA_DLG_BEFORE, new Integer(counter))); 
					}

					recordLabelsList.add(NLS.bind(UiPluginResourceBundle.DATA_DLG_AFTER, new Integer(counter)));
				}
			}

			if(recordLabelsList.size() == 0){
				recordLabels = new String[]{UiPluginResourceBundle.DATA_DLG_FIRST_RECORD}; 
			}
			else{
				recordLabels = ((String[])(recordLabelsList.toArray(new String[recordLabelsList.size()])));				
			}
		}

		return recordLabels;
	}

	private int getPreviousRecordLastIndex(String[] recordLabels){
		
		int index = -1;
		
		if((recordLabels != null) && (previousRecord != null)){

			String previousRecordIndex = String.valueOf(getRecordIndex(previousRecord.getEquivalenceClass(), previousRecord)); 
			
			for(int counter = 0; counter < recordLabels.length; counter++){

				if(recordLabels[counter].endsWith(previousRecordIndex)){										
					
					if(index != -1){
						
						index = counter;

						break;
					}
					else{
						index = counter;
					}
				}
			}
		}

		return index;
	}

	private int getRecordIndex(IDatapoolEquivalenceClass equivalenceClass, IDatapoolRecord record){

		for(int counter = 0; counter < equivalenceClass.getRecordCount(); counter++){

			if(equivalenceClass.getRecord(counter).equals(record)){
				return counter;
			}
		}

		return -1;
	}

	public int getInsertionRecordIndex(){
		return insertionRecordIndex;
	}
}
