/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CSVImportExportUtil.java,v 1.25 2010/03/31 19:18:06 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.datapool.internal.util;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.edit.datapool.IDatapoolCell;
import org.eclipse.hyades.edit.datapool.IDatapoolEquivalenceClass;
import org.eclipse.hyades.edit.datapool.IDatapoolRecord;
import org.eclipse.hyades.edit.datapool.IDatapoolSuggestedType;
import org.eclipse.hyades.edit.datapool.IDatapoolVariable;
import org.eclipse.hyades.models.common.datapool.internal.util.DatapoolSuggestedTypeChecker;
import org.eclipse.hyades.test.ui.datapool.internal.dialog.DatapoolConstants;
import org.eclipse.hyades.test.ui.datapool.util.DatapoolUtilities;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.osgi.util.NLS;

/**
 * <p>CSV import and export utilities.</p>
 * 
 * 
 * @author  Peter Sun
 * @author  Bianca Xue Jiang
 * @author  Paul E. Slauenwhite
 * @author  Jerome Bozier
 * @version March 31, 2010
 * @since   January 27, 2005
 */
public class CSVImportExportUtil{
	
	public static final String ASCII = "US-ASCII"; //$NON-NLS-1$
	public static final String ISONLATIN = "ISO-8859-1"; //$NON-NLS-1$
	public static final String UTF8 = "UTF-8"; //$NON-NLS-1$
	public static final String UTF16 = "UTF-16"; //$NON-NLS-1$
	public static final String UTF16LE = "UTF-16LE"; //$NON-NLS-1$
	public static final String UTF16BE = "UTF-16BE"; //$NON-NLS-1$
		
	private static final String SEPERATOR = ","; //$NON-NLS-1$
	private static final String LINEFEED = "\n"; //$NON-NLS-1$
	private static final String CARRIAGE_RETURN = "\r"; //$NON-NLS-1$

	private static final CSVImportExportUtil instance = new CSVImportExportUtil();

	public static CSVImportExportUtil getInstance()
	{
		return instance;
	}
	
	/**
	 * Private constructor.
	 */
	private CSVImportExportUtil(){
		//No-operation.
	}
	
	/**
	 * <p>Validates a Comma Separated Values (CSV) file containing datapool records.</p>
	 * 
	 * <p>Validation is required when importing a CSV file into a new or existing datapool.</p>
	 * 
	 * @param  csvFilePath The absolute path of the CSV file.
	 * @param  isFirstRowVariableInfo If the first row of the CSV file contains the variable information (name/type), otherwise <code>false</code>.
	 * @param  isFirstColumnEquivalenceClassInfo If the first column of the CSV file contains the equivalence class information (name/row index), otherwise <code>false</code>.
	 * @param  encoding The encoding of the CSV file, otherwise <code>null</null>.
	 * @return <code>true</code> if the CSV file is valid, otherwise <code>false</code>.
	 * @throws FileNotFoundException The CSV file could not be found.
	 * @throws IOException An error occurs while reading the CSV file.
	 * @throws CorruptCSVFileException The CSV is invalid.
	 * @see    #importCSV(IDatapool, String, boolean, boolean, String)
	 * @see    #importCSV(IDatapool, String, boolean, boolean, String, int, int)
	 * @see    #appendFromCSV(IDatapool, String, boolean, boolean, String)
	 */
	public boolean validateCSVFile(String csvFilePath, 
			boolean isFirstRowVariableInfo, 
			boolean isFirstColumnEquivalenceClassInfo, 
			String encoding) throws CorruptCSVFileException {
		return (validateCSVFileWithDatapool(csvFilePath, null, isFirstRowVariableInfo, isFirstColumnEquivalenceClassInfo, false, encoding));
	}
	
	/**
	 * <p>Validates a Comma Separated Values (CSV) file containing datapool records against a datapool.</p>
	 * 
	 * <p>Validation is required when importing a CSV file into a new or existing datapool.</p>
	 * 
	 * @param  csvFilePath The absolute path of the CSV file.
	 * @param  datapool The datapool that the CSV file is validated against, otherwise <code>null</null>.
	 * @param  isFirstRowVariableInfo If the first row of the CSV file contains the variable information (name/type), otherwise <code>false</code>.
	 * @param  isFirstColumnEquivalenceClassInfo If the first column of the CSV file contains the equivalence class information (name/row index), otherwise <code>false</code>.
	 * @param  isReplaceExistingRecords If the CSV file will replace the existing records in the datapool.
	 * @param  encoding The encoding of the CSV file, otherwise <code>null</null>.
	 * @return <code>true</code> if the CSV file is valid, otherwise <code>false</code>.
	 * @throws FileNotFoundException The CSV file could not be found.
	 * @throws IOException An error occurs while reading the CSV file.
	 * @throws CorruptCSVFileException The CSV is invalid.
	 * @see    #importCSV(IDatapool, String, boolean, boolean, String)
	 * @see    #importCSV(IDatapool, String, boolean, boolean, String, int, int)
	 * @see    #appendFromCSV(IDatapool, String, boolean, boolean, String)
	 */
	public boolean validateCSVFileWithDatapool(String csvFilePath, 
			IDatapool datapool, 
			boolean isFirstRowVariableInfo, 
			boolean isFirstColumnEquivalenceClassInfo,
			boolean isReplaceExistingRecords,
			String encoding) throws CorruptCSVFileException {

		//Determine if the datapool should be validated :
		boolean validateDatapool = ((datapool != null) && (datapool.getEquivalenceClassCount() > 0) && (datapool.getVariableCount() > 0));
		
		//Resolve the input reader:
		BufferedReader lineReader = null;
        
		try {
			
			if((encoding == null) || (encoding.trim().length() == 0)) {
				lineReader = new BufferedReader(new InputStreamReader(new FileInputStream(csvFilePath)));
	        } 
			else {
				lineReader = new BufferedReader(new InputStreamReader(new FileInputStream(csvFilePath), encoding));
	        }
			
			//Read the first line:
			String line = lineReader.readLine();
			int lineCount = 1;
			int variableCount = -1;
			
			if(validateDatapool){
				
				variableCount = datapool.getVariableCount();
				
				if(variableCount > DatapoolConstants.MAXIMUM_VARIABLE_LIMIT){
					throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount),NLS.bind(UiPluginResourceBundle.DATA_COL_DLG_ERROR_VARIABLE_LIMIT_EXCEEDED, DatapoolConstants.MAXIMUM_VARIABLE_LIMIT)}));
				}
			}
			
			//Check if the file is empty:
			if(line == null){
				throw new CorruptCSVFileException(UiPluginResourceBundle.DATA_COL_DLG_ERROR_EMPTY_CSV_FILE);	
			}
	
			//Validate the variable names and optional suggested types in the first row:
			if(isFirstRowVariableInfo){

				//Check if the first line is empty:
				if(line.trim().length() == 0){
					throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount),UiPluginResourceBundle.DATA_COL_DLG_ERROR_MISSING_CVS_LINE}));	
				}

				//Parse the first row containing the variable names and optional suggested types:
				CSVTokenizer tokenizer = new CSVTokenizer(line);
				
				//Check if the first column is the equivalence class cell (e.g. empty value):
				if((isFirstColumnEquivalenceClassInfo) && ((!tokenizer.hasMoreTokens()) || (tokenizer.nextToken().trim().length() > 0))){
					throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount),UiPluginResourceBundle.DATA_COL_DLG_ERROR_MISSING_EQUIVALENCE}));	
				}
				
				List variableNames = new ArrayList();
				
				//Iterate the columns containing the variable names and optional suggested types:
				while(tokenizer.hasMoreTokens()){
								
					//Resolve the variable name:
					String suggestedType = null;
					String variableName = tokenizer.nextToken().trim();
					int twoColonsIndex = variableName.indexOf(DatapoolConstants.VARIABLE_NAME_TYPE_DELIMITER); 
	
					//Resolve the optional suggested type within the variable name:
					if(twoColonsIndex != -1){					
	
						suggestedType = variableName.substring(twoColonsIndex + DatapoolConstants.VARIABLE_NAME_TYPE_DELIMITER.length()).trim();
						variableName = variableName.substring(0, twoColonsIndex).trim();
					}
					
					//Validate the variable name:
					if(variableName.length() == 0){
						throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount), UiPluginResourceBundle.DATA_COL_DLG_ERROR_MISSING_VARIABLE}));	
					}
					else if(!DatapoolUtilities.getInstance().isVariableNameValid(variableName)){
						throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount), UiPluginResourceBundle.DATA_COL_DLG_ERROR_NAME_NOT_VALID}));	
					}				
					else if(variableNames.contains(variableName)){
						throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount), UiPluginResourceBundle.DATA_COL_DLG_ERROR_NAME_NOT_UNIQUE}));	
					}
					else{
						variableNames.add(variableName);
					}
					
					//Validate the optional suggested type:
					if((suggestedType != null) && (!DatapoolUtilities.getInstance().isVariableTypeValid(suggestedType))){
						throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount), UiPluginResourceBundle.DATA_COL_DLG_ERROR_TYPE_NOT_VALID}));	
					}
										
					//Validate the variable name and optional suggested type against the datapool:
					if(validateDatapool){

						if(datapool.getVariableIndex(variableName) == -1){
							throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount), UiPluginResourceBundle.DATA_COL_DLG_ERROR_INVALID_VARIABLE}));	
						}

						try {
							
							org.eclipse.hyades.execution.runtime.datapool.IDatapoolVariable variable = datapool.getVariable(variableNames.size() - 1);
	
							if((!variableName.equals(variable.getName())) || ((suggestedType != null) && (!DatapoolSuggestedTypeChecker.getInstance().isTypeMatch(suggestedType, variable.getSuggestedType())))){
								throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount), UiPluginResourceBundle.DatapoolImportWizard_diffVariableInfoWarning}));	
							}
						} 
						catch (Exception e) {
							throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount), UiPluginResourceBundle.DatapoolImportWizard_diffVariableInfoWarning}));	
						}
					}
				}
				
				int columnCount = variableNames.size();
				
				//Check if any variables were validated:
				if(columnCount == 0){
					throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount),UiPluginResourceBundle.DATA_COL_DLG_ERROR_MISSING_VARIABLE}));
				}
				else if(columnCount > DatapoolConstants.MAXIMUM_VARIABLE_LIMIT){
					throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount),NLS.bind(UiPluginResourceBundle.DATA_COL_DLG_ERROR_VARIABLE_LIMIT_EXCEEDED, DatapoolConstants.MAXIMUM_VARIABLE_LIMIT)}));
				}
				
				if(variableCount == -1){
					variableCount = columnCount;
				}
				else if(columnCount != variableCount){
					throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount), NLS.bind(UiPluginResourceBundle.DATA_COL_DLG_ERROR_INCONSISTENT_COLUMN_COUNT, String.valueOf(columnCount))}));					
				}
				
				line = lineReader.readLine();
				lineCount++;
			}
	
			Map equivalenceClassNames = new HashMap();
			String defaultEquivalenceClassName = null;
			int emptyLineCount = 0;
	
			//Validate the remaining lines:
			while(line != null){
	
				//Validate the next line that is not empty:
				if(line.trim().length() > 0){
	
					//Parse the row containing the remaining columns:
					CSVTokenizer tokenizer = new CSVTokenizer(line);
	
					//Validate the equivalence class names and optional row numbers in the first column:
					if(isFirstColumnEquivalenceClassInfo){
	
						//Resolve the equivalence class name:					
						String rowNumber = null;
						String equivalenceClassName = tokenizer.nextToken().trim();
						int twoColonsIndex = equivalenceClassName.indexOf(DatapoolConstants.VARIABLE_NAME_TYPE_DELIMITER); 
	
						//Resolve the optional row number within the equivalence class name:
						if(twoColonsIndex != -1){					
	
							rowNumber = equivalenceClassName.substring(twoColonsIndex + DatapoolConstants.VARIABLE_NAME_TYPE_DELIMITER.length()).trim();
							equivalenceClassName = equivalenceClassName.substring(0, twoColonsIndex).trim();
						}
	
						//Validate the equivalence class name:
						if(equivalenceClassName.length() == 0){
							throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount), UiPluginResourceBundle.DATA_COL_DLG_ERROR_MISSING_EQUIVALENCE}));	
						}
						else if(!DatapoolUtilities.getInstance().isEquivalenceClassNameValid(equivalenceClassName)){
							throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount), UiPluginResourceBundle.DATA_ROW_GRP_DLG_ERROR_NAME_NOT_VALID}));	
						}
						else if(equivalenceClassNames.containsKey(equivalenceClassName)){					
							equivalenceClassNames.put(equivalenceClassName, new Integer(((Integer)(equivalenceClassNames.get(equivalenceClassName))).intValue() + 1));						
						}
						else{

							int recordCount = 0;
							
							//Resolve the record count from the equivalence class:
							if(validateDatapool){

								if(datapool.getEquivalenceClassIndex(equivalenceClassName) == -1){
									throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount), UiPluginResourceBundle.DATA_COL_DLG_ERROR_INVALID_EQUIVALENCE}));	
								}

								org.eclipse.hyades.execution.runtime.datapool.IDatapoolEquivalenceClass equivalenceClass = datapool.getEquivalenceClass(datapool.getEquivalenceClassIndex(equivalenceClassName));
								
								if(equivalenceClass != null){
									recordCount = equivalenceClass.getRecordCount();
								}
							}

							equivalenceClassNames.put(equivalenceClassName, new Integer(recordCount));

							if(defaultEquivalenceClassName == null){
								defaultEquivalenceClassName = equivalenceClassName;
							}
						}
	
						//Count the empty lines as empty rows for the default equivalence class:
						if(emptyLineCount > 0){
	
							equivalenceClassNames.put(defaultEquivalenceClassName, new Integer(((Integer)(equivalenceClassNames.get(defaultEquivalenceClassName))).intValue() + emptyLineCount));
	
							emptyLineCount = 0;
						}
	
						//Validate the optional row number:
						if((!isReplaceExistingRecords) && (rowNumber !=  null)){
	
							try {
	
								int recordIndex = Integer.parseInt(rowNumber);
								
								if(recordIndex > (((Integer)(equivalenceClassNames.get(equivalenceClassName))).intValue())){
									throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount), NLS.bind(UiPluginResourceBundle.DATA_COL_DLG_ERROR_INCONSISTENT_ROW_ORDER, rowNumber)}));	
								}		
								else if(validateDatapool){
									
									org.eclipse.hyades.execution.runtime.datapool.IDatapoolEquivalenceClass equivalenceClass = datapool.getEquivalenceClass(datapool.getEquivalenceClassIndex(equivalenceClassName));
									
									if((equivalenceClass != null) && (recordIndex < equivalenceClass.getRecordCount()) && (equivalenceClass.getRecord(recordIndex) != null)){
										throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[] {String.valueOf(lineCount), UiPluginResourceBundle.DATA_CSV_ERROR_RECORD_INDEX_DUP}));										
									}
								}
							} 
							catch (NumberFormatException n) {
								throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount), NLS.bind(UiPluginResourceBundle.DATA_COL_DLG_ERROR_INVALID_ROW_COUNT, rowNumber)}));	
							}
						}
					}
	
					//Validate the number of remaining columns:		
					int columnCount = 0;
	
					while(tokenizer.hasMoreTokens()){
	
						columnCount++;
	
						tokenizer.nextToken();
					}
						
					if(columnCount > DatapoolConstants.MAXIMUM_VARIABLE_LIMIT){
						throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount),NLS.bind(UiPluginResourceBundle.DATA_COL_DLG_ERROR_VARIABLE_LIMIT_EXCEEDED, DatapoolConstants.MAXIMUM_VARIABLE_LIMIT)}));
					}

					if(variableCount == -1){
						variableCount = columnCount;
					}
					else if(columnCount != variableCount){
						throw new CorruptCSVFileException(NLS.bind(UiPluginResourceBundle.DATA_CSV_LINE, new String[]{String.valueOf(lineCount), NLS.bind(UiPluginResourceBundle.DATA_COL_DLG_ERROR_INCONSISTENT_COLUMN_COUNT, String.valueOf(columnCount))}));	
					}
				}
				else{
					emptyLineCount++;
				}
	
				line = lineReader.readLine();
				lineCount++;
			}
						
			return true;
		}
		catch (FileNotFoundException f) {
			throw new CorruptCSVFileException(UiPluginResourceBundle.DATA_COL_DLG_ERROR_MISSING_CSV_FILE);
		}
		catch (IOException i) {
			throw new CorruptCSVFileException(UiPluginResourceBundle.DATA_COL_DLG_ERROR_CORRUPT_CSV_FILE);
		}
		finally{
	
			//Close the line reader:
			try {
				
				if(lineReader != null){
					lineReader.close();		
				}
			} 
			catch (IOException i) {
				//Ignore since closing input reader.
			}
		}
	}
		
	/**
	 * <p>Imports a Comma Separated Values (CSV) file into a datapool.</p>
	 * 
	 * <p><b>Note</b>: The CSV file is assumed to be valid.  To validate the CSV file 
	 * with the datapool, call the {@link #validateCSVFileWithDatapool(String, IDatapool, boolean, boolean, String)}
	 * method.</p>
	 * 
	 * @param  datapool The datapool that the CSV file is imported into.
	 * @param  csvFilePath The absolute path of the CSV file.
	 * @param  isFirstRowVariableInfo If the first row of the CSV file contains the variable information (name/type), otherwise <code>false</code>.
	 * @param  isFirstColumnEquivalenceClassInfo If the first column of the CSV file contains the equivalence class information (name/row index), otherwise <code>false</code>.
	 * @param  encoding The encoding of the CSV file, otherwise <code>null</null>.
	 * @throws IOException An error occurs while reading the CSV file.
	 * @see    #validateCSVFileWithDatapool(String, IDatapool, boolean, boolean, String)
	 */
	public void importCSV(IDatapool datapool, 
			String csvFileName, 
			boolean isFirstRowVariableInfo, 
			boolean isFirstColumnEquivalenceClassInfo,
			String encoding)throws IOException {
		importCSV(datapool, 
				csvFileName, 
				isFirstRowVariableInfo, 
				isFirstColumnEquivalenceClassInfo,
				encoding, 
				-1, 
				-1);
	}
				
	/**
	 * <p>Imports a Comma Separated Values (CSV) file into a datapool.</p>
	 * 
	 * <p><b>Note</b>: The CSV file is assumed to be valid.  To validate the CSV file 
	 * with the datapool, call the {@link #validateCSVFileWithDatapool(String, IDatapool, boolean, boolean, String)}
	 * method.</p>
	 * 
	 * @param  datapool The datapool that the CSV file is imported into.
	 * @param  csvFilePath The absolute path of the CSV file.
	 * @param  isFirstRowVariableInfo If the first row of the CSV file contains the variable information (name/type), otherwise <code>false</code>.
	 * @param  isFirstColumnEquivalenceClassInfo If the first column of the CSV file contains the equivalence class information (name/row index), otherwise <code>false</code>.
	 * @param  encoding The encoding of the CSV file, otherwise <code>null</null>.
	 * @param  numberOfColumns The number of columns of the CSV file to import into the datapool, otherwise <code>-1</code>.
	 * @param  numberOfRows The number of rows of the CSV file to import into the datapool, otherwise <code>-1</code>.
	 * @throws IOException An error occurs while reading the CSV file.
	 * @see    #validateCSVFileWithDatapool(String, IDatapool, boolean, boolean, String)
	 */
	public void importCSV(IDatapool datapool, 
			String csvFilePath, 
			boolean isFirstRowVariableInfo, 
			boolean isFirstColumnEquivalenceClassInfo, 
			String encoding, 
			int numberOfColumns, 
			int numberOfRows) throws IOException{
		
		if((datapool != null) && (csvFilePath != null)){
		
			BufferedReader bufferedReader = null;

			try{			
					
				if((encoding != null) && (encoding.trim().length() > 0)){
					bufferedReader = new CSVBufferedReader(new InputStreamReader(new FileInputStream(csvFilePath), (encoding.trim())));
				}
				else{
					bufferedReader = new CSVBufferedReader(new InputStreamReader(new FileInputStream(csvFilePath)));
				}
					
				String line = bufferedReader.readLine();

				//Initialize to 1 since once line has already been read:
				int lineCounter = 1;

				if(isFirstRowVariableInfo){
					
					createVariablesFromFile(datapool, line, lineCounter, isFirstColumnEquivalenceClassInfo, numberOfColumns);
					
					line = bufferedReader.readLine();
					
					lineCounter++;
				}
				else{
					createVariables(datapool, line, isFirstColumnEquivalenceClassInfo, numberOfColumns);
				}
				
				int recordCounter = 0;
				
				while((line != null) && ((numberOfRows == -1) || (recordCounter < numberOfRows))){
					
					createRecord(datapool, line, lineCounter, isFirstColumnEquivalenceClassInfo);
					
					recordCounter++;
					
					line = bufferedReader.readLine();
					
					lineCounter++;
				}
			}
			finally{
				
				//Close the buffered reader:
				try {

					if(bufferedReader != null){
						bufferedReader.close();
					}
				} 
				catch (IOException i) {
					//Ignore since closing input stream reader.
				}
			}
		}
	}
	
	/**
	 * <p>Appends a Comma Separated Values (CSV) file to a datapool.</p>
	 * 
	 * <p><b>Note</b>: The CSV file is assumed to be valid.  To validate the CSV file 
	 * with the datapool, call the {@link #validateCSVFileWithDatapool(String, IDatapool, boolean, boolean, String)}
	 * method.</p>
	 * 
	 * @param  datapool The datapool that the CSV file is appended to.
	 * @param  csvFilePath The absolute path of the CSV file.
	 * @param  isFirstRowVariableInfo If the first row of the CSV file contains the variable information (name/type), otherwise <code>false</code>.
	 * @param  isFirstColumnEquivalenceClassInfo If the first column of the CSV file contains the equivalence class information (name/row index), otherwise <code>false</code>.
	 * @param  encoding The encoding of the CSV file, otherwise <code>null</null>.
	 * @throws IOException An error occurs while reading the CSV file.
	 * @see    #validateCSVFileWithDatapool(String, IDatapool, boolean, boolean, String)
	 */
	public void appendFromCSV(IDatapool datapool, 
							String csvFilePath, 
							boolean isFirstRowVariableInfo, 
							boolean isFirstColumnEquivalenceClassInfo,
							String encoding) throws IOException {
		
		if((datapool != null) && (csvFilePath != null)){
			
			BufferedReader bufferedReader = null;

			try{			
					
				if((encoding != null) && (encoding.trim().length() > 0)){
					bufferedReader = new CSVBufferedReader(new InputStreamReader(new FileInputStream(csvFilePath), (encoding.trim())));
				}
				else{
					bufferedReader = new CSVBufferedReader(new InputStreamReader(new FileInputStream(csvFilePath)));
				}
					
				String line = bufferedReader.readLine();

				//Initialize to 1 since once line has already been read:
				int lineCounter = 1;

				if(isFirstRowVariableInfo){
					
					line = bufferedReader.readLine();
					
					lineCounter++;
				}
				
				while(line != null){
					
					createRecord(datapool, line, lineCounter, isFirstColumnEquivalenceClassInfo);
										
					line = bufferedReader.readLine();
					
					lineCounter++;
				}
			}
			finally{
				
				//Close the buffered reader:
				try {

					if(bufferedReader != null){
						bufferedReader.close();
					}
				} 
				catch (IOException i) {
					//Ignore since closing input stream reader.
				}
			}
		}
	}
	
	/**
	 * <p>Exports a datapool to a Comma Separated Values (CSV) file.</p>
	 * 
	 * @param  datapool The datapool that is exported to the CSV file.
	 * @param  csvFilePath The absolute path of the CSV file.
	 * @param  includeVariableInfo If the first row of the CSV file contains the variable information (name/type), otherwise <code>false</code>.
	 * @param  includeEquivalenceClassInfo If the first column of the CSV file contains the equivalence class information (name/row index), otherwise <code>false</code>.
	 * @param  includeTags If the cell values are enclosed in tags, otherwise <code>false</code>.
	 * @param  encoding The encoding of the CSV file, otherwise <code>null</null>.
	 * @throws IOException An error occurs while writing the CSV file.
	 */
	public void exportCSV(IDatapool datapool, 
							String csvFilePath, 
							boolean includeVariableInfo, 
							boolean includeEquivalenceClassInfo,
							boolean includeTags,
							String encoding) throws IOException{
		
		if((datapool != null) && (csvFilePath != null)){
			
			BufferedWriter bufferedWriter = null;

			try{			
					
				if((encoding != null) && (encoding.trim().length() > 0)){
					bufferedWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(csvFilePath), (encoding.trim())));
				}
				else{
					bufferedWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(csvFilePath)));
				}
					
				if(includeVariableInfo){
					writeVariables(datapool, bufferedWriter, true, includeEquivalenceClassInfo);
				}
				
				writeRecords(datapool, bufferedWriter, includeEquivalenceClassInfo, includeTags);
			}
			finally{
				
				//Close the buffered writer:
				try {

					if(bufferedWriter != null){
						bufferedWriter.close();
					}
				} 
				catch (IOException i) {
					//Ignore since closing input stream reader.
				}
			}
		}
	}
	
	private void createVariablesFromFile(IDatapool datapool, String variableString, int lineNumber, boolean ignoreFirst, int numberOfColumns)
	{
		boolean first = true;
		int counter = 1;
		CSVTokenizer tokenizer = new CSVTokenizer(variableString);
		while(tokenizer.hasMoreTokens() && ((numberOfColumns == -1 || counter <= numberOfColumns)))
		{
			String nameTypePair = tokenizer.nextToken();
			if(ignoreFirst && first)
			{
				first = false;
				continue;
			}
			counter++;
			int separatorIndex = nameTypePair.indexOf(DatapoolConstants.VARIABLE_NAME_TYPE_DELIMITER); 
			String name = new String();
			String type = new String();
			if(separatorIndex == -1 )
			{
				name = nameTypePair;
			}
			else
			{
				name = nameTypePair.substring(0, separatorIndex);
				type = nameTypePair.substring(separatorIndex + 2);
			}
			
			
			IDatapoolVariable variable = datapool.constructVariable();
			variable.setName(name);
			IDatapoolSuggestedType suggestedType = (IDatapoolSuggestedType)variable.getSuggestedType();
			DatapoolSuggestedTypeChecker.getInstance().setVariableType(suggestedType, type);
			variable.setSuggestedType(suggestedType);
			datapool.appendVariable(variable);
		}
	}

	private void createVariables(IDatapool datapool, String varaibleString, boolean ignoreFirst, int numberOfColumns)
	{
		CSVTokenizer tokenizer = new CSVTokenizer(varaibleString);
		int counter = 1;
		if(ignoreFirst && tokenizer.hasMoreTokens())
			tokenizer.nextToken();
		while(tokenizer.hasMoreTokens() && (numberOfColumns == -1 || counter <= numberOfColumns))
		{
			tokenizer.nextToken();
			String name = NLS.bind(UiPluginResourceBundle.DATA_VARIABLE_NAME, String.valueOf(counter)); 
			IDatapoolVariable variable = datapool.constructVariable();
			variable.setName(name);
			datapool.appendVariable(variable);
			counter++;
		}
	}
	
	private void createRecord(IDatapool datapool, String recordString, int lineNumber, boolean useSpecifiedEquivalenceClass) {

		CSVTokenizer tokenizer = new CSVTokenizer(recordString);
		IDatapoolEquivalenceClass equivalenceClass = null;
		int recordIndex = -1;
		
		//Note: Check if the current token is not empty since the tokenizer returns empty tokens.
		if((useSpecifiedEquivalenceClass) && (tokenizer.hasMoreTokens()) && (tokenizer.currentToken().trim().length() > 0)){

			String equivalenceClassName = tokenizer.nextToken();
			int separatorIndex = equivalenceClassName.indexOf(DatapoolConstants.VARIABLE_NAME_TYPE_DELIMITER); 
			
			if(separatorIndex != -1){
				
				try{
					recordIndex = Integer.parseInt(equivalenceClassName.substring(separatorIndex + DatapoolConstants.VARIABLE_NAME_TYPE_DELIMITER.length()));
				}
				catch(NumberFormatException n){
					//Ignore and append the record. 
				}
				
				equivalenceClassName = equivalenceClassName.substring(0, separatorIndex);
			}
						
			int equivalenceClassIndex = datapool.getEquivalenceClassIndex(equivalenceClassName);
			
			if(equivalenceClassIndex == -1){

				equivalenceClass = datapool.constructEquivalenceClass();
				equivalenceClass.setName(equivalenceClassName);
				
				datapool.appendEquivalenceClass(equivalenceClass);
			}
			else{
				equivalenceClass = ((IDatapoolEquivalenceClass)(datapool.getEquivalenceClass(equivalenceClassIndex)));
			}		
		}
		else{
			
			//By default, import all records into the default equivalence class:
			if(datapool.getEquivalenceClassCount() == 0){
				
				equivalenceClass = datapool.constructEquivalenceClass();
				
				datapool.appendEquivalenceClass(equivalenceClass);
			}
			else{				
				equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(Math.max(0, datapool.getDefaultEquivalenceClassIndex()));
			}
		}
		
		//Resolve the remaining cells from the record:
		List cells = new ArrayList();
		
		while(tokenizer.hasMoreTokens()){
			cells.add(tokenizer.nextToken());
		}
		
		IDatapoolRecord record = equivalenceClass.constructRecord(((String[])(cells.toArray(new String[cells.size()]))));
		int recordCount = equivalenceClass.getRecordCount();
		
		if((recordIndex > -1) && (recordIndex < recordCount) && (equivalenceClass.getRecord(recordIndex) == null)){
			equivalenceClass.insertRecord(record, recordIndex);				
		}
		else{
			equivalenceClass.appendRecord(record);
		}
	}
	
	private void writeVariables(IDatapool datapool, BufferedWriter writer, boolean includeTypes, boolean includeEquivalenceClasses)
	{
		String output = new String();
		int count = datapool.getVariableCount();
		if(count != 0)
		{			
			if(includeEquivalenceClasses)
			{
				output = output.concat(SEPERATOR);
			}
			for(int i = 0; i < count; i++)
			{
				IDatapoolVariable variable = (IDatapoolVariable)datapool.getVariable(i);
				String name = variable.getName();
				String separator = new String();
				if(i != count - 1)
					separator = SEPERATOR;
				if(includeTypes)
				{
					String type = variable.getSuggestedType().getSuggestedClassName();
					String preprocessedOutput = name.concat(DatapoolConstants.VARIABLE_NAME_TYPE_DELIMITER).concat(type);
					preprocessedOutput = processString(preprocessedOutput);
					output = output.concat(preprocessedOutput).concat(separator);
				}
				else
				{
					name = processString(name);
					output = output.concat(name).concat(separator);
				}	
			}
			output = output.concat(LINEFEED);
		}
		else
		{
			output = LINEFEED;
		}
		try
		{
			writer.write(output);
		}
		catch(IOException e)
		{
		}
	}
	
	private void writeRecords(IDatapool datapool, BufferedWriter writer, boolean includeEquivalenceClasses, boolean includeTags)
	{
		StringBuffer output = new StringBuffer();
		for(int i = 0; i < datapool.getEquivalenceClassCount(); i++)
		{
			IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(i);
			String equivalenceClassName = new String();
			if(includeEquivalenceClasses)
			{
				equivalenceClassName = equivalenceClass.getName();
				equivalenceClassName = equivalenceClassName.concat(DatapoolConstants.VARIABLE_NAME_TYPE_DELIMITER);
			}
			for(int j = 0; j < equivalenceClass.getRecordCount(); j++)
			{
				IDatapoolRecord record = (IDatapoolRecord)equivalenceClass.getRecord(j);
				if(includeEquivalenceClasses)
				{
					String preprocessedOutput = equivalenceClassName.concat(String.valueOf(j));
					preprocessedOutput = processString(preprocessedOutput);
					output.append(preprocessedOutput).append(SEPERATOR);
				}
				int count = record.getCellCount();
				for(int k = 0; k < count; k++)
				{
					IDatapoolCell cell = (IDatapoolCell)record.getCell(k);
					String cellValue = new String();
					if(includeTags)
						cellValue = cell.getPersistedRepresentation();
					else
						cellValue = cell.getStringValue();
					
					output.append(processString(cellValue));
					if(k != count - 1)
						output.append(SEPERATOR);
				}
				output.append(CSVBufferedReader.LINE_SEPARATOR);
			}
		}
		try
		{
			writer.write(output.toString());
		}
		catch(IOException e)
		{
		}				
	}
	
	private String processString(String value)
	{
		if(value == null)
			return null;

		// https://bugs.eclipse.org/bugs/show_bug.cgi?id=117932
		// performance optimization
		if(value.indexOf(CSVTokenizer.DOUBLEQUOTE) > -1)
		{
			value = value.replaceAll(CSVTokenizer.DOUBLEQUOTE, CSVTokenizer.TWO_DOUBLEQUOTES);
			return (CSVTokenizer.DOUBLEQUOTE + value + CSVTokenizer.DOUBLEQUOTE);
		}
		
		if(value.indexOf(CSVTokenizer.COMMA) > -1 ||
			value.indexOf(LINEFEED) > -1 ||
			value.indexOf(CARRIAGE_RETURN) > -1 ||
			value.trim().length() < value.length())
			return (CSVTokenizer.DOUBLEQUOTE + value + CSVTokenizer.DOUBLEQUOTE);
	
		return value;
	}
}
