/**********************************************************************
 * Copyright (c) 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DatapoolUtilities.java,v 1.1 2008/11/03 13:30:31 bjerome Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.datapool.util;

import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.edit.datapool.IDatapoolEquivalenceClass;
import org.eclipse.hyades.edit.datapool.IDatapoolVariable;


/** 
 * various utilities for name check on datapools
 * @author jbozier
 * @since October 7, 2008
 * @version October 7, 2008
 * @provisional As of TPTP 4.5.2, this is stable provisional API
 */

public class DatapoolUtilities {
	private static final DatapoolUtilities instance = new DatapoolUtilities();

	/**
	 * list of invalid characters for variable and equivalence class name
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public static final String DATA_DLG_INVALID_NAME_CHARACTERS = "\t\n\r\"\\\',./<>?;:|!@#%^&*()-=+[]{}`~";
	/**
	 * list of invalid characters for type name
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */	
	public static final String DATA_DLG_INVALID_TYPE_CHARACTERS = "\t\n\r\"\\\' ,/<>?;:|!@#%^&*()-=+[]{}`~";
	/**
	 * list of invalid characters for first char of type name
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */	
	public static final String DATA_DLG_INVALID_TYPE_FIRST_CHARACTER	= "01234567890";

	/**
	 * singleton class
	 * @return instance of DatapoolUtilities
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public static DatapoolUtilities getInstance() {
		return instance;
	}
	
	/**
	 * protected constructor (singleton class)
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	protected DatapoolUtilities() {
	}
	
	/**
	 * check if a value contains some invalid characters
	 * @param value : value to check
	 * @param invalidCharacters : list of invalid characters
	 * @return true if value contains at least one invalid character
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public boolean hasInvalidCharacters(String value, String invalidCharacters) {
		if(invalidCharacters == null) {
			return false;
		}
		for(int i = 0; i < invalidCharacters.length(); i++) {
			String character = invalidCharacters.substring(i, i+1);
			if(value.lastIndexOf(character) != -1) {
				return true;	
			}
		}
		return false;		
	}

	/**
	 * check if datapool have an unique equivalence class name
	 * @param datapool : checked datapool
	 * @param name : checked name
	 * @param excludeEquivalenceClass : equivalence class to exclude from the check
	 * @return true if name is not used as equivalence class name, outside excludeEquivalenceClass
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public boolean isEquivalenceClassNameUnique(IDatapool datapool, String name, IDatapoolEquivalenceClass excludeEquivalenceClass) {
		if(datapool == null) {
			return true;
		}
		for(int i = 0; i < datapool.getEquivalenceClassCount(); i++) {
			IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(i);
			if(!equivalenceClass.equals(excludeEquivalenceClass) && name.equals(equivalenceClass.getName())) {							
				return false;
			}
		}
		return true;
	
	}
	
	/**
	 * check if name is valid as an equivalence class name
	 * (check DATA_DLG_INVALID_NAME_CHARACTERS for list of invalid char)
	 * @param name : name to check
	 * @return true if name is valid 
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public boolean isEquivalenceClassNameValid(String name) {
		if(name.length() == 0) {
			return false;		
		}
		return !hasInvalidCharacters(name, DATA_DLG_INVALID_NAME_CHARACTERS);
	}
	
	/**
	 * check if datapool have an unique variable class name
	 * @param datapool : checked datapool
	 * @param name : checked name
	 * @param excludeVariable : variable class to exclude from the check
	 * @return true if name is not used as variable class name, outside excludeVariable
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public boolean isVariableNameUnique(IDatapool datapool, String name, IDatapoolVariable excludeVariable) {
		if(datapool == null) {
			return true;
		}
		for(int i = 0; i < datapool.getVariableCount(); i++) {
			IDatapoolVariable variable = (IDatapoolVariable)datapool.getVariable(i);
			if(!variable.equals(excludeVariable) && name.equals(variable.getName())) {
				return false;
			}
		}
		return true;			
	}
	
	/**
	 * check if name is valid as a variable name
	 * (check DATA_DLG_INVALID_NAME_CHARACTERS for list of invalid char)
	 * @param name : name to check
	 * @return true if name is valid 
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public boolean isVariableNameValid(String name) {
		if(name.length() == 0) {
			return false;		
		}
		return !hasInvalidCharacters(name, DATA_DLG_INVALID_NAME_CHARACTERS);
	}
	
	/**
	 * check if type is valid as a type name
	 * (check DATA_DLG_INVALID_TYPE_CHARACTERS for list of invalid char 
	 *  and DATA_DLG_INVALID_TYPE_FIRST_CHARACTER for list of invalid first char)
	 * @param type : type name to check
	 * @return true if type name is valid
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public boolean isVariableTypeValid(String type) {
		if(type.length() == 0) {
			return true;
		}
		if(hasInvalidCharacters(type.substring(0,1), DATA_DLG_INVALID_TYPE_FIRST_CHARACTER)) {
			return false;
		}
		return !hasInvalidCharacters(type, DATA_DLG_INVALID_TYPE_CHARACTERS);
	}
}
