/********************************************************************** 
 * Copyright (c) 2008, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: DatapoolCheck.java,v 1.8 2010/03/31 19:18:07 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.ui.dialog;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.hyades.models.common.datapool.DPLDatapool;
import org.eclipse.hyades.models.common.util.EncryptionManager;
import org.eclipse.hyades.test.core.launch.extensions.IPassInfoCollector;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.datapool.internal.dialog.DatapoolBaseDialog;
import org.eclipse.hyades.test.ui.datapool.internal.dialog.DatapoolConstants;
import org.eclipse.hyades.test.ui.datapool.internal.util.GridDataUtil;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

/**
 * <p>Implementation of <code>launchDatapoolCheckPass</code> extension point that provides
 * a dialog to prompt user input password for an encrypted datapool when launching a test 
 * from the workbench.</p>
 *
 * 
 * @author  Huang Xin Ying 
 * @author  Paul E. Slauenwhite
 * @version March 31, 2010
 * @since   March 20, 2008
 */
public class DatapoolCheck implements IPassInfoCollector {

	public DatapoolCheck() {
	}
    
	/**
	 * Create a dialog to prompt user input the passwords for encrypted datapools
	 *  while running a test suite .
	 * @param dp
	 * @param pass
	 * @pass isContinue
	 */
	public void execute(DPLDatapool[] dp, Map pass, List isContinue) {
		DialogRun dr = new DialogRun();
		dr.setDp(dp);
		dr.setContinue(isContinue);
		dr.setPass(pass);
		Display.getDefault().syncExec(dr);
	}

	private class DialogRun implements Runnable {
		private DPLDatapool[] dp;
		private Map pass;
		private List isContinue;

		public void setDp(DPLDatapool[] dp) {
			this.dp = dp;
		}

		public void setPass(Map pass) {
			this.pass = pass;
		}

		public void setContinue(List isContinue) {
			this.isContinue = isContinue;
		}

		public void run() {
			String testId = null;
			if(isContinue != null && isContinue.size() == 1){
				testId = (String)isContinue.get(0);
				isContinue.clear();
			}
			DatapoolCheckDialog db = new DatapoolCheckDialog(null, 
					dp, 
					UiPluginResourceBundle.DatapoolDialog_INPUTKEYDIALGOTITLE,
					pass,
					testId);
			db.setBlockOnOpen(true);
			if (db.open() == IDialogConstants.OK_ID) {
				isContinue.add(Boolean.TRUE.toString());
			}
		}
	}
    
	private class DatapoolCheckDialog extends DatapoolBaseDialog implements KeyListener {

		private DPLDatapool[] dp;
		private String title;
		private Label[] name;
		private Text[] pass;
		private boolean[] status;
		private List unCorrectList = new ArrayList();
		private Map passed;
		private String testId;

		/**
		 * @param parentShell
		 */
		public DatapoolCheckDialog(Shell parentShell) {
			super(parentShell);
		}

		public DatapoolCheckDialog(Shell parentShell, DPLDatapool[] dp,
				String title, Map passed,String testId) {
			super(parentShell);
			this.dp = dp;
			this.title = title;
			this.passed = passed;
			this.testId = testId;
		}

		/* modified the content */
		protected Control createDialogArea(Composite parent) {
			
			getShell().setText(title);
			
			Composite superComposite = ((Composite)(super.createDialogArea(parent)));

			GridData gridData = GridDataUtil.createFill();
			gridData.minimumWidth = 300;
			
			superComposite.setLayoutData(gridData);		
			superComposite.setLayout(new GridLayout());

			GridLayout gridLayout = new GridLayout();
			gridLayout.numColumns = 2;

			Group descriptionGroup = new Group(superComposite, SWT.SHADOW_ETCHED_IN);
			descriptionGroup.setText(UiPluginResourceBundle.DatapoolDialog_INPUTPASSHINT);
			descriptionGroup.setLayoutData(gridData);
			descriptionGroup.setLayout(gridLayout);

			if ((dp != null) && (dp.length > 0)) {

				name = new Label[dp.length];
				pass = new Text[dp.length];
				status = new boolean[dp.length];
				
				for (int counter = 0; counter < dp.length; counter++) {
					
					name[counter] = new Label(descriptionGroup, SWT.NONE);
					
					String datapoolName = dp[counter].getName();
					int splitMarkIndex = datapoolName.indexOf(DatapoolConstants.VARIABLE_NAME_TYPE_DELIMITER);
					
					if(splitMarkIndex != -1){
						name[counter].setText(NLS.bind(UiPluginResourceBundle.DatapoolDialog_DATAPOOLNAMELABEL, datapoolName.substring(0, splitMarkIndex)));						
					}
					else{
						name[counter].setText(NLS.bind(UiPluginResourceBundle.DatapoolDialog_DATAPOOLNAMELABEL, datapoolName));												
					}

					pass[counter] = new Text(descriptionGroup, SWT.PASSWORD | SWT.BORDER);
					pass[counter].setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
					pass[counter].addKeyListener(this);

					status[counter] = false;		
				}
			}
			
			superComposite.pack();
			
			return superComposite;
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.jface.dialogs.Dialog#createContents(org.eclipse.swt.widgets.Composite)
		 */
		protected Control createContents(Composite parent){
			
			Control control = super.createContents(parent);
			
			getOKButton().setEnabled(false);
			
			return control;
		}
		
		public void okPressed(){
			enableOK();
		}
		
		private void enableOK(){
			// check all the passwords
			for (int i = 0; i < dp.length; i++) {
				try {
					if (status[i]) {
						continue;
					} else if (dp[i].getChallenge().equals(
							EncryptionManager.EncoderByMd5(pass[i].getText()))) {
						
						if(passed.get(testId) == null){
							passed.put(testId, new HashMap());
						}
						HashMap thm = (HashMap)passed.get(testId);
						thm.put(dp[i].getName(), pass[i].getText());
						
						pass[i].setEnabled(false);
						status[i] = true;
						if (unCorrectList != null
								&& unCorrectList.contains(dp[i])) {
							unCorrectList.remove(dp[i]);
						}
					} else {
						pass[i].setText("");
						if (!unCorrectList.contains(dp[i]))
							unCorrectList.add(dp[i]);
					}
				} catch (Exception er) {
					UiPlugin.logError(er);
				}
			}
			// if all status is true, then set ok button can be pressed
			if (unCorrectList.size() == 0) {
				super.okPressed();
				return;
			}

			StringBuffer toShow = new StringBuffer();
			toShow
					.append(UiPluginResourceBundle.DatapoolDialog_PASSWORDSHINT);
			for (int i = 0; i < unCorrectList.size(); i++) {
				toShow.append(((DPLDatapool) unCorrectList.get(i))
						.getName().split(DatapoolConstants.VARIABLE_NAME_TYPE_DELIMITER)[0]);
				if (i < unCorrectList.size() - 1) {
					toShow.append(",\n");
				}
			}

			MessageDialog.openError(null, UiPluginResourceBundle.W_ERROR, toShow.toString());
		}

		public void keyPressed(KeyEvent e) {}

		public void keyReleased(KeyEvent e) {
			
			if(e.keyCode == SWT.CR){
				enableOK();			
			}
			else{
				getOKButton().setEnabled(true);
			}
		}
	}
}
