/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BaseEditorExtension.java,v 1.30 2010/04/12 12:38:50 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.editor.extension;

import java.lang.reflect.InvocationTargetException;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.loaders.util.Guid;
import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.editor.extension.resources.EditorExtensionMessages;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.FileProxyManager;
import org.eclipse.hyades.test.ui.navigator.CMNNamedElementProxyNode;
import org.eclipse.hyades.test.ui.navigator.IFileProxyManager;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.util.TestUIUtil;
import org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter;
import org.eclipse.hyades.ui.editor.EditorExtension;
import org.eclipse.hyades.ui.editor.IHyadesEditorExtension;
import org.eclipse.hyades.ui.editor.IHyadesEditorPart;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.ModalContext;
import org.eclipse.jface.util.ListenerList;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorActionBarContributor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.dialogs.SaveAsDialog;
import org.eclipse.ui.part.EditorActionBarContributor;

/**
 * <p>BaseEditorExtension.java</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Bertrand Cormier
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite
 * @version April 12, 2010
 * @since   February 1, 2005
 * @deprecated The implementation of this class is not based on Eclipse Forms(a.k.a. "flat" look) 
 * framework that was made available publicly in Eclipse 3.0. If you would like your editor(s) to be 
 * built with the Eclipse Forms technology, this is NOT the class to extend. To know more about 
 * Eclipse Forms: <br>
 * <a href="http://dev.eclipse.org/viewcvs/index.cgi/%7Echeckout%7E/pde-ui-home/working/EclipseForms/EclipseForms.html">Eclipse Forms Programming Guide</a>
 */
public abstract class BaseEditorExtension extends EditorExtension implements ISynchronizedEditorAdapter, IHyadesEditorExtension {
	
	private ListenerList writeAccessChangeListeners = null;
	private Clipboard clipboard = null;
	private boolean dirty = false;
	 
	/**
	 * The constructor for the class.
	 */
	public BaseEditorExtension()
	{
		writeAccessChangeListeners = new ListenerList();
	}
	
	/**
	 * Disposes any resource used by this object. There are no guarantees that 
	 * this method is invoked by this object's container or by the Hyades framework.
	 */
	public void dispose()
	{
		if(clipboard != null)
		{
			clipboard.dispose();
			clipboard = null;
		}
		writeAccessChangeListeners.clear();
		
		super.dispose();
	}
	
	/**
	 * Returns an object which is an instance of the given class
	 * associated with this object. Returns <code>null</code> if
	 * no such object can be found.
	 *
	 * @param adapter the adapter class to look up
	 * @return a object castable to the given class, 
	 *    or <code>null</code> if this object does not
	 *    have an adapter for the given class
	 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
	 */
	public Object getAdapter(Class adapter)
	{
		if(ISynchronizedEditorAdapter.class == adapter)
			return this;
			
		return super.getAdapter(adapter);
	}

	/**
	 * Returns this editor extension's resource set.
	 * @return ResourceSet
	 */
	protected ResourceSet getResourceSet()
	{
		return ((EObject)getHyadesEditorPart().getEditorObject()).eResource().getResourceSet();	
	}
	
	/**
	 * Returns the extension editor's clipboard.
	 * @return Clipboard
	 */
	public Clipboard getClipboard()
	{
		if(clipboard == null)
			clipboard = new Clipboard(getHyadesEditorPart().getEditorPart().getEditorSite().getShell().getDisplay());
		return clipboard;
	}
	
	/**
	 * Adds a write access change listener to the list of registered listeners.
	 * @param listener the write access change listener to add
	 */
	public void addListener(IWriteAccessChangeListener listener)
	{
		writeAccessChangeListeners.add(listener);
	}
	
	/**
	 * Removes a write access change listener from the list of registered listeners.
	 * @param listener the write access change listener to remove
	 */
	public void removeListener(IWriteAccessChangeListener listener)
	{
		writeAccessChangeListeners.remove(listener);
	}
	
	/**
	 * Notifies a write access change to the registered listeners.
	 * @param file the resource from which the write access change has been raised
	 * @param isReadOnly the write access change that has been raised
	 */
	protected void notifyWriteAccessChangeListeners(IFile file, boolean isReadOnly)
	{
		if(file == null)
			return;
			
		Object[] listeners = writeAccessChangeListeners.getListeners();
		for(int i = 0, maxi = listeners.length; i < maxi; i++)
			((IWriteAccessChangeListener)listeners[i]).writeAccessChanged(file, isReadOnly);
	}
	
	/**
	 * Marks the editor as dirty.
	 */
	public synchronized void markDirty()
	{
		if(!dirty)
		{
			dirty = true;
			getHyadesEditorPart().firePropertyChange(IHyadesEditorPart.PROP_DIRTY);
		}
	}
	
	/**
	 * Clears the dirty flag without refreshing the editor.
	 */
	protected void clearDirty()
	{
		dirty = false;
	}
	
	/**
	 * Returns whether the contents of this editor extension have changed since the 
	 * last save operation.
	 * @return <code>true</code> if the contents have been modified and need
	 * saving, and <code>false</code> otherwise.
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#isDirty()
	 */
	public boolean isDirty()
	{
		return dirty;
	}

	/**
	* called by doSave() to allow the subclass to verify the editor contents before saving the
	* file.
	* 
	* @return true to continue the save operation, false to abort the save
	*/
	protected boolean isOkToSave()
	{
		/*
		 * Always return true here.  A subclass could do some processing of its own
		 * and return false to abort the save if it needs to...
		 */
		return true;
	}
	
	/**
	 * Saves the contents of this editor extension.
	 * @param monitor the progress monitor
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#doSave(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public final void doSave(IProgressMonitor monitor) {
		if (!isValidToSave()) {
			return;
		}
		if (this.isOkToSave()) {
			WorkspaceModifyOperation operation = new WorkspaceModifyOperation() {
				protected void execute(IProgressMonitor progressMonitor)
						throws CoreException, InvocationTargetException,
						InterruptedException {
					try {
						save(progressMonitor);
						clearDirty();
					} catch (OperationCanceledException e) {
						// OK
					} catch (Exception e) {
						handleException(e);
					}
				}
			};
			IActionBars actionBars = getHyadesEditorPart().getEditorPart()
					.getEditorSite().getActionBars();
			if ((actionBars != null)
					&& (actionBars.getStatusLineManager() != null)) {
				actionBars.getStatusLineManager().setErrorMessage(null);
				actionBars.getStatusLineManager().setMessage(null);
				monitor = actionBars.getStatusLineManager()
						.getProgressMonitor();
				actionBars.getStatusLineManager().setCancelEnabled(false);
			} else if (monitor == null) {
				monitor = new NullProgressMonitor();
			}
			try {
				ModalContext.run(operation, false, monitor, Display
						.getCurrent());
			} catch (Exception e) {
				handleException(e);
			}
		}
	}
	
	/**
	 * Saves the contents of this editor extension.
	 * @param monitor the progress monitor to report the work
	 * @throws Exception this method is supposed to throw any exception.
	 */
	public void save(IProgressMonitor monitor)
	throws Exception
	{
		Resource[] resources = (Resource[])getResourceSet().getResources().toArray(new Resource[getResourceSet().getResources().size()]);
		int length = resources.length;
		int counter = 0;
		monitor.beginTask("", length); //$NON-NLS-1$
		
		IFileProxyManager fileProxyManager = TestNavigator.getFileProxyManager();

		if (fileProxyManager == null) {
			fileProxyManager = new FileProxyManager();
		} 
		
		for(int i = 0, maxi = resources.length; i < maxi; i++)
		{
			if(resources[i].isModified())
			{
				logSaveMessage(monitor, resources[i], length, counter++);
				
				EMFUtil.save(resources[i]);

				if (resources[i] instanceof IFile) {
					fileProxyManager.updateProxy(((IFile)(resources[i])), null);
				}
			}
			monitor.worked(1);
		}
		monitor.done();
	}
	
	private void displayErrorOnSaveValidation(String message) {
		MessageDialog.openError(getHyadesEditorPart().getEditorPart().getSite().getShell(), EditorExtensionMessages._8, message);
	}
	
	/**
	 * <p>Determines if the new logical name is valid.</p>
	 *  
	 * <p>Validity is based on the following conditions:</p>
	 * 
	 * <ul>
	 * <li>The editor object is not <code>null</code> and an instance of {@link CMNNamedElement}.</li>
	 * <li>The new logical name is not <code>null</code> or empty.</li>
	 * <li>The new logical name is valid according to {@link TestUIUtil#isLogicalNameLengthValid(String)}.</li>
	 * <li>The new logical name is valid according to {@link TestUIUtil#isLogicalNameValid(String)}.</li>
	 * <li>The new logical name is valid according to {@link TestUIUtil#isLogicalNameAvailable(String, org.eclipse.core.resources.IContainer)}.</li>
	 * </ul>
	 * 
	 * <p>This method will return <code>true</code> is the new logical name is the same as the old logical name (case insensitive on Windows).</p>

	 * <p>Note: Subclasses should override this method and always return <code>true</code> to disable rename validation.</p>
	 * 
	 * @return <code>true</code> if all of the validity conditions are satisfied (see above), otherwise <code>false</code>.
	 * @see TestUIUtil#isLogicalNameLengthValid(String)
	 * @see TestUIUtil#isLogicalNameValid(String)
	 * @see TestUIUtil#isLogicalNameAvailable(String, org.eclipse.core.resources.IContainer)
	 */
	protected boolean isValidToSave() {
		
		EObject editorEObject = ((EObject)(getHyadesEditorPart().getEditorObject()));
		
		if ((editorEObject != null) && (editorEObject instanceof CMNNamedElement)) {
			
			CMNNamedElement editorNamedElement = ((CMNNamedElement)(editorEObject));
			
			String logicalName = editorNamedElement.getName();
			
			//Check 1: Check if the logical name is missing:
			if ((logicalName == null) || (logicalName.trim().length() == 0)){
				
				displayErrorOnSaveValidation(EditorExtensionMessages.SaveValidationErrorMsg_0);
				
				return false;
			}

			//Check 2:  Check if the logical name is the same:			
			//Assumption: The workspace file exists since the editor is open on the workspace file.
			IFileProxyManager fileProxyManager = TestNavigator.getFileProxyManager();

			if (fileProxyManager == null) {
				fileProxyManager = new FileProxyManager(); 
			}
			
			IFile editorFile = EMFUtil.getWorkspaceFile(editorNamedElement.eResource());
			IProxyNode proxy = fileProxyManager.getProxy(editorFile, null);
			
			if ((proxy != null) && (proxy instanceof CMNNamedElementProxyNode)) {

				if (TestUIUtil.IS_WINDOWS_OS){

					if (((CMNNamedElementProxyNode)(proxy)).getName().equalsIgnoreCase(logicalName)) {
						return true;
					}					
				} 
				else if (((CMNNamedElementProxyNode)(proxy)).getName().equals(logicalName)) {
					return true;
				}
			}

			//Check 3:  Check if the logical name length is invalid, logical name is invalid, and logical name is available:
			if ((!TestUIUtil.isLogicalNameLengthValid(logicalName)) || (!TestUIUtil.isLogicalNameValid(logicalName)) || (!TestUIUtil.isLogicalNameAvailable(logicalName, editorFile.getParent()))) {

				displayErrorOnSaveValidation(EditorExtensionMessages.SaveValidationErrorMsg_3);

				return false;
			}
		}
		
		return true;
	}
	
	/**
	 * Displays a message when saving a resource.
	 * e.g : (3 of 5) myResource.res
	 * @param monitor the progress monitor
	 * @param resource the current resource to save
	 * @param total the total number of resources to save
	 * @param current the number of the current resource
	 */
	protected void logSaveMessage(IProgressMonitor monitor, Resource resource, int total, int current)
	{
		monitor.subTask(""); //$NON-NLS-1$
	}
	
	/**
	 * Handles any exception thrown during the 
	 * {@link #save(IProgressMonitor)} execution.
	 * @param exception
	 */
	protected void handleException(Exception exception)
	{
		UiPlugin.logError(exception);
		String filePath = EMFUtil.getFilePath((EObject)getHyadesEditorPart().getEditorObject());
		UIUtil.openSaveFileErrorDialog(getHyadesEditorPart().getEditorPart().getSite().getShell(), filePath, exception);
	}
	
	/**
	 * Reloades the editor's object.  This method is invoked by the synchronizer
	 * adapter and should return <code>true</code> if the synchronizer is not
	 * supposed to continue with the default treatment.
	 * @return boolean
	 */
	protected boolean reloadEditorObject()
	{
		Resource resource = ((EObject)getHyadesEditorPart().getEditorObject()).eResource(); 
		
		resource.unload();
		try
		{
			resource.load(EMFUtil.RESOURCE_OPTIONS);
		}
		catch(Exception e)
		{
			UiPlugin.logError(e);
		}
		
		if(!resource.getContents().isEmpty())
			getHyadesEditorPart().setEditorObject(resource.getContents().get(0));
		else
			getHyadesEditorPart().setEditorObject(null);

		return false;
	}
	
	/**
	 * Returns the complete EMF URI which includes the resource information.
	 * @param object
	 * @return
	 */
	protected URI getObjectURI(Object object)
	{
		if(object != null)
		{
			if(object instanceof EObject)
			{
				EObject eObject = (EObject)object;
				if(eObject.eResource() != null)
					return eObject.eResource().getURI().appendFragment(eObject.eResource().getURIFragment(eObject));
			}
		}
		
		return null;
	}
	
	/**
	 * This method is invoked when the editor file has been changed.
	 * @return <code>true</code> if the editor synchronizer should not
	 * perform the default treatment.
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#editorFileChanged()
	 */
	public boolean editorFileChanged()
	{
		return false;
	}

	/**
	 * This method is invoked when the editor file has been deleted.
	 * @return <code>true</code> if the editor synchronizer should not
	 * perform the default treatment.
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#editorFileDeleted()
	 */
	public boolean editorFileDeleted()
	{
		return false;
	}

	/**
	 * This method is invoked when the editor file has been changed.
	 * @return <code>true</code> if the editor synchronizer should not
	 * perform the default treatment.
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#editorFileWriteAccessChanged(boolean)
	 */
	public boolean editorFileWriteAccessChanged(boolean isReadOnly)
	{
		notifyWriteAccessChangeListeners(EMFUtil.getWorkspaceFile((EObject)getHyadesEditorPart().getEditorObject()), isReadOnly);
		return false;
	}

	/**
	 * Handles modifications in one or more monitored files.  The
	 * lists contains the modified IFiles.
	 * @param changedFiles
	 * @param removedFiles
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#handle(java.util.List, java.util.List)
	 */
	public void handle(List changedFiles, List removedFiles)
	{

	}

	/**
	 * Triggers a reload in the editor.
	 * @return <code>true</code> if the editor was reloaded
	 * or <code>false</code> otherwise. 
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#reload()
	 */
	public boolean reload()
	{
		String fragment = null;
		IStructuredSelection structuredSelection = getSelection();
		if(structuredSelection.size() == 1)
		{
			Object selection = structuredSelection.getFirstElement();
			if(selection instanceof EObject)
				fragment = ((EObject)selection).eResource().getURIFragment((EObject)selection);
		}
		
		if(!reloadEditorObject())
		{
			Object editorObject = getHyadesEditorPart().getEditorObject();
			// clean up resource set to avoid ghost elements
			if (editorObject instanceof EObject) {
				Resource res = ((EObject)editorObject).eResource();
				ResourceSet resset = getResourceSet();
				Object allres[] = getResourceSet().getResources().toArray();
				for (int i=0; i<allres.length; i++) {
					if (!((Resource)allres[i]).getURI().equals(res.getURI())) {
						resset.getResources().remove(allres[i]);
					}
				}
			}
			refreshContent(editorObject);
			
			if((editorObject instanceof EObject) && (fragment != null))
			{
				Object selection = ((EObject)(editorObject)).eResource().getEObject(fragment);
				if(selection != null)
					structuredSelection = new StructuredSelection(selection);
				else
					structuredSelection = StructuredSelection.EMPTY;
					
				setSelection(structuredSelection);
			}
			
			dirty = false;
			getHyadesEditorPart().firePropertyChange(IHyadesEditorPart.PROP_DIRTY);			
		}
		
		return true;
	}
	
	/**
	 * This method is invoked when the editor content should be saved.
	 * @param wasDeleted indicates whether the editor file has been deleted.
	 * @return <code>true</code> if the editor synchronizer should not
	 * perform the default treatment.
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#doSaveEditorFile(boolean)
	 */
	public boolean doSaveEditorFile(boolean wasDeleted)
	{
		getHyadesEditorPart().getEditorPart().doSave(new NullProgressMonitor());
		return false;
	}

    /**
     * Returns the status line manager of this editor.
     * @return the status line manager of this editor
     */
    private IStatusLineManager getStatusLineManager() {

        IEditorActionBarContributor contributor = getHyadesEditorPart().getEditorPart().getEditorSite().getActionBarContributor();
        if (!(contributor instanceof EditorActionBarContributor)) {
            return null;
        }

        IActionBars actionBars = ((EditorActionBarContributor) contributor).getActionBars();
        if (actionBars == null) {
            return null;
        }

        return actionBars.getStatusLineManager();
    }

    protected IProgressMonitor getProgressMonitor() {
        IProgressMonitor pm = null;

        IStatusLineManager manager = getStatusLineManager();
        if (manager != null)
            pm = manager.getProgressMonitor();

        return pm != null ? pm : new NullProgressMonitor();
    }

    public void doSaveAs() {
        performSaveAs(getProgressMonitor());
    }
    
    private void performSaveAs(IProgressMonitor progressMonitor) {
        
    	//See BaseFormEditor.doSaveAs() for a similar algorithm.
    	boolean isCanceled = false;

    	try {
    		
    		//Create, configure, and open the Save As dialog: 
    		SaveAsDialog dialog = new SaveAsDialog(getHyadesEditorPart().getEditorPart().getEditorSite().getShell());        
    		
    		//Resolve the file editor input and configure the Save As dialog:
    		IEditorInput editorInput = getHyadesEditorPart().getEditorPart().getEditorInput();

    		if(editorInput instanceof IFileEditorInput){

    			IFile originalFile = ((IFileEditorInput)(editorInput)).getFile();

    			if(originalFile != null){
    				dialog.setOriginalFile(originalFile);
    			}
    		}

    		dialog.create();

    		if (dialog.open() == Window.OK) {

    			//Resolve the new file path:
    			IPath newFilePath = dialog.getResult();

    			if (newFilePath != null) {

        			//Resolve the new file:
    				IFile newFile = ResourcesPlugin.getWorkspace().getRoot().getFile(newFilePath);

    				try {

    					//Resolve editor object:
    					Object editorObject = getHyadesEditorPart().getEditorObject();

    					if (editorObject instanceof EObject) {

    						//Resolve editor eObject and resource:
    						EObject eObject = ((EObject)(editorObject));
    						Resource resource = eObject.eResource();

    						if((resource != null) && (eObject instanceof CMNNamedElement)) {

    							CMNNamedElement namedElement = ((CMNNamedElement)(eObject));

    							//Capture the original URI, ID, and name:
    							URI originalURI = resource.getURI();
    							String originalId = namedElement.getId();
    							String originalName = namedElement.getName();
    							
    							try{

    								//Disable notifications:
    								namedElement.eSetDeliver(false);

    								//Set the new URI, ID, and name:            		
    								resource.setURI(URI.createPlatformResourceURI(newFilePath.toString(), false));
    								namedElement.setId(new Guid().toString());
    								namedElement.setName(newFilePath.removeFileExtension().lastSegment());

    								//Save the new test asset:
    								EMFUtil.save(resource);
    							}
    							finally{

    								//Set the original URI, ID, and name:    
    								resource.setURI(originalURI);
    								namedElement.setId(originalId);
    								namedElement.setName(originalName);       

    								//Re-enable notifications:
    								namedElement.eSetDeliver(true);
    							}

    							//Attempt to open the new test asset in its editor:
    							TestUIUtil.openEditor(newFile, getHyadesEditorPart().getEditorPart().getEditorSite().getId());

    							isCanceled = false;
    						} 
    					}
    				} 
    				catch (Exception e) {
    					TestCorePlugin.getDefault().logError(e);
    				}
    			}
    		}
    	}
    	finally{

    		if (progressMonitor != null) {
    			progressMonitor.setCanceled(isCanceled);
    		}
    	}
    }

    public boolean isSaveAsAllowed() {
        return true;
    }
}
