/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DatapoolChildrenTreeSection.java,v 1.6 2010/03/29 13:52:36 paules Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.editor.form.util;

import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.test.ui.adapter.NamedElementPropertySource;
import org.eclipse.hyades.test.ui.adapter.PropertySourceLabelProvider;
import org.eclipse.hyades.test.ui.internal.editor.form.util.TreeSection;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TreeEditor;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.actions.SelectionListenerAction;

/**
 * <p>The <i>Equivalence Classes</i> and <i>Variables</i> sections in the Overview tab 
 * of the Datapool editor.</p>
 * 
 * 
 * @author  Bianca Xue Jiang
 * @author  Paul Slauenwhite
 * @version March 29, 2010
 * @since   March 2, 2005
 */
public abstract class DatapoolChildrenTreeSection extends TreeSection implements MouseListener, KeyListener {
	
	private IAction[] buttonActions;
	private ITreeContentProvider contentProvider;
	private ILabelProvider labelProvider;
	private TreeEditor treeEditor;
	
	class DatapoolChildrenContentProvider implements ITreeContentProvider
	{
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
		 */
		public Object[] getChildren(Object parentElement) {
			if(parentElement instanceof IDatapool)
				return DatapoolChildrenTreeSection.this.getChildren((IDatapool)parentElement);
			return new Object[0];
		}
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
		 */
		public Object getParent(Object element) {
			return DatapoolChildrenTreeSection.this.getParent(element);
		}
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
		 */
		public boolean hasChildren(Object element) {
			if(element instanceof IDatapool)
				return DatapoolChildrenTreeSection.this.hasChildren((IDatapool)element);
			return false;
		}
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
		 */
		public Object[] getElements(Object inputElement) {
			if(inputElement instanceof IDatapool)
				return DatapoolChildrenTreeSection.this.getElements((IDatapool)inputElement);
			else
				return new Object[0];
		}
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
		 */
		public void dispose() {
			// TODO Auto-generated method stub
		}
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			DatapoolChildrenTreeSection.this.inputChanged(viewer, oldInput, newInput);
		}
	}
	
	/**
	 * @param editorForm
	 * @param buttonLabels
	 */
	public DatapoolChildrenTreeSection(EditorForm editorForm, IAction[] buttonActions, int treeStyle) {
		super(editorForm);

		this.buttonActions = buttonActions;
		String[] buttonLabels = new String[buttonActions.length];
		for(int i = 0; i < buttonActions.length; i++)
		{
			buttonLabels[i] = buttonActions[i].getText();
		}

		super.setTreeStryle(treeStyle);
		super.setButtonLabels(buttonLabels);
	}

	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.FormSection#dispose()
	 */
	public void dispose() {
		for(int i = 0; i < buttonActions.length; i++)
		{
			if(buttonActions[i] != null)
				buttonActions[i] = null;
		}
		super.dispose();
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseTreeSection#adjustClient(org.eclipse.swt.widgets.Composite)
	 */
	protected void adjustClient(Composite parent)
	{		
		TreeViewer treeViewer = getTreeViewer();
		Tree tree = treeViewer.getTree();
		treeEditor = new TreeEditor(tree);
		treeEditor.horizontalAlignment = SWT.LEFT;
		treeEditor.grabHorizontal = true;
		treeEditor.minimumWidth = 50;
		
		tree.addMouseListener(this);
		tree.addKeyListener(this);
		
		treeViewer.setContentProvider(getContentProvider());
		treeViewer.setLabelProvider(getLabelProvider());
		treeViewer.setAutoExpandLevel(0);
	
		for(int i = 0; i < buttonActions.length; i++)
		{
			if(buttonActions[i] != null)
			{
				getTreePart().setButtonEnabled(i, buttonActions[i].isEnabled());
			}
		}
	}
	
	protected void startTreeItemEditing()
	{		
		Tree tree = getTreeViewer().getTree();
		TreeItem[] items = tree.getSelection();
		if(items == null || items.length < 1)
			return;
		
		Text editor = new Text(tree, SWT.BORDER);
		editor.setText(items[0].getText());
		editor.addKeyListener(this);
		editor.addFocusListener(new FocusAdapter() {
			public void focusLost(FocusEvent e) {
				applyTreeItemEditing();
			}
		});
		editor.selectAll();
		editor.setFocus();
		treeEditor.setEditor(editor, items[0]);
	}
	
	/**
	 * Set the new string from editor to the tree item.
	 * @return the new string value from the editor or <code>null</code> if unsuccessful retrieving from the editor.
	 */
	protected String applyTreeItemEditing()
	{
		String newValue = null;
		Control editor = treeEditor.getEditor();
		if(editor == null)
			return newValue;
		if(editor instanceof Text)
		{
			newValue = ((Text)editor).getText();
			if(newValue.length() == 0 || newValue.equals(treeEditor.getItem().getText()))
			{
				cancelTreeItemEditing();
				return null;
			}
			setNewTreeItemName(treeEditor.getItem(), newValue);
		}
		editor.dispose();
		treeEditor.setEditor(null);
		return newValue;
	}
	
	public void setNewTreeItemName(TreeItem item, String newName)
	{
		item.setText(newName);
		if(item.getData() instanceof NamedElementPropertySource)
		{
			NamedElementPropertySource namedElementPropSrc = (NamedElementPropertySource)item.getData();
			namedElementPropSrc.getNamedElement().setName(newName);
		}
	}
	
	protected void cancelTreeItemEditing()
	{
		Control editor = treeEditor.getEditor();
		if(editor == null)
			return;
		editor.dispose();
		treeEditor.setEditor(null);
	}
	
	protected ITreeContentProvider getContentProvider()
	{
		if(this.contentProvider == null)
			contentProvider = new DatapoolChildrenContentProvider();
		
		return contentProvider;
	}
	
	protected ILabelProvider getLabelProvider()
	{
		if(this.labelProvider == null)
			labelProvider = new PropertySourceLabelProvider();
		
		return labelProvider;
	}

	public abstract Object[] getChildren(IDatapool datapool);
	public abstract Object getParent(Object element);
	public abstract boolean hasChildren(IDatapool element);
	public abstract Object[] getElements(IDatapool inputElement);
	public abstract void inputChanged(Viewer viewer, Object oldInput, Object newInput);
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseStructuredViewerSection#buttonSelected(int)
	 */
	protected void buttonSelected(int index)
	{
		if(!isReadOnly())
		{
			if( index >=0 && index < buttonActions.length )
				buttonActions[index].run();
		}
	
		updateActionsAndButtons(getStructuredSelection());
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseStructuredViewerSection#updateActionsAndButtons(org.eclipse.jface.viewers.IStructuredSelection)
	 */
	protected void updateActionsAndButtons(IStructuredSelection structuredSelection)
	{
		if(buttonActions == null || buttonActions.length < 1)
			return;
			
		//if(isEditable())
		{
			for(int i = 0; i < buttonActions.length; i++)
			{
				if(buttonActions[i] != null)
				{
					if(buttonActions[i] instanceof SelectionListenerAction)
						((SelectionListenerAction)buttonActions[i]).selectionChanged(structuredSelection);
					getTreePart().setButtonEnabled(i, buttonActions[i].isEnabled());
				}
			}
		}
	}
	
	protected void selectionChanged(IStructuredSelection structuredSelection) {
		super.selectionChanged(structuredSelection);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.MouseListener#mouseDoubleClick(org.eclipse.swt.events.MouseEvent)
	 */
	public void mouseDoubleClick(MouseEvent e) {
		
		if(e.widget instanceof Tree){
			
			//Only process this mouse event if the editor is not opened in read-only mode:
			if(!isReadOnly()){
				startTreeItemEditing();
			}
		}
	}
	public void mouseDown(MouseEvent e) {
		// TODO Auto-generated method stub

	}
	public void mouseUp(MouseEvent e) {
	}
	public void keyPressed(KeyEvent e) {
		// TODO Auto-generated method stub

	}
	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyReleased(KeyEvent e) {
		
		if(e.widget instanceof Tree){
			
			if(e.keyCode == SWT.F2){
			
				//Only process this key event if the editor is not opened in read-only mode:
				if(!isReadOnly()){
					startTreeItemEditing();
				}
			}
		}
		else if(e.widget instanceof Text){
			
			if(e.character == SWT.CR){
				applyTreeItemEditing();
			}
			else if(e.character == SWT.ESC){
				cancelTreeItemEditing();	
			}
		}
	}
}
