/********************************************************************** 
 * Copyright (c) 2005, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: OpenJavaScriptAction.java,v 1.6 2008/03/20 19:49:56 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.ui.forms.actions;

import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.hyades.loaders.util.RegistryReader;
import org.eclipse.hyades.models.common.common.CMNExtendedProperty;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.core.util.JavaUtil;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.forms.extensions.IEventAction;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;

/**
 * <p>This action class can be used as a sample action as is for opening java
 * script file from an execution event in TPTP Test Log Viewer. It is written as a proof-of-concept for
 * extension point <b>{@link org.eclipse.hyades.test.ui.executionHistoryExtension executionHistoryExtension}</b>.
 * It is NOT used by TPTP.</p>
 * 
 * <p>Activating this action opens a java class in eclipse java editor and highlights a line. The java project, java class,
 * and line number are specified as properties in {@link TPFExecutionEvent}. To have this action to work, two things need to happen at minimum:
 * <ol>
 * <li>Fully qualified class name of the java script presents as a property among {@link TPFExecutionEvent#getProperties()}</li>
 * <li>Register this class in an extension of <i>executionHistoryExtension.actionExtension</i>.</li>
 * </ol>
 * 
 * There are default property names defined in this class, but should be the property names generated by your test runner.
 * By extending <i>executionHistoryExtension.actionExtension</i>, actions will show up as buttons next to the Events tree of
 * Test Log Viewer. There are two attributes of <i>acionExtension</i>:
 * <li>text, button text</li>
 * <li>action, an implementation of {@link org.eclipse.hyades.test.ui.forms.extensions.IEventAction IEventAction}</li><br>. </p>
 * 
 * <p>Here's an extension example to add this "Go To Script" button action in Test Log Viewer:<br>
 * 
 * <pre>
 * <extension
         point="org.eclipse.hyades.test.ui.executionHistoryExtension">
      <actionExtension testType="org.eclipse.hyades.test.junit">
         <eventAction
               action="org.eclipse.hyades.test.ui.forms.actions.OpenJavaScriptAction"
               asButton="true"
               asContextMenu="true"
               text="Open Script"/>
      </actionExtension>
 * </extension>
 * </pre></p>
 * 
 * @provisional
 * @see extension point "org.eclipse.hyades.test.ui.executionHistoryExtension".
 * @author bjiang
 * @since 4.2
 */
public class OpenJavaScriptAction extends Action implements IEventAction {

	/**
	 * Property name for java project name used in {@link TPFExecutionEvent#getProperties()}. 
	 * Should make it your own property name generated by your test runner.
	 * This property is optional for this action. When value not available, the project of the current
	 * selected execution history will be used.
	 */
	public static String propNamejavaProject = "javaProjectName"; //$NON-NLS-1$
	/**
	 * Property name for fully qualified class name of your java script used in {@link TPFExecutionEvent#getProperties()}.
	 * Should make it your own property name generated by your test runner.
	 * This property is required by this action.
	 */
	public static String propNameJavaClass = "qualifiedJavaClassName";//$NON-NLS-1$
	/**
	 * Property name for line number in the java script used in {@link TPFExecutionEvent#getProperties()}. 
	 * Should make it your own property name generated by your test runner.
	 * This property is optional for this action. When value not available, the first line will be highlighted.
	 */
	public static String propNameLineNumber = "javaClassLineNumber";//$NON-NLS-1$
	
	private static boolean isRCP = false;
	private TPFExecutionEvent executionEvent;
	private	String javaProject = null, javaClassName = null, javaClassLineNumber = null;
	
	/**
	 * Default constructor which takes no arguments. This is necessary for extension instantiation.
	 */
	public OpenJavaScriptAction() {
		setEnabled(false);
		if (!RegistryReader.isPlatformMode() || !RegistryReader.isWorkspaceMode())
			isRCP = true;
	}
	
	/**
	 * This action listens to selection changes on Test Log Viewer Events tree and this method gets called on such selection changes.
	 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
	 */
	public void selectionChanged(SelectionChangedEvent event) {
		IStructuredSelection selection = (IStructuredSelection)event.getSelection();
		
		setEnabled(false);
		if(!isRCP && !selection.isEmpty())
		{
			executionEvent = null;
			javaProject = javaClassName = javaClassLineNumber = null;
			boolean checkedProject = false, checkedClass = false, checkedLine = false;
			Object sel = selection.getFirstElement();
			if(sel instanceof TPFExecutionEvent)
			{
				executionEvent = (TPFExecutionEvent)sel;
				List properties = ((TPFExecutionEvent)sel).getProperties();
				if(properties != null && !properties.isEmpty())
				{
					for(Iterator it = properties.iterator(); it.hasNext(); )
					{
						Object next = it.next();
						if(next instanceof CMNExtendedProperty)
						{
							if(propNamejavaProject.equals(((CMNExtendedProperty)next).getName()))
							{
								javaProject = ((CMNExtendedProperty)next).getValue();
								checkedProject = true;
								if(checkedClass == true && checkedLine == true)
									break;
							}
							else if(propNameJavaClass.equals(((CMNExtendedProperty)next).getName()))
							{
								javaClassName = ((CMNExtendedProperty)next).getValue();
								checkedClass = true;
								if(checkedProject == true && checkedLine == true)
									break;
							}
							else if(propNameLineNumber.equals(((CMNExtendedProperty)next).getName()))
							{
								javaClassLineNumber = ((CMNExtendedProperty)next).getValue();
								checkedLine = true;
								if(checkedProject == true && checkedClass == true)
									break;
							}
						}
					}
					
					if(javaClassName != null)
					{
						setEnabled(true);
					}
				}
			}
		}
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.action.Action#run()
	 */
	public void run() {
		openJavaTypeAtLine(javaProject, javaClassName, javaClassLineNumber);
	}
	
	public boolean updateButtonStatus() {
		return isEnabled();
	}
	
	public void menuAboutToShow(IMenuManager manager) {
		manager.add(this);
		manager.add(new Separator());
	}

	private void openJavaTypeAtLine(String javaProject, String qualifiedJavaClassName, String lineNumber)
	{						
		if(isRCP || qualifiedJavaClassName == null)
			return;
		
		IJavaProject project = null;
		if(javaProject != null)
			project = JavaCore.create(ResourcesPlugin.getWorkspace().getRoot().getProject(javaProject));
		// get the project of this execution history.
		project = getJavaProject();
		
		String cuName = null, className = qualifiedJavaClassName;
		int innerClassSeparater = qualifiedJavaClassName.indexOf('$'); 
		if(innerClassSeparater > -1)
			className = qualifiedJavaClassName.substring(0, innerClassSeparater);
		else if(qualifiedJavaClassName.endsWith(".java"))//$NON-NLS-1$
			className = qualifiedJavaClassName.substring(0, qualifiedJavaClassName.length() - 5);		
		if(className.indexOf('.') > -1)
			cuName = className.substring(className.lastIndexOf('.') + 1, className.length());
		cuName = cuName + ".java";//$NON-NLS-1$
		
		int line = 1;
		if(lineNumber != null)
			line = Integer.parseInt(lineNumber);
		
		try {
			openCompilationUnitAtLine(project, cuName, className, line);
		}
		catch(Exception e) {
			UiPlugin.logError(e);
		}
	}
	
	private void openCompilationUnitAtLine(IJavaProject javaProject, String cuName, String className, int line) throws CoreException {
		IJavaElement javaElement = JavaUtil.findElement(javaProject, cuName, className);
		if (javaElement != null) {
			org.eclipse.hyades.ui.internal.util.JavaUtil.revealJavaElementAtLine(javaElement, line);
		}
	}

	private IJavaProject getJavaProject() {
		if (executionEvent != null) {
			IFile file = EMFUtil.getWorkspaceFile(executionEvent);
			if (file != null) {
				return JavaCore.create(file.getProject());
			}
		}
		return null;
	}
}
