/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: VerdictNavigationAction.java,v 1.8 2010/04/07 19:37:41 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.forms.actions;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionHistory;
import org.eclipse.hyades.models.common.testprofile.TPFVerdictEvent;
import org.eclipse.hyades.test.ui.forms.base.IVerdictTypeSelectionListener;
import org.eclipse.hyades.test.ui.forms.util.ITestLogVerdictTraversal;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.ui.actions.SelectionListenerAction;

/**
 * <p>Verdict navigation action to navigate typed verdicts in the Events 
 * tree in the Events page of the Test Log editor.</p>
 * 
 * 
 * @author      Bianca Xue Jiang 
 * @author      Paul E. Slauenwhite
 * @author 		Blazej Kroll
 * @version     April 7, 2010
 * @since       September 14, 2005
 */
public class VerdictNavigationAction extends SelectionListenerAction implements IVerdictTypeSelectionListener
{
	protected ITestLogVerdictTraversal verdictCollection;
	private int currentVerdictType;
	private TPFVerdictEvent currentVerdict;
	private StructuredViewer viewer;

	public VerdictNavigationAction(VerdictTypesPullDownAction verdictTypesAction, ITestLogVerdictTraversal collection, StructuredViewer structuredViewer)
	{
		super(null);
		this.currentVerdictType = verdictTypesAction.getCurrentVerdictType();
		this.verdictCollection = collection;
		verdictTypesAction.addVerdictTypeSelectionListener(this);
		setStructuredViewer(structuredViewer);
		
		//Initially disable the verdict navigation action if there are no verdicts of the current type:
		setEnabled(verdictCollection.contains(currentVerdictType));
	}
	
	/**
	 * Sets the structuredViewer that should be updated after running this action.
	 * @param structuredViewer
	 */
	public void setStructuredViewer(StructuredViewer structuredViewer)
	{
		if(this.viewer != null)
			viewer.removeSelectionChangedListener(this);
		
		this.viewer = structuredViewer;
		structuredViewer.addSelectionChangedListener(this);
	}

	/**
	 * Returns the structuredViewer to be updated after running this action.
	 * @return structuredViewer
	 */
	public StructuredViewer getStructuredViewer()
	{
		return viewer;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.actions.BaseSelectionListenerAction#updateSelection(org.eclipse.jface.viewers.IStructuredSelection)
	 */
	protected boolean updateSelection(IStructuredSelection selection)
	{
		if(!selection.isEmpty() && (selection.getFirstElement() instanceof TPFVerdictEvent))
			currentVerdict = (TPFVerdictEvent)selection.getFirstElement();
		else
			currentVerdict = null;
			
		if(super.updateSelection(selection) == false)
			return false;
		
		if(!verdictCollection.contains(currentVerdictType))
			return false;
		
		return true;
	}

	public void verdictTypeSelected(int type) {
		this.currentVerdictType = type;
	}
	
	public int getCurrentVerdictType()
	{
		return currentVerdictType;
	}
	
	public TPFVerdictEvent getCurrentVerdictSelection()
	{
		return currentVerdict;
	}

	/**
	 * <p>Reveals and selects the verdict event in the the tree viewer.</p>
	 * 
	 * @param verdictEvent The verdict event to be revealed and selected.
	 * @param treeViewer The tree viewer containing the verdict event.
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	public static void revealEvent(TPFVerdictEvent verdictEvent, TreeViewer treeViewer) {
		
		//Resolve the container events of the verdict event:
		List<EObject> containerEvents = new ArrayList<EObject>();
		EObject currentEvent = verdictEvent;
		EObject currentEventContainer = currentEvent.eContainer();
		
		while (currentEventContainer != null) {
			
			//Note: TPFExecutionHistory events are not displayed in the Test Log editor.
			if (!(currentEventContainer instanceof TPFExecutionHistory)) {
			
				containerEvents.add(0, currentEvent);
			
				currentEvent = currentEventContainer;
			}

			currentEventContainer = currentEventContainer.eContainer();
		}
		
		//Select and reveal the verdict event:
		treeViewer.setSelection(new TreeSelection(new TreePath(containerEvents.toArray())), true);
  }	
}
