/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: VerdictTypesPullDownAction.java,v 1.11 2008/07/11 13:44:46 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.forms.actions;

import java.util.ArrayList;

import org.eclipse.core.runtime.ListenerList;
import org.eclipse.hyades.models.common.testprofile.TPFVerdict;
import org.eclipse.hyades.models.common.testprofile.TPFVerdictEvent;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.forms.base.IVerdictTypeSelectionListener;
import org.eclipse.hyades.test.ui.forms.util.ITestLogVerdictTraversal;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuCreator;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.IWorkbenchWindowPulldownDelegate2;

/**
 * Verdict type pull down action to select the type of verdict to navigate through 
 * in the Events tree in the Events page of the Test Log Viewer.
 * 
 * 
 * @author      Bianca Xue Jiang 
 * @author      Paul E. Slauenwhite
 * @version     July 11, 2008
 * @since       September 14, 2005
 */
public class VerdictTypesPullDownAction extends Action implements IWorkbenchWindowPulldownDelegate2, IMenuCreator 
{	
	/**
	 * Cascading menu 
	 */
	private Menu fCreatedMenu;
	private ITestLogVerdictTraversal verdictCollection;
	private VerdictTypeAction[] verdictTypeActions;
	private StructuredViewer viewer;
	public static final int VERDICT_TYPE_ALL = ITestLogVerdictTraversal.VERDICT_TYPE_ALL;
	private int verdictType = VERDICT_TYPE_ALL;
	
	private ListenerList verdictTypeSelectionListeners = new ListenerList();
	
	private class VerdictTypeAction extends Action
	{
		private int type;
		
		public VerdictTypeAction(int verdictType, String text, int style)
		{
			super(text, style);
			type = verdictType;
		}
		
		public void run()
		{
			VerdictTypesPullDownAction.this.run(this);
		}
		
		public int getVerdictType()
		{
			return type;
		}
	}
	
	public VerdictTypesPullDownAction(ITestLogVerdictTraversal collection, StructuredViewer structuredViewer)
	{
		super();
		setMenuCreator(this);
		this.verdictCollection = collection;
		
		
		this.viewer = structuredViewer;
		setToolTipText(UiPluginResourceBundle.TestLogViewer_SelectNavigationVerdictType); 
		this.setImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_VERDICT));
	}

	public void addVerdictTypeSelectionListener(IVerdictTypeSelectionListener listener)
	{
		verdictTypeSelectionListeners.add(listener);
	}
	
	public void removeVerdictTypeSelectionListener(IVerdictTypeSelectionListener listener)
	{
		verdictTypeSelectionListeners.remove(listener);
	}
	
	protected void fireVerdictTypeChanged(int type)
	{
		Object[] listeners = verdictTypeSelectionListeners.getListeners();
		for(int i = 0; i < listeners.length; i++)
		{
			((IVerdictTypeSelectionListener)listeners[i]).verdictTypeSelected(type);
		}
	}
	
	public Menu getMenu(Menu parent) {
		return null;
	}

	public Menu getMenu(Control parent) {
		if(fCreatedMenu == null)
		{
			fCreatedMenu = new Menu(parent);
			fillMenu(fCreatedMenu);
		}
		
		return fCreatedMenu;
	}
	
	private void fillMenu(Menu parent)
	{
		Action[] actions = createVerdictTypeActions(parent);
		for(int i = 0; i < actions.length; i++)
		{
			ActionContributionItem item= new ActionContributionItem(actions[i]);
			item.fill(parent, -1);
		}
	}
	
	protected Action[] createVerdictTypeActions(Menu parent)
	{
		if(verdictTypeActions == null)
		{
			ArrayList actions = new ArrayList();
			
			if(verdictCollection.getFirst(TPFVerdict.ERROR_LITERAL.getValue()) != null){
				
				Action errorAction = new VerdictTypeAction(TPFVerdict.ERROR_LITERAL.getValue(), TPFVerdict.ERROR_LITERAL.getLabel(), Action.AS_PUSH_BUTTON);
				errorAction.setImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_VERDICT_ERROR));
				actions.add(errorAction);
			}
			
			if(verdictCollection.getFirst(TPFVerdict.FAIL_LITERAL.getValue()) != null){
			
				Action failAction = new VerdictTypeAction(TPFVerdict.FAIL_LITERAL.getValue(), TPFVerdict.FAIL_LITERAL.getLabel(), Action.AS_PUSH_BUTTON);
				failAction.setImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_VERDICT_FAIL));
				actions.add(failAction);
			}			
			
			if(verdictCollection.getFirst(TPFVerdict.INCONCLUSIVE_LITERAL.getValue()) != null){
				
				Action inclAction = new VerdictTypeAction(TPFVerdict.INCONCLUSIVE_LITERAL.getValue(), TPFVerdict.INCONCLUSIVE_LITERAL.getLabel(), Action.AS_PUSH_BUTTON);
				inclAction.setImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_VERDICT_INCONCLUSIVE));
				actions.add(inclAction);
			}
			
			if(verdictCollection.getFirst(TPFVerdict.PASS_LITERAL.getValue()) != null){
				
				Action passAction = new VerdictTypeAction(TPFVerdict.PASS_LITERAL.getValue(), TPFVerdict.PASS_LITERAL.getLabel(), Action.AS_PUSH_BUTTON);
				passAction.setImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_VERDICT_PASS));
				actions.add(passAction);
			}
			
			//for all verdict types
			VerdictTypeAction allVerdictAction = new VerdictTypeAction(VERDICT_TYPE_ALL, UiPluginResourceBundle.W_ALL, Action.AS_PUSH_BUTTON);
			allVerdictAction.setImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_VERDICT));
			verdictType = VERDICT_TYPE_ALL;
			actions.add(allVerdictAction);
			
			verdictTypeActions = (VerdictTypeAction[])actions.toArray(new VerdictTypeAction[actions.size()]);
		}
		return verdictTypeActions; 
	}
	
	public void dispose() {
		if(fCreatedMenu != null && !fCreatedMenu.isDisposed())
		{
			fCreatedMenu.dispose();
			fCreatedMenu = null;
		}
	}

	public void init(IWorkbenchWindow window) {
	}
	
	public void run()
	{
		TPFVerdictEvent event = verdictCollection.getFirst(getCurrentVerdictType());

		if(event != null) {
			VerdictNavigationAction.revealEvent(event, (TreeViewer) getStructuredViewer());
		}
	}

	public void run(IAction action) {
		
		int newVerdictType = ((VerdictTypeAction)(action)).getVerdictType();
		
		//Only change the verdict type, notify the listeners, and run the action when the verdict type has changed:
		if(newVerdictType != verdictType){
			
			selectVerdcitType(newVerdictType);
			
			run();
		}
	}
	
	/**
	 * Select the specified verdict type menu item without running the action. As a result, all other
	 * verdict types will be unselected since there is only one action can be checked at a time.
	 * 
	 * @param verdictType
	 */
	public void selectVerdcitType(int verdictType)
	{
		fireVerdictTypeChanged(verdictType);
		VerdictTypeAction action;
		for(int i = 0; i < verdictTypeActions.length; i++)
		{
			action = verdictTypeActions[i];
			if(action.getVerdictType() == verdictType)
			{
				this.verdictType = verdictType;
				//currentAction = action;
				this.setImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_VERDICT_NON));
				this.setImageDescriptor(action.getImageDescriptor());
			}
		}
	}

	public int getCurrentVerdictType()
	{
		return verdictType;
	}

	public void selectionChanged(IAction action, ISelection selection) {
		// TODO Auto-generated method stub

	}

	/**
	 * @return Returns the viewer.
	 */
	public StructuredViewer getStructuredViewer()
	{
		return viewer;
	}

	/**
	 * @param viewer The viewer to set.
	 */
	public void setStructuredViewer(StructuredViewer viewer)
	{
		this.viewer = viewer;
	}

}
