/********************************************************************** 
 * Copyright (c) 2005, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: ExecutionEventDetailsPart.java,v 1.45 2009/05/08 18:51:57 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.ui.forms.base;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.models.common.common.CMNAnnotation;
import org.eclipse.hyades.models.common.common.CMNExtendedProperty;
import org.eclipse.hyades.models.common.interactions.BVRInteractionFragment;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFRepositoryRecord;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.forms.editor.TestLogViewer;
import org.eclipse.hyades.test.ui.forms.extensions.IPropertyLabelProvider;
import org.eclipse.hyades.test.ui.forms.extensions.provisional.IRecordRepositoryProvider;
import org.eclipse.hyades.test.ui.forms.extensions.provisional.IRepositoryRecordListener;
import org.eclipse.hyades.test.ui.forms.util.FormsUtil;
import org.eclipse.hyades.test.ui.internal.editor.form.util.ExecutionHistoryExtensionsManager;
import org.eclipse.hyades.test.ui.internal.editor.form.util.TestLogExtensionsManager;
import org.eclipse.hyades.test.ui.internal.editor.form.util.TestLogExtensionsManager.DefectProviderExtension;
import org.eclipse.hyades.test.ui.internal.model.EventUtil;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.util.TestUIUtil;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.osgi.util.TextProcessor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.browser.Browser;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.program.Program;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.forms.AbstractFormPart;
import org.eclipse.ui.forms.IDetailsPage;
import org.eclipse.ui.forms.IFormPart;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormPage;
import org.eclipse.ui.forms.events.HyperlinkEvent;
import org.eclipse.ui.forms.events.IHyperlinkListener;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.Hyperlink;
import org.eclipse.ui.forms.widgets.Section;

import com.ibm.icu.text.Collator;
import com.ibm.icu.text.NumberFormat;

/**
 * <p>Details page of the page book used for {@link org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent TPFExecutionEvent}
 * in the events tree of the Test Log Viewer.</p>
 * 
 * <p>This is an Eclipse forms page derived from the original TPTP Test Log viewer.</p>
 * 
 * 
 * @author  Bianca Xue Jiang
 * @author  Marcelo Paternostro
 * @author  Paul E. Slauenwhite
 * @version May 8, 2009
 * @since   August 9, 2008
 */
public abstract class ExecutionEventDetailsPart extends AbstractFormPart implements IDetailsPage, IHyperlinkListener, SelectionListener, IRepositoryRecordListener //, ModifyListener
{
	private FormPage editorPage;
	private Composite parent;
	protected Section commonPropSection;
	protected Section extendedPropSection;
	private IRecordRepositoryProvider[] defectProviders;
	
	private Text timeText;
	private Hyperlink modelElementLink;
	//private Text descriptionText;
	protected Control messageText;
	private Table propertiesTable;
	private Table attachmentsTable = null;	
	private Button attachmentsOpenButton = null;
	private Button attachmentsSaveButton = null;
	private static final String HTML_TAG_START = "<html>"; //$NON-NLS-1$
	private static final String HTML_TAG_END = "</html>"; //$NON-NLS-1$
	private static final String HTML_TAG_ANCHOR_START = "<a "; //$NON-NLS-1$
	private static final String HTML_ATTR_HREF = " href"; //$NON-NLS-1$
	private static final String HTML_ATTR_TARGET = " target"; //$NON-NLS-1$
	private static final String HTML_ATTR_TARGET_BLANK = "target=\"_blank\" "; //$NON-NLS-1$
	private static final String HTML_ATTR_TARGET_SELF = "target=\"_self\" "; //$NON-NLS-1$
	
	private TPFExecutionEvent executionEvent;
	
	/**
	 * Default Constructor
	 */
	public ExecutionEventDetailsPart()
	{		
	}

	/**
	 * Creates an instance of this class with the editor page.
	 * @param page forms editor page of this part.
	 */
	public ExecutionEventDetailsPart(FormPage page)
	{
		editorPage = page;
	}
	
	public void initialize(IManagedForm mForm)
	{
		super.initialize(mForm);
	}
	
	public void setFormPage(FormPage page)
	{
		this.editorPage = page;
	}
	
	public FormPage getFormPage()
	{
		return this.editorPage;
	}

	public void createContents(Composite parent)
	{		
		if(parent == this.parent)
			return;
		
		// add this form part to the managed form the first time this part is created
		if(this.parent == null)
			getManagedForm().addPart(this);
		
		this.parent = parent;
		// recreating the content of this page
		disposeContent();
		
		commonPropSection = FormsUtil.createSection(getManagedForm(), parent, UiPluginResourceBundle.TTL_PROPERTIES, ""); 
		Composite commonSectionClient = (Composite)commonPropSection.getClient();
		createCommonContents(commonSectionClient);	
		getManagedForm().getToolkit().paintBordersFor(commonSectionClient);
		
		if(editorPage.getEditor() instanceof TestLogViewer)
		{
			EObject input = ((TestLogViewer)editorPage.getEditor()).getEditorObject();
			if(input instanceof TPFExecutionResult)
			{
				try {
					defectProviders = createDefectSectionFromExtension(((TPFExecutionResult)input).getType(), parent);
				}
				catch(Throwable t)
				{
					// log the error from creating the defect section but let the UI finish.
					UiPlugin.logError(t);
				}
			}
		}
		
		extendedPropSection = FormsUtil.createSection(getManagedForm(), parent, UiPluginResourceBundle.TTL_EXT_PROPERTIES, ""); 
		Composite extendedSectionClient = (Composite)extendedPropSection.getClient();
		createExtendedContents(extendedSectionClient);
		getManagedForm().getToolkit().paintBordersFor(extendedSectionClient);
	}

	protected void createCommonContents(Composite parent)
	{	
		FormToolkit toolkit = getManagedForm().getToolkit();
		
		createModelElementHyperlink(parent, toolkit);		
		createDetailsContents(parent);		
		createTimeText(parent, toolkit);
		
		toolkit.createLabel(parent, UiPluginResourceBundle.LBL_TEXT); 
		
		toolkit.paintBordersFor(parent);
	}
	
	protected IRecordRepositoryProvider[] createDefectSectionFromExtension(String testType, Composite parent)
	{
		IRecordRepositoryProvider[] providers = new IRecordRepositoryProvider[0];
		DefectProviderExtension[] extensions = TestLogExtensionsManager.getInstance().getDefectProviderExtensions();
		if(extensions == null || extensions.length == 0)
			return providers;

		String[] disabledProviders = new String[0];
		if(testType != null)
			disabledProviders = TestLogExtensionsManager.getInstance().getDisabledProviders(testType);
		
		List providerList = new ArrayList();
		for(int i = 0; i < extensions.length; i++)
		{
			if(Arrays.asList(disabledProviders).contains(extensions[i].getType()))
				continue;
		
			IRecordRepositoryProvider provider = extensions[i].getProvider(true);
			provider.createContent(getManagedForm(), parent);
			provider.addRecordListener(this);
			providerList.add(provider);
		}
		
		providers = (IRecordRepositoryProvider[])providerList.toArray(new IRecordRepositoryProvider[providerList.size()]);
		return providers;
	}
	
	private void createModelElementHyperlink(Composite parent, FormToolkit toolkit)
	{
		modelElementLink = toolkit.createHyperlink(parent, "", SWT.WRAP); //$NON-NLS-1$
		modelElementLink.addHyperlinkListener(this);
		modelElementLink.setLayoutData(new GridData());
		modelElementLink.setText(UiPluginResourceBundle.NO_INTF_TO_INV); 
		modelElementLink.setToolTipText(UiPluginResourceBundle.NO_INTF_TO_INV); 
	}
	
	private void createTimeText(Composite parent, FormToolkit toolkit)
	{
		toolkit.createLabel(parent, UiPluginResourceBundle.LBL_TIME); 
		timeText = toolkit.createText(parent, "", SWT.FULL_SELECTION | SWT.SINGLE); //$NON-NLS-1$
		timeText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		timeText.setEditable(false);
	}
	
	protected void createDetailsContents(Composite parent)
	{
	}
	
	protected void createExtendedContents(Composite parent)
	{
		createPropertiesTable(parent, getManagedForm().getToolkit());
		createAttachmentTable(parent, getManagedForm().getToolkit());
	}
	
	protected void setMessageText(String text)
	{
		text = (text == null ? "" : text); //$NON-NLS-1$
		FormToolkit toolkit = getManagedForm().getToolkit();
		Composite messageTextParent = (Composite)commonPropSection.getClient();
		if(isHTMLString(text))
		{
			String html = parseHTMLString(text);
			if(messageText == null || messageText.isDisposed() || !(messageText instanceof Browser))
			{			
				if(messageText != null)
				{
					 messageText.dispose();
					 messageText = null;
				}
				messageText = new Browser(messageTextParent, SWT.NONE);
				messageText.setEnabled(true);
				GridData gd = new GridData(SWT.FILL, SWT.FILL, true, true);
				gd.heightHint = 104;
				messageText.setLayoutData(gd);
				messageText.setData(FormToolkit.KEY_DRAW_BORDER, FormToolkit.TEXT_BORDER);
				toolkit.adapt((Browser)messageText);
				toolkit.paintBordersFor(messageTextParent);
			}			
			((Browser)messageText).setText(html);
		}
		else
		{
			
			//Replace any carriage return characters with new line characters since 
			//these characters are not rendered (e.g. boxes):
			String normalizedText = text.replace('\r', '\n');

			String processedText = TextProcessor.process(normalizedText);
			
			if(messageText == null || messageText.isDisposed() || !(messageText instanceof Text))
			{							
				if(messageText != null)
				{
					 messageText.dispose();
					 messageText = null;
				}
				messageText = toolkit.createText(messageTextParent, processedText, SWT.WRAP | SWT.V_SCROLL);
				messageText.setEnabled(true);
				((Text)messageText).setEditable(false);
				GridData gd = new GridData(GridData.FILL_HORIZONTAL);
				gd.heightHint = 8*((Text)messageText).getLineHeight();
				messageText.setLayoutData(gd);
				//messageText.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_RED));
				toolkit.paintBordersFor(messageTextParent);
			}
			
			((Text)messageText).setText(processedText);
		}
	}
	
	/*private void createMessageText(String text)
	{
		if(messageText != null)
		{
			 messageText.dispose();
			 messageText = null;
		}
		
		text = text == null ? "" : text;
		FormToolkit toolkit = getManagedForm().getToolkit();
		if(isHTMLString(text))
		{
			String html = parseHTMLString(text);
			messageText = new Browser(commonSectionClient, SWT.NONE);
			messageText.setEnabled(true);
			GridData gd = new GridData(SWT.FILL, SWT.FILL, true, true);
			gd.heightHint = 104;
			messageText.setLayoutData(gd);
			messageText.setData(FormToolkit.KEY_DRAW_BORDER, FormToolkit.TEXT_BORDER);
			toolkit.adapt((Browser)messageText);
			((Browser)messageText).setText(html);
		}
		else
		{
			messageText = toolkit.createText(commonSectionClient, text, SWT.WRAP | SWT.V_SCROLL);
			messageText.setEnabled(true);
			((Text)messageText).setEditable(false);
			GridData gd = new GridData(GridData.FILL_HORIZONTAL);
			gd.heightHint = 8*((Text)messageText).getLineHeight();
			messageText.setLayoutData(gd);
		}			
		toolkit.paintBordersFor(commonSectionClient);
	}	*/

	protected void createPropertiesTable(Composite parent, FormToolkit toolkit)
	{
		Composite propParent = toolkit.createComposite(parent);
		propParent.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL));
		GridLayout layout = new GridLayout();
		layout.marginHeight = layout.marginWidth = 1;
		propParent.setLayout(layout);		

		getManagedForm().getToolkit().createLabel(propParent, UiPluginResourceBundle.LBL_PROPERTIES); 
		propertiesTable = toolkit.createTable(propParent, SWT.SINGLE);
		propertiesTable.setLinesVisible(true);
		propertiesTable.setHeaderVisible(true);
		propertiesTable.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		TableColumn nameColumn = new TableColumn(propertiesTable, SWT.NONE);
		nameColumn.setText(UiPluginResourceBundle.Word_Name); 
		nameColumn.setMoveable(true);
		nameColumn.setResizable(true);
		nameColumn.setWidth(100);
		
		TableColumn valueColumn = new TableColumn(propertiesTable, SWT.NONE);
		valueColumn.setText(UiPluginResourceBundle.Word_Value); 
		valueColumn.setMoveable(true);
		valueColumn.setResizable(true);
		valueColumn.setWidth(100);
		
		toolkit.paintBordersFor(propParent);
	}
	
	protected void createAttachmentTable(final Composite parent, FormToolkit toolkit) {
		toolkit.createLabel(parent, UiPluginResourceBundle.LBL_ATTACHMENTS); 

		GridLayout gridLayout = new GridLayout();
		gridLayout.marginHeight = 1;
		gridLayout.marginWidth = 1;
		gridLayout.numColumns = 2;
		
		Composite attachmentsComposite = toolkit.createComposite(parent);
		attachmentsComposite.setLayoutData(new GridData(SWT.FILL,SWT.FILL, true, false));
		attachmentsComposite.setLayout(gridLayout);
		
		attachmentsTable = toolkit.createTable(attachmentsComposite, (SWT.SINGLE | SWT.V_SCROLL | SWT.H_SCROLL));
		attachmentsTable.setLayoutData(new GridData(SWT.FILL,SWT.FILL, true, false));
		attachmentsTable.setHeaderVisible(true);
		attachmentsTable.setLinesVisible(true);
		
		 Listener columnSorter = new Listener() {
		       
		    	public void handleEvent(Event event) {

		    		TableItem[] rows = attachmentsTable.getItems();
		            TableColumn sortColumn = attachmentsTable.getSortColumn();
		            TableColumn selectedColumn = ((TableColumn)(event.widget));

		            if(selectedColumn.equals(sortColumn)){
		            	
		            	List reversedRows = Arrays.asList(rows);
		            	
		            	Collections.reverse(reversedRows);

		            	for (int currentRowIndex = 0; currentRowIndex < reversedRows.size(); currentRowIndex++) {
							
		            		TableItem oldRow = ((TableItem)(reversedRows.get(currentRowIndex)));
		            		
	            			Object data = oldRow.getData("FILE"); //$NON-NLS-1$
	            			Image image = oldRow.getImage();
	            			String[] text = new String[attachmentsTable.getColumnCount()];
	            			
	            			for (int counter = 0; counter < text.length; counter++) {
								text[counter] = oldRow.getText(counter);
							}
	            			
	            			oldRow.dispose();
	                        
	                        TableItem newRow = new TableItem(attachmentsTable, SWT.LEFT, currentRowIndex);
	                        newRow.setData("FILE", data); //$NON-NLS-1$
	                        newRow.setImage(image);
	                        newRow.setText(text);
						}
		            	
		            	if(attachmentsTable.getSortDirection() == SWT.UP){
		            		attachmentsTable.setSortDirection(SWT.DOWN);
		            	}
		            	else{
		            		attachmentsTable.setSortDirection(SWT.UP);	            		
		            	}
		            }
		            else{

		            	//Note: The sort direction default is UP.
			            int selectedColumnIndex = Arrays.asList(attachmentsTable.getColumns()).indexOf(selectedColumn);
			            Collator collator = Collator.getInstance();
			            
			            for (int currentRowIndex = 1; currentRowIndex < rows.length; currentRowIndex++) {

							String currentRowCell = rows[currentRowIndex].getText(selectedColumnIndex);

							for (int sampleRowIndex = 0; sampleRowIndex < currentRowIndex; sampleRowIndex++) {

								if (collator.compare(currentRowCell, rows[sampleRowIndex].getText(selectedColumnIndex)) < 0) {

									Object data = rows[currentRowIndex].getData("FILE"); //$NON-NLS-1$
									Image image = rows[currentRowIndex].getImage();
									String[] text = new String[attachmentsTable.getColumnCount()];

									for (int counter = 0; counter < text.length; counter++) {
										text[counter] = rows[currentRowIndex].getText(counter);
									}

									rows[currentRowIndex].dispose();

									TableItem row = new TableItem(attachmentsTable, SWT.LEFT, sampleRowIndex);
									row.setData("FILE", data); //$NON-NLS-1$
									row.setImage(image);
									row.setText(text);

									rows = attachmentsTable.getItems();

									break;
								}
							}
						}

						attachmentsTable.setSortDirection(SWT.UP);
						attachmentsTable.setSortColumn(selectedColumn);
		            }
		    	}
		    };
		    
	    final TableColumn attachmentsNameTableColumn = new TableColumn(attachmentsTable, SWT.LEFT);
	    attachmentsNameTableColumn.setMoveable(true);
	    attachmentsNameTableColumn.setResizable(true);
	    attachmentsNameTableColumn.setText(UiPluginResourceBundle.LBL_ATTACHMENTS_NAME); 
	    attachmentsNameTableColumn.addListener(SWT.Selection, columnSorter);

	    final TableColumn attachmentsSizeTableColumn = new TableColumn(attachmentsTable, SWT.LEFT);
	    attachmentsSizeTableColumn.setMoveable(true);
	    attachmentsSizeTableColumn.setResizable(true);
	    attachmentsSizeTableColumn.setText(UiPluginResourceBundle.LBL_ATTACHMENTS_SIZE); 
	    attachmentsSizeTableColumn.addListener(SWT.Selection, columnSorter);

	    final TableColumn attachmentsTypeTableColumn = new TableColumn(attachmentsTable, SWT.LEFT);
	    attachmentsTypeTableColumn.setMoveable(true);
	    attachmentsTypeTableColumn.setResizable(true);
	    attachmentsTypeTableColumn.setText(UiPluginResourceBundle.LBL_ATTACHMENTS_TYPE); 
	    attachmentsTypeTableColumn.setWidth(100);
	    attachmentsTypeTableColumn.addListener(SWT.Selection, columnSorter);

	    //Set the default sort column and direction of the table:
	    attachmentsTable.setSortDirection(SWT.UP);
        attachmentsTable.setSortColumn(attachmentsNameTableColumn);

	    //Listener to set the width of the table columns when the table has been initially painted:
	    attachmentsTable.addListener(SWT.Paint, new Listener(){
	    	 
	    	public void handleEvent(Event event){

				int tableWidth = attachmentsTable.getClientArea().width;
					
				attachmentsNameTableColumn.setWidth(((int)(tableWidth * 0.5)));
				attachmentsSizeTableColumn.setWidth(((int)(tableWidth * 0.20)));
				attachmentsTypeTableColumn.setWidth(tableWidth - (attachmentsNameTableColumn.getWidth() + attachmentsSizeTableColumn.getWidth()));
						
				//Remove the listener since the width of the table columns is only set once when the table has been initially painted:
				attachmentsTable.removeListener(SWT.Paint, this);
			}
		});

		gridLayout = new GridLayout();
		gridLayout.marginHeight = 0;
		gridLayout.marginWidth = 0;
		gridLayout.numColumns = 1;

		Composite attachmentsButtonComposite = toolkit.createComposite(attachmentsComposite);
		attachmentsButtonComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, true));
		attachmentsButtonComposite.setLayout(gridLayout);

		attachmentsOpenButton = toolkit.createButton(attachmentsButtonComposite, UiPluginResourceBundle.LBL_ATTACHMENTS_OPEN, SWT.PUSH); 
		attachmentsOpenButton.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false));

		attachmentsSaveButton = toolkit.createButton(attachmentsButtonComposite, UiPluginResourceBundle.LBL_ATTACHMENTS_SAVE_AS, SWT.PUSH); 
		attachmentsSaveButton.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false));

		attachmentsOpenButton.addListener(SWT.Selection, new Listener(){
	    	 
	    	public void handleEvent(Event event){	    		

	    		//Assumption:  Single selection.
	    		if(attachmentsTable.getSelectionCount() > 0){
	    			
	    			TableItem[] attachmnets = attachmentsTable.getSelection();

	    			for (int counter = 0; counter < attachmnets.length; counter++) {
	    				Program.launch(((File)(attachmnets[counter].getData("FILE"))).getAbsolutePath()); //$NON-NLS-1$
					}	    

	    			attachmentsTable.deselect(attachmentsTable.getSelectionIndices());
	    		}
	    		
	    		attachmentsOpenButton.setEnabled(false);
	    		attachmentsSaveButton.setEnabled(false);
	    	}
	    });
		
		attachmentsSaveButton.addListener(SWT.Selection, new Listener(){
	    	 
	    	public void handleEvent(Event event){	    		

	    		//Assumption:  Single selection.
	    		if(attachmentsTable.getSelectionCount() > 0){
	    			
	    			TableItem[] attachmnets = attachmentsTable.getSelection();
	    			
	    			for (int counter = 0; counter < attachmnets.length; counter++) {
	    				
	    	    		FileDialog fileDialog = new FileDialog(parent.getShell(), (SWT.SAVE | SWT.SINGLE | SWT.APPLICATION_MODAL));	 
	    	    		fileDialog.setFileName(attachmnets[counter].getText(0));
	    	    		fileDialog.setFilterExtensions(new String[]{"*.*"}); //$NON-NLS-1$
	    	    		fileDialog.setFilterNames(new String[]{UiPluginResourceBundle.ATTACHMENTS_TABLE_GROUP_SAVE_FILE_DIALOG_FILTER_NAMES}); 

	    	    		File newFile = null;
	    	    		
	    				while(fileDialog.open() != null){

		    				newFile = new File(fileDialog.getFilterPath(), fileDialog.getFileName());

		    				if(newFile.exists()){

		    					MessageBox messageBox = new MessageBox(parent.getShell(), (SWT.ICON_ERROR | SWT.OK | SWT.APPLICATION_MODAL));
		    					messageBox.setText(UiPluginResourceBundle.W_ERROR); 
		    					messageBox.setMessage(NLS.bind(UiPluginResourceBundle.LBL_ATTACHMENTS_SAVE_NEW_ERR_MSG, (new String[]{newFile.getName(), newFile.getParentFile().getAbsolutePath()}))); 
		    					messageBox.open();
		    				}
		    				else{
		    					break;
		    				}
	    				}
		    			
	    				if(newFile != null){
	    				
	    					FileInputStream fileInputStream  = null;
		    				FileOutputStream fileOutputStream = null;
		    				
	    					try {
		    					
			    				newFile.createNewFile();

			    				fileInputStream  = new FileInputStream(((File)(attachmnets[counter].getData("FILE")))); //$NON-NLS-1$
			    				fileOutputStream = new FileOutputStream(newFile);
				    				    
			    				byte[] buffer = new byte[1024];
			    				int bufferLength = 0;

			    				while((bufferLength = fileInputStream.read(buffer)) != -1) {
				    				fileOutputStream.write(buffer, 0, bufferLength);
				    			}				    			
			    			} 
	    					catch (Throwable t) {

	    						MessageBox messageBox = new MessageBox(parent.getShell(), (SWT.ICON_ERROR | SWT.OK | SWT.APPLICATION_MODAL));
		    					messageBox.setText(UiPluginResourceBundle.W_ERROR); 
		    					messageBox.setMessage(NLS.bind(UiPluginResourceBundle.LBL_ATTACHMENTS_SAVE_WRITE_ERR_MSG, (new String[]{newFile.getName(), newFile.getParentFile().getAbsolutePath(),t.getLocalizedMessage()}))); 
		    					messageBox.open();
							}
	    					finally{
	    						
	    						if(fileInputStream != null){
	    							
	    							try {
										fileInputStream.close();
									} 
	    							catch (IOException i) {
										//Ignore since closing file input stream.
									}
	    						}

	    						if(fileOutputStream != null){
	    							
	    							try {
	    								fileOutputStream.close();
									} 
	    							catch (IOException i) {
										//Ignore since closing file output stream.
									}
	    						}
	    					}
	    				}
	    				
					}
	    
	    			attachmentsTable.deselect(attachmentsTable.getSelectionIndices());
	    		}
	    		
	    		attachmentsOpenButton.setEnabled(false);
	    		attachmentsSaveButton.setEnabled(false);
	    	}
	    });
		
		attachmentsTable.addListener(SWT.Selection, new Listener(){
			    	 
			public void handleEvent(Event event){	    		
				
				attachmentsOpenButton.setEnabled(attachmentsTable.getSelectionCount() > 0);
				attachmentsSaveButton.setEnabled(attachmentsTable.getSelectionCount() > 0);
			}
		});
		
		toolkit.paintBordersFor(attachmentsButtonComposite);
	    
		toolkit.paintBordersFor(attachmentsComposite);
	}
	
	public void commit(boolean onSave) {
		// only change dirty state after saving.
		if(onSave)
			super.commit(onSave);
	}
	
	protected void updateCommonSectionState(Section commonSection)
	{
		Composite commonSectionClient = (Composite)commonSection.getClient();
		commonSectionClient.getParent().layout(true);
		commonSectionClient.getParent().getParent().layout(true);
	}
	
	protected void updateExtendedSectionState(Section extendedSection)
	{
		if(propertiesTable.getItemCount() == 0 && attachmentsTable.getItemCount() == 0)
			extendedSection.setExpanded(false);
		else
			extendedSection.setExpanded(true);
	}
	
	public void dispose()
	{
		getManagedForm().removePart(this);
		super.dispose();
		disposeContent();
	}
	
	protected void disposeContent()
	{
		if(commonPropSection != null && !commonPropSection.isDisposed())
		{
			Composite client = (Composite)commonPropSection.getClient();
			client.dispose();
			timeText = null;
			messageText = null;
			modelElementLink = null;
			commonPropSection.dispose();
			commonPropSection = null;
		}
		if(extendedPropSection != null && !extendedPropSection.isDisposed())
		{
			Composite client = (Composite)extendedPropSection.getClient();
			client.dispose();
			propertiesTable = null;
			attachmentsTable = null;
			extendedPropSection.dispose();
			extendedPropSection = null;
		}
		executionEvent = null;
	}
	
	protected TPFExecutionEvent getExecutionEvent()
	{
		return executionEvent;
	}

	public boolean setFormInput(Object input)
	{
		// TODO Auto-generated method stub
		return false;
	}
	
	protected void setInput(Object object)
	{
		executionEvent = null;

		if(object instanceof IStructuredSelection)
		{
			IStructuredSelection structuredSelection = (IStructuredSelection)object;
			if(structuredSelection.size() == 1)
				object = structuredSelection.getFirstElement();
		}
		
		/*if(object instanceof CMNNamedElement)
		{
			CMNNamedElement namedElement = (CMNNamedElement)object;
			descriptionText.removeModifyListener(this);
			String description = namedElement.getDescription();
			if(description != null && description.length() > 0)
				descriptionText.setText(namedElement.getDescription());
			descriptionText.addModifyListener(this);
		}*/

		if(object instanceof TPFExecutionEvent)
		{
			executionEvent = (TPFExecutionEvent)object;
			
			timeText.setText(EventUtil.getTime(getExecutionEvent().getTimestamp()));
			
			String value = getExecutionEvent().getText();
			setMessageText(value);

			modelElementLink.setVisible(true);

			TPFTest test = null;
			if((executionEvent.getExecutionHistory() != null) && (executionEvent.getExecutionHistory().getExecutionResult() != null))
				test = executionEvent.getExecutionHistory().getExecutionResult().getTest();

			BVRInteractionFragment interactionFragment = executionEvent.getInteractionFragment();
			if((interactionFragment != null) && (interactionFragment.eResource() != null))
			{
				if(interactionFragment.getName() != null){
					modelElementLink.setText(interactionFragment.getName());
					modelElementLink.setToolTipText(interactionFragment.getName());
				}
				else{
					modelElementLink.setText(UiPluginResourceBundle.W_ELEMENT); 
					modelElementLink.setToolTipText(UiPluginResourceBundle.W_ELEMENT);
				}
					
				modelElementLink.setData(executionEvent);
			}
			else if((test != null) && (test.eResource() != null))
			{
				if(test.getName() != null){
					modelElementLink.setText(test.getName());
					modelElementLink.setToolTipText(test.getName());
				}
				else{
					modelElementLink.setText(UiPluginResourceBundle.W_TEST);
					modelElementLink.setToolTipText(UiPluginResourceBundle.W_TEST);
				}
				
				modelElementLink.setData(executionEvent.getExecutionHistory().getExecutionResult());
			}
			else
			{
				modelElementLink.setText(UiPluginResourceBundle.NO_INTF_TO_INV); 
				modelElementLink.setToolTipText(UiPluginResourceBundle.NO_INTF_TO_INV);
				modelElementLink.setData(null);
			}
			
			//display properties and attachments of this execution event
			setPropertiesInput(executionEvent);
			setAttachmentsInput(executionEvent);			
		}
		else
		{
			timeText.setText(""); //$NON-NLS-1$
			setMessageText(""); //$NON-NLS-1$
			modelElementLink.setVisible(false);
		}		

		if(defectProviders != null)
		{
			for(int i = 0; i < defectProviders.length; i++)
			{
				defectProviders[i].setInput(executionEvent);
			}
		}

		updateCommonSectionState(commonPropSection);
		updateExtendedSectionState(extendedPropSection);
	}
	
	private void setPropertiesInput(TPFExecutionEvent event)
	{				
		if(propertiesTable == null || propertiesTable.isDisposed())
			return;
		propertiesTable.removeAll();
		List properties = event.getProperties();
		if(properties != null && !properties.isEmpty())
		{
			String eventType = event.getEventType();			
			for(Iterator it = properties.iterator(); it.hasNext();)
			{
				Object obj = it.next();
				if(obj instanceof CMNExtendedProperty)
				{
					String name = ((CMNExtendedProperty)obj).getName();
					String value = ((CMNExtendedProperty)obj).getValue();
					value = (value == null) ? "" : value; //$NON-NLS-1$
					if(eventType != null && name != null)
					{
						// customize the property display from extensions.
						ExecutionHistoryExtensionsManager.Property prop = ExecutionHistoryExtensionsManager.getInstance().getEventProperty(eventType, name);
						if(prop != null)
						{
							if(!prop.isVisible())
								continue;
							
							if(value.equals(prop.getVisibleUnless()))
								continue;
							
							IPropertyLabelProvider provider = prop.getProvider();
							if(provider != null)
							{
								name = (provider.getName((CMNExtendedProperty)obj) == null) ? name : provider.getName((CMNExtendedProperty)obj);
								value = (provider.getValue((CMNExtendedProperty)obj) == null) ? value : provider.getValue((CMNExtendedProperty)obj);
							}
							
							if(prop.getUnit() != null)
							{
								value = value.concat(" ").concat(prop.getUnit());
							}
						}
					}
						
					TableItem item = new TableItem(propertiesTable, SWT.NONE);
					item.setText(new String[]{name, value});
				}
			}

			propertiesTable.update();
			getManagedForm().getToolkit().paintBordersFor(propertiesTable.getParent());
		}
	}
	
	protected void setAttachmentsInput(TPFExecutionEvent event)
	{
		attachmentsTable.removeAll();
		
		attachmentsOpenButton.setEnabled(false);
		attachmentsSaveButton.setEnabled(false);

		Iterator annotationsIterator = event.getAnnotations().iterator();

		while(annotationsIterator.hasNext()){
		
			CMNAnnotation annotation = ((CMNAnnotation)(annotationsIterator.next()));
			
			try {
				
				File file = new File(annotation.getFileAnnotation().toFileString());
				
				if(file != null){

					//Resolve the file properties:
					String fileName = file.getName().trim();
					int lastDotIndex = fileName.lastIndexOf('.');		   
					String fileExtension = ""; //$NON-NLS-1$
					
					if(lastDotIndex != -1){	    					
						fileExtension = fileName.substring(lastDotIndex + 1);
					}
					
					Program program = Program.findProgram(fileExtension);
					ImageData icon = null;
					String programName = null;

					if(program != null){
						
						icon = program.getImageData();
						programName = program.getName().trim();
					}

					//Resolve the cells:
					Image image = null;
					String[] cells = new String[3];
					
					//Resolve the image:
					if(icon != null){
						image = new Image(Display.getCurrent(), icon);
					}		    			  
					else{
						image = TestUIImages.INSTANCE.getImage(TestUIImages.IMG_FILE);
					}

					//Resolve the name:
					cells[0] = fileName;

					//Resolve the file size:
					if(file.length() > 0){
						cells[1] = NumberFormat.getIntegerInstance().format(Math.max(Math.round((double)(((double)(file.length())) / 1024.0)),1)).concat(" KB");  //$NON-NLS-1$
					}
					else{
						cells[1] =  "0 KB"; //$NON-NLS-1$
					}
					
					//Resolve the type:
					if((icon != null) && (programName.trim().length() > 0)){
						cells[2] = programName;
					}
					else{
						cells[2] = fileExtension.toUpperCase().concat(" File").trim(); //$NON-NLS-1$
					}
					
					//Resolve the row index:
					int newRowIndex = attachmentsTable.getItemCount();
					
					if((newRowIndex > 0) && (attachmentsTable.getSortDirection() != SWT.NONE)){
							
			    		TableItem[] rows = attachmentsTable.getItems();
						int sortColumnIndex = Arrays.asList(attachmentsTable.getColumns()).indexOf(attachmentsTable.getSortColumn());
				        Collator collator = Collator.getInstance();
				        
						if(attachmentsTable.getSortDirection() == SWT.UP){
								
				        	for (int counter = 0; counter < rows.length; counter++){
				                    
				            	if (collator.compare(cells[sortColumnIndex], rows[counter].getText(sortColumnIndex)) < 0) {
				                    
				            		newRowIndex = counter;    
				            		
				                    break;
				                }
							}
						}
						else{
							
							for (int counter = 0; counter < rows.length; counter++){

								if (collator.compare(cells[sortColumnIndex], rows[counter].getText(sortColumnIndex)) > 0) {
				                    
									newRowIndex = counter; 
				            		
				                    break;
				                }
							}
						}
					}
					
					//Create the row:
					TableItem fileTableItem = new TableItem (attachmentsTable, SWT.NONE, newRowIndex);
					fileTableItem.setData("FILE", file); //$NON-NLS-1$
					fileTableItem.setImage(image);
					fileTableItem.setText(cells);
				}
			} 
			catch (Exception e) {
				//Ignore since annotation cannot be added to the table.
			}
		}
	}
	
	protected boolean isHTMLString(String text)
	{
		if(text == null)
			return false;
		
		int length = text.length();
		if(length > (HTML_TAG_START + HTML_TAG_END).length() &&
			HTML_TAG_START.compareToIgnoreCase(text.substring(0, HTML_TAG_START.length())) == 0 && 
			HTML_TAG_END.compareToIgnoreCase(text.substring(length - HTML_TAG_END.length(), length)) == 0)
			return true;
		else
			return false;
	}

	/**
	 * Purpose of the browser text box is to display html formatted text. 
	 * However it is not a full-blown browser and does not expose any functions of a real browser.
	 * Thus add target="_blank" to anchors with hyperlinks to open the link in a new system default browser window.
	 * @param html
	 * @return
	 */
	protected String parseHTMLString(String html) {
		if(!isHTMLString(html))
			return html;

		int anchorStart = html.indexOf(HTML_TAG_ANCHOR_START);
		while(anchorStart > -1)
		{
			int anchorEnd = html.indexOf('>', anchorStart);
			int href = html.indexOf(HTML_ATTR_HREF, anchorStart);
			if(href > -1)
			{
				String anchor = html.substring(anchorStart, anchorEnd);
				
				if(anchor.indexOf(HTML_ATTR_TARGET) == -1)
				{
					int insertIndex = anchorStart + HTML_TAG_ANCHOR_START.length();
					html = html.substring(0, insertIndex) + HTML_ATTR_TARGET_BLANK + html.substring(insertIndex, html.length());
				}
				else
				{
					// only replace *target* attribute that's *_self*.
					if(anchor.indexOf(HTML_ATTR_TARGET_SELF) > -1)
					{
						html = html.substring(0, anchorStart) + 
								anchor.replaceAll(HTML_ATTR_TARGET_SELF, HTML_ATTR_TARGET_BLANK) + 
								html.substring(anchorEnd, html.length());
					}
				}
			}
				
			anchorStart = html.indexOf(HTML_TAG_ANCHOR_START, anchorEnd);
		}
		
		return html;
	}

	public void setFocus()
	{
	}

	public void selectionChanged(IFormPart part, ISelection selection)
	{
		if(selection instanceof IStructuredSelection)
		{
			IStructuredSelection sel = (IStructuredSelection)selection;
			if(!sel.isEmpty())
				setInput(sel.getFirstElement());
		}

	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.events.IHyperlinkListener#linkActivated(org.eclipse.ui.forms.events.HyperlinkEvent)
	 */
	public void linkActivated(HyperlinkEvent e)
	{
		if(e.widget == modelElementLink)
		{
			Object data = modelElementLink.getData();
			if(data != null)
			{
				if (data instanceof TPFExecutionEvent) {
					data = ((TPFExecutionEvent)data).getInteractionFragment();
				} else if (data instanceof TPFExecutionResult) {
					data = ((TPFExecutionResult)data).getTest();
				}
				EObject eObject = (EObject)data;
				Resource resource = eObject.eResource();
				IEditorPart editorPart = TestUIUtil.openEditor(resource, null, false);
				if(editorPart instanceof ISelectionProvider)
					((ISelectionProvider)editorPart).setSelection(new StructuredSelection(eObject));
				
				return;
			}
		}
		
		IActionBars actionBars = editorPage.getEditorSite().getActionBars();
		IStatusLineManager manager = actionBars.getStatusLineManager();
		if (manager != null)
			manager.setErrorMessage(UiPluginResourceBundle._ERROR_MSG_UN_OPEN_OBJ); 
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.events.IHyperlinkListener#linkEntered(org.eclipse.ui.forms.events.HyperlinkEvent)
	 */
	public void linkEntered(HyperlinkEvent e)
	{
		IActionBars actionBars = editorPage.getEditorSite().getActionBars();
		IStatusLineManager manager = actionBars.getStatusLineManager();
		if (manager != null)
			manager.setMessage(modelElementLink.getText());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.events.IHyperlinkListener#linkExited(org.eclipse.ui.forms.events.HyperlinkEvent)
	 */
	public void linkExited(HyperlinkEvent e)
	{
		IActionBars actionBars = editorPage.getEditorSite().getActionBars();
		IStatusLineManager manager = actionBars.getStatusLineManager();
		if (manager != null)
		{
			manager.setMessage(""); //$NON-NLS-1$
			manager.setErrorMessage(""); //$NON-NLS-1$
		}
	}

	/*public void modifyText(ModifyEvent e) {		
		if(e.widget == descriptionText)
		{
			if(executionEvent != null)
			{
				String description = descriptionText.getText();
				executionEvent.setDescription(description);
				commonSectionPart.markDirty();
			}
		}
	}*/
	
	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}
	
	public void widgetSelected(SelectionEvent e) {
		/*if(e.widget == recordsTable)
		{
			TableItem[] selections = recordsTable.getSelection();
			if(selections != null && selections.length > 0)
			{
				remove.setEnabled(true);
				open.setEnabled(true);
			}
			else
			{
				remove.setEnabled(false);
				open.setEnabled(false);
			}
		}*/
	}
	
	public void recordAdded(TPFRepositoryRecord record) {
		if(executionEvent != null && !executionEvent.getDefectRecords().contains(record))
		{
			executionEvent.getDefectRecords().add(record);	
			TPFExecutionResult result = getRootExecutionResult(executionEvent);
			if(result != null && !result.getRecords().contains(record))
				result.getRecords().add(record);
			
			markDirty();
		}
	}
	public void recordRemoved(TPFRepositoryRecord record) {
		if(executionEvent != null && executionEvent.getDefectRecords().contains(record))
		{
			executionEvent.getDefectRecords().remove(record);	
			TPFExecutionResult result = getRootExecutionResult(executionEvent);
			if(result != null && result.getRecords().contains(record))
				result.getRecords().remove(record);
			
			markDirty();
		}
	}

	/**
	 * @provisional As of TPTP V4.5.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	protected Hyperlink getModelElementLink(){
		return modelElementLink;
	}
	
	private TPFExecutionResult getRootExecutionResult(TPFExecutionEvent event)
	{
		
		EObject container = event.eContainer();
		if(container != null)
		{
			while(container.eContainer() != null)
				container = container.eContainer();
		}
		if(container instanceof TPFExecutionResult)
			return (TPFExecutionResult)container;
		
		return null;
	}
}