/********************************************************************** 
 * Copyright (c) 2005, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: ExecutionResultDetailsPart.java,v 1.27 2010/05/28 14:17:03 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.ui.forms.base;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.common.CMNNodeType;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.util.ConfigurationUtil;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.forms.editor.TestLogViewer;
import org.eclipse.hyades.test.ui.forms.util.FormsUtil;
import org.eclipse.hyades.test.ui.forms.util.ITestLogVerdictTraversal;
import org.eclipse.hyades.test.ui.internal.model.EventUtil;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.util.TestUIUtil;
import org.eclipse.hyades.ui.extension.IAssociationConstants;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.forms.IDetailsPage;
import org.eclipse.ui.forms.IFormPart;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormPage;
import org.eclipse.ui.forms.events.HyperlinkEvent;
import org.eclipse.ui.forms.events.IHyperlinkListener;
import org.eclipse.ui.forms.widgets.FormText;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ImageHyperlink;
import org.eclipse.ui.forms.widgets.Section;
import org.eclipse.ui.model.IWorkbenchAdapter;

/**
 * <p>Details page of the page book used for {@link org.eclipse.hyades.models.common.testprofile.TPFExecutionResult TPFExecutionResult}
 * in the Overview page as well as Events page of Test Log Viewer.</p>
 * 
 * <p>This is an Eclipse forms page derived from the original TPTP Test Log viewer.</p>
 * 
 * 
 * @author      Paul E. Slauenwhite
 * @author      Bianca Xue Jiang 
 * @author		Marcelo Paternostro
 * @version     May 28, 2010
 * @since       August 9, 2005
 */
public class ExecutionResultDetailsPart implements IDetailsPage, IHyperlinkListener {

	protected FormPage formPage;
	protected IManagedForm mForm;
	protected FormToolkit toolkit;
	
	private Composite testComposite;
	private ImageHyperlink testLink;
	private FormText testType;
	private FormText testFile;
	private FormText hostname;
	private ImageHyperlink deploymentLink;
	private ImageHyperlink locationLink;
	
	private Text verdictText;
	private Text startText;
	private Text stopText;
	
	/**
	 * Default Constructor
	 */
	public ExecutionResultDetailsPart()
	{
	}
	
	/**
	 * Creates an instance of this class with the editor page.
	 * @param page forms editor page of this part.
	 */
	public ExecutionResultDetailsPart(FormPage page)
	{
		this.formPage = page;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.IFormPart#initialize(org.eclipse.ui.forms.IManagedForm)
	 */
	public void initialize(IManagedForm form)
	{
		this.mForm = form;
		toolkit = form.getToolkit();
	}
	
	public void setFormPage(FormPage page)
	{
		this.formPage = page;
	}
	
	public FormPage getFormPage()
	{
		return this.formPage;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.IDetailsPage#createContents(org.eclipse.swt.widgets.Composite)
	 */
	public void createContents(Composite parent)
	{				
		createCommonPropSection(parent);
		mForm.addPart(this);
	}

	protected void createCommonPropSection(Composite parent) {
		Section detailsSection = FormsUtil.createSection(mForm, parent, UiPluginResourceBundle.TTL_CMN_PROPS, ""); 
		Composite client = (Composite)detailsSection.getClient();
		//client.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_BLACK));
		
		Composite verdictComposite = toolkit.createComposite(client);
		verdictComposite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		GridLayout layout = new GridLayout(2, false);
		layout.marginBottom = layout.marginTop = layout.marginLeft = layout.marginRight = 0;
		verdictComposite.setLayout(layout);
		//verdictComposite.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_DARK_GRAY));
		
		Label label = toolkit.createLabel(verdictComposite, UiPluginResourceBundle.LBL_VERD); 
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 2;
		label.setLayoutData(gd);
		verdictText = toolkit.createText(verdictComposite, "", SWT.FULL_SELECTION | SWT.SINGLE); //$NON-NLS-1$
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 2;
		verdictText.setLayoutData(gd);
		verdictText.setEditable(false);
		
		label = toolkit.createLabel(verdictComposite, UiPluginResourceBundle.LBL_START); 
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 2;
		label.setLayoutData(gd);
		startText = toolkit.createText(verdictComposite, "", SWT.FULL_SELECTION | SWT.SINGLE); //$NON-NLS-1$
		//startText.setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB));
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 2;
		startText.setLayoutData(gd);
		startText.setEditable(false);

		label = toolkit.createLabel(verdictComposite, UiPluginResourceBundle.LBL_STOP); 
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 2;
		label.setLayoutData(gd);
		stopText = toolkit.createText(verdictComposite, "", SWT.FULL_SELECTION | SWT.SINGLE); //$NON-NLS-1$
		//stopText.setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB));
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 2;
		stopText.setLayoutData(gd);
		stopText.setEditable(false);
		
		toolkit.paintBordersFor(verdictComposite);
		
		createTestDetails(client);
		toolkit.paintBordersFor(client);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.IPartSelectionListener#selectionChanged(org.eclipse.ui.forms.IFormPart, org.eclipse.jface.viewers.ISelection)
	 */
	public void selectionChanged(IFormPart part, ISelection selection)
	{
		if(selection instanceof IStructuredSelection)
		{
			IStructuredSelection sel = (IStructuredSelection)selection;
			if(!sel.isEmpty())
				setFormInput(sel.getFirstElement());
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.IFormPart#commit(boolean)
	 */
	public void commit(boolean onSave)
	{
		// TODO Auto-generated method stub
		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.IFormPart#dispose()
	 */
	public void dispose()
	{
		// TODO Auto-generated method stub
		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.IFormPart#isDirty()
	 */
	public boolean isDirty()
	{
		// TODO Auto-generated method stub
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.IFormPart#isStale()
	 */
	public boolean isStale()
	{
		// TODO Auto-generated method stub
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.IFormPart#refresh()
	 */
	public void refresh()
	{
		setFormInput(formPage.getManagedForm().getInput());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.IFormPart#setFocus()
	 */
	public void setFocus()
	{
		// TODO Auto-generated method stub
	}
	
	protected void createTestDetails(Composite parent){

		//Reuseable layout data:
		GridLayout layout = new GridLayout(2, false);
		layout.marginTop = 10;

		GridData gridData = new GridData();
		gridData.horizontalSpan = 2;
		
		//Test composite:
		testComposite = toolkit.createComposite(parent);
		testComposite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		testComposite.setLayout(layout);

		//Test:
		testLink = toolkit.createImageHyperlink(testComposite, SWT.WRAP);
		testLink.setImage(TestUIImages.INSTANCE.getImage(TestUIImages.IMG_ERROR));
		testLink.setText(UiPluginResourceBundle.NO_TST_TO_INV);
		testLink.addHyperlinkListener(this);
		testLink.setLayoutData(gridData);
		
		//Type:
		FormText typeLabel = toolkit.createFormText(testComposite, false); 
		typeLabel.setText(UiPluginResourceBundle.L_TYPE, false, false); 
		typeLabel.setLayoutData(new GridData());

		testType = toolkit.createFormText(testComposite, false);
		testType.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		//File:
		FormText fileLabel = toolkit.createFormText(testComposite, false); 
		fileLabel.setText(UiPluginResourceBundle.L_FILE, false, false); 
		fileLabel.setLayoutData(new GridData());

		testFile = toolkit.createFormText(testComposite, false);
		testFile.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		//Host:
		FormText hostnameLabel = toolkit.createFormText(testComposite, false); 
		hostnameLabel.setText(UiPluginResourceBundle.LBL_HOST_COLON, false, false); 
		hostnameLabel.setLayoutData(new GridData());
		
		hostname = toolkit.createFormText(testComposite, false);
		hostname.setText(ConfigurationUtil.LOCALHOST, false, false);
		hostname.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		toolkit.paintBordersFor(testComposite);
	}
	
	protected void createDeploymentLinks(Composite parent, TPFDeployment deployment, TPFTestSuite test){
		
		if((deployment != null) && (test != null)){
			
			//Create the deployment label:
			if(deployment.eResource() != null){
			
				if((deploymentLink == null) || (deploymentLink.isDisposed())){
					
					toolkit.createLabel(parent, UiPluginResourceBundle.L_DEPLOYMENT); 
					
					deploymentLink = toolkit.createImageHyperlink(parent, SWT.WRAP);
					deploymentLink.setImage(TestUIImages.INSTANCE.getImage(TestUIImages.IMG_DEPLOYMENT));
					deploymentLink.setText(deployment.getName());
					deploymentLink.setToolTipText(UiPluginResourceBundle.TIP_DEPLOYMENT_USED); 
					deploymentLink.addHyperlinkListener(this);
				}
				
				deploymentLink.setData(deployment);
			}
			
			//Create the location label:
			CFGLocation location = null;		
			
			//Bugzilla_146905 Special handling of deployment that only contains a single location. 
			if(ConfigurationUtil.isDefaultLocation(deployment)){
				location = ConfigurationUtil.getDefaultLocation(deployment);
			}
			else{
				location = ConfigurationUtil.searchLocationWithTestAsset(test, deployment);
			}
			
			if((location != null) && (location.eResource() != null)){
				
				if((locationLink == null) || (locationLink.isDisposed())){
					
					toolkit.createLabel(parent, UiPluginResourceBundle.L_LOCATION); 
					
					locationLink = toolkit.createImageHyperlink(parent, SWT.WRAP);
					locationLink.setImage(TestUIImages.INSTANCE.getImage(TestUIImages.IMG_LOCATION));
					locationLink.setText(location.getName());
					locationLink.setToolTipText(UiPluginResourceBundle.TIP_LOCATION_USED); 
					locationLink.addHyperlinkListener(this);
				}
				
				locationLink.setData(location);			
				
				if(location instanceof CMNNodeType){
					hostname.setText(((CMNNodeType)location).getHostname(), false, false);
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.IFormPart#setFormInput(java.lang.Object)
	 */
	public boolean setFormInput(Object input) {
		TPFExecutionResult executionResult;
		
		if(input instanceof IStructuredSelection)
		{
			IStructuredSelection structuredSelection = (IStructuredSelection)input;
			if(structuredSelection.size() == 1)
				input = structuredSelection.getFirstElement();
		}

		boolean doPack = false;
		if(input instanceof TPFExecutionResult)
		{
			executionResult = (TPFExecutionResult)input;
			TPFTest test = executionResult.getTest();
			doPack = true;
			
			if(test != null)
			{		
				
				if(test.eResource() != null){

					ImageDescriptor imageDescriptor = null;
					if(test instanceof IAdaptable)
					{
						IWorkbenchAdapter workbenchAdapter = (IWorkbenchAdapter)((IAdaptable)test).getAdapter(IWorkbenchAdapter.class);
						if((workbenchAdapter != null))
							imageDescriptor = workbenchAdapter.getImageDescriptor(test);
					}
					if(imageDescriptor == null)
						imageDescriptor = TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_DEFAULT);
	
					testLink.setImage(imageDescriptor.createImage());
					
					if(test.getName() != null)
						testLink.setText(test.getName());
					else
						testLink.setText(UiPluginResourceBundle.W_TEST); 
					testLink.setData(executionResult);
					
					TPFTestSuite testSuite = null;
					if(test instanceof TPFTestSuite)
						testSuite = (TPFTestSuite)test;
					else
						testSuite = ((TPFTestCase)test).getTestSuite();
					if (testSuite!=null) {
						String value = testSuite.getType();
						if(value != null)
						{
							IAssociationDescriptor descriptor = TestUIExtension.getTestSuiteMappingRegistry().getAssociationMapping(IAssociationConstants.EP_TYPE_DESCRIPTIONS).getDefaultAssociationDescriptor(value);
							if((descriptor != null) && (descriptor.getName() != null))
								value = descriptor.getName();
							testType.setText(value, false, false);
						}
						testFile.setText(EMFUtil.getFilePath(testSuite), false, false);
					}					
				}
				else{
					
					testLink.setImage(TestUIImages.INSTANCE.getImage(TestUIImages.IMG_ERROR));
					testLink.setText(UiPluginResourceBundle.NO_TST_TO_INV); 
					testLink.setData(null);
					testFile.setText("", false, false); //$NON-NLS-1$
					testType.setText("", false, false); //$NON-NLS-1$
				}
				
				//bugzilla_144045 Show "no verdict" when there is no actually verdict events in the test log.
				boolean noVerdict = true;
				TestLogVerdictTraversalQuery query = (TestLogVerdictTraversalQuery) ((TestLogViewer)formPage.getEditor()).getTestLogVerdictTraversal();
				if (query.getCount(ITestLogVerdictTraversal.VERDICT_TYPE_ALL) != 0)
					noVerdict = false;
				
				if(noVerdict)
					verdictText.setText(UiPluginResourceBundle.LogOverview_NoVerdict); 
				else
					verdictText.setText(executionResult.getVerdict().getLabel());
					
				startText.setText(EventUtil.getTime(EventUtil.getStartTimeStamp(executionResult)));
				stopText.setText(EventUtil.getTime(EventUtil.getStopTimeStamp(executionResult)));	
								
				// bugzilla_63734 Provide location info. in test log viewer.
				TPFDeployment deployment = executionResult.getDeployment();
				if(deployment != null && (test instanceof TPFTestSuite))
				{
					createDeploymentLinks(testComposite, deployment, (TPFTestSuite)test);
				}
			}
		}
		else
		{
			verdictText.setText(UiPluginResourceBundle.LogOverview_NoVerdict); 
			startText.setText(""); //$NON-NLS-1$
			stopText.setText(""); //$NON-NLS-1$
		}
		
		if(doPack)
		{
			testLink.pack(true);
			testLink.redraw();			
		}
		
		return true;
	}
	
	/**
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	protected ImageHyperlink getTestLink()
	{
		return testLink;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.events.IHyperlinkListener#linkActivated(org.eclipse.ui.forms.events.HyperlinkEvent)
	 */
	public void linkActivated(HyperlinkEvent e)
	{
		
		EObject eObject = null;

		if(e.widget == testLink)
		{
			Object data = testLink.getData();
			
			if((data != null) && (data instanceof TPFExecutionResult)){
				eObject = ((TPFExecutionResult)(data)).getTest();
			}
		}
		else if(e.widget == deploymentLink)
		{
			
			Object data = deploymentLink.getData();

			if((data != null) && (data instanceof TPFDeployment)){
				eObject = ((TPFDeployment)(data));
			}
		}
		else if(e.widget == locationLink)
		{
			
			Object data = locationLink.getData();
			
			if((data != null) && (data instanceof CFGLocation)){
				eObject = ((CFGLocation)(data));
			}
		}	
		
		if(eObject != null){

			IEditorPart editorPart = TestUIUtil.openEditor(eObject.eResource(), null, false);
			
			if(editorPart instanceof ISelectionProvider){
				((ISelectionProvider)editorPart).setSelection(new StructuredSelection(eObject));
			}
		
			return;
		}
		
		IActionBars actionBars = formPage.getEditorSite().getActionBars();
		IStatusLineManager manager = actionBars.getStatusLineManager();

		if (manager != null){
			manager.setErrorMessage(UiPluginResourceBundle._ERROR_MSG_UN_OPEN_OBJ);
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.events.IHyperlinkListener#linkEntered(org.eclipse.ui.forms.events.HyperlinkEvent)
	 */
	public void linkEntered(HyperlinkEvent e)
	{
		IActionBars actionBars = formPage.getEditorSite().getActionBars();
		IStatusLineManager manager = actionBars.getStatusLineManager();
		if (manager != null)
		{
			if(e.widget == testLink)
				manager.setMessage(testLink.getText());
			else if(e.widget == deploymentLink)
				manager.setMessage(deploymentLink.getText());
			else if(e.widget == locationLink)
				manager.setMessage(locationLink.getText());
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.events.IHyperlinkListener#linkExited(org.eclipse.ui.forms.events.HyperlinkEvent)
	 */
	public void linkExited(HyperlinkEvent e)
	{
		IActionBars actionBars = formPage.getEditorSite().getActionBars();
		IStatusLineManager manager = actionBars.getStatusLineManager();
		if (manager != null)
		{
			manager.setMessage(""); //$NON-NLS-1$
			manager.setErrorMessage(""); //$NON-NLS-1$
		}
	}
}
