/********************************************************************** 
 * Copyright (c) 2008, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: ExecutionResultEventDetailsPart.java,v 1.7 2010/05/28 14:17:03 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.ui.forms.base;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.forms.editor.TestLogViewer;
import org.eclipse.hyades.test.ui.forms.util.FormsUtil;
import org.eclipse.hyades.test.ui.forms.util.ITestLogVerdictTraversal;
import org.eclipse.hyades.test.ui.internal.model.EventUtil;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.util.TestUIUtil;
import org.eclipse.hyades.ui.extension.IAssociationConstants;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.forms.AbstractFormPart;
import org.eclipse.ui.forms.IDetailsPage;
import org.eclipse.ui.forms.IFormPart;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormPage;
import org.eclipse.ui.forms.events.HyperlinkEvent;
import org.eclipse.ui.forms.events.IHyperlinkListener;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.Hyperlink;
import org.eclipse.ui.forms.widgets.Section;

/**
 * <p>Details page of the page book used for {@link org.eclipse.hyades.models.common.testprofile.TPFExecutionResult TPFExecutionResults}
 * in the Events page of the Test Log Viewer.</p>
 *
 * 
 * @author      Paul E. Slauenwhite
 * @version     May 28, 2010
 * @since       July 9, 2008
 * @provisional As of TPTP V4.5.1, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
 */
public class ExecutionResultEventDetailsPart extends AbstractFormPart implements IDetailsPage, IHyperlinkListener {
	
	private FormPage formPage = null;
	private Composite parent = null;
	private Section commonPropertiesSection = null;
	private Hyperlink modelElementLink = null;
	private Text testTypeText = null;
	private Text testFileText = null;
	private Text verdictText = null;
	private Text startTimeText = null;
	private Text stopTimeText = null;

	/**
	 * Creates an instance of this class with the editor page.
	 * 
	 * @param page forms editor page of this part.
	 */
	public ExecutionResultEventDetailsPart(FormPage formPage){
		this.formPage = formPage;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.IDetailsPage#createContents(org.eclipse.swt.widgets.Composite)
	 */
	public void createContents(Composite parent){		
		
		if(parent != this.parent){
		
			//Add this form part to the managed form the first time this part is created:
			if(this.parent == null){
				getManagedForm().addPart(this);
			}
			
			this.parent = parent;

			//Recreate the content of this page:
			disposeContent();
			
			commonPropertiesSection = FormsUtil.createSection(getManagedForm(), parent, UiPluginResourceBundle.TTL_PROPERTIES, "");  //$NON-NLS-1$
			
			Composite commonSectionClient = (Composite)commonPropertiesSection.getClient();
			
			FormToolkit toolkit = getManagedForm().getToolkit();
			
			modelElementLink = toolkit.createHyperlink(commonSectionClient, "", SWT.WRAP); //$NON-NLS-1$
			modelElementLink.addHyperlinkListener(this);
			modelElementLink.setLayoutData(new GridData());
			modelElementLink.setText(UiPluginResourceBundle.NO_INTF_TO_INV); 
			modelElementLink.setToolTipText(UiPluginResourceBundle.NO_INTF_TO_INV); 
			
			toolkit.createLabel(commonSectionClient, UiPluginResourceBundle.LBL_TYPE); 
			testTypeText = toolkit.createText(commonSectionClient, "", SWT.FULL_SELECTION | SWT.SINGLE); //$NON-NLS-1$
			testTypeText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
			testTypeText.setEditable(false);

			toolkit.createLabel(commonSectionClient, UiPluginResourceBundle.LBL_FILE); 
			testFileText = toolkit.createText(commonSectionClient, "", SWT.FULL_SELECTION | SWT.SINGLE); //$NON-NLS-1$
			testFileText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
			testFileText.setEditable(false);

			toolkit.createLabel(commonSectionClient, UiPluginResourceBundle.LBL_VERD); 
			verdictText = toolkit.createText(commonSectionClient, "", SWT.FULL_SELECTION | SWT.SINGLE); //$NON-NLS-1$
			verdictText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
			verdictText.setEditable(false);
			
			toolkit.createLabel(commonSectionClient, UiPluginResourceBundle.LBL_START); 
			startTimeText = toolkit.createText(commonSectionClient, "", SWT.FULL_SELECTION | SWT.SINGLE); //$NON-NLS-1$
			startTimeText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
			startTimeText.setEditable(false);

			toolkit.createLabel(commonSectionClient, UiPluginResourceBundle.LBL_STOP); 
			stopTimeText = toolkit.createText(commonSectionClient, "", SWT.FULL_SELECTION | SWT.SINGLE); //$NON-NLS-1$
			stopTimeText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
			stopTimeText.setEditable(false);

			toolkit.paintBordersFor(commonSectionClient);
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.AbstractFormPart#initialize(org.eclipse.ui.forms.IManagedForm)
	 */
	public void initialize(IManagedForm managedForm){
		super.initialize(managedForm);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.AbstractFormPart#setFormInput(java.lang.Object)
	 */
	public boolean setFormInput(Object input){
		return false;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.AbstractFormPart#setFocus()
	 */
	public void setFocus(){
		//No-operation.
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.AbstractFormPart#commit(boolean)
	 */
	public void commit(boolean onSave) {
		
		//Only change dirty state after saving:
		if(onSave){
			super.commit(onSave);
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.AbstractFormPart#dispose()
	 */
	public void dispose(){
		
		getManagedForm().removePart(this);
		
		super.dispose();
		
		disposeContent();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.IPartSelectionListener#selectionChanged(org.eclipse.ui.forms.IFormPart, org.eclipse.jface.viewers.ISelection)
	 */
	public void selectionChanged(IFormPart part, ISelection selection){
		
		if(selection instanceof IStructuredSelection){
			
			IStructuredSelection structuredSelection = ((IStructuredSelection)(selection));
			
			if(!structuredSelection.isEmpty()){
				
				Object object = structuredSelection.getFirstElement();
				
				if((object instanceof TPFExecutionResult) && (((TPFExecutionResult)(object)).getTest() != null)){
					
					TPFExecutionResult executionResult = ((TPFExecutionResult)(object));
										
					TPFTest test = executionResult.getTest();
					
					modelElementLink.setVisible(true);

					//Handle the case where the referenced test cannot be resolved (e.g. moved/deleted):
					if((test != null) && (test.eResource() != null)){
						
						if(test.getName() != null){
							modelElementLink.setText(test.getName());
							modelElementLink.setToolTipText(test.getName()); 
						}
						else{
							modelElementLink.setText(UiPluginResourceBundle.W_TEST);
							modelElementLink.setToolTipText(UiPluginResourceBundle.W_TEST); 
						}

						modelElementLink.setData(executionResult);
					}
					else{
						
						modelElementLink.setText(UiPluginResourceBundle.NO_INTF_TO_INV);
						modelElementLink.setToolTipText(UiPluginResourceBundle.NO_INTF_TO_INV); 
						modelElementLink.setData(null);
					}
					
					TPFTestSuite testSuite = null;

					if(test instanceof TPFTestSuite){
						testSuite = ((TPFTestSuite)(test));
					}
					else{
						testSuite = ((TPFTestCase)(test)).getTestSuite();
					}

					//Handle the case where the referenced test suite cannot be resolved (e.g. moved/deleted):
					if(testSuite != null){
					
						String value = testSuite.getType();

						if(value != null){
	
							IAssociationDescriptor descriptor = TestUIExtension.getTestSuiteMappingRegistry().getAssociationMapping(IAssociationConstants.EP_TYPE_DESCRIPTIONS).getDefaultAssociationDescriptor(value);
	
							if((descriptor != null) && (descriptor.getName() != null)){
								value = descriptor.getName();
							}
	
							testTypeText.setText(value);
						}
						else{
							testTypeText.setText(""); //$NON-NLS-1$
						}
	
						testFileText.setText(EMFUtil.getFilePath(testSuite));
					}
					else{
						
						testTypeText.setText(""); //$NON-NLS-1$
						testFileText.setText(""); //$NON-NLS-1$
					}
					
					//Show "no verdict" when there is no actually verdict events in the test log:
					TestLogVerdictTraversalQuery query = (TestLogVerdictTraversalQuery) ((TestLogViewer)formPage.getEditor()).getTestLogVerdictTraversal();

					if (query.getCount(ITestLogVerdictTraversal.VERDICT_TYPE_ALL) != 0){
						verdictText.setText(executionResult.getVerdict().getLabel());
					}
					else{
						verdictText.setText(UiPluginResourceBundle.LogOverview_NoVerdict);
					}

					startTimeText.setText(EventUtil.getTime(EventUtil.getStartTimeStamp(executionResult)));
					stopTimeText.setText(EventUtil.getTime(EventUtil.getStopTimeStamp(executionResult)));	
				}
				else{
					
					modelElementLink.setVisible(false);
					testTypeText.setText(""); //$NON-NLS-1$
					testFileText.setText(""); //$NON-NLS-1$
					verdictText.setText(""); //$NON-NLS-1$
					startTimeText.setText(""); //$NON-NLS-1$
					stopTimeText.setText(""); //$NON-NLS-1$
				}		

				Composite commonSectionClient = ((Composite)(commonPropertiesSection.getClient()));
				commonSectionClient.getParent().layout(true);
				commonSectionClient.getParent().getParent().layout(true);
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.events.IHyperlinkListener#linkActivated(org.eclipse.ui.forms.events.HyperlinkEvent)
	 */
	public void linkActivated(HyperlinkEvent e){
		
		if(e.widget == modelElementLink){
			
			Object data = modelElementLink.getData();
			
			if((data != null) && (data instanceof TPFExecutionResult)){
				
				EObject eObject = ((EObject)(((TPFExecutionResult)(data)).getTest()));
				IEditorPart editorPart = TestUIUtil.openEditor( eObject.eResource(), null, false);
				
				if(editorPart instanceof ISelectionProvider){
					((ISelectionProvider)(editorPart)).setSelection(new StructuredSelection(eObject));
				}
				
				return;
			}
		}
		
		IActionBars actionBars = formPage.getEditorSite().getActionBars();
		IStatusLineManager manager = actionBars.getStatusLineManager();

		if (manager != null){
			manager.setErrorMessage(UiPluginResourceBundle._ERROR_MSG_UN_OPEN_OBJ);
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.events.IHyperlinkListener#linkEntered(org.eclipse.ui.forms.events.HyperlinkEvent)
	 */
	public void linkEntered(HyperlinkEvent e){
		
		IActionBars actionBars = formPage.getEditorSite().getActionBars();
		IStatusLineManager manager = actionBars.getStatusLineManager();
		
		if (manager != null){
			manager.setMessage(modelElementLink.getText());
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.events.IHyperlinkListener#linkExited(org.eclipse.ui.forms.events.HyperlinkEvent)
	 */
	public void linkExited(HyperlinkEvent e){
		
		IActionBars actionBars = formPage.getEditorSite().getActionBars();
		IStatusLineManager manager = actionBars.getStatusLineManager();
		
		if (manager != null){
			
			manager.setMessage(""); //$NON-NLS-1$
			manager.setErrorMessage(""); //$NON-NLS-1$
		}
	}
	
	private void disposeContent(){
		
		if((commonPropertiesSection != null) && (!commonPropertiesSection.isDisposed())){
			
			((Composite)(commonPropertiesSection.getClient())).dispose();
			
			modelElementLink = null;
			testTypeText = null;
			testFileText = null;
			verdictText = null;
			startTimeText = null;
			stopTimeText = null;

			commonPropertiesSection.dispose();
			commonPropertiesSection = null;
		}
	}
}