/********************************************************************** 
 * Copyright (c) 2005, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id$ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.ui.forms.base;

import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.forms.editor.BaseFormEditor;
import org.eclipse.hyades.test.ui.forms.util.FormsUtil;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.swt.SWT;
import org.eclipse.swt.accessibility.AccessibleAdapter;
import org.eclipse.swt.accessibility.AccessibleEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.VerifyEvent;
import org.eclipse.swt.events.VerifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.forms.AbstractFormPart;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormPage;
import org.eclipse.ui.forms.widgets.FormText;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.Section;

/**
 * <p>This is the General Information section of the Overview page in the TPTP Test Log Viewer.</p>
 * 
 * 
 * @author  Bianca Xue Jiang 
 * @author  Paul Slauenwhite
 * @version April 8, 2010
 * @since   October 14, 2005
 */
public class GeneralInfoSectionPart extends AbstractFormPart implements ModifyListener, VerifyListener{
	
	private FormPage formPage;
	private Section section;
	
	private Text nameText;
	private Text descriptionText;
	private FormText fileText;

	public GeneralInfoSectionPart(FormPage page, Composite parent) {
		this.formPage = page;
		initialize(page.getManagedForm());
		createSection(page.getManagedForm(), parent);
	}

	protected void createSection(IManagedForm managedForm, Composite parent){
		
		this.section = FormsUtil.createSection(managedForm, parent, UiPluginResourceBundle.EDT_GENERAL_INFO, UiPluginResourceBundle.LogOverview_OverviewDescription);
		Composite client = ((Composite)(section.getClient()));
		FormToolkit toolkit = managedForm.getToolkit();
		
		//Name:
		toolkit.createLabel(client, UiPluginResourceBundle.LBL_NAME); 
		
		nameText = toolkit.createText(client, "", SWT.FULL_SELECTION);
		nameText.setLayoutData(GridDataUtil.createHorizontalFill());
		nameText.getAccessible().addAccessibleListener(new AccessibleAdapter() {
			
			/* (non-Javadoc)
			 * @see org.eclipse.swt.accessibility.AccessibleAdapter#getName(org.eclipse.swt.accessibility.AccessibleEvent)
			 */
			public void getName(AccessibleEvent e) {
				e.result = UiPluginResourceBundle.W_NAME; 
			}
		});
		
		//Description:
		toolkit.createLabel(client, UiPluginResourceBundle.LBL_DESC); 

		descriptionText = toolkit.createText(client, "", SWT.FULL_SELECTION | SWT.MULTI | SWT.WRAP | SWT.V_SCROLL);
		
		GridData gridData = GridDataUtil.createHorizontalFill();
		gridData.heightHint = (4 * descriptionText.getLineHeight());
		
		descriptionText.setLayoutData(gridData);
		descriptionText.getAccessible().addAccessibleListener(new AccessibleAdapter() {
			
			/* (non-Javadoc)
			 * @see org.eclipse.swt.accessibility.AccessibleAdapter#getName(org.eclipse.swt.accessibility.AccessibleEvent)
			 */
			public void getName(AccessibleEvent e) {
				e.result = UiPluginResourceBundle.W_DESCRIPTION; 
			}
		});
		
		//File:
		GridLayout gridLayout = new GridLayout(2, false);
		gridLayout.marginHeight = 0;
		gridLayout.marginWidth = 0;
		
		Composite fileComposite = toolkit.createComposite(client);
		fileComposite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		fileComposite.setLayout(gridLayout);
		
		toolkit.createLabel(fileComposite, UiPluginResourceBundle.L_FILE); 

		fileText = toolkit.createFormText(fileComposite, false);
		fileText.setLayoutData(GridDataUtil.createHorizontalFill());
		
		toolkit.paintBordersFor(client);

		managedForm.addPart(this);
	}
	
	public Section getSection()
	{
		return this.section;
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.eclipse.ui.forms.IFormPart#setFormInput(java.lang.Object)
	 */
	public boolean setFormInput(Object input) {
		
		if(input instanceof CMNNamedElement){
			
			CMNNamedElement namedElement = ((CMNNamedElement)(input));
			
			nameText.removeVerifyListener(this);
			nameText.removeModifyListener(this);

			nameText.setText(namedElement.getName());

			nameText.addModifyListener(this);
			nameText.addVerifyListener(this);

			descriptionText.removeVerifyListener(this);
			descriptionText.removeModifyListener(this);

			if(namedElement.getDescription() != null){
				descriptionText.setText(namedElement.getDescription());	
			}

			//Note: The modify listener is always added when the form input is set irrespective of the description value. 
			descriptionText.addModifyListener(this);
			descriptionText.addVerifyListener(this);
			
			fileText.setText(EMFUtil.getFilePath(namedElement), false, false);

			//Recalculate the size of file field in case the new file path is longer than the original file path:
			fileText.getParent().layout(true);
			
			return true;
		}
		
		return false;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.VerifyListener#verifyText(org.eclipse.swt.events.VerifyEvent)
	 */
	public void verifyText(VerifyEvent verifyEvent) {

		//Cancel this key event if the editor is opened in read-only mode:
		verifyEvent.doit = (!((BaseFormEditor)(formPage.getEditor())).isReadOnly());
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
	 */
	public void modifyText(ModifyEvent e)
	{	
		CMNNamedElement editorObject = (CMNNamedElement)formPage.getManagedForm().getInput();
		if(e.widget == nameText)
		{
			String name = nameText.getText();
			if(editorObject != null)
				editorObject.setName(name);
			((BaseFormEditor)formPage.getEditor()).setEditorTitle(name);
		}
		else if(e.widget == descriptionText)
		{
			String description = descriptionText.getText();
			if(editorObject != null)
				editorObject.setDescription(description);
		}
		else
			return;

		markDirty();
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.eclipse.ui.forms.IFormPart#refresh()
	 */
	public void refresh() {
		super.refresh();
		setFormInput(formPage.getManagedForm().getInput());
	}
}
