/********************************************************************** 
 * Copyright (c) 2005, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: LogEventsBlock.java,v 1.26 2010/04/07 19:37:41 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.ui.forms.base;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.testprofile.Common_TestprofilePackage;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.forms.actions.CollapseTreeAction;
import org.eclipse.hyades.test.ui.forms.actions.ExpandTreeAction;
import org.eclipse.hyades.test.ui.forms.actions.TestLogSearchAction;
import org.eclipse.hyades.test.ui.forms.actions.VerdictFirstAction;
import org.eclipse.hyades.test.ui.forms.actions.VerdictLastAction;
import org.eclipse.hyades.test.ui.forms.actions.VerdictNavigationAction;
import org.eclipse.hyades.test.ui.forms.actions.VerdictNextAction;
import org.eclipse.hyades.test.ui.forms.actions.VerdictPreviousAction;
import org.eclipse.hyades.test.ui.forms.actions.VerdictTypesPullDownAction;
import org.eclipse.hyades.test.ui.forms.editor.TestLogViewer;
import org.eclipse.hyades.test.ui.forms.extensions.IEventAction;
import org.eclipse.hyades.test.ui.forms.util.FormsUtil;
import org.eclipse.hyades.test.ui.forms.util.ITestLogVerdictTraversal;
import org.eclipse.hyades.test.ui.forms.util.TestLogEventsLazyPathContentProvider;
import org.eclipse.hyades.test.ui.internal.editor.form.base.Partition;
import org.eclipse.hyades.test.ui.internal.editor.form.util.EventLabelProvider;
import org.eclipse.hyades.test.ui.internal.editor.form.util.ExecutionHistoryExtensionsManager;
import org.eclipse.hyades.test.ui.internal.editor.form.util.ExecutionHistoryExtensionsManager.EventAction;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITreeViewerListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeExpansionEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.DetailsPart;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.MasterDetailsBlock;
import org.eclipse.ui.forms.SectionPart;
import org.eclipse.ui.forms.editor.FormPage;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;

/**
 * <p>This is the master details block class used in <i>Events</i> page of 
 * the TPTP Test Log Viewer.</p>
 * 
 * <p>This is an Eclipse forms class derived from the original TPTP Test 
 * Log Viewer.</p>
 * 
 * 
 * @author  Bianca Xue Jiang 
 * @author  Marcelo Paternostro
 * @author  Paul E. Slauenwhite
 * @author 	Blazej Kroll
 * @version April 7, 2010
 * @since   August 9, 2005
 */
public class LogEventsBlock extends MasterDetailsBlock implements ISelectionChangedListener, SelectionListener, IMenuListener
{
	private FormPage formPage;
	private IManagedForm mForm;
	private SectionPart eventsSectionPart;
	private TreeViewer viewer;
	private EventAction[] extensionActions;
	private Button[] extensionButtons;
	private ITestLogVerdictTraversal verdictCollection;
	
	private VerdictTypesPullDownAction verdictTypesAction;		
	private VerdictNavigationAction firstVerdict;		
	private VerdictNavigationAction prevVerdict;		
	private VerdictNavigationAction nextVerdict;		
	private VerdictNavigationAction lastVerdict;
	private CollapseTreeAction collapseAction;
	private ExpandTreeAction expandAction;
	private TestLogSearchAction testLogSearchAction;
	
	/**
	 * Max. number of details pages can be cached.
	 * @see DetailsPart 
	 */
	public static final int DETAILS_PAGE_CACHE_SIZE = 20;
	
	/**
	 * Creates an instance of this class with the editor page.
	 * @param page forms editor page of this block.
	 */
	public LogEventsBlock(FormPage page)
	{
		this.formPage = page;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.MasterDetailsBlock#createMasterPart(org.eclipse.ui.forms.IManagedForm, org.eclipse.swt.widgets.Composite)
	 */
	protected void createMasterPart(IManagedForm managedForm, Composite parent)
	{
		this.mForm = managedForm;
		FormToolkit toolkit = managedForm.getToolkit();
		Section section = FormsUtil.createSection(managedForm, parent, UiPluginResourceBundle.W_EVENTS, "");		  
		Composite client = (Composite)section.getClient();	
		
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.makeColumnsEqualWidth = false;
		client.setLayout(layout);
		
		Tree tree = toolkit.createTree(client, SWT.FULL_SELECTION | SWT.SINGLE | SWT.VIRTUAL);
		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.grabExcessVerticalSpace = true;
		gd.widthHint = 100;
		gd.heightHint = 200;
		tree.setLayoutData(gd);
		
		viewer = new TreeViewer(tree);		
		viewer.setContentProvider(new TestLogEventsLazyPathContentProvider(Common_TestprofilePackage.eINSTANCE.getTPFExecutionHistory_ExecutionEvents()));
		viewer.setLabelProvider(new EventLabelProvider());
		viewer.setUseHashlookup(true);
		viewer.addSelectionChangedListener(this);

		TPFExecutionResult result = (TPFExecutionResult)((TestLogViewer)formPage.getEditor()).getEditorObject();
		
		eventsSectionPart = new SectionPart(section)
		{
			//Bugzilla_152218 refresh input after reload editor object.
			public boolean setFormInput(Object input) {
				if(input instanceof TPFExecutionResult)
					viewer.setInput(((TPFExecutionResult)input).getExecutionHistory());
				
				return true;
			}
		};
		managedForm.addPart(eventsSectionPart);
		
		createActions(result);		
		createContextMenu();
		
		createExtensionActionButtons(toolkit, client);
		toolkit.paintBordersFor(client);
		
		viewer.addTreeListener(new ITreeViewerListener() {

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ITreeViewerListener#treeCollapsed(org.eclipse.jface.viewers.TreeExpansionEvent)
			 */
			public void treeCollapsed(TreeExpansionEvent event){
				//No-operation.
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ITreeViewerListener#treeExpanded(org.eclipse.jface.viewers.TreeExpansionEvent)
			 */
			public void treeExpanded(final TreeExpansionEvent event){	

				final Object expandedObject = event.getElement();

				if (expandedObject instanceof Partition) {

					BusyIndicator.showWhile(PlatformUI.getWorkbench().getDisplay(), new Runnable(){

						public void run() {

							//Collapse all other partition node: 
							Object[] expandedElements = viewer.getExpandedElements();						

							for (int counter = 0; counter < expandedElements.length; counter++){

								if ((expandedElements[counter] instanceof Partition) && (expandedElements[counter] != expandedObject)){
									
									//Note: When parent nodes are collapsed in a virtual tree, the child nodes are removed from tree and need to be recreated.
									viewer.collapseToLevel(expandedElements[counter], 1);
								}
							}		
						}
					});
				}	
			} 
		});

		viewer.addSelectionChangedListener(new ISelectionChangedListener(){

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
			 */
			public void selectionChanged(SelectionChangedEvent event) {	

				final Object selectedObject = ((StructuredSelection)(event.getSelection())).getFirstElement();

				if (selectedObject instanceof Partition){

					BusyIndicator.showWhile(PlatformUI.getWorkbench().getDisplay(), new Runnable(){

						public void run() {

							List children = ((Partition)(selectedObject)).getChildren();

							if (children.size() > 0){

								//Partition node is expanded:
								if(viewer.getExpandedState(selectedObject)){

									//Select the first child element of the selected partition:
									viewer.setSelection(new StructuredSelection(children.get(0)), true);
								} 

								//Partition node is collapsed:
								else {

									//Select the first parent element of the selected partition:
									viewer.setSelection(new StructuredSelection(((EObject)(children.get(0))).eContainer()), true);
								}
							}
						}
					});
				}
			}
		});
	}

	protected void createActions(TPFExecutionResult result)
	{
		//		if (result.eResource() instanceof FacadeResourceImpl) {
		//			verdictCollection = new TestLogVerdictCollection(result);
		//		} else {
		//			verdictCollection = new TestLogVerdictTraversalQuery(result);
		//		}

		verdictCollection = ((TestLogViewer)formPage.getEditor()).getTestLogVerdictTraversal();

		// verdict navigation actions
		verdictTypesAction = new VerdictTypesPullDownAction(verdictCollection, viewer);		
		firstVerdict = new VerdictFirstAction(verdictTypesAction, verdictCollection, viewer);		
		prevVerdict = new VerdictPreviousAction(verdictTypesAction, verdictCollection, viewer);		
		nextVerdict = new VerdictNextAction(verdictTypesAction, verdictCollection, viewer);		
		lastVerdict = new VerdictLastAction(verdictTypesAction, verdictCollection, viewer);
		
		//Create the test log search action:
		testLogSearchAction = new TestLogSearchAction();
		
		// collapse & expand actions
		collapseAction = new CollapseTreeAction(UiPluginResourceBundle.TestLogViewer_Collapse, viewer);
		expandAction = new ExpandTreeAction(UiPluginResourceBundle.TestLogViewer_Expand, viewer);
		collapseAction.setExpandAction(expandAction);
		expandAction.setCollapseAction(collapseAction);
		
		createExtensionActions(result);
	}
	
	protected void createContextMenu()
	{
		MenuManager menuMgr = new MenuManager("#testLogViewerEventsContextMenu");
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(this);
		Menu menu = menuMgr.createContextMenu(viewer.getTree());
		viewer.getTree().setMenu(menu);
		//formPage.getEditorSite().registerContextMenu(menuMgr, viewer);
	}
	
	protected void addContextMenuActions(IMenuManager menuManager)
	{
		/*menuManager.add(firstVerdict);
		menuManager.add(prevVerdict);
		menuManager.add(nextVerdict);
		menuManager.add(lastVerdict);
		menuManager.add(new Separator());*/
		
		//Add the test log search action to the context menu:
		menuManager.add(testLogSearchAction);
		menuManager.add(new Separator());

		menuManager.add(collapseAction);
		menuManager.add(expandAction);
		menuManager.add(new Separator());
	}
	
	public TreeViewer getTreeViewer()
	{
		return viewer;
	}
	
	public void addEventsSelectionChangedListener(ISelectionChangedListener listener)
	{
		if(viewer != null)
			viewer.addSelectionChangedListener(listener);
	}
	
	public void removeEventsSelectionChangedListener(ISelectionChangedListener listener)
	{
		if(viewer != null)
			viewer.remove(listener);
	}

	protected void createExtensionActions(TPFExecutionResult result) 
	{
		String testType = result.getType();
		testType = (testType == null) ? ExecutionHistoryExtensionsManager.ALL_TEST_TYPES : testType;
		extensionActions = ExecutionHistoryExtensionsManager.getInstance().getEventActions(testType);
	}

	protected void createExtensionActionButtons(FormToolkit toolkit, Composite parent) 
	{
		if(extensionActions == null)
		{
			extensionButtons = new Button[0];
			return;
		}
		
		Composite buttonComposite = toolkit.createComposite(parent);
		GridData gd = new GridData();
		gd.verticalAlignment = SWT.FILL;
		gd.grabExcessVerticalSpace = true;
		buttonComposite.setLayoutData(gd);
		GridLayout layout = new GridLayout();
		layout.marginHeight = layout.marginWidth = 0;
		buttonComposite.setLayout(layout);	
		
		List buttons = new ArrayList();
		for(int i = 0; i < extensionActions.length; i++)
		{			
			if(extensionActions[i].isActionButton())
			{
				IEventAction action = extensionActions[i].getAction();
				Button actionButton = toolkit.createButton(buttonComposite, extensionActions[i].getActionText(), SWT.PUSH);
				actionButton.setData(action);
				actionButton.addSelectionListener(this);
				buttons.add(actionButton);
				actionButton.setEnabled(action.isEnabled());
				gd = new GridData(GridData.FILL_HORIZONTAL);
				gd.verticalAlignment = SWT.TOP;
				actionButton.setLayoutData(gd);
			}
			else{

				//Notify the action of selection events from the viewer:
				viewer.addSelectionChangedListener(extensionActions[i].getAction());
			}
		}
		extensionButtons = (Button[])buttons.toArray(new Button[buttons.size()]);
		
		toolkit.paintBordersFor(buttonComposite);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.MasterDetailsBlock#createToolBarActions(org.eclipse.ui.forms.IManagedForm)
	 */
	protected void createToolBarActions(IManagedForm managedForm)
	{
		final ScrolledForm form = managedForm.getForm();
		// add verdict navigation toolbar actions in this order
		form.getToolBarManager().add(firstVerdict);
		form.getToolBarManager().add(prevVerdict);
		form.getToolBarManager().add(verdictTypesAction);
		form.getToolBarManager().add(nextVerdict);
		form.getToolBarManager().add(lastVerdict);
		form.getToolBarManager().add(new Separator());

		//Add the test log search action to the tool bar:
		form.getToolBarManager().add(testLogSearchAction);
		form.getToolBarManager().add(new Separator());

		form.getToolBarManager().add(collapseAction);
		form.getToolBarManager().add(expandAction);
		form.getToolBarManager().add(new Separator());
		
		createOrientationToolBarActions(form);
	}
	
	private void createOrientationToolBarActions(final ScrolledForm form)
	{
		Action haction = new Action(null, Action.AS_RADIO_BUTTON) {
			public void run() {
				sashForm.setOrientation(SWT.HORIZONTAL);
				form.reflow(true);
			}
		};
		haction.setChecked(true);
		haction.setToolTipText(UiPluginResourceBundle.Editor_horizontal_orientation); 
		haction.setImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor("e", TestUIImages.IMG_HORIZONTAL));
		Action vaction = new Action(null, Action.AS_RADIO_BUTTON) {
			public void run() {
				sashForm.setOrientation(SWT.VERTICAL);
				form.reflow(true);
			}
		};
		vaction.setChecked(false);
		vaction.setToolTipText(UiPluginResourceBundle.Editor_vertical_orientation); 
		vaction.setImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor("e", TestUIImages.IMG_VERTICAL));
		form.getToolBarManager().add(haction);
		form.getToolBarManager().add(vaction);
	}

	/*protected TestLogVerdictCollection getVerdictCollection()
	{
		return this.verdictCollection;
	}*/

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.MasterDetailsBlock#registerPages(org.eclipse.ui.forms.DetailsPart)
	 */
	protected void registerPages(DetailsPart detailsPart)
	{
		detailsPart.setPageLimit(DETAILS_PAGE_CACHE_SIZE);
		detailsPart.setPageProvider(new ExecutionEventDetailsProvider(formPage));
	}
	
	public void selectionChanged(SelectionChangedEvent event) {
		ISelection selection = event.getSelection();
		mForm.fireSelectionChanged(eventsSectionPart, selection);
		
		for(int i = 0; i < extensionButtons.length; i++)
		{
			IEventAction action = (IEventAction)extensionButtons[i].getData();
			if(action != null)				
			{
				// the order is important: notify the action of selection changed first for it to update its status.
				action.selectionChanged(event);
				extensionButtons[i].setEnabled(action.isEnabled());
			}
		}
	}
	
	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}
	
	public void widgetSelected(SelectionEvent e) {
		if(e.widget instanceof Button)
		{
			Object action = e.widget.getData();
			if(action instanceof IAction)
				((IAction)action).run();
		}
	}
	
	public void menuAboutToShow(IMenuManager manager) 
	{		
		if(extensionActions != null)
		{
			for(int i = 0; i < extensionActions.length; i++)
			{
				if(extensionActions[i].isActionContextMenu())
					extensionActions[i].getAction().menuAboutToShow(manager);
			}
		}

		addContextMenuActions(manager);
	}
}
	
