/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.test.ui.forms.base;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.loaders.util.HyadesResourceExtensions;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.common.testprofile.Common_TestprofilePackage;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFVerdict;
import org.eclipse.hyades.models.common.testprofile.TPFVerdictEvent;
import org.eclipse.hyades.models.hierarchy.extensions.BinaryExpression;
import org.eclipse.hyades.models.hierarchy.extensions.ExtensionsFactory;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.QueryResult;
import org.eclipse.hyades.models.hierarchy.extensions.RelationalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.ResultEntry;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleOperand;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.hierarchy.util.IHyadesResourceExtension;
import org.eclipse.hyades.test.ui.forms.util.ITestLogVerdictTraversal;

/**
 * TestLogVerdictTraversalQuery.java
 * 
 * 
 * @author      Jospeh P. Toomey
 * @author      Paul E. Slauenwhite
 * @version     July 24, 2008
 * @since       April 24, 2007
 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
 */
public class TestLogVerdictTraversalQuery implements ITestLogVerdictTraversal {
	
	private TPFExecutionResult result = null;
	private QueryResult[] verdictQueries = null;
	private int[] verdictCursors = null;

	public TestLogVerdictTraversalQuery(TPFExecutionResult result) {
		init(result);	
	}
	
	public void init(TPFExecutionResult result) {
		
		this.result = result;
		
		verdictQueries = new QueryResult[5];
		verdictQueries[TPFVerdict.PASS] = createVerdictQueryResult(result, TPFVerdict.PASS_LITERAL); 
		verdictQueries[TPFVerdict.FAIL] = createVerdictQueryResult(result, TPFVerdict.FAIL_LITERAL); 
		verdictQueries[TPFVerdict.ERROR] = createVerdictQueryResult(result, TPFVerdict.ERROR_LITERAL); 
		verdictQueries[TPFVerdict.INCONCLUSIVE] = createVerdictQueryResult(result, TPFVerdict.INCONCLUSIVE_LITERAL); 
		verdictQueries[ITestLogVerdictTraversal.VERDICT_TYPE_ALL] = createVerdictQueryResult(result, null);
		
		verdictCursors = new int[5];

		resetCursors();
	}

	private QueryResult createVerdictQueryResult(TPFExecutionResult result, TPFVerdict verdict) {
		SimpleSearchQuery verdictQuery = ExtensionsFactory.eINSTANCE.createSimpleSearchQuery();
		verdictQuery.getSources().add(EcoreUtil.getURI(result).toString());

		// Minimum paths required to traverse the complete execution history.
		verdictQuery.getRequiredPaths().add(Common_TestprofilePackage.eINSTANCE.getTPFExecutionResult_ExecutionHistory());
		verdictQuery.getRequiredPaths().add(Common_TestprofilePackage.eINSTANCE.getTPFExecutionHistory_ExecutionEvents());
		verdictQuery.getRequiredPaths().add(Common_TestprofilePackage.eINSTANCE.getTPFExecutionEvent_Children());
		verdictQuery.getRequiredPaths().add(Common_TestprofilePackage.eINSTANCE.getTPFInvocationEvent_InvokedExecutionResult());

		SimpleOperand Operand = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		Operand.setType(Common_TestprofilePackage.eINSTANCE.getTPFVerdictEvent());
		verdictQuery.getOutputElements().add(Operand);
		
		LogicalExpression whereExpression = createORExpression();

		if (verdict != null) {
			BinaryExpression verdictTypeFilter = createTPFVerdictFilter(verdict);		
			whereExpression.getArguments().add(verdictTypeFilter);
		}
				
		verdictQuery.setWhereExpression(whereExpression);
		
		QueryResult queryResult = null;
		IHyadesResourceExtension hyadesResourceExtension = (IHyadesResourceExtension) HyadesResourceExtensions.getInstance().get(LoadersUtils.getPostfix(result.eResource().getURI().toString()));
		if (hyadesResourceExtension != null) {
			List notLoadedTypes = new ArrayList();
			ResourceSet resourceSet = result.eResource().getResourceSet();
			queryResult = hyadesResourceExtension.executeQuery(verdictQuery, resourceSet, notLoadedTypes);
		}
		return queryResult;
	}

	protected BinaryExpression createTPFVerdictFilter(TPFVerdict verdict) {
		return createBinaryExpression(Common_TestprofilePackage.eINSTANCE.getTPFVerdictEvent_Verdict(), Common_TestprofilePackage.eINSTANCE.getTPFVerdictEvent_Verdict().getEAttributeType(), verdict);		
	}
	
	protected BinaryExpression createBinaryExpression(EAttribute feature, EDataType valueType, Object value) {
		SimpleOperand operand;
		BinaryExpression expression;
		expression = ExtensionsFactory.eINSTANCE.createBinaryExpression();
		operand = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		operand.setFeature(feature);
		expression.setLeftOperand(operand);				
		operand = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		operand.setValueType(valueType);
		operand.setValue(value);
		expression.getRightOperands().add(operand);
		expression.setOperator(RelationalOperators.LIKE_LITERAL);
		return expression;
	}
	
	protected LogicalExpression createORExpression()
	{
		LogicalExpression or = ExtensionsFactory.eINSTANCE.createLogicalExpression();
		or.setOperator(LogicalOperators.OR_LITERAL);
		return or;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.forms.util.ITestLogVerdictTraversal#contains(int)
	 */
	public boolean contains(int verdictType) {
		return (getCount(verdictType) > 0);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.forms.util.ITestLogVerdictTraversal#getExecutionResult()
	 */
	public TPFExecutionResult getExecutionResult() {
		return result;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.forms.util.ITestLogVerdictTraversal#getFirst(int)
	 */
	public TPFVerdictEvent getFirst(int verdictType) {
		List result = verdictQueries[verdictType].getResultEntries();
		if (!result.isEmpty()) {
			List resultSet = (List) ((ResultEntry) result.get(0)).getValue();
			if (!resultSet.isEmpty()) {
				verdictCursors[verdictType] = 0;
				return (TPFVerdictEvent) resultSet.get(verdictCursors[verdictType]);
			}
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.forms.util.ITestLogVerdictTraversal#getLast(int)
	 */
	public TPFVerdictEvent getLast(int verdictType) {
		List result = verdictQueries[verdictType].getResultEntries();
		if (!result.isEmpty()) {
			List resultSet = (List) ((ResultEntry) result.get(0)).getValue();
			if (!resultSet.isEmpty()) {
				verdictCursors[verdictType] = resultSet.size()-1;
				return (TPFVerdictEvent) resultSet.get(verdictCursors[verdictType]);
			}
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.forms.util.ITestLogVerdictTraversal#getNext(int, java.lang.Object)
	 */
	public TPFVerdictEvent getNext(int verdictType, Object current) {
		List result = verdictQueries[verdictType].getResultEntries();
		if (!result.isEmpty()) {
			List resultSet = (List) ((ResultEntry) result.get(0)).getValue();
			if (!resultSet.isEmpty()) {
				
				if(current != null){
					
					int currentIndex = resultSet.indexOf(current);
					
					if(currentIndex != -1){

						if (currentIndex == (resultSet.size() - 1)){
							currentIndex = 0;
						}
						else{
							currentIndex++;
						}
						
						verdictCursors[verdictType] = currentIndex;
						
						return ((TPFVerdictEvent)(resultSet.get(currentIndex)));
					}
				}
				else{

					if (verdictCursors[verdictType] == (resultSet.size() - 1)){
						verdictCursors[verdictType] = 0;
					}
					else{
						verdictCursors[verdictType]++;
					}
					
					return ((TPFVerdictEvent)(resultSet.get(verdictCursors[verdictType])));
				}
			}
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.forms.util.ITestLogVerdictTraversal#getPrevious(int, java.lang.Object)
	 */
	public TPFVerdictEvent getPrevious(int verdictType, Object current) {
		List result = verdictQueries[verdictType].getResultEntries();
		if (!result.isEmpty()) {
			List resultSet = (List) ((ResultEntry) result.get(0)).getValue();
			if (!resultSet.isEmpty()) {
				
				if(current != null){
					
					int currentIndex = resultSet.indexOf(current);
					
					if(currentIndex != -1){

						if (currentIndex == 0){
							currentIndex = (resultSet.size() - 1);
						}
						else{ 
							currentIndex--;
						}
											
						verdictCursors[verdictType] = currentIndex;

						return ((TPFVerdictEvent)(resultSet.get(currentIndex)));						
					}
				}
				else{

					//Account for the initialized cursor value (-1):
					if ((verdictCursors[verdictType] == -1) || (verdictCursors[verdictType] == 0)){
						verdictCursors[verdictType] = (resultSet.size() - 1);
					}
					else{ 
						verdictCursors[verdictType]--;
					}
										
					return ((TPFVerdictEvent)(resultSet.get(verdictCursors[verdictType])));
				}
			}
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.forms.util.ITestLogVerdictTraversal#hasPrevious(int, java.lang.Object)
	 */
	public boolean hasPrevious(int verdictType, Object current)
	{
		List result = verdictQueries[verdictType].getResultEntries();
		if (!result.isEmpty()) {
			List resultSet = (List) ((ResultEntry) result.get(0)).getValue();
			if (!resultSet.isEmpty()) {
				
				if(current != null){

					int currentIndex = resultSet.indexOf(current);
					
					return ((currentIndex != -1) && (currentIndex != 0));										
				}
				else{
				
   				    //Account for the initialized cursor value (-1):
					return ((verdictCursors[verdictType] != -1) && (verdictCursors[verdictType] != 0));					
				}
			}
		}
		return false;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.forms.util.ITestLogVerdictTraversal#hasNext(int, java.lang.Object)
	 */
	public boolean hasNext(int verdictType, Object current)
	{
		List result = verdictQueries[verdictType].getResultEntries();
		if (!result.isEmpty()) {
			List resultSet = (List) ((ResultEntry) result.get(0)).getValue();
			if (!resultSet.isEmpty()) {

				if(current != null){
					
					int currentIndex = resultSet.indexOf(current);
					
					return ((currentIndex != -1) && (currentIndex != (resultSet.size() - 1)));										
				}
				else{
					return (verdictCursors[verdictType] != (resultSet.size() - 1));					
				}
			}
		}
		return false;
	}

	public int getCount(int verdictType) {
		List result = verdictQueries[verdictType].getResultEntries();
		if (!result.isEmpty()) {
			List resultSet = (List) ((ResultEntry) result.get(0)).getValue();
			if (!resultSet.isEmpty()) {
				return resultSet.size();
			}
		}
		return 0;
	}
	
	public List getQueryResults(int verdictType) {
		List result = verdictQueries[verdictType].getResultEntries();
		if (!result.isEmpty()) {
			return (List) ((ResultEntry) result.get(0)).getValue();
		}
		return new ArrayList();
	}
	
	public void resetCursors() {
		for (int i=0; i< verdictCursors.length; i++) {
			verdictCursors[i] = -1;
		}
	}
}
