/********************************************************************** 
 * Copyright (c) 2005, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: VerdictEventDetailsPart.java,v 1.27 2009/05/08 18:51:57 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.ui.forms.base;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.hyades.log.ui.internal.navigator.EMFUtil;
import org.eclipse.hyades.models.common.testprofile.TPFVerdictEvent;
import org.eclipse.hyades.test.core.util.JavaUtil;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jface.util.IOpenEventListener;
import org.eclipse.jface.util.OpenStrategy;
import org.eclipse.jface.window.Window;
import org.eclipse.osgi.util.TextProcessor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.forms.editor.FormPage;
import org.eclipse.ui.forms.widgets.FormToolkit;

/**
 * <p>This is the details page of the page book used for 
 * {@link org.eclipse.hyades.models.common.testprofile.TPFVerdictEvent TPFVerdictEvent}
 * in the events tree of the TPTP Test Log Viewer.</p>
 * 
 * <p>This is an Eclipse forms page derived from the original TPTP Test Log Viewer.</p>
 * 
 * 
 * @author  Bianca Xue Jiang
 * @author  Marcelo Paternostro
 * @author  Paul E. Slauenwhite
 * @version May 8, 2009
 * @since   August 9, 2005
 */
public class VerdictEventDetailsPart extends ExecutionEventDetailsPart implements IOpenEventListener
{
	private Text verdictText;
	//private Text reasonText;
	private TPFVerdictEvent verdictEvent;
	private String stackTraceAsText;

	private final static String TRACE_LINE_PATTERN = "at "; //$NON-NLS-1$
	private final static String STACK_FRAME_PATTERN_WIN = "\r\n\t" + TRACE_LINE_PATTERN; //$NON-NLS-1$
	private final static String STACK_FRAME_PATTERN_LINUX = "\n\t" + TRACE_LINE_PATTERN; //$NON-NLS-1$
	private static final String JUNIT_FRAMEWORK_STACK_PATTERN = "junit.framework."; //$NON-NLS-1$
	
	public VerdictEventDetailsPart()
	{
		super();
	}

	/**
	 * Creates an instance of this class with the editor page.
	 * @param page forms editor page of this part.
	 */
	public VerdictEventDetailsPart(FormPage page)
	{
		super(page);
	}

	protected void createDetailsContents(final Composite parent)
	{
		FormToolkit toolkit = getManagedForm().getToolkit();
		toolkit.createLabel(parent, UiPluginResourceBundle.LBL_VERD); 
		verdictText = toolkit.createText(parent, "", SWT.FULL_SELECTION | SWT.SINGLE); //$NON-NLS-1$
		//verdictText.setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB));
		verdictText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		verdictText.setEditable(false);

		toolkit.paintBordersFor(parent);
	}
	
	protected void setInput(Object object)
	{
		super.setInput(object);
		
		if(object instanceof TPFVerdictEvent)
		{
			verdictEvent = (TPFVerdictEvent)object;
				
			if(verdictEvent.getVerdict() != null)
				verdictText.setText(verdictEvent.getVerdict().getLabel());
			else
				verdictText.setText(""); //$NON-NLS-1$

			/*if(verdictEvent.getReason() != null)
				reasonText.setText(verdictEvent.getReason().getLabel());
			else
				reasonText.setText(""); //$NON-NLS-1$
*/			
		}
		else
		{
			verdictText.setText(""); //$NON-NLS-1$
			verdictText.setEditable(false);
			//reasonText.setText(""); //$NON-NLS-1$
			//reasonText.setEditable(false);
		}
	}
	
	protected void setMessageText(String text) {
		if(isStackTrace(text))
		{
			if(!(messageText instanceof Table))
			{
				if(messageText != null)
				{
					 messageText.dispose();
					 messageText = null;
				}
				Composite parent = (Composite)commonPropSection.getClient();
				messageText = createStackTraceControl(parent, getManagedForm().getToolkit());
				getManagedForm().getToolkit().paintBordersFor(parent);
			}
			setStackTraceInput(text);
		}
		else
			super.setMessageText(text);
	}
	
	private boolean isStackTrace(String text)
	{
		// bugzilla https://bugs.eclipse.org/bugs/show_bug.cgi?id=133129
		// acknowlege test log generated on either systems.
		if(text != null && (text.indexOf(STACK_FRAME_PATTERN_WIN) > -1 || text.indexOf(STACK_FRAME_PATTERN_LINUX) > -1))
			return true;
		return false;
	}
	
	private Control createStackTraceControl(Composite parent, FormToolkit toolkit) {
		final Table stackTrace = toolkit.createTable(parent, SWT.FULL_SELECTION | SWT.SINGLE);
		GridData data = GridDataUtil.createHorizontalFill();
		data.heightHint = stackTrace.getItemHeight() * 10;
		stackTrace.setLayoutData(data);
		OpenStrategy handler = new OpenStrategy(stackTrace);
		handler.addOpenListener(this);
		toolkit.paintBordersFor(parent);
		createStackTracePopupMenu(stackTrace);
		return stackTrace;
	}

	
	private void createStackTracePopupMenu(Table stackTrace) {
		Menu popUpMenu = new Menu(stackTrace.getShell(), SWT.POP_UP);
		stackTrace.setMenu(popUpMenu);
		MenuItem copyItem = new MenuItem(popUpMenu, SWT.PUSH);
		copyItem.setText(UiPluginResourceBundle.TestLogViewer_CopyStackTrace); 
		copyItem.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				Clipboard clipboard = new Clipboard(getFormPage().getSite().getShell().getDisplay());
				try {
					clipboard.setContents(new Object[] { stackTraceAsText }, new Transfer[] { TextTransfer.getInstance() });
				} finally {
					clipboard.dispose();
				}
			}
		});
	}

	private void setStackTraceInput(String trace) {
		stackTraceAsText = trace;
		if(!(messageText instanceof Table) || !isStackTrace(trace))
			return;
		
		Table stackTrace = (Table)messageText; 
		stackTrace.removeAll();
		
		// bugzilla https://bugs.eclipse.org/bugs/show_bug.cgi?id=133129
		int end = (trace.indexOf(STACK_FRAME_PATTERN_WIN) == -1) ? 
					trace.indexOf(STACK_FRAME_PATTERN_LINUX) : trace.indexOf(STACK_FRAME_PATTERN_WIN);
		if (end > -1) {
			addStackItem(stackTrace, trace.substring(0, end), false);
		}
		boolean junitFrameworkStack = true;
		while (end != -1) {
			int start = end;
			end = (trace.indexOf(STACK_FRAME_PATTERN_WIN) == -1) ? 
					trace.indexOf(STACK_FRAME_PATTERN_LINUX, start + STACK_FRAME_PATTERN_LINUX.length()) : 
					trace.indexOf(STACK_FRAME_PATTERN_WIN, start + STACK_FRAME_PATTERN_WIN.length());
			String line;
			if (end == -1) {
				line = trace.substring(start);
			} else {
				line = trace.substring(start, end);
			}
			// bugzilla_142483 [Usability] JUnit assertion fail stack trace should omit JUnit internal stack
			if (junitFrameworkStack && line.indexOf(JUNIT_FRAMEWORK_STACK_PATTERN) == -1) {
				junitFrameworkStack = false;
			}
			if (!junitFrameworkStack) {
				addStackItem(stackTrace, line, true);
			}
		}
	}
	
	private void addStackItem(Table stackTrace, String text, boolean isException) {
		
		TableItem item = new TableItem(stackTrace, SWT.NONE);
		
		//Replace any tab, new line, or carriage return characters with space characters since 
		//the text is displayed on one line and these characters are not rendered (e.g. boxes):
		String normalizedText = text.replace('\t', ' ').replace('\n', ' ').replace('\r', ' ').trim();
		
		//Process the stack trace line to change the orientation of the stack trace 
		//segments to LTR (left to right) so the stack trace line is rendered correctly 
		//in RTL (right to left) environments:
		if(Window.getDefaultOrientation() == SWT.RIGHT_TO_LEFT){
			
			//Note: Processing does not alter the stack trace line unless current locale 
			//is a Bidi locale, despite running in a RTL (right to left) environment: 
			item.setText(TextProcessor.process(normalizedText, " .():$")); //$NON-NLS-1$
		}
		else{
			item.setText(normalizedText);			
		}

		if (isException) {
			item.setImage(TestUIImages.INSTANCE.getImage(TestUIImages.IMG_STACK_FRAME));
		} else {
			item.setImage(TestUIImages.INSTANCE.getImage(TestUIImages.IMG_CATCH_CLAUSE));
		}
	}		

	private void openStackFrame(String traceLine) {
		try { 
			String testName= traceLine;
			testName= testName.substring(testName.indexOf(TRACE_LINE_PATTERN));
			testName= testName.substring(TRACE_LINE_PATTERN.length(), testName.lastIndexOf('(')).trim();
			testName= testName.substring(0, testName.lastIndexOf('.'));
			int innerSeparatorIndex= testName.indexOf('$');
			if (innerSeparatorIndex != -1)
				testName= testName.substring(0, innerSeparatorIndex);
			
			String lineNumber= traceLine;
			lineNumber= lineNumber.substring(lineNumber.indexOf(':') + 1, lineNumber.lastIndexOf(')'));
			int line= Integer.valueOf(lineNumber).intValue();
			String cuName= traceLine.substring(traceLine.lastIndexOf('(') + 1, traceLine.lastIndexOf(':'));
			openCompilationUnitAtLine(cuName, testName, line);
		} catch(NumberFormatException e) {
			// Problem in parsing line: don't do anything
		}
		catch(IndexOutOfBoundsException e) {	
			// Ditto
		}
		catch(CoreException e) {
			// Problem when opening the editor
			UiPlugin.logError(e);
		}
	}
	
	private void openCompilationUnitAtLine(String cuName, String className, int line) throws CoreException {
		IJavaElement javaElement = JavaUtil.findElement(getJavaProject(), cuName, className);
		if (javaElement != null) {
			org.eclipse.hyades.ui.internal.util.JavaUtil.revealJavaElementAtLine(javaElement, line);
		}
	}

	private IJavaProject getJavaProject() {
		if (verdictEvent != null) {
			IFile file = EMFUtil.getWorkspaceFile(verdictEvent);
			if (file != null) {
				return JavaCore.create(file.getProject());
			}
		}
		return null;
	}
	
	public void handleOpen(SelectionEvent e) {
		if(!(messageText instanceof Table))
			return;
		TableItem[] selection = ((Table)messageText).getSelection();
		if (selection.length > 0) {
			openStackFrame(selection[0].getText());
		}
	}
}
