/*******************************************************************************
 * Copyright (c) 2006, 2009 IBM Corporation and others. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse
 * Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: VerdictSummaryChart.java,v 1.18 2009/08/11 20:11:11 paules Exp $
 * 
 * Contributors: IBM - Initial API and implementation
 ******************************************************************************/
package org.eclipse.hyades.test.ui.forms.base;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.birt.chart.computation.DataPointHints;
import org.eclipse.birt.chart.device.ICallBackNotifier;
import org.eclipse.birt.chart.device.IDeviceRenderer;
import org.eclipse.birt.chart.event.StructureSource;
import org.eclipse.birt.chart.event.StructureType;
import org.eclipse.birt.chart.exception.ChartException;
import org.eclipse.birt.chart.factory.GeneratedChartState;
import org.eclipse.birt.chart.factory.Generator;
import org.eclipse.birt.chart.factory.RunTimeContext;
import org.eclipse.birt.chart.model.Chart;
import org.eclipse.birt.chart.model.ChartWithoutAxes;
import org.eclipse.birt.chart.model.attribute.ActionType;
import org.eclipse.birt.chart.model.attribute.Bounds;
import org.eclipse.birt.chart.model.attribute.CallBackValue;
import org.eclipse.birt.chart.model.attribute.ChartDimension;
import org.eclipse.birt.chart.model.attribute.DataPointComponent;
import org.eclipse.birt.chart.model.attribute.DataPointComponentType;
import org.eclipse.birt.chart.model.attribute.LegendItemType;
import org.eclipse.birt.chart.model.attribute.TooltipValue;
import org.eclipse.birt.chart.model.attribute.TriggerCondition;
import org.eclipse.birt.chart.model.attribute.impl.BoundsImpl;
import org.eclipse.birt.chart.model.attribute.impl.CallBackValueImpl;
import org.eclipse.birt.chart.model.attribute.impl.ColorDefinitionImpl;
import org.eclipse.birt.chart.model.attribute.impl.DataPointComponentImpl;
import org.eclipse.birt.chart.model.attribute.impl.JavaNumberFormatSpecifierImpl;
import org.eclipse.birt.chart.model.attribute.impl.TooltipValueImpl;
import org.eclipse.birt.chart.model.component.Series;
import org.eclipse.birt.chart.model.component.impl.SeriesImpl;
import org.eclipse.birt.chart.model.data.Action;
import org.eclipse.birt.chart.model.data.NumberDataSet;
import org.eclipse.birt.chart.model.data.SeriesDefinition;
import org.eclipse.birt.chart.model.data.TextDataSet;
import org.eclipse.birt.chart.model.data.Trigger;
import org.eclipse.birt.chart.model.data.impl.ActionImpl;
import org.eclipse.birt.chart.model.data.impl.NumberDataSetImpl;
import org.eclipse.birt.chart.model.data.impl.SeriesDefinitionImpl;
import org.eclipse.birt.chart.model.data.impl.TextDataSetImpl;
import org.eclipse.birt.chart.model.data.impl.TriggerImpl;
import org.eclipse.birt.chart.model.impl.ChartWithoutAxesImpl;
import org.eclipse.birt.chart.model.layout.Legend;
import org.eclipse.birt.chart.model.layout.Plot;
import org.eclipse.birt.chart.model.type.PieSeries;
import org.eclipse.birt.chart.model.type.impl.PieSeriesImpl;
import org.eclipse.birt.chart.render.IActionRenderer;
import org.eclipse.birt.chart.util.PluginSettings;
import org.eclipse.core.runtime.ListenerList;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFVerdictList;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.forms.extensions.provisional.DefaultVerdictCategoryProvider;
import org.eclipse.hyades.test.ui.forms.extensions.provisional.IVerdictCategoryProvider;
import org.eclipse.hyades.test.ui.forms.extensions.provisional.VerdictCategory;
import org.eclipse.hyades.test.ui.forms.util.ITestLogVerdictTraversal;
import org.eclipse.hyades.test.ui.internal.editor.form.util.ExecutionHistoryExtensionsManager;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.ui.forms.widgets.FormText;
import org.eclipse.ui.forms.widgets.Section;

/**
 * Verdict summary pie displayed in the Overview tab of the Test Log Viewer.
 * 
 * 
 * @author  Bianca Xue Jiang
 * @author  Paul E. Slauenwhite
 * @version August 11, 2009
 * @since   March 15, 2006
 */
public class VerdictSummaryChart extends Canvas implements PaintListener, ICallBackNotifier, IActionRenderer, ISelectionProvider {
	
	private IDeviceRenderer dr = null;
	private RunTimeContext runtimeContext;

	private Chart cm = null;
	
	private GeneratedChartState chartState = null;
	
	private VerdictCategory[] verdictCategories;
	private ListenerList selectionListeners = new ListenerList();
	private IStructuredSelection currentSelection;
	
	private Section chartSection;
	
	public VerdictSummaryChart(Composite parent, int style)
	{
		super(parent, style);
		addPaintListener(this);
		final PluginSettings ps = PluginSettings.instance();
		try {
			dr = ps.getDevice("dv.SWT");//$NON-NLS-1$
			dr.setProperty(IDeviceRenderer.UPDATE_NOTIFIER, this);
		} catch (ChartException pex) {
			UiPlugin.logError(pex);
		}

		runtimeContext = new RunTimeContext( );
		runtimeContext.setActionRenderer(this);
		
		//Handle tab group traversal:
		addListener(SWT.Traverse, new Listener() {

			public void handleEvent(Event event) {

				switch (event.detail) {
				case SWT.TRAVERSE_ESCAPE:
				case SWT.TRAVERSE_RETURN:
				case SWT.TRAVERSE_TAB_NEXT:
				case SWT.TRAVERSE_TAB_PREVIOUS:
				case SWT.TRAVERSE_PAGE_NEXT:
				case SWT.TRAVERSE_PAGE_PREVIOUS:
					event.doit = true;
					break;
				}
			}
		});
	}
	
	public void setChartSection(Section section)
	{
		this.chartSection = section;
	}
	
	/**
	 * Returns <code>false</code> if there is no verdict in this input, <b>true</b> if chart is drawn successfully.
	 * @param input
	 * @return
	 */
	public boolean setInput(TPFExecutionResult input)
	{
		String testType = input.getType();
		VerdictCategory[] categories= null;
		IVerdictCategoryProvider provider = ExecutionHistoryExtensionsManager.getInstance().getVerdictProvider(testType);
		if(provider != null)
		{
			if(provider instanceof DefaultVerdictCategoryProvider) 
			{
				categories = ((DefaultVerdictCategoryProvider)provider).getVerdictCategories(input);
			}
			else
			{
				List verLists = input.getVerdictLists();

				//Do not display the verdict chart when no verdicts are in the test log (e.g. 'no verdict') or only a consolidated verdict exists:
				if((verLists == null) || (verLists.isEmpty()) || (verLists.size() == 1)){
					
					((FormText)chartSection.getDescriptionControl()).setText(UiPluginResourceBundle.LogOverview_NoVerdictInLog, false, false);
					
					return false;
				}
	
				TPFVerdictList list = null;
				for(Iterator it = verLists.iterator(); it.hasNext();)
				{
					list = (TPFVerdictList)it.next();
					if(TPFVerdictList.VERDICT_TYPE_ALL.equals(list.getType()))
						break;
					else
						list = null;
				}
				if(list == null)
					return false;
				
				List allVerdicts = list.getVerdictEvents();
				categories = provider.getVerdictCategories(allVerdicts);
			}
		}
		
		if(categories == null)
		{
			provider = new DefaultVerdictCategoryProvider();
			categories = ((DefaultVerdictCategoryProvider)provider).getVerdictCategories(input);
		}
		
		//this.input = input;
		cm = createPieChart(categories);
		if(cm == null)
			return false;
		
		cm.setDimension(ChartDimension.TWO_DIMENSIONAL_WITH_DEPTH_LITERAL);		
		//initial selection
		setSelection(new StructuredSelection(provider.getDefaultCategory(categories)));
		return true;
	}

	/**
	 * Returns <code>false</code> if there is no verdict in this input, <b>true</b> if chart is drawn successfully.
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	public boolean setInput(TPFExecutionResult input, TestLogVerdictTraversalQuery verdictTraversal)
	{
		String testType = input.getType();
		VerdictCategory[] categories= null;
		IVerdictCategoryProvider provider = ExecutionHistoryExtensionsManager.getInstance().getVerdictProvider(testType);
		try {
			if(provider != null)
			{
				int verdictCount = verdictTraversal.getCount(ITestLogVerdictTraversal.VERDICT_TYPE_ALL);
				
				//Do not display the verdict chart when no verdicts are in the test log (e.g. 'no verdict') or only a consolidated verdict exists:
				if (verdictCount < 2){
					
					((FormText)chartSection.getDescriptionControl()).setText(UiPluginResourceBundle.LogOverview_NoVerdictInLog, false, false);
					
					return false;
				}
				
				try {
					categories = provider.getVerdictCategories(verdictTraversal);
				} catch (RuntimeException e) {
					// Deliberately fall through here and retry with the 
					// default category provider
				}
			}
			
			if(categories == null)
			{
				provider = new DefaultVerdictCategoryProvider();
				categories = ((DefaultVerdictCategoryProvider)provider).getVerdictCategories(verdictTraversal);
			}
		} catch (UnsupportedOperationException e) {
			// If we can't find the correct verdict categories, we won't create
			// the pie chart.
			return false;
		}
		
		//this.input = input;
		cm = createPieChart(categories);
		if(cm == null)
			return false;
		
		cm.setDimension(ChartDimension.TWO_DIMENSIONAL_WITH_DEPTH_LITERAL);		
		//initial selection
		setSelection(new StructuredSelection(provider.getDefaultCategory(categories)));
		return true;
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.eclipse.swt.events.PaintListener#paintControl(org.eclipse.swt.events.PaintEvent)
	 */
	public final void paintControl(PaintEvent pe) {
		if(cm == null)
			return;
		
		dr.setProperty(IDeviceRenderer.GRAPHICS_CONTEXT, pe.gc);

		Composite co = (Composite) pe.getSource();
		Rectangle re = co.getClientArea();
		Bounds bo = BoundsImpl.create(re.x, re.y, re.width, re.height);
		bo.scale(72d / dr.getDisplayServer().getDpiResolution());
		Generator gr = Generator.instance();

		try {
			chartState = gr.build(dr.getDisplayServer( ), cm, bo, null, runtimeContext, new SystemStyleProcessor());
		}
		catch(ChartException ce) {
			ce.printStackTrace();
		}
		
		try { 
			gr.render(dr, chartState);
		} catch (ChartException ce) {
			UiPlugin.logError(ce);
		}
	}

	private final Chart createPieChart(VerdictCategory[] vCategories) {
		
		if((vCategories != null) && (vCategories.length > 0)){
		
			//Remove the verdict categories with no verdicts so the pie chart does not displays 0.0% labels and their legend category:
			List verdictCategoriesList = new ArrayList();
			
			for (int counter = 0; counter < vCategories.length; counter++) {
				
				if(vCategories[counter].getNumVerdicts() != 0){
					verdictCategoriesList.add(vCategories[counter]);
				}
			}
			
			this.verdictCategories = ((VerdictCategory[])(verdictCategoriesList.toArray(new VerdictCategory[verdictCategoriesList.size()])));
			
			ChartWithoutAxes pie = ChartWithoutAxesImpl.create();
			pie.getBlock().setBackground(null);
			
			//Plot		
			Plot p = pie.getPlot();
			p.getClientArea().setBackground(null);
			p.getClientArea().getOutline().setVisible(false);
			p.getOutline().setVisible(false);
	
			//Legend
			Legend lg = pie.getLegend();
			lg.setItemType(LegendItemType.CATEGORIES_LITERAL);
			lg.getClientArea().getOutline().setVisible(true);
			lg.getTitle().setVisible(false);

			//Increase the top inset to account for the height of the colored box:
			lg.getClientArea().getInsets().set(6.0, 4.0, 4.0, 4.0);
	
			//Title
			pie.getTitle().setVisible(false);
	
			//Data Set
			int length = verdictCategories.length;
			String[] texts = new String[length];
			double[] values = new double[length];
			List fills = new ArrayList();
			
			double totalSize = 0.0;
			for(int i = 0; i < length; i++)
				totalSize += verdictCategories[i].getNumVerdicts();
			
			double size;
			RGB color;
			for(int i = 0; i < length; i++)
			{
				texts[i] = verdictCategories[i].getText();
				color = verdictCategories[i].getColor();
				fills.add(ColorDefinitionImpl.create(color.red, color.green, color.blue));
				size = verdictCategories[i].getNumVerdicts();
				values[i] = size / totalSize * 100;
			}
	
			//Base Series
			SeriesDefinition sd = SeriesDefinitionImpl.create();
			pie.getSeriesDefinitions().add(sd);
			sd.getSeriesPalette().getEntries().clear();
			sd.getSeriesPalette().getEntries().addAll(fills);
	
			Series seCategory = (Series) SeriesImpl.create();
			TextDataSet categoryValues = TextDataSetImpl.create(texts);
			seCategory.setDataSet(categoryValues);
			sd.getSeries().add(seCategory);
	
			//Orthogonal Series
			PieSeries sePie = (PieSeries) PieSeriesImpl.create();
			NumberDataSet seriesOneValues = NumberDataSetImpl.create(values);
			sePie.setDataSet(seriesOneValues);
			//sePie.setLabelPosition(Position.INSIDE_LITERAL);
			sePie.setSeriesIdentifier(""); //$NON-NLS-1$
			
			// triggers
			// tooltip trigger
			Action tooltipAction = ActionImpl.create(ActionType.SHOW_TOOLTIP_LITERAL, TooltipValueImpl.create(500, null));
			Trigger tooltipTrigger = TriggerImpl.create(TriggerCondition.ONMOUSEOVER_LITERAL, tooltipAction);
			sePie.getTriggers().add(tooltipTrigger);
			
			//pie.getInteractivity().setEnable(true);
			//pie.getInteractivity().setLegendBehavior(LegendBehaviorType.HIGHLIGHT_SERIE_LITERAL);
			/*Action highlightAction = ActionImpl.create(ActionType.HIGHLIGHT_LITERAL, SeriesValueImpl.create( String.valueOf( sePie.getSeriesIdentifier() ) ) ) ;
			Trigger highlightTrigger = TriggerImpl.create(TriggerCondition.ONCLICK_LITERAL, highlightAction);
			//Trigger highlightTrigger = TriggerImpl.create(TriggerCondition.ONMOUSEDOWN_LITERAL, highlightAction);
			sePie.getTriggers().add(highlightTrigger);	*/
			
			// callback on click trigger
			Action action = ActionImpl.create(ActionType.CALL_BACK_LITERAL, CallBackValueImpl.create(VerdictSummaryChart.class.getName()));
			Trigger tg = TriggerImpl.create(TriggerCondition.ONCLICK_LITERAL, action);
			//Trigger tg = TriggerImpl.create(TriggerCondition.ONMOUSEDOWN_LITERAL, action);
			sePie.getTriggers().add(tg);	
	
			SeriesDefinition sdVerdicts = SeriesDefinitionImpl.create();
			sdVerdicts.getQuery().setDefinition( "Verdict Types" ); //$NON-NLS-1$
			sd.getSeriesDefinitions( ).add( sdVerdicts );
			sdVerdicts.getSeries().add(sePie);
			
			// create percentage values
			DataPointComponent dpc = DataPointComponentImpl.create( DataPointComponentType.ORTHOGONAL_VALUE_LITERAL,
					JavaNumberFormatSpecifierImpl.create( "0.0'%'" ) );//$NON-NLS-1$
			sePie.getDataPoint( ).getComponents( ).clear( );
			sePie.getDataPoint( ).getComponents( ).add( dpc );
			
			//Explosion
			sePie.setExplosion(0);
			//sePie.setRatio(0.5);
	
			//Min Slice
			/*pie.setMinSlice(10);
			pie.setMinSlicePercent(true);
			pie.setMinSliceLabel("Others");//$NON-NLS-1$*/		
			return pie;
		}
		
		return null;
	}
	
	public void callback(Object event, Object source, CallBackValue value) {
		DataPointHints data = null;
		if(source instanceof StructureSource)
		{
			if(((StructureSource)source).getSource() instanceof DataPointHints)
			data = (DataPointHints)((StructureSource)source).getSource();
		}
		else if(source instanceof DataPointHints)
			data = (DataPointHints)source;
		
		if(data != null)
		{
			String text = data.getBaseDisplayValue();
			if(text != null)
			{
				for(int i = 0; i < verdictCategories.length; i++)
				{
					if(text.equals(verdictCategories[i].getText()))
						fireSelectionChangedEvent(new StructuredSelection(verdictCategories[i]));
				}
			}
			
			/*//new api available since 3/15/06. need new birt build adopted by TPTP 
			int index = data.getIndex();
			if(index >=0 && index < verdictCategories.length)
			{
				fireSelectionChangedEvent(new StructuredSelection(verdictCategories[index]));
			}*/
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.birt.chart.device.IUpdateNotifier#getDesignTimeModel()
	 */
	public Chart getDesignTimeModel() {
		return cm;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.birt.chart.device.IUpdateNotifier#getRunTimeModel()
	 */
	public Chart getRunTimeModel() {

		return chartState.getChartModel( );
	}

	/* (non-Javadoc)
	 * @see org.eclipse.birt.chart.device.IUpdateNotifier#peerInstance()
	 */
	public Object peerInstance() {
		return this;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.birt.chart.device.IUpdateNotifier#regenerateChart()
	 */
	public void regenerateChart() {
		redraw();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.birt.chart.device.IUpdateNotifier#repaintChart()
	 */
	public void repaintChart() {
		redraw();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.birt.chart.device.IUpdateNotifier#getContext(java.lang.Object)
	 */
	public Object getContext(Object key) {
		// TODO Auto-generated method stub
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.birt.chart.device.IUpdateNotifier#putContext(java.lang.Object, java.lang.Object)
	 */
	public Object putContext(Object key, Object value) {
		// TODO Auto-generated method stub
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.birt.chart.device.IUpdateNotifier#removeContext(java.lang.Object)
	 */
	public Object removeContext(Object key) {
		// TODO Auto-generated method stub
		return null;
	}

	public void processAction(Action action, StructureSource source) {
		
		if(ActionType.SHOW_TOOLTIP_LITERAL.equals(action.getType())){
			
			if(StructureType.SERIES_DATA_POINT.equals(source.getType())){
				
				String text = ((DataPointHints)(source.getSource())).getBaseDisplayValue().trim();
				
				for(int counter = 0; counter < verdictCategories.length; counter++){
					
					if(text.equals(verdictCategories[counter].getText().trim())){

						((TooltipValue)(action.getValue())).setText(NLS.bind(UiPluginResourceBundle.LogOverview_VerdictSummaryToolTip, text, String.valueOf(verdictCategories[counter].getNumVerdicts())));

						break;
					}
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.ISelectionProvider#addSelectionChangedListener(org.eclipse.jface.viewers.ISelectionChangedListener)
	 */
	public void addSelectionChangedListener(ISelectionChangedListener listener) {
		selectionListeners.add(listener);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.ISelectionProvider#getSelection()
	 */
	public ISelection getSelection() {
		return currentSelection;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.ISelectionProvider#removeSelectionChangedListener(org.eclipse.jface.viewers.ISelectionChangedListener)
	 */
	public void removeSelectionChangedListener(ISelectionChangedListener listener) {
		selectionListeners.remove(listener);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.ISelectionProvider#setSelection(org.eclipse.jface.viewers.ISelection)
	 */
	public void setSelection(ISelection selection) {
		if(selection instanceof IStructuredSelection)
		{
			//TODO select the pie slice for (TPFVerdictList)obj.
			fireSelectionChangedEvent((IStructuredSelection)selection);
		}		
	}
	
	protected void fireSelectionChangedEvent(IStructuredSelection selection)
	{
		// must have a selection
		if(selection == null || selection.isEmpty())
			return;
		
		if(currentSelection != null && currentSelection.getFirstElement().equals(selection.getFirstElement()))
			return;

		currentSelection = selection;
		Object[] listeners = selectionListeners.getListeners();
		for(int i = 0; i < listeners.length; i++)
		{
			((ISelectionChangedListener)listeners[i]).selectionChanged(new SelectionChangedEvent(this, currentSelection));
		}
	}
}
