/********************************************************************** 
 * Copyright (c) 2005, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id$ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.ui.forms.editor;

import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.ListenerList;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.loaders.util.Guid;
import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.models.common.util.ResourceCache;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.internal.util.EObjectEditorInput;
import org.eclipse.hyades.test.ui.navigator.EObjectProxyNode;
import org.eclipse.hyades.test.ui.util.RCPFileEditorInput;
import org.eclipse.hyades.test.ui.util.TestUIUtil;
import org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter;
import org.eclipse.hyades.ui.internal.editor.EditorSynchronizer;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IPathEditorInput;
import org.eclipse.ui.IURIEditorInput;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.dialogs.SaveAsDialog;
import org.eclipse.ui.editors.text.ILocationProvider;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.editor.IFormPage;
import org.eclipse.ui.forms.widgets.FormToolkit;

/**
 * <p>Test Editor built on the Eclipse Forms framework.</p>
 * 
 * 
 * @author   Bianca Xue Jiang 
 * @author   Paul E. Slauenwhite
 * @author   Marcelo Paternostro
 * @author   Jerome Bozier
 * @version  August 11, 2010
 * @since    October 14, 2005
 */
public abstract class BaseFormEditor extends FormEditor implements ISynchronizedEditorAdapter, ISelectionProvider
{
	private Class editorObjectClass;
	private EObject editorEObject;
	private ResourceSet sharedResourceSet;
	private Adapter thisAdapter;
	private URI cachedURI;	
	EditorSynchronizer editorSynchronizer;
	private ListenerList selectionChangedListeners;
	private boolean isReadOnly = false;

	/**
	 * Instantiate this class with the <code>Class</code> of the editor object.
	 * @param editorObjectClass
	 */
	public BaseFormEditor(Class editorObjectClass)
	{
		super();
		this.editorObjectClass = editorObjectClass;
		this.selectionChangedListeners = new ListenerList();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.forms.editor.FormEditor#init(org.eclipse.ui.IEditorSite, org.eclipse.ui.IEditorInput)
	 */
	public void init(IEditorSite site, IEditorInput input) throws PartInitException
	{
		super.init(site, input);
		site.setSelectionProvider(this);
		
		EObject eObject = null;
		if(input instanceof EObjectEditorInput){
			eObject = getEObjectContent(((EObjectEditorInput)(input)).getEObject());
		}
		else if(input instanceof IFileEditorInput)
			eObject = getFileContent(((IFileEditorInput)input).getFile());
		else if (input instanceof ILocationProvider)
		{
			ILocationProvider javaFileEditorInput = (ILocationProvider)input;
			eObject = getFileContent(javaFileEditorInput.getPath(javaFileEditorInput).toFile());
		}
		else if(input instanceof RCPFileEditorInput)
			eObject = getFileContent(((RCPFileEditorInput)input).getIOFile());
		else if(input instanceof IPathEditorInput)
		{
			IPath path = ((IPathEditorInput)input).getPath();
			eObject = getFileContent(path.toFile());
		} else if (input instanceof IURIEditorInput) {
			eObject = getFileContent(new File(((IURIEditorInput)input).getURI()));
		}
		
		if(eObject == null)
			throw new PartInitException(UiPluginResourceBundle._ERROR_EDT_INPUT); 
		
		setEditorObject(eObject);
		editorSynchronizer = new EditorSynchronizer(this);
		isReadOnly = EditorSynchronizer.isReadOnly(editorSynchronizer.getEditorFile());//EMFUtil.getWorkspaceFile(eObject).isReadOnly();
	}
	
	/*
	 *  (non-Javadoc)
	 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
	 */
	public Object getAdapter(Class adapter)
	{
		if(ISynchronizedEditorAdapter.class == adapter)
			return this;
			
		return super.getAdapter(adapter);
	}
	
	/**
	 * @see org.eclipse.ui.IWorkbenchPart#dispose()
	 */
	public void dispose()
	{
		if(getSite() != null)
			getSite().setSelectionProvider(null);
		
		if(editorSynchronizer != null)
		{
			editorSynchronizer.dispose();
			editorSynchronizer = null;
		}
		
		if(sharedResourceSet != null)
		{
			EMFUtil.removeAdapters(sharedResourceSet.getAllContents(), thisAdapter);
			sharedResourceSet.eAdapters().remove(thisAdapter);
			sharedResourceSet = null;
			ResourceCache.getInstance().releaseSharedResource(cachedURI);
		}
		
		if(selectionChangedListeners != null)
		{
			selectionChangedListeners.clear();
			selectionChangedListeners = null;
		}
		
		super.dispose();
	}
	
	/*
	 *  (non-Javadoc)
	 * @see org.eclipse.ui.forms.editor.FormEditor#createToolkit(org.eclipse.swt.widgets.Display)
	 */
	protected FormToolkit createToolkit(Display display) {
		return new FormToolkit(UiPlugin.getDefault().getFormColors(display));
	}
	
	/**
	 * @deprecated As of TPTP 4.7.0, this API is internal.
	 */
	public void setEditorObject(EObject editorObject)
	{
		this.editorEObject = editorObject;
		if(editorObjectClass.isInstance(editorObject))
		{
			if(editorObject instanceof CMNNamedElement)
				this.setEditorTitle(((CMNNamedElement)editorObject).getName());
			else
				this.setEditorTitle(editorObject.toString());
		}
		
		for (int i = 0; i < pages.size(); i++) {
			Object page = pages.get(i);
			if(page instanceof IFormPage)
			{
				IManagedForm mform = ((IFormPage)page).getManagedForm();
				if(mform != null)
					mform.setInput(editorObject);
			}
		}
	}
	
	public EObject getEditorObject()
	{
		return this.editorEObject;
	}
	
	public void setEditorTitle(String title)
	{
		this.setPartName(title);
	}
	
	public String getEditorTitle()
	{
		return getPartName();
	}
	
	/**
	 * @return Returns the <code>Class</code> of the editor object.
	 */
	protected Class getEditorObjectClass() {
		return editorObjectClass;
	}

	protected EObject getEObjectContent(EObject eObject){

		cachedURI = eObject.eResource().getURI();
		
		if(getEditorObjectClass().isInstance(eObject)){

			sharedResourceSet = ResourceCache.getInstance().createResourceSet();
			
			//Add the eObject's resource to the shared resource set:
			sharedResourceSet.getResources().add(eObject.eResource());

			return eObject;
		}

		return null;
	}
	 
	/**
	 * Returns the "meaningful" content of a given IFile.
	 * @param file
	 * @return Object
	 */
	protected EObject getFileContent(IFile file)
	{
		cachedURI = URI.createPlatformResourceURI(file.getFullPath().toString(), false);		
		return getFileContent(cachedURI);
	}

	protected EObject getFileContent(File file)
	{
		cachedURI = URI.createFileURI(file.getAbsolutePath());
		return getFileContent(cachedURI);
	}

	private EObject getFileContent(URI uri) {
		Resource resource = ResourceCache.getInstance().getSharedResource(uri);
		if (resource.getContents().size() != 1
			|| !getEditorObjectClass().isInstance(resource.getContents().get(0))) {
			return null;
		}
		
		sharedResourceSet = resource.getResourceSet();
		
		// adapter already added in this ResourceSet in ResourceCache.
		/*thisAdapter = new AdapterImpl()
		{
			public void notifyChanged(Notification msg)
			{
				switch(msg.getEventType())
				{
					case Notification.ADD:
						if(msg.getNewValue() instanceof Resource)
							((Resource)msg.getNewValue()).setTrackingModification(true);
						break;
					
					case Notification.ADD_MANY:
						Collection collection = (Collection)msg.getNewValue();
						for (Iterator i = collection.iterator(); i.hasNext();)
						{
							Object element = i.next();
							if(element instanceof Resource)
								((Resource)element).setTrackingModification(true);							
						}
						break;	
				}
			}
		};
		sharedResourceSet.eAdapters().add(thisAdapter);*/
		
		return (EObject)resource.getContents().get(0);
	}
	
	public ResourceSet getResourceSet()
	{
		return sharedResourceSet;
	}
	

	/*
	 *  (non-Javadoc)
	 * @see org.eclipse.ui.ISaveablePart#doSave(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void doSave(IProgressMonitor monitor)
	{
		save(monitor);
		commitFormPages(true);
		super.editorDirtyStateChanged();
	}
	
	protected void save(IProgressMonitor monitor)
	{
		Resource[] resources = (Resource[])getResourceSet().getResources().toArray(new Resource[getResourceSet().getResources().size()]);
		int length = resources.length;
		int counter = 0;
		monitor.beginTask("", length); //$NON-NLS-1$
		
		for(int i = 0, maxi = resources.length; i < maxi; i++)
		{
			if(resources[i].isModified())
			{
				logSaveMessage(monitor, resources[i], length, counter++);
				try
				{
					EMFUtil.save(resources[i]);
				}
				catch(Exception e)
				{
					handleException(e);
				}
			}
			monitor.worked(1);
		}
		monitor.done();
	}
	
	private void commitFormPages(boolean onSave) {
		for (int i = 0; i < pages.size(); i++) {
			Object page = pages.get(i);
			if(page instanceof IFormPage)
			{
				IManagedForm mform = ((IFormPage)page).getManagedForm();
				if (mform != null && mform.isDirty())
					mform.commit(true);
			}
		}
	}
	
	/*public void refresh()
	{
		for (int i = 0; i < pages.size(); i++) {
			Object page = pages.get(i);
			if(page instanceof IFormPage)
			{
				IManagedForm mform = ((IFormPage)page).getManagedForm();
				if (mform != null && mform.isStale())
					mform.refresh();
			}
		}
	}*/
	
	protected IFormPage[] getPages() {
		ArrayList formPages = new ArrayList();
		for (int i = 0; i < pages.size(); i++) {
			Object page = pages.get(i);
			if (page instanceof IFormPage)
				formPages.add(page);
		}
		return (IFormPage[]) formPages.toArray(new IFormPage[formPages.size()]);
	}
	
	/**
	 * Displays a message when saving a resource.
	 * e.g : (3 of 5) myResource.res
	 * @param monitor the progress monitor
	 * @param resource the current resource to save
	 * @param total the total number of resources to save
	 * @param current the number of the current resource
	 */
	protected void logSaveMessage(IProgressMonitor monitor, Resource resource, int total, int current)
	{
		//String resourceName = EMFUtil.getFilePath(resource);
		monitor.subTask(""); //$NON-NLS-1$
	}
	
	/**
	 * Handles any exception thrown during the 
	 * {@link #save(IProgressMonitor)} execution.
	 * @param exception
	 */
	protected void handleException(Exception exception)
	{
		UiPlugin.logError(exception);
		String filePath = EMFUtil.getFilePath(getEditorObject());
		UIUtil.openSaveFileErrorDialog(getSite().getShell(), filePath, exception);
	}
	
	/**
	 * Reloades the editor's object.  This method is invoked by the synchronizer
	 * adapter and should return <code>true</code> if the synchronizer is not
	 * supposed to continue with the default treatment.
	 * @return boolean
	 */
	protected boolean reloadEditorObject()
	{
		Resource resource = ((EObject)getEditorObject()).eResource(); 		
		try
		{
			resource.unload();
			resource.load(EMFUtil.RESOURCE_OPTIONS);
			
			if(!resource.getContents().isEmpty())
				setEditorObject((EObject)resource.getContents().get(0));
			else
				setEditorObject(null);
		}
		catch(Exception e)
		{
			UiPlugin.logError(e);
		}

		return false;
	}

	/**
	 * Notifies this listener that the selection has changed.
	 * @param part the workbench part containing the selection
	 * @param selection the current selection
	 * @see org.eclipse.ui.ISelectionListener#selectionChanged(org.eclipse.ui.IWorkbenchPart, org.eclipse.jface.viewers.ISelection)
	 */
	public void selectionChanged(IWorkbenchPart part, ISelection selection)
	{
		if(part instanceof TestNavigator)
		{
			if(!(selection instanceof IStructuredSelection))
				return;

			IStructuredSelection structuredSelection = (IStructuredSelection)selection;
			if(structuredSelection.size() != 1) 
				return;
			
			EObject obj2Select = null;
			
			Object first = structuredSelection.getFirstElement();
			if (getEditorObjectClass().isInstance(first)) {
				URI uri = EMFUtil.getObjectURI(first);
				if(uri == null)
					return;
				ResourceSet resSet = getResourceSet();
			    Resource res = resSet.getResource(uri.trimFragment(), false);
			    if (res != null)
			    {
					String id = uri.fragment();
				    for (Iterator i = res.getAllContents(); i.hasNext(); )
				    {
				      EObject eObject = (EObject)i.next();
				      if (id.equals(EcoreUtil.getID(eObject)) && getEditorObjectClass().isInstance(eObject))
				      {
				      	obj2Select = eObject;
				      }
				    }
			    }
			} else if (first instanceof EObjectProxyNode) {
				EObjectProxyNode proxy = (EObjectProxyNode) first;
				URI uri = proxy.getOriginatorURI();
				if(uri != null) {
					obj2Select = getResourceSet().getEObject(uri, false);
				}
			}
	
			if(obj2Select == null)
				return;
			
			//setSelection(new StructuredSelection(obj2Select));
		}
	}

	/*
	 *  (non-Javadoc)
	 * @see org.eclipse.ui.ISaveablePart#doSaveAs()
	 */
	public void doSaveAs() {

		//See BaseEditorExtension.performSaveAs(IProgressMonitor) for a similar algorithm.
		//Create, configure, and open the Save As dialog: 
		SaveAsDialog dialog = new SaveAsDialog(getEditorSite().getShell());        

		//Resolve the file editor input and configure the Save As dialog:
		IEditorInput editorInput = getEditorInput();

		if(editorInput instanceof IFileEditorInput){

			IFile originalFile = ((IFileEditorInput)(editorInput)).getFile();

			if(originalFile != null){
				dialog.setOriginalFile(originalFile);
			}
		}

		dialog.create();

		if (dialog.open() == Window.OK) {

			//Resolve the new file path:
			IPath newFilePath = dialog.getResult();

			if (newFilePath != null) {

				//Resolve the new file:
				IFile newFile = ResourcesPlugin.getWorkspace().getRoot().getFile(newFilePath);

				try {

					//Resolve resource:
					Resource resource = editorEObject.eResource();

					if((resource != null) && (editorEObject instanceof CMNNamedElement)) {

						CMNNamedElement namedElement = ((CMNNamedElement)(editorEObject));

						//Capture the original URI, ID, and name:
						URI originalURI = resource.getURI();
						String originalId = namedElement.getId();
						String originalName = namedElement.getName();

						try{

							//Disable notifications:
							namedElement.eSetDeliver(false);

							//Set the new URI, ID, and name:            		
							resource.setURI(URI.createPlatformResourceURI(newFilePath.toString(), false));
							namedElement.setId(new Guid().toString());
							namedElement.setName(newFilePath.removeFileExtension().lastSegment());

							//Save the new test asset:
							EMFUtil.save(resource);
						}
						finally{

							//Set the original URI, ID, and name:    
							resource.setURI(originalURI);
							namedElement.setId(originalId);
							namedElement.setName(originalName);       

							//Re-enable notifications:
							namedElement.eSetDeliver(true);
						}
						
						//Attempt to open the new test asset in its editor:
						TestUIUtil.openEditor(newFile, getEditorSite().getId());

						//Release the shared resource (see dispose()) BEFORE loading the new resource:
						ResourceCache.getInstance().releaseSharedResource(cachedURI);
					} 
				} 
				catch (Exception e) {
					TestCorePlugin.getDefault().logError(e);
				}
			}
		}
	}

	/*
	 *  (non-Javadoc)
	 * @see org.eclipse.ui.ISaveablePart#isSaveAsAllowed()
	 */
	public boolean isSaveAsAllowed()
	{
		return true;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#doSaveEditorFile(boolean)
	 */
	public boolean doSaveEditorFile(boolean wasDeleted) {
		this.doSave(new NullProgressMonitor());
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#editorFileChanged()
	 */
	public boolean editorFileChanged() {
		reload();
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#editorFileDeleted()
	 */
	public boolean editorFileDeleted() {
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#editorFileWriteAccessChanged(boolean)
	 */
	public boolean editorFileWriteAccessChanged(boolean isReadOnly) {
		
		this.isReadOnly = isReadOnly;
		
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#handle(java.util.List, java.util.List)
	 */
	public void handle(List changedFiles, List removedFiles) {
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#reload()
	 */
	public boolean reload() {
		// Bugzilla_152218 preserve the selection after reload.
		String fragment = null;
		IStructuredSelection structuredSelection = (IStructuredSelection)getSelection();
		if(structuredSelection != null && structuredSelection.size() == 1)
		{
			Object selection = structuredSelection.getFirstElement();
			if(selection instanceof EObject)
				fragment = ((EObject)selection).eResource().getURIFragment((EObject)selection);
		}
		
		if(!reloadEditorObject())
		{
			Object editorObject = getEditorObject();
			if((editorObject instanceof EObject) && (fragment != null))
			{
				Object selection = ((EObject)(editorObject)).eResource().getEObject(fragment);
				if(selection != null)
					structuredSelection = new StructuredSelection(selection);
				else
					structuredSelection = StructuredSelection.EMPTY;
					
				setSelection(structuredSelection);
			}	
		}
		
		return true;
	}

	public void addSelectionChangedListener(ISelectionChangedListener listener) {
		selectionChangedListeners.add(listener);
	}

	public void removeSelectionChangedListener(ISelectionChangedListener listener) {
		selectionChangedListeners.remove(listener);
	}

	public abstract ISelection getSelection();

	public abstract void setSelection(ISelection selection);
	
	public void fireSelectionChanged(SelectionChangedEvent event)
	{
		Object[] listeners = selectionChangedListeners.getListeners();
		for(int i = 0; i < listeners.length; i++)
			((ISelectionChangedListener)listeners[i]).selectionChanged(new SelectionChangedEvent(this, event.getSelection()));
	}
	
	/**
	 * Resolves if the editor is opened in read-only mode.
	 * 
	 * @return <code>true</code> if the editor is opened in read-only mode, otherwise <code>false</code>.
	 */
	public boolean isReadOnly(){
		return isReadOnly;
	}
}
