/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LogOverviewPage.java,v 1.20 2010/05/07 17:21:27 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.forms.editor.page;

import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.forms.base.ExecutionResultDetailsPart;
import org.eclipse.hyades.test.ui.forms.base.GeneralInfoSectionPart;
import org.eclipse.hyades.test.ui.forms.base.LogOverviewDetailsPart;
import org.eclipse.hyades.test.ui.forms.base.TestLogVerdictTraversalQuery;
import org.eclipse.hyades.test.ui.forms.base.VerdictSummaryChart;
import org.eclipse.hyades.test.ui.forms.editor.BaseFormEditor;
import org.eclipse.hyades.test.ui.forms.editor.TestLogViewer;
import org.eclipse.hyades.test.ui.forms.util.FormsUtil;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.editor.FormPage;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;
import org.osgi.framework.Bundle;

/**
 * <p>This is the <i>Overview</i> page class of the TPTP Test Log Viewer.</p>
 *
 * <p>This is an Eclipse forms class derived from the original TPTP Test Log Viewer.</p>
 * 
 * 
 * @author  Bianca Xue Jiang
 * @author  Marcelo Paternostro
 * @author  Paul E. Slauenwhite
 * @version May 7, 2010
 * @since   August 9, 2005
 */
public class LogOverviewPage extends FormPage {
	
	public static final String PAGE_ID = "org.eclipse.tptp.test.ui.logviewer.overview"; //$NON-NLS-1$
	private TPFExecutionResult input;
	private SashForm sashForm;
	private SashForm overviewSash;
	private SashForm detailsSash;
	private ExecutionResultDetailsPart detailsPart;
	
	/**
	 * <p><code>true<code> if the BIRT chart run-time is available, otherwise <code>false</code>.</p>
	 * 
	 * <p>Note: This value must default to <code>false</code>.
	 */
	private boolean isBIRTAvailable = false;
	private Section chartSection;
	private VerdictSummaryChart chart;

	public LogOverviewPage(FormEditor editor) {
		super(editor, PAGE_ID, UiPluginResourceBundle.W_OVERVIEW);
	}

	protected void createFormContent(IManagedForm managedForm) {
		
		//Check if the BIRT chart run-time is available.
		try {

			//See /META-INF/MANIFEST.MF for the dependent BIRT bundle:
			Bundle birtChartBundle = Platform.getBundle("org.eclipse.birt.chart.engine"); //$NON-NLS-1$

			if(birtChartBundle != null){				
				isBIRTAvailable = ((birtChartBundle.getState() == Bundle.RESOLVED) || (birtChartBundle.getState() == Bundle.STARTING) || (birtChartBundle.getState() == Bundle.ACTIVE));
			}
		}
		catch (Throwable t) {
			//Ignore since the BIRT chart run-time is not available.
		}

		//Reuseable layout data:
		GridLayout layout = new GridLayout();
		layout.marginWidth = 0;
		layout.marginHeight = 0;

		//Overview page:
		ScrolledForm scrolledForm = managedForm.getForm();
		scrolledForm.setText(UiPluginResourceBundle.W_TEST_LOG);
		scrolledForm.getBody().setLayout(layout);
		
		//Vertical sash:
		sashForm = new SashForm(scrolledForm.getBody(), SWT.NULL);
		
		FormToolkit toolkit = managedForm.getToolkit();
		toolkit.adapt(sashForm, false, false);

		sashForm.setMenu(scrolledForm.getBody().getMenu());
		sashForm.setLayoutData(new GridData(GridData.FILL_BOTH));

		//Overview:
		createOverviewPart(managedForm, sashForm);
		
		//Details:
		createDetailsPart(managedForm, sashForm);

		//Toolbar:
		createToolBarActions(managedForm);
		
		scrolledForm.updateToolBar();
		
		Object input = ((BaseFormEditor)getEditor()).getEditorObject();
		TestLogVerdictTraversalQuery query = (TestLogVerdictTraversalQuery) ((TestLogViewer)getEditor()).getTestLogVerdictTraversal();
		managedForm.setInput(input);
		
		if((chart != null) && (input instanceof TPFExecutionResult)){
			
			try {
				
				boolean isChartDrawn = chart.setInput((TPFExecutionResult)input, query);
				
				if((!isChartDrawn) && (chartSection != null)){
					
					chartSection.setExpanded(false);
					
					if(detailsPart instanceof LogOverviewDetailsPart){
						
						((LogOverviewDetailsPart)detailsPart).setInput(null);
						Section verdictListSection = ((LogOverviewDetailsPart)(detailsPart)).getVerdictListSection();
						
						if(verdictListSection != null){
							verdictListSection.setExpanded(false);
						}
					}
				}
			}
			catch (Throwable t) {
				UiPlugin.logError(t);
			}
		}
	}
	
	protected void createOverviewPart(IManagedForm managedForm, Composite parent)
	{
		overviewSash = new SashForm(parent, SWT.VERTICAL);
		managedForm.getToolkit().adapt(overviewSash, false, false);
		overviewSash.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		//generalInfoPart = new GeneralInfoSectionPart(this, leftPartSash);
		new GeneralInfoSectionPart(this, overviewSash);
		
		if(isBIRTAvailable)
		{
			chartSection = FormsUtil.createSection(managedForm, overviewSash, UiPluginResourceBundle.LogOverview_VerdictSummaryTitle, UiPluginResourceBundle.LogOverview_VerdictSummaryDescription); 
			overviewSash.setWeights(new int[]{35, 65});
			Composite client = (Composite)chartSection.getClient();
			chart = new VerdictSummaryChart(client, SWT.NONE);
			chart.setChartSection(chartSection);
			chart.setLayoutData(new GridData(GridData.FILL_BOTH));
		}
	}
	
	protected void createDetailsPart(IManagedForm managedForm, Composite parent)
	{
		if(isBIRTAvailable)
		{
			detailsPart = new LogOverviewDetailsPart(this);
			detailsPart.initialize(managedForm);		
			detailsPart.createContents(parent);
			detailsSash = ((LogOverviewDetailsPart)detailsPart).getParent();
			addSelectionChangedListener((LogOverviewDetailsPart)detailsPart);
		}
		else
		{
			detailsPart = new ExecutionResultDetailsPart(this);
			detailsPart.initialize(managedForm);		
			detailsPart.createContents(parent);			
		}
	}
	
	protected void createToolBarActions(IManagedForm managedForm)
	{
		final ScrolledForm form = managedForm.getForm();
		Action haction = new Action(null, Action.AS_RADIO_BUTTON) {
			public void run() {
				sashForm.setOrientation(SWT.HORIZONTAL);
				overviewSash.setOrientation(SWT.VERTICAL);
				if(detailsSash != null)
					detailsSash.setOrientation(SWT.VERTICAL);
				form.reflow(true);
			}
		};
		haction.setChecked(true);
		haction.setToolTipText(UiPluginResourceBundle.Editor_horizontal_orientation);
		haction.setImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor("e", TestUIImages.IMG_HORIZONTAL)); //$NON-NLS-1$
		Action vaction = new Action(null, Action.AS_RADIO_BUTTON) {
			public void run() {
				sashForm.setOrientation(SWT.VERTICAL);
				overviewSash.setOrientation(SWT.HORIZONTAL);
				if(detailsSash != null)
					detailsSash.setOrientation(SWT.HORIZONTAL);
				form.reflow(true);
			}
		};
		vaction.setChecked(false);
		vaction.setToolTipText(UiPluginResourceBundle.Editor_vertical_orientation);
		vaction.setImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor("e", TestUIImages.IMG_VERTICAL)); //$NON-NLS-1$
		form.getToolBarManager().add(haction);
		form.getToolBarManager().add(vaction);
	}
	
	/**
	 * Could return <code>null</code> when BIRT is not available and thus no chart will be created.
	 * @return
	 */
	public VerdictSummaryChart getVerdictSummaryChart()
	{
		return chart;
	}
	
	/**
	 * Resolves the selection of the chart, otherwise <code>null</code>.
	 * <p/>
	 * Note, no chart is created with BIRT/DTP/GMF are not available.  As such,
	 * this method returns <code>null</code>.
	 * <p/>
	 * 
	 * @return The selection of the chart, otherwise <code>null</code>.
	 * @deprecated As of TPTP V4.5.1, use {@link LogEventsPage#getSelection()}.
	 */
	public ISelection getSelection(){
		return (chart == null ? null : chart.getSelection());
	}
	
	/**
	 * Sets the selection of the chart.
	 * <p/>
	 * Note, this method is empty.
	 * <p/>
	 * 
	 * @param selection The selection of the chart.
	 * @deprecated As of TPTP V4.5.1, use {@link LogEventsPage#setSelection(ISelection)}.
	 */
	public void setSelection(ISelection selection){		
		//No-operation.
	}
	
	/**
	 * Adds a selection changed listener to BIRT chart. No-op when BIRT is not available and thus no chart created.
	 * @param listener
	 */
	public void addSelectionChangedListener(ISelectionChangedListener listener) {
		if(chart != null)
			chart.addSelectionChangedListener(listener);
	}
	
	/**
	 * Removes a selection changed listener to BIRT chart. No-op when BIRT is not available and thus no chart created.
	 * @param listener
	 */
	public void removeSelectionChangedListener(ISelectionChangedListener listener) {
		if(chart != null)
			chart.removeSelectionChangedListener(listener);
	}
	
	public boolean selectReveal(Object object) {
		if(object == input)
		{
			if(getEditor().getActivePageInstance() != this)
				getEditor().setActivePage(PAGE_ID);
			return true;
		}
		return false;
	}
}
