/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionEventSearchProvider.java,v 1.7 2007/05/03 01:44:43 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.forms.extensions.provisional;

import java.util.Map;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.common.common.CommonPackage;
import org.eclipse.hyades.models.common.testprofile.Common_TestprofilePackage;
import org.eclipse.hyades.models.hierarchy.extensions.BinaryExpression;
import org.eclipse.hyades.models.hierarchy.extensions.ExtensionsFactory;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.RelationalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleOperand;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.internal.search.TestLogSearchPage;
import org.eclipse.search.ui.ISearchPageContainer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

/**
 * Base class of search provider classes for Execution Event element in Test Log Search.
 * 
 * @see extension point "org.eclipse.hyades.test.ui.testLogSearchProvider"
 * @see ITestLogSearchProvider
 * @author bjiang
 * @since 4.2
 */
public class ExecutionEventSearchProvider implements ITestLogSearchProvider, SelectionListener {

	private ISearchPageContainer pageContainer;
	private boolean searchText = true;
	private Button text;
	private boolean searchTime;
	private Button time;
	private boolean searchAttachment;
	private Button attachment;
	private boolean searchPropName;
	private Button propName;
	private boolean searchPropValue;
	private Button propValue;
	
	public void createContents(Composite parent) {
		GridLayout layout = new GridLayout();
		layout.marginHeight = layout.marginWidth = 0;
		parent.setLayout(layout);
		Composite comp = new Composite(parent, SWT.NONE);
		comp.setLayoutData(new GridData(GridData.FILL_HORIZONTAL | GridData.VERTICAL_ALIGN_BEGINNING));
		comp.setLayout(new GridLayout(3, false));
		
		Label searchIn = new Label(comp, SWT.NONE);
		searchIn.setText(UiPluginResourceBundle.TestLogSearch_SearchInFields); 
		GridData gd = new GridData();
		gd.horizontalSpan = 3;
		searchIn.setLayoutData(gd);
		text = new Button(comp, SWT.CHECK);
		text.setText(UiPluginResourceBundle.TestLogSearch_Text); 
		text.addSelectionListener(this);
		text.setSelection(searchText);
		time = new Button(comp, SWT.CHECK);
		time.setText(UiPluginResourceBundle.TestLogSearch_Time); 
		time.addSelectionListener(this);
		attachment = new Button(comp, SWT.CHECK);
		attachment.setText(UiPluginResourceBundle.TestLogSearch_AttachmentFileName);		 
		attachment.addSelectionListener(this);
		propName = new Button(comp, SWT.CHECK);
		propName.setText(UiPluginResourceBundle.TestLogSearch_PropertyName); 
		propName.addSelectionListener(this);
		propValue = new Button(comp, SWT.CHECK);
		propValue.setText(UiPluginResourceBundle.TestLogSearch_PropertyValue); 
		propValue.addSelectionListener(this);
	}

	public void setContainer(ISearchPageContainer container) {
		this.pageContainer = container;
	}
	
	public ISearchPageContainer getContainer()
	{
		return this.pageContainer;
	}
	
	public SimpleSearchQuery createSearchQuery(EObject rootElement, String text, Map options)
	{
		SimpleSearchQuery query = createQuery(rootElement);		
		setQueryOutputType(query, Common_TestprofilePackage.eINSTANCE.getTPFExecutionEvent());
		LogicalExpression or = createORExpression();
		
		boolean caseSensitive = Boolean.TRUE.equals(options.get(TestLogSearchPage.SEARCH_OPTION_CASE_SENSITIVE));
		// TODO if(!recursive) see example in SymptomTests.java 
			
		if(searchText)
		{
			BinaryExpression textSearch = createBinaryExpression(Common_TestprofilePackage.eINSTANCE.getTPFExecutionEvent_Text(), text);		
			textSearch.setCaseInsensitive(!caseSensitive);
			or.getArguments().add(textSearch);
		}
		if(searchTime)
		{
			BinaryExpression timeSearch = createBinaryExpression(Common_TestprofilePackage.eINSTANCE.getTPFExecutionEvent_Timestamp(), text);		
			timeSearch.setCaseInsensitive(!caseSensitive);
			or.getArguments().add(timeSearch);
		}
		if(searchAttachment)
		{
			query.getRequiredPaths().add(Common_TestprofilePackage.eINSTANCE.getTPFExecutionEvent_Annotations());
			BinaryExpression attachmentSearch = createBinaryExpression(CommonPackage.eINSTANCE.getCMNAnnotation_URI(), text);		
			attachmentSearch.setCaseInsensitive(!caseSensitive);
			or.getArguments().add(attachmentSearch);
		}
		if(searchPropName)
		{
			query.getRequiredPaths().add(Common_TestprofilePackage.eINSTANCE.getTPFExecutionEvent_Properties());
			BinaryExpression propNameSearch = createBinaryExpression(CommonPackage.eINSTANCE.getCMNExtendedProperty_Name(), text);		
			propNameSearch.setCaseInsensitive(!caseSensitive);
			or.getArguments().add(propNameSearch);
		}
		if(searchPropValue)
		{
			query.getRequiredPaths().add(Common_TestprofilePackage.eINSTANCE.getTPFExecutionEvent_Properties());
			BinaryExpression propValueSearch = createBinaryExpression(CommonPackage.eINSTANCE.getCMNExtendedProperty_Value(), text);		
			propValueSearch.setCaseInsensitive(!caseSensitive);
			or.getArguments().add(propValueSearch);
		}
		
		if(or.getArguments().size() > 0)
			query.setWhereExpression(or);
		
		return query;
	}
	
	protected SimpleSearchQuery createQuery(EObject rootElement)
	{
		SimpleSearchQuery query = ExtensionsFactory.eINSTANCE.createSimpleSearchQuery();
		query.setDistinct(true);
		query.getSources().add(EcoreUtil.getURI(rootElement).toString());

		// Minimum paths required to traverse the complete execution hitory.
		query.getRequiredPaths().add(Common_TestprofilePackage.eINSTANCE.getTPFExecutionResult_ExecutionHistory());
		query.getRequiredPaths().add(Common_TestprofilePackage.eINSTANCE.getTPFExecutionHistory_ExecutionEvents());
		query.getRequiredPaths().add(Common_TestprofilePackage.eINSTANCE.getTPFExecutionEvent_Children());
		query.getRequiredPaths().add(Common_TestprofilePackage.eINSTANCE.getTPFInvocationEvent_InvokedExecutionResult());
		
		return query;
	}
	
	protected void setQueryOutputType(SimpleSearchQuery query, EClass type)
	{
		SimpleOperand outputType = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		outputType.setType(type);
		
		if(query.getOutputElements().size() > 0)
			query.getOutputElements().clear();
		
		query.getOutputElements().add(outputType);
	}
	
	protected LogicalExpression createORExpression()
	{
		LogicalExpression or = ExtensionsFactory.eINSTANCE.createLogicalExpression();
		or.setOperator(LogicalOperators.OR_LITERAL);
		return or;
	}
	
	protected LogicalExpression createANDExpression()
	{
		LogicalExpression and = ExtensionsFactory.eINSTANCE.createLogicalExpression();
		and.setOperator(LogicalOperators.AND_LITERAL);
		return and;
	}
	
	protected void combineExecutionEventAndElementExpressions(SimpleSearchQuery query, LogicalExpression executionEventORExpression, LogicalExpression elementORExpression)
	{
		if(executionEventORExpression != null && !executionEventORExpression.getArguments().isEmpty() && 
				elementORExpression!= null && !elementORExpression.getArguments().isEmpty())
		{
			LogicalExpression and = createANDExpression();
			and.getArguments().add(executionEventORExpression);
			and.getArguments().add(elementORExpression);
			query.setWhereExpression(and);
		}
		else if(elementORExpression != null && !elementORExpression.getArguments().isEmpty())
			query.setWhereExpression(elementORExpression);
		else if(executionEventORExpression != null && !executionEventORExpression.getArguments().isEmpty())
			query.setWhereExpression(executionEventORExpression);
		else // both empty, set OR
			query.setWhereExpression(createORExpression());
	}
	
	protected BinaryExpression createBinaryExpression(EAttribute feature, String value) {
		SimpleOperand operand;
		BinaryExpression expression;
		expression = ExtensionsFactory.eINSTANCE.createBinaryExpression();
		operand = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		operand.setFeature(feature);
		expression.setLeftOperand(operand);				
		operand = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		operand.setValueType(EcorePackage.eINSTANCE.getEString());
		operand.setValue(value);
		expression.getRightOperands().add(operand);
		expression.setOperator(RelationalOperators.LIKE_LITERAL);
		return expression;
	}
	
	protected BinaryExpression createBinaryExpression(EAttribute feature, EDataType valueType, Object value) {
		SimpleOperand operand;
		BinaryExpression expression;
		expression = ExtensionsFactory.eINSTANCE.createBinaryExpression();
		operand = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		operand.setFeature(feature);
		expression.setLeftOperand(operand);				
		operand = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		operand.setValueType(valueType);
		operand.setValue(value);
		expression.getRightOperands().add(operand);
		expression.setOperator(RelationalOperators.LIKE_LITERAL);
		return expression;
	}
	
	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}
	public void widgetSelected(SelectionEvent e) {
		if(e.widget == text)
			searchText = text.getSelection();
		else if(e.widget == time)
			searchTime = time.getSelection();
		else if(e.widget == attachment)
			searchAttachment = attachment.getSelection();
		else if(e.widget == propName)
			searchPropName = propName.getSelection();
		else if(e.widget == propValue)
			searchPropValue = propValue.getSelection();
	}
}
