/********************************************************************** 
 * Copyright (c) 2007, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: TPTPVerdictCategoryProvider.java,v 1.4 2009/02/20 17:49:22 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.ui.forms.extensions.provisional;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFVerdict;
import org.eclipse.hyades.models.common.testprofile.TPFVerdictEvent;
import org.eclipse.hyades.test.ui.TestUIConstants;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.forms.base.TestLogVerdictTraversalQuery;
import org.eclipse.hyades.test.ui.forms.util.ITestLogVerdictTraversal;

/**
 * <p>This class is the verdict category provider for TPTP test types 
 * implementing the <code>executionHistoryExtension.verdictProvider</code>
 * extension point.</p>
 * 
 * <p>The purpose of this class is to handle the special rolled up verdict in 
 * TPTP test logs.</p> 
 * 
 * 
 * @author  	Bianca Xue Jiang
 * @author  	Paul Slauenwhite
 * @version 	February 20, 2009
 * @since   	March 15, 2006
 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
 */
public class TPTPVerdictCategoryProvider extends DefaultVerdictCategoryProvider {

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.forms.extensions.provisional.DefaultVerdictCategoryProvider#getVerdictCategories(org.eclipse.hyades.models.common.testprofile.TPFExecutionResult)
	 */
	public VerdictCategory[] getVerdictCategories(TPFExecutionResult executionResult) {
		
		List execEvents = executionResult.getExecutionHistory().getExecutionEvents();
		TPFVerdictEvent verdict = null;
		if(execEvents != null) 
		{
			for(int i = (execEvents.size() - 1); i >=0; i--)
			{
				if(execEvents.get(i) instanceof TPFVerdictEvent)
					// This is the rolled up verdict to be filtered out.
					verdict = (TPFVerdictEvent)execEvents.get(i);
			}
		}
		
		VerdictCategory[] vc = super.getVerdictCategories(executionResult);
		if(vc != null && verdict != null)
		{
			for(int i = 0; i < vc.length; i++)
			{
				if(vc[i].getText().equals(verdict.getVerdict().getLabel()))
				{
					List verdicts = vc[i].getVerdicts();
					verdicts.remove(verdict);
					break;
				}
			}
		}
		return vc;
	}

	public VerdictCategory[] getVerdictCategories(ITestLogVerdictTraversal traversal) {
		VerdictCategory[] categories = new VerdictCategory[4];
		if (traversal instanceof TestLogVerdictTraversalQuery) {
			TestLogVerdictTraversalQuery query = (TestLogVerdictTraversalQuery) traversal;
			
			// Find the last verdict
			TPFVerdictEvent verdict = query.getLast(ITestLogVerdictTraversal.VERDICT_TYPE_ALL);
			List verdictList = query.getQueryResults(verdict.getVerdict().getValue());
			int indexOfIgnoredVerdict = verdictList.size()-1;
			
			// Make a list (containing a single element) with indices of verdicts
			// which should be ignored, and pass that into the VerdictCatgeory
			// constructor (only for the category of the verdict we're ignoring)
			ArrayList ignoreList = new ArrayList();
			ignoreList.add(new Integer(indexOfIgnoredVerdict));
			
			categories[0] = (new VerdictCategory(query.getQueryResults(TPFVerdict.ERROR), 
					TPFVerdict.ERROR_LITERAL.getLabel(), TestUIConstants.COLOR_ERROR, 
					TestUIImages.INSTANCE.getImage(TestUIImages.IMG_EVENT_VERDICT_ERROR),
					verdict.getVerdict().getValue() == TPFVerdict.ERROR ? ignoreList : null));
			categories[1] = (new VerdictCategory(query.getQueryResults(TPFVerdict.FAIL), 
					TPFVerdict.FAIL_LITERAL.getLabel(), TestUIConstants.COLOR_FAIL, 
					TestUIImages.INSTANCE.getImage(TestUIImages.IMG_EVENT_VERDICT_FAIL),
					verdict.getVerdict().getValue() == TPFVerdict.FAIL ? ignoreList : null));
			categories[2] = (new VerdictCategory(query.getQueryResults(TPFVerdict.INCONCLUSIVE), 
					TPFVerdict.INCONCLUSIVE_LITERAL.getLabel(), TestUIConstants.COLOR_INCONCLUSIVE, 
					TestUIImages.INSTANCE.getImage(TestUIImages.IMG_EVENT_VERDICT_INCONCLUSIVE),
					verdict.getVerdict().getValue() == TPFVerdict.INCONCLUSIVE ? ignoreList : null));
			categories[3] = (new VerdictCategory(query.getQueryResults(TPFVerdict.PASS), 
					TPFVerdict.PASS_LITERAL.getLabel(), TestUIConstants.COLOR_PASS, 
					TestUIImages.INSTANCE.getImage(TestUIImages.IMG_EVENT_VERDICT_PASS),
					verdict.getVerdict().getValue() == TPFVerdict.PASS ? ignoreList : null));
			return categories;
		}
		else throw new UnsupportedOperationException();
	}
}
